﻿#!/usr/bin/env python
# -*- coding: utf-8 -*-

#-------------------------------------------------------------------------------
# Name:        TkPyms Modules
# Purpose:     Convert Excel file to XML ISO 19139/Inspire file using Python and XLRD and XLWT
#
# Author:      Guillaume Ryckelynck
#
# Created:     25/07/2011
# Modified:    25/01/2012
# Version:     0.3
# Copyright:   (c) Guillaume Ryckelynck / CIGAL 2011
# Licence:     <your licence>
#-------------------------------------------------------------------------------


"""
Programme python de conversion de fichiers de métadonnées du format MS Excel vers le format XML ISO 19139 / Inspire.
"""

__author__ = "Guillaume Ryckelynck"
__copyright__ = "Copyright 2011, Guillaume Ryckelynck"
__credits__ = ["Guillaume Ryckelynck"]
__license__ = "GPL"
__version__ = "0.3"
__maintainer__ = "Guillaume Ryckelynck"
__email__ = "guillaume@ryckelynck.info"
__status__ = "Developement"

import os
import sys
import glob
import uuid
import datetime
import time
import ConfigParser
import urlparse

try:
    import xlrd
    import xlwt
except ImportError, e:
    print "Import error! Application cannot start:", e
    sys.exit(1)


### Initialisation des variables ###
APP_NAME = "TkPyms Module"
APP_VERSION = "0.3"
PATH_SCRIPT = os.path.dirname(sys.argv [0])

''' Tableaux de correspondance '''
# LanguageCode: liste des langues
LanguageCode = { u'1': u'fre',
                 u'2': u'eng',
                 u'3': u'ger',
                 # u'Erreur': u'fre',
                }

# CharacterSetCode : liste des encodage de caractères
# B5.10
CharacterSetCode = { u'1' : u'ucs2',
                     u'2' : u'ucs4',
                     u'3' : u'utf7',
                     u'4' : u'utf8',
                     u'5' : u'utf16',
                     u'6' : u'8859part1',
                     u'7' : u'8859part2',
                     u'8' : u'8859part3',
                     u'9' : u'8859part4',
                     u'10' : u'8859part5',
                     u'11' : u'8859part6',
                     u'12' : u'8859part7',
                     u'13' : u'8859part8',
                     u'14' : u'8859part9',
                     u'15' : u'8859part10',
                     u'16' : u'8859part11',
                     u'17' : u'reserved',
                     u'18' : u'8859part13',
                     u'19' : u'8859part14',
                     u'20' : u'8859part15',
                     u'21' : u'8859part16',
                     u'22' : u'jis',
                     u'23' : u'shiftJIS',
                     u'24' : u'eucJP',
                     u'25' : u'usAscii',
                     u'26' : u'ebcdic',
                     u'27' : u'eucKR',
                     u'28' : u'big5',
                     u'29' : u'GB2312',
                    }

# RoleCode: liste des rôles
# B5.5
RoleCode = { u'1': u'resourceProvider',
             u'2': u'custodian',
             u'3': u'owner',
             u'4': u'user',
             u'5': u'distributor',
             u'6': u'originator',
             u'7': u'pointOfContact',
             u'8': u'principalInvestigator',
             u'9': u'processor',
             u'10': u'publisher',
             u'11': u'author',
            }

# ClassificationCode: liste des type de mots-clés
# B5.11
ClassificationCode = { u'1': u'unclassified',
                       u'2': u'restricted',
                       u'3': u'confidential',
                       u'4': u'secret',
                       u'5': u'topSecret',
                     }

# KeywordTypeCode: liste des type de mots-clés
# B5.17
KeywordTypeCode = { u'1': u'discipline',
                    u'2': u'place',
                    u'3': u'stratum',
                    u'4': u'temporal',
                    u'5': u'theme',
                  }

# MaintenanceFrequencyCode
# B.5.18
MaintenanceFrequencyCode = { u'1': u'continual',
                             u'2': u'daily',
                             u'3': u'weekly',
                             u'4': u'fortnightly',
                             u'5': u'monthly',
                             u'6': u'quaterly',
                             u'7': u'biannualy',
                             u'8': u'annualy',
                             u'9': u'adNeeded',
                             u'10': u'irregular',
                             u'11': u'notPlanned',
                             u'12': u'unknown',
                           }

# DateTypeCode
# B.5.2
DateTypeCode = { u'1': u'creation',
                 u'2': u'publication',
                 u'3': u'revision',
                 u'4': u'',
               }

# RestrictionCode: liste des contraintes d'accès
# B5.24
RestrictionCode = { u'1': u'copyright',
                    u'2': u'copyright',
                    u'3': u'patent',
                    u'4': u'patentPending',
                    u'5': u'trademark',
                    u'6': u'license',
                    u'7': u'intellectualPropertyRights',
                    u'8': u'restricted',
                    u'9': u'otherRestrictions',
                  }

# ScopeCode: liste des type de données décrites
# B5.25
ScopeCode = {  u'1': u'attribute',
               u'2': u'attributeType',
               u'3': u'collectionHardware',
               u'4': u'collectionSession',
               u'5': u'dataset',
               u'6': u'series',
               u'7': u'nonGeographicDataset',
               u'8': u'dimensionGroup',
               u'9': u'feature',
               u'10': u'featureType',
               u'11': u'propertyType',
               u'12': u'software',
               u'13': u'fieldSession',
               u'14': u'service',
               u'15': u'model',
               u'16': u'tile',
               u'17': u'fieldCampaign',
             }

# SpatialRepresentationTypeCode: liste des type de représentation
# B5.26
SpatialRepresentationTypeCode = { u'1': u'vector',
                                  u'2': u'grid',
                                  u'3': u'textTable',
                                  u'4': u'tin',
                                  u'5': u'stereoModel',
                                  u'6': u'video',
                                }

# TopicCategoryCode: liste des thèmes ISO
# B5.27
TopicCategoryCode = { u'1': u'farming',
                      u'2': u'biota',
                      u'3': u'boundaries',
                      u'4': u'climatologyMeteorologyAtmosphere',
                      u'5': u'economy',
                      u'6': u'elevation',
                      u'7': u'environment',
                      u'8': u'geoscientificInformation',
                      u'9': u'health',
                      u'10': u'imageryBaseMapsEarthCover',
                      u'11': u'intelligenceMilitary',
                      u'12': u'inlandWaters',
                      u'13': u'location',
                      u'14': u'oceans',
                      u'15': u'planningCadastre',
                      u'16': u'society',
                      u'17': u'structure',
                      u'18': u'transportation',
                      u'19': u'utilitiesCommunication',
                    }

# InspireThemeCode: liste des thèmes inspire
#
InspireThemeCode = { u'1': u'Coordinate reference systems',
                     u'2': u'Geographical grid systems',
                     u'3': u'Geographical names',
                     u'4': u'Administrative units',
                     u'5': u'Addresses',
                     u'6': u'Cadastral parcels',
                     u'7': u'Transport networks',
                     u'8': u'Hydrography',
                     u'9': u'Protected sites',
                     u'10': u'Elevation',
                     u'11': u'Land cover',
                     u'12': u'Orthoimagery',
                     u'13': u'Geology',
                     u'14': u'Statistical units',
                     u'15': u'Buildings',
                     u'16': u'Soil',
                     u'17': u'Land use',
                     u'18': u'Human health and safety',
                     u'19': u'Utility and governmental services',
                     u'20': u'Environmental monitoring facilities',
                     u'21': u'Production and industrial facilities',
                     u'22': u'Agricultural and aquaculture facilities',
                     u'23': u'Population distribution - demography',
                     u'24': u'Area management/restriction/regulation zones and reporting units',
                     u'25': u'Natural risk zones',
                     u'26': u'Atmospheric conditions',
                     u'27': u'Meteorological geographical features',
                     u'28': u'Oceanographic geographical features',
                     u'29': u'Sea regions',
                     u'30': u'Bio-geographical regions',
                     u'31': u'Habitats and biotopes',
                     u'32': u'Species distribution',
                     u'33': u'Energy resources',
                     u'34': u'Mineral resources',
                   }

# PassCode
#
PassCode = { u'1': u'true',
             u'2': u'false',
             u'3': u'',
           }

# InspireRestrictionCode : liste des constraintes légales d'accès public selon INSPIRE
InspireRestrictionCode = {  u"0" : u"Pas de restriction d'accès selon INSPIRE",
                            u"1" : u"L124-4-I-1 du code de l'environnement (Directive 2007/2/CE (INSPIRE), Article 13.1.a)",
                            u"2" : u"L124-5-II-1 du code de l'environnement (Directive 2007/2/CE (INSPIRE), Article 13.1.b)",
                            u"3" : u"L124-5-II-2 du code de l'environnement (Directive 2007/2/CE (INSPIRE), Article 13.1.c)",
                            u"4" : u"L124-4-I-1 du code de l'environnement (Directive 2007/2/CE (INSPIRE), Article 13.1.d)",
                            u"5" : u"L124-5-II-3 du code de l'environnement (Directive 2007/2/CE (INSPIRE), Article 13.1.e)",
                            u"6" : u"L124-4-I-1 du code de l'environnement (Directive 2007/2/CE (INSPIRE), Article 13.1.f)",
                            u"7" : u"L124-4-I-3 du code de l'environnement (Directive 2007/2/CE (INSPIRE), Article 13.1.g)",
                            u"8" : u"L124-4-I-2 du code de l'environnement (Directive 2007/2/CE (INSPIRE), Article 13.1.h)"
}

# InspireSpecificationCode : liste des spécifications INSPIRE
InspireSpecificationCode = { 
    u"1" : u"COMMISSION REGULATION (EC) No 1205/2008 of 3 December 2008 implementing Directive 2007/2/EC of the European Parliament and of the Council as regards metadata (publication:2008-12-04)",
    u"2" : u"Corrigendum to INSPIRE Metadata Regulation published in the Official Journal of the European Union, L 328, page 83 (publication:2009-12-15))",
    u"3" : u"COMMISSION REGULATION (EU) No 1089/2010 of 23 November 2010 implementing Directive 2007/2/EC of the European Parliament and of the Council as regards interoperability of spatial data sets and services (publication:2010-12-08)",
    u"4" : u"COMMISSION REGULATION (EU) No 1088/2010 of 23 November 2010 amending Regulation (EC) No 976/2009 as regards download services and transformation services (publication:2010-12-08)",
    u"5" : u"COMMISSION REGULATION (EC) No 976/2009 of 19 October 2009 implementing Directive 2007/2/EC of the European Parliament and of the Council as regards the Network Services (publication:2009-10-20)",
    u"6" : u"COMMISSION REGULATION (EU) No 268/2010 of 29 March 2010 implementing Directive 2007/2/EC of the European Parliament and of the Council as regards the access to spatial data sets and services of the Member States by Community institutions and bodies under harmonised conditions (publication:2010-03-30)",
    u"7" : u"Commission Decision of 5 June 2009 implementing Directive 2007/2/EC of the European Parliament and of the Council as regards monitoring and reporting (notified under document number C(2009) 4199) (2009/442/EC) (publication:2009-06-11)"
}

# ------------------------------------------------------------------------------
# CONVERT_XLS2XML
#
# Fonction de conversion d'un fichier MS Excel vers XML
# ------------------------------------------------------------------------------
def convert_xls2xml(xls_file, xml_dir):

        # Geston des erreurs
        Error = u''
        Error_CreateFile = u''

        wb = xlrd.open_workbook(xls_file, encoding_override="utf-8")
        # Liste des cellules nommées dans le fichier MS Excel
        lst_name = wb.name_map
        # print len(lst_name), u'cellules nommées dans le fichier Excel.'

        str19139 = u'<?xml version="1.0" encoding="UTF-8"?>\n'
        str19139 += u'<gmd:MD_Metadata xsi:schemaLocation="http://www.isotc211.org/2005/gmd http://schemas.opengis.net/iso/19139/20060504/gmd/gmd.xsd" xmlns:gmd="http://www.isotc211.org/2005/gmd" xmlns:gco="http://www.isotc211.org/2005/gco" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:gml="http://www.opengis.net/gml" xmlns:xlink="http://www.w3.org/1999/xlink">\n'

        # MD_FileIdentifier
        e, MD_FileIdentifier = _get_xls_value('md_fileidentifier', lst_name, 'string')
        if not MD_FileIdentifier:
            Error += u"MD_FileIdentifier: aucun UUID renseignée. Utilisation du nom de fichier XLS comme UUID... "
            MD_FileIdentifier, ext =  os.path.splitext(os.path.basename(xls_file))
            Error += u"MD_FileIdentifier = " + MD_FileIdentifier + "\n"
        else:
            str19139 += u'<gmd:fileIdentifier><gco:CharacterString>' + MD_FileIdentifier + u'</gco:CharacterString></gmd:fileIdentifier>\n'

        # MD_Language
        MD_Language = u'1' # Valeur par défaut : "Français"
        e, MD_Language = _get_xls_value('md_language', lst_name, 'string')
        if MD_Language: MD_Language = MD_Language.split()[0]
        
        if MD_Language in LanguageCode:
            str19139 += u'<gmd:language><gmd:LanguageCode codeList="http://www.loc.gov/standards/iso639-2/" codeListValue="' + LanguageCode[MD_Language] + u'">' + LanguageCode[MD_Language] + u'</gmd:LanguageCode></gmd:language>\n'
        else:
            Error += u'MD_Language: erreure de lecture. Une langue pour les métadonnées doit être renseignée.\n'

        # MD_CharacterSet
        e, MD_CharacterSet = _get_xls_value('md_characterset', lst_name, 'string')
        if MD_CharacterSet: MD_CharacterSet = MD_CharacterSet.split()[0]
        if MD_CharacterSet not in CharacterSetCode:
            MD_CharacterSet = '4' # Valeur par defaut : "utf8"

        str19139 += u'<gmd:characterSet><gmd:MD_CharacterSetCode codeList="http://www.isotc211.org/2005/resources/codeList.xml#MD_CharacterSetCode" codeListValue="' + CharacterSetCode[MD_CharacterSet] + u'">' + CharacterSetCode[MD_CharacterSet] + u'</gmd:MD_CharacterSetCode></gmd:characterSet>\n'

        # MD_HierarchyLevel
        MD_HierarchyLevel = u'6' # Valeur par defaut : "dataset"
        e, MD_HierarchyLevel = _get_xls_value('md_hierarchylevel', lst_name, 'string')
        if MD_HierarchyLevel: MD_HierarchyLevel = MD_HierarchyLevel.split()[0]
        
        if not MD_HierarchyLevel:
            Error += u'MD_HierarchyLevel: un niveau de description pour les métadonnées doit être renseignée.\n'
        else:
            str19139 += u'<gmd:hierarchyLevel><gmd:MD_ScopeCode codeList="http://standards.iso.org/ittf/PubliclyAvailableStandards/ISO_19139_Schemas/resources/Codelist/ML_gmxCodelists.xml#MD_ScopeCode" codeListValue="' + ScopeCode[MD_HierarchyLevel] + u'">' +  ScopeCode[MD_HierarchyLevel] + u'</gmd:MD_ScopeCode></gmd:hierarchyLevel>\n'

        # MD_Contacts
        Error_MdContacts = 1
        #MD_CntRole = u'11' # Valeur par defaut: "pointofContact"
        for i in range(1, 20):
            e, MD_CntName = _get_xls_value('md_cnt'+str(i)+'_name', lst_name, 'string')
            e, MD_CntFunction = _get_xls_value('md_cnt'+str(i)+'_fct', lst_name, 'string')
            e, MD_CntOrganism = _get_xls_value('md_cnt'+str(i)+'_org', lst_name, 'string')
            e, MD_CntAddress = _get_xls_value('md_cnt'+str(i)+'_address', lst_name, 'string')
            e, MD_CntPostalCode = _get_xls_value('md_cnt'+str(i)+'_cp', lst_name, 'integer')
            e, MD_CntCity = _get_xls_value('md_cnt'+str(i)+'_city', lst_name, 'string')
            e, MD_CntPhone = _get_xls_value('md_cnt'+str(i)+'_tel', lst_name, 'string')
            e, MD_CntEmail = _get_xls_value('md_cnt'+str(i)+'_email', lst_name, 'string')
            e, MD_CntRole = _get_xls_value('md_cnt'+str(i)+'_role', lst_name, 'string')
            if MD_CntRole: MD_CntRole = MD_CntRole.split()[0]

            if MD_CntOrganism and MD_CntEmail and MD_CntRole in RoleCode:
                Error_MdContacts = 0
                str19139 += u'<gmd:contact>\n'
                str19139 += u'<gmd:CI_ResponsibleParty>\n'
                if MD_CntName:
                    str19139 += u'<gmd:individualName><gco:CharacterString>' + MD_CntName + '</gco:CharacterString></gmd:individualName>\n'
                if MD_CntOrganism:
                    str19139 += u'<gmd:organisationName><gco:CharacterString>' + MD_CntOrganism + '</gco:CharacterString></gmd:organisationName>\n'
                if MD_CntFunction:
                    str19139 += u'<gmd:positionName><gco:CharacterString>' + MD_CntFunction + '</gco:CharacterString></gmd:positionName>\n'
                str19139 += u'<gmd:contactInfo><gmd:CI_Contact>\n'
                if MD_CntPhone:
                    str19139 += u'<gmd:phone><gmd:CI_Telephone><gmd:voice><gco:CharacterString>' + MD_CntPhone + '</gco:CharacterString></gmd:voice></gmd:CI_Telephone></gmd:phone>\n'
                str19139 += u'<gmd:address><gmd:CI_Address>\n'
                if MD_CntAddress:
                    str19139 += u'<gmd:deliveryPoint><gco:CharacterString>' + MD_CntAddress + '</gco:CharacterString></gmd:deliveryPoint>\n'
                if MD_CntCity:
                    str19139 += u'<gmd:city><gco:CharacterString>' + MD_CntCity + '</gco:CharacterString></gmd:city>\n'
                if MD_CntPostalCode:
                    str19139 += u'<gmd:postalCode><gco:CharacterString>' + MD_CntPostalCode + '</gco:CharacterString></gmd:postalCode>\n'
                str19139 += u'<gmd:electronicMailAddress><gco:CharacterString>' + MD_CntEmail + '</gco:CharacterString></gmd:electronicMailAddress>\n'
                str19139 += u'</gmd:CI_Address></gmd:address>\n'
                str19139 += u'</gmd:CI_Contact></gmd:contactInfo>\n'
                str19139 += u'<gmd:role><gmd:CI_RoleCode codeList="http://standards.iso.org/ittf/PubliclyAvailableStandards/ISO_19139_Schemas/resources/Codelist/ML_gmxCodelists.xml#CI_RoleCode" codeListValue="' + RoleCode[MD_CntRole] + '">' + RoleCode[MD_CntRole] + '</gmd:CI_RoleCode></gmd:role>\n'
                str19139 += u'</gmd:CI_ResponsibleParty>\n'
                str19139 += u'</gmd:contact>\n'

        if Error_MdContacts :
            Error += u'MD_Contact: au moins 1 contact pour les métadonnées doit être renseigné (un organisme et une adresse email minimum).\n'

        # MD_DateStamp
        e, MD_DateStamp = _get_xls_value('md_datestamp', lst_name, 'date')
        if MD_DateStamp:
            str19139 += u'<gmd:dateStamp><gco:Date>' + MD_DateStamp + u'</gco:Date></gmd:dateStamp>\n'
        else:
            str19139 += u'<gmd:dateStamp><gco:Date>' + time.strftime('%Y-%m-%d',time.localtime()) + u'</gco:Date></gmd:dateStamp>\n'
            Error += u'MD_DateStamp: erreur de lecture de la date. Une date pour les métadonnées doit être renseignée. La date du jour a été utilisée à la place.\n'
            
        # MD_StandardName
        e, MD_StandardName = _get_xls_value('md_standardname', lst_name, 'string')
        if MD_StandardName :
            str19139 += u'<gmd:metadataStandardName><gco:CharacterString>' + MD_StandardName + '</gco:CharacterString></gmd:metadataStandardName>\n'

        # MD_StandardVersion
        e, MD_StandardVersion = _get_xls_value('md_standardversion', lst_name, 'string')
        if MD_StandardVersion :
            str19139 += u'<gmd:metadataStandardVersion><gco:CharacterString>' + MD_StandardVersion + '</gco:CharacterString></gmd:metadataStandardVersion>\n'

        # Data_ReferenceSystem
        Error_DataReferenceSystem = 1
        for i in range(1, 20):
            e, Data_ReferenceSystem = _get_xls_value('data_referencesystem'+str(i), lst_name, 'string')
            if Data_ReferenceSystem :
                Error_DataReferenceSystem = 0
                str19139 += u'<gmd:referenceSystemInfo><gmd:MD_ReferenceSystem>\n'
                str19139 += u'<gmd:referenceSystemIdentifier><gmd:RS_Identifier>\n'
                str19139 += u'<gmd:code><gco:CharacterString>' + Data_ReferenceSystem + '</gco:CharacterString></gmd:code>\n'
                str19139 += u'</gmd:RS_Identifier></gmd:referenceSystemIdentifier>\n'
                str19139 += u'</gmd:MD_ReferenceSystem></gmd:referenceSystemInfo>\n'

        if Error_DataReferenceSystem :
            Error += u'Data_ReferenceSystem: un systeme de projection pour les données doit être renseignée.\n'

        # IDENTIFICATION INFO
        str19139 += u'<gmd:identificationInfo><gmd:MD_DataIdentification>\n'
        str19139 += u'<gmd:citation><gmd:CI_Citation>\n'
        
        # Data_Title
        e, Data_Title = _get_xls_value('data_title', lst_name, 'string')
        if Data_Title:
            str19139 = str19139 + u'<gmd:title><gco:CharacterString>' + Data_Title + u'</gco:CharacterString></gmd:title>\n'
        else:
            Error += u'Data_Title: un titre pour les données doit être renseignée.\n'

        # Data_DateCreation / Data_DateRevision / Data_DatePublication
        Error_date = 1
        DateType = (u'creation', u'revision', u'publication')
        for type in DateType:
            e, Data_Date = _get_xls_value('data_date'+type, lst_name, 'date')
            
            if Data_Date:
                Error_date = 0
                str19139 += u'<gmd:date><gmd:CI_Date>\n'
                str19139 += u'<gmd:date><gco:Date>' + Data_Date + '</gco:Date></gmd:date>\n'
                str19139 += u'<gmd:dateType><gmd:CI_DateTypeCode codeList="http://standards.iso.org/ittf/PubliclyAvailableStandards/ISO_19139_Schemas/resources/Codelist/ML_gmxCodelists.xml#CI_DateTypeCode" codeListValue="' + type + '">' + type + '</gmd:CI_DateTypeCode></gmd:dateType>\n'
                str19139 += u'</gmd:CI_Date></gmd:date>\n'

        if Error_date: Error += u'Data_Date: une date pour les données doit être renseignée.\n'

        # Data_Identifier
        # Renseigner MD_FileIdentifier comme Data_Identifier par défaut
        if MD_FileIdentifier != '':
            Data_Code0 = MD_FileIdentifier
            Data_CodeSpace0 = 'MD_URI'
            str19139 += u'<gmd:identifier><gmd:RS_Identifier>\n'
            str19139 += u'<gmd:code><gco:CharacterString>' + Data_Code0 + '</gco:CharacterString></gmd:code>\n'
            str19139 += u'<gmd:codeSpace><gco:CharacterString>' + Data_CodeSpace0 + '</gco:CharacterString></gmd:codeSpace>\n'
            str19139 += u'</gmd:RS_Identifier></gmd:identifier>\n'
        
        for i in range(1, 20):
            e, Data_Code = _get_xls_value('data_code'+str(i), lst_name, 'string')
            e, Data_CodeSpace = _get_xls_value('data_codespace'+str(i), lst_name, 'string')
            
            if Data_Code :
                str19139 += u'<gmd:identifier><gmd:RS_Identifier>\n'
                str19139 += u'<gmd:code><gco:CharacterString>' + Data_Code + '</gco:CharacterString></gmd:code>\n'
                str19139 += u'<gmd:codeSpace><gco:CharacterString>' + Data_CodeSpace + '</gco:CharacterString></gmd:codeSpace>\n'
                str19139 += u'</gmd:RS_Identifier></gmd:identifier>\n'

        str19139 += u'</gmd:CI_Citation></gmd:citation>\n'

        # Data_Abstract
        e, Data_Abstract = _get_xls_value('data_abstract', lst_name, 'string')
        if Data_Abstract:
            str19139 += u'<gmd:abstract><gco:CharacterString>' + Data_Abstract + '</gco:CharacterString></gmd:abstract>\n'
        else:
            Error += u'Data_Abstract: un résumé pour les données doit être renseignée.\n'

        # Data_PointOfContact
        Error_DataPointOfContacts = 1
        #Data_CntRole = u'11' # Valeur par defaut: "pointofContact"
        for i in range(1, 20):
            e, Data_CntName = _get_xls_value('data_cnt'+str(i)+'_name', lst_name, 'string')
            e, Data_CntFunction = _get_xls_value('data_cnt'+str(i)+'_fct', lst_name, 'string')
            e, Data_CntOrganism = _get_xls_value('data_cnt'+str(i)+'_org', lst_name, 'string')
            e, Data_CntAddress = _get_xls_value('data_cnt'+str(i)+'_address', lst_name, 'string')
            e, Data_CntPostalCode = _get_xls_value('data_cnt'+str(i)+'_cp', lst_name, 'integer')
            e, Data_CntCity = _get_xls_value('data_cnt'+str(i)+'_city', lst_name, 'string')
            e, Data_CntPhone = _get_xls_value('data_cnt'+str(i)+'_tel', lst_name, 'string')
            e, Data_CntEmail = _get_xls_value('data_cnt'+str(i)+'_email', lst_name, 'string')
            e, Data_CntRole = _get_xls_value('data_cnt'+str(i)+'_role', lst_name, 'string')
            if Data_CntRole: Data_CntRole = Data_CntRole.split()[0]

            if Data_CntOrganism and Data_CntEmail and Data_CntRole in RoleCode:
                Error_DataPointOfContacts = 0
                str19139 += u'<gmd:pointOfContact>\n'
                str19139 += u'<gmd:CI_ResponsibleParty>\n'
                if Data_CntName:
                    str19139 += u'<gmd:individualName><gco:CharacterString>' + Data_CntName + '</gco:CharacterString></gmd:individualName>\n'
                if Data_CntOrganism:
                    str19139 += u'<gmd:organisationName><gco:CharacterString>' + Data_CntOrganism + '</gco:CharacterString></gmd:organisationName>\n'
                if Data_CntFunction:
                    str19139 += u'<gmd:positionName><gco:CharacterString>' + Data_CntFunction + '</gco:CharacterString></gmd:positionName>\n'
                str19139 += u'<gmd:contactInfo><gmd:CI_Contact>\n'
                if Data_CntPhone:
                    str19139 += u'<gmd:phone><gmd:CI_Telephone><gmd:voice><gco:CharacterString>' + Data_CntPhone + '</gco:CharacterString></gmd:voice></gmd:CI_Telephone></gmd:phone>\n'
                str19139 += u'<gmd:address><gmd:CI_Address>\n'
                if Data_CntAddress:
                    str19139 += u'<gmd:deliveryPoint><gco:CharacterString>' + Data_CntAddress + '</gco:CharacterString></gmd:deliveryPoint>\n'
                if Data_CntCity:
                    str19139 += u'<gmd:city><gco:CharacterString>' + Data_CntCity + '</gco:CharacterString></gmd:city>\n'
                if Data_CntPostalCode:
                    str19139 += u'<gmd:postalCode><gco:CharacterString>' + Data_CntPostalCode + '</gco:CharacterString></gmd:postalCode>\n'
                str19139 += u'<gmd:electronicMailAddress><gco:CharacterString>' + Data_CntEmail + '</gco:CharacterString></gmd:electronicMailAddress>\n'
                str19139 += u'</gmd:CI_Address></gmd:address>\n'
                str19139 += u'</gmd:CI_Contact></gmd:contactInfo>\n'
                str19139 += u'<gmd:role><gmd:CI_RoleCode codeList="http://standards.iso.org/ittf/PubliclyAvailableStandards/ISO_19139_Schemas/resources/Codelist/ML_gmxCodelists.xml#CI_RoleCode" codeListValue="' + RoleCode[Data_CntRole] + '">' + RoleCode[Data_CntRole] + '</gmd:CI_RoleCode></gmd:role>\n'
                str19139 += u'</gmd:CI_ResponsibleParty>\n'
                str19139 += u'</gmd:pointOfContact>\n'

        if Error_DataPointOfContacts :
            Error += u'Data_PointOfContacts: au moins 1 contact pour les données doit être renseigné (un organisme et une adresse email minimum).\n'

        # Data_MaintenanceFrequency
        Data_MaintenanceFrequency = u'12' # Valeur par defaut: "unknown"
        e, Data_MaintenanceFrequency = _get_xls_value('data_maintenancefrequency', lst_name, 'string')
        if Data_MaintenanceFrequency: Data_MaintenanceFrequency = Data_MaintenanceFrequency.split()[0]
        
        if Data_MaintenanceFrequency and Data_MaintenanceFrequency in MaintenanceFrequencyCode:
            str19139 += u'<gmd:resourceMaintenance>\n'
            str19139 += u'<gmd:MD_MaintenanceInformation>\n'
            str19139 += u'<gmd:maintenanceAndUpdateFrequency>\n'
            str19139 += u'<gmd:MD_MaintenanceFrequencyCode codeList="http://standards.iso.org/ittf/PubliclyAvailableStandards/ISO_19139_Schemas/resources/Codelist/ML_gmxCodelists.xml#MD_MaintenanceFrequencyCode" codeListValue="' + MaintenanceFrequencyCode[Data_MaintenanceFrequency] + '">' + MaintenanceFrequencyCode[Data_MaintenanceFrequency] + '</gmd:MD_MaintenanceFrequencyCode>\n'
            str19139 += u'</gmd:maintenanceAndUpdateFrequency>\n'
            str19139 += u'</gmd:MD_MaintenanceInformation>\n'
            str19139 += u'</gmd:resourceMaintenance>\n'

        # Data_BrowseGraphic
        for i in range(1, 20):
            e, Data_BrowseGraphic_Name = _get_xls_value('data_browsegraphic'+str(i)+'_filename', lst_name, 'string')
            e, Data_BrowseGraphic_Description = _get_xls_value('data_browsegraphic'+str(i)+'_description', lst_name, 'string')
            Data_BrowseGraphic_Type = os.path.splitext(urlparse.urlparse(Data_BrowseGraphic_Name).path)[1]

            if Data_BrowseGraphic_Name:
                str19139 += u'<gmd:graphicOverview>\n'
                str19139 += u'<gmd:MD_BrowseGraphic>\n'
                str19139 += u'<gmd:fileName>\n'
                str19139 += u'<gco:CharacterString>' + Data_BrowseGraphic_Name + '</gco:CharacterString>\n'
                str19139 += u'</gmd:fileName>\n'
                str19139 += u'<gmd:fileDescription>\n'
                str19139 += u'<gco:CharacterString>' + Data_BrowseGraphic_Description + '</gco:CharacterString>\n'
                str19139 += u'</gmd:fileDescription>\n'
                str19139 += u'<gmd:fileType>\n'
                str19139 += u'<gco:CharacterString>'+ Data_BrowseGraphic_Type +'</gco:CharacterString>\n'
                str19139 += u'</gmd:fileType>\n'
                str19139 += u'</gmd:MD_BrowseGraphic>\n'
                str19139 += u'</gmd:graphicOverview>\n'

        
        # Data_Keywords
        Error_DataKeywords = 1
        
        for i in range(1, 20):
            e, Data_Keyword = _get_xls_value('data_keyword'+str(i), lst_name, 'string')
            e, Data_KeywordType = _get_xls_value('data_keyword'+str(i)+'_type', lst_name, 'string')
            if Data_KeywordType: Data_KeywordType = Data_KeywordType.split()[0]
            e, Data_ThesaurusName = _get_xls_value('data_keyword'+str(i)+'_thesaurusname', lst_name, 'string')
            e, Data_ThesaurusDateCreation = _get_xls_value('data_keyword'+str(i)+'_thesaurusdatecreation', lst_name, 'date')
            e, Data_ThesaurusDatePublication = _get_xls_value('data_keyword'+str(i)+'_thesaurusdatepublication', lst_name, 'date')
            e, Data_ThesaurusDateRevision = _get_xls_value('data_keyword'+str(i)+'_thesaurusdaterevision', lst_name, 'date')

            if Data_Keyword :
                Error_DataKeywords = 0
                str19139 += u'<gmd:descriptiveKeywords><gmd:MD_Keywords>\n'
                str19139 += u'<gmd:keyword><gco:CharacterString>' + Data_Keyword + '</gco:CharacterString></gmd:keyword>\n'
                if Data_KeywordType :
                    str19139 += u'<gmd:type><gmd:MD_KeywordTypeCode codeListValue="' + KeywordTypeCode[Data_KeywordType] + '" codeList="http://www.isotc211.org/2005/resources/codeList.xml#MD_KeywordTypeCode" /></gmd:type>\n'
                if Data_ThesaurusName :
                    str19139 += u'<gmd:thesaurusName><gmd:CI_Citation>\n'
                    str19139 += u'<gmd:title><gco:CharacterString>' + Data_ThesaurusName + '</gco:CharacterString></gmd:title>\n'
                    if Data_ThesaurusDateCreation:
                        str19139 += u'<gmd:date><gmd:CI_Date>\n'
                        str19139 += u'<gmd:date><gco:Date>' + Data_ThesaurusDateCreation + '</gco:Date></gmd:date>\n'
                        str19139 += u'<gmd:dateType><gmd:CI_DateTypeCode codeList="http://standards.iso.org/ittf/PubliclyAvailableStandards/ISO_19139_Schemas/resources/Codelist/ML_gmxCodelists.xml#CI_DateTypeCode" codeListValue="creation">creation</gmd:CI_DateTypeCode></gmd:dateType>\n'
                        str19139 += u'</gmd:CI_Date></gmd:date>\n'
                    if Data_ThesaurusDateRevision:
                        str19139 += u'<gmd:date><gmd:CI_Date>\n'
                        str19139 += u'<gmd:date><gco:Date>' + Data_ThesaurusDateRevision + '</gco:Date></gmd:date>\n'
                        str19139 += u'<gmd:dateType><gmd:CI_DateTypeCode codeList="http://standards.iso.org/ittf/PubliclyAvailableStandards/ISO_19139_Schemas/resources/Codelist/ML_gmxCodelists.xml#CI_DateTypeCode" codeListValue="revision">revision</gmd:CI_DateTypeCode></gmd:dateType>\n'
                        str19139 += u'</gmd:CI_Date></gmd:date>\n'
                    if Data_ThesaurusDatePublication:
                        str19139 += u'<gmd:date><gmd:CI_Date>\n'
                        str19139 += u'<gmd:date><gco:Date>' + Data_ThesaurusDatePublication + '</gco:Date></gmd:date>\n'
                        str19139 += u'<gmd:dateType><gmd:CI_DateTypeCode codeList="http://standards.iso.org/ittf/PubliclyAvailableStandards/ISO_19139_Schemas/resources/Codelist/ML_gmxCodelists.xml#CI_DateTypeCode" codeListValue="publication">publication</gmd:CI_DateTypeCode></gmd:dateType>\n'
                        str19139 += u'</gmd:CI_Date></gmd:date>\n'
                    str19139 += u'</gmd:CI_Citation></gmd:thesaurusName>\n'
                str19139 += u'</gmd:MD_Keywords></gmd:descriptiveKeywords>\n'

        # INSPIRE Keywords
        Error_DataKeyword = 1
        Data_InspireThesaurusName = u"GEMET - INSPIRE themes, version 1.0"
        Data_InspireThesaurusDatePublication = u"2008-06-01"

        for i in range(1, 20):
            e, Data_InspireKeyword = _get_xls_value('data_inspirekeyword'+str(i), lst_name, 'string')
            if Data_InspireKeyword: Data_InspireKeyword = Data_InspireKeyword.split()[0]
            
            if Data_InspireKeyword:
                Error_DataKeyword = 0
                str19139 += u'<gmd:descriptiveKeywords><gmd:MD_Keywords>\n'
                str19139 += u'<gmd:keyword><gco:CharacterString>' + InspireThemeCode[Data_InspireKeyword] + '</gco:CharacterString></gmd:keyword>\n'
                str19139 += u'<gmd:thesaurusName><gmd:CI_Citation>\n'
                str19139 += u'<gmd:title><gco:CharacterString>' + Data_InspireThesaurusName + '</gco:CharacterString></gmd:title>\n'
                str19139 += u'<gmd:date><gmd:CI_Date>\n'
                str19139 += u'<gmd:date><gco:Date>' + Data_InspireThesaurusDatePublication + '</gco:Date></gmd:date>\n'
                str19139 += u'<gmd:dateType><gmd:CI_DateTypeCode codeList="http://standards.iso.org/ittf/PubliclyAvailableStandards/ISO_19139_Schemas/resources/Codelist/ML_gmxCodelists.xml#CI_DateTypeCode" codeListValue="publication">publication</gmd:CI_DateTypeCode></gmd:dateType>\n'
                str19139 += u'</gmd:CI_Date></gmd:date>\n'
                str19139 += u'</gmd:CI_Citation></gmd:thesaurusName>\n'
                str19139 += u'</gmd:MD_Keywords></gmd:descriptiveKeywords>\n'
        if Error_DataKeyword: Error += u'Data_InspireKeyword: au moins 1 mot-clé INSPIRE pour les données doit être renseigné .\n'

        # Data_ResourceConstraints
        # Data_UseLimitation
        Error_DataUseLimitation = 1
        
        for i in range(1, 20):
            e, Data_UseLimitation = _get_xls_value('data_uselimitation'+str(i), lst_name, 'string')            
            if Data_UseLimitation:
                Error_DataUseLimitation = 0
                str19139 += u'<gmd:resourceConstraints><gmd:MD_Constraints>\n'
                str19139 += u'<gmd:useLimitation><gco:CharacterString>' + Data_UseLimitation + '</gco:CharacterString></gmd:useLimitation>\n'
                str19139 += u'</gmd:MD_Constraints></gmd:resourceConstraints>\n'
        if Error_DataUseLimitation: Error += u'Data_UseLimitation: une limite d\'utilisation pour les données doit être renseignée (indiquer: "Aucune limite." le cas échéant).\n'

        str19139 += u'<gmd:resourceConstraints><gmd:MD_LegalConstraints>\n'

        # Data_AccessConstraints
        # Principe d'implémentation retenu: n'est pas obligatoire car par défaut la valeur Data_AccessConstraints = "otherRestrictions" est ajouté et le champ Data_OtherConstraints est lui considéré comme obligatoire
        # Error_DataAccessConstraints = 1
        for i in range(1, 20):
            e, Data_AccessConstraints = _get_xls_value('data_accessconstraints'+str(i), lst_name, 'string')
            if Data_AccessConstraints: Data_AccessConstraints = Data_AccessConstraints.split()[0]
            
            if Data_AccessConstraints and Data_AccessConstraints in RestrictionCode:
                str19139 += u'<gmd:accessConstraints>\n'
                str19139 += u'<gmd:MD_RestrictionCode codeList="http://standards.iso.org/ittf/PubliclyAvailableStandards/ISO_19139_Schemas/resources/Codelist/ML_gmxCodelists.xml#MD_RestrictionCode" codeListValue="' + RestrictionCode[Data_AccessConstraints] + '">' + RestrictionCode[Data_AccessConstraints] + '</gmd:MD_RestrictionCode>\n'
                str19139 += u'</gmd:accessConstraints>\n'
        # if Error_DataAccessConstraints: Error += u'Data_AccessConstraints: une contrainte d\'accès doit être précisée pour les données.\n'
        
        # Ajout de Data_AccessConstraints = "otherRestrictions" par défaut
        str19139 += u'<gmd:accessConstraints>\n'
        str19139 += u'<gmd:MD_RestrictionCode codeList="http://standards.iso.org/ittf/PubliclyAvailableStandards/ISO_19139_Schemas/resources/Codelist/ML_gmxCodelists.xml#MD_RestrictionCode" codeListValue="otherRestrictions">otherRestrictions</gmd:MD_RestrictionCode>\n'
        str19139 += u'</gmd:accessConstraints>\n'      
        
        """
        # Data_UseConstraints
        # Error_DataUseConstraints = 1
        for i in range(1, 20):
            e, Data_UseConstraints = _get_xls_value('data_useconstraints'+str(i), lst_name, 'string')
            if Data_UseConstraints and Data_UseConstraints in RestrictionCode:
                # Error_DataUseConstraints = 0
                str19139 += u'<gmd:useConstraints>\n'
                str19139 += u'<gmd:MD_RestrictionCode codeList="http://standards.iso.org/ittf/PubliclyAvailableStandards/ISO_19139_Schemas/resources/Codelist/ML_gmxCodelists.xml#MD_RestrictionCode" codeListValue="' + RestrictionCode[Data_UseConstraints] + '">' + RestrictionCode[Data_UseConstraints] + '</gmd:MD_RestrictionCode>\n'
                str19139 += u'</gmd:useConstraints>\n'
        # if Error_DataUseConstraints: Error += u'Data_UseConstraints: une contrainte d\'utilisation doit être précisée pour les données.\n'
        """
        
        # Data_OtherConstraints
        Error_DataOtherConstraints = 1
        for i in range(1, 20):
            e, Data_OtherConstraints = _get_xls_value('data_otherconstraints'+str(i), lst_name, 'string')
            if Data_OtherConstraints: Data_OtherConstraints = Data_OtherConstraints.split()[0]
            
            if Data_OtherConstraints and Data_OtherConstraints in InspireRestrictionCode:
                Error_DataOtherConstraints = 0
                str19139 += u'<gmd:otherConstraints><gco:CharacterString>' + InspireRestrictionCode[Data_OtherConstraints] + '</gco:CharacterString></gmd:otherConstraints>\n'
        if Error_DataOtherConstraints: Error += u'Data_OtherConstraints: une contrainte d\'accès doit être précisée pour les données (autres contraintes).\n'

        str19139 += u'</gmd:MD_LegalConstraints></gmd:resourceConstraints>\n'

        # Data_Classification
        Data_Classification = u'1' # Valeur par défaut = "Non classifié (unclassified)"
        e, Data_Classification = _get_xls_value('data_classification', lst_name, 'string')
        if Data_Classification: Data_Classification = Data_Classification.split()[0]
        
        if Data_Classification and Data_Classification in ClassificationCode:
            str19139 += u'<gmd:resourceConstraints><gmd:MD_SecurityConstraints><gmd:classification>\n'
            str19139 += u'<gmd:MD_ClassificationCode codeList="http://standards.iso.org/ittf/PubliclyAvailableStandards/ISO_19139_Schemas/resources/Codelist/ML_gmxCodelists.xml#MD_ClassificationCode" codeListValue="' + ClassificationCode[Data_Classification] + '">' + ClassificationCode[Data_Classification] + '</gmd:MD_ClassificationCode>\n'
            str19139 += u'</gmd:classification></gmd:MD_SecurityConstraints></gmd:resourceConstraints>\n'
        else:
            Error += u'Data_Classification: le champ "Restriction d\'utilisation" doit être renseigné pour les données.\n'

        # Fin de Data_ResourceConstraints


        # Data_SpatialRepresentationType
        Data_SpatialRepresentationType = u'1' # Valeur par defaut: "vector"
        e, Data_SpatialRepresentationType = _get_xls_value('data_spatialrepresentationtype', lst_name, 'string')
        if Data_SpatialRepresentationType: Data_SpatialRepresentationType = Data_SpatialRepresentationType.split()[0]
        
        if Data_SpatialRepresentationType and Data_SpatialRepresentationType in SpatialRepresentationTypeCode:
            str19139 = str19139 + '<gmd:spatialRepresentationType>\n'
            str19139 = str19139 + '<gmd:MD_SpatialRepresentationTypeCode codeList="http://www.isotc211.org/2005/resources/codeList.xml#MD_SpatialRepresentationTypeCode" codeListValue="' + SpatialRepresentationTypeCode[Data_SpatialRepresentationType] + '">' + SpatialRepresentationTypeCode[Data_SpatialRepresentationType] + '</gmd:MD_SpatialRepresentationTypeCode>\n'
            str19139 = str19139 + '</gmd:spatialRepresentationType>\n'
        else:
            Error += u'Data_SpatialRepresentationType: erreure de lecture du type de représentation spatiale. Un type de représentation spatiale doit être renseignée pour les données.\n'

        # Data_ScaleDenominator / Data_ScaleDistance
        e, Data_ScaleDenominator = _get_xls_value('data_scaledenominator', lst_name, 'integer')
        e, Data_ScaleDistance = _get_xls_value('data_scaledistance', lst_name, 'integer')
        if not Data_ScaleDenominator and not Data_ScaleDistance:
            Error += u'Data_ScaleDenominator / Data_ScaleDistance: une échelle ou une résolution doit être renseignée pour les données.\n'
        else:
            str19139 = str19139 + '<gmd:spatialResolution><gmd:MD_Resolution>\n'
            if Data_ScaleDenominator:
                str19139 = str19139 + '<gmd:equivalentScale><gmd:MD_RepresentativeFraction>\n'
                str19139 = str19139 + '<gmd:denominator><gco:Integer>' + Data_ScaleDenominator + '</gco:Integer></gmd:denominator>\n'
                str19139 = str19139 + '</gmd:MD_RepresentativeFraction></gmd:equivalentScale>\n'
            if Data_ScaleDistance:
                str19139 = str19139 + '<gmd:distance><gco:Distance uom="http://standards.iso.org/ittf/PublicityAvailableStandards/ISO_19139_Schemas/resources.uom/ML_gmxUom.xml#m">' + Data_ScaleDistance + '</gco:Distance></gmd:distance>\n'
            str19139 = str19139 + '</gmd:MD_Resolution></gmd:spatialResolution>\n'

        # Data_Language
        Error_DataLanguage = 1
        #Data_Language = u'1' # Valeur par defaut: "français"
        for i in range(1, 20):
            e, Data_Language = _get_xls_value('data_language'+str(i), lst_name, 'string')
            if Data_Language: Data_Language = Data_Language.split()[0]
            if Data_Language and Data_Language in LanguageCode:
                Error_DataLanguage = 0
                str19139 += u'<gmd:language>\n'
                str19139 += u'<gmd:LanguageCode codeList="http://www.loc.gov/standards/iso639-2/" codeListValue="' + LanguageCode[Data_Language] + '">' + LanguageCode[Data_Language] + '</gmd:LanguageCode>\n'
                str19139 += u'</gmd:language>\n'
        if Error_DataLanguage: Error += u'Data_Language: une langue pour les données doit être renseignée.\n'

        # Data_CharacterSet
        e, Data_CharacterSet = _get_xls_value('data_characterset', lst_name, 'string')
        if Data_CharacterSet: Data_CharacterSet = Data_CharacterSet.split()[0]
        if Data_CharacterSet not in CharacterSetCode:
            Data_CharacterSet = '4' # Valeur par defaut: "utf8"
        else :
            str19139 += u'<gmd:characterSet><gmd:MD_CharacterSetCode codeList="http://www.isotc211.org/2005/resources/codeList.xml#MD_CharacterSetCode" codeListValue="' + CharacterSetCode[Data_CharacterSet] + '">' + CharacterSetCode[Data_CharacterSet] + '</gmd:MD_CharacterSetCode></gmd:characterSet>\n'

        # Data_TopicCategory
        Error_TopicCategory = 1
        for i in range(1, 20):
            e, Data_TopicCategory = _get_xls_value('data_topiccategory'+str(i), lst_name, 'string')
            if Data_TopicCategory: Data_TopicCategory = Data_TopicCategory.split()[0]
            if Data_TopicCategory and Data_TopicCategory in TopicCategoryCode:
                Error_TopicCategory = 0
                str19139 += u'<gmd:topicCategory><gmd:MD_TopicCategoryCode>' + TopicCategoryCode[Data_TopicCategory] + '</gmd:MD_TopicCategoryCode></gmd:topicCategory>\n'
        if Error_TopicCategory: Error += u'Data_TopicCategory: un thème ISO pour les données doit être renseigné.\n'
        
        ### Data_Extent
        # Data_GeographicExtent
        Error_DataGeographicExtent = 1
        for i in range(1, 20):
            e, Data_ExtentName = _get_xls_value('data_ext'+str(i)+'_name', lst_name, 'string')
            e, Data_ExtentNorthbound = _get_xls_value('data_ext'+str(i)+'_n', lst_name, 'string')
            e, Data_ExtentSouthbound = _get_xls_value('data_ext'+str(i)+'_s', lst_name, 'string')
            e, Data_ExtentEastbound = _get_xls_value('data_ext'+str(i)+'_e', lst_name, 'string')
            e, Data_ExtentWestbound = _get_xls_value('data_ext'+str(i)+'_w', lst_name, 'string')
            
            if Data_ExtentNorthbound and Data_ExtentSouthbound and Data_ExtentEastbound and Data_ExtentWestbound:
                Error_DataGeographicExtent = 0
                str19139 += u'<gmd:extent><gmd:EX_Extent>\n'
                str19139 += u'<gmd:description><gco:CharacterString>' + Data_ExtentName + '</gco:CharacterString></gmd:description>\n'
                str19139 += u'<gmd:geographicElement><gmd:EX_GeographicBoundingBox>\n'
                str19139 += u'<gmd:westBoundLongitude><gco:Decimal>' + Data_ExtentWestbound + '</gco:Decimal></gmd:westBoundLongitude>\n'
                str19139 += u'<gmd:eastBoundLongitude><gco:Decimal>' + Data_ExtentEastbound + '</gco:Decimal></gmd:eastBoundLongitude>\n'
                str19139 += u'<gmd:southBoundLatitude><gco:Decimal>' + Data_ExtentSouthbound + '</gco:Decimal></gmd:southBoundLatitude>\n'
                str19139 += u'<gmd:northBoundLatitude><gco:Decimal>' + Data_ExtentNorthbound + '</gco:Decimal></gmd:northBoundLatitude>\n'
                str19139 += u'</gmd:EX_GeographicBoundingBox></gmd:geographicElement>\n'
                #str19139 += u'<gmd:geographicElement><gmd:EX_GeographicDescription>\n'
                #str19139 += u'<extentTypeCode><gmd:Boolean>1</gmd:Boolean></extentTypeCode>\n'
                #str19139 += u'<geographicIdentifier><gmd:MD_Identifier>\n'
                #str19139 += u'<code><gco:CharacterString>' + Data_ExtentName + '</gco:CharacterString></code>\n'
                #str19139 += u'</gmd:MD_Identifier></geographicIdentifier>\n'
                #str19139 += u'</gmd:EX_GeographicDescription></gmd:geographicElement>\n'
                str19139 += u'</gmd:EX_Extent></gmd:extent>\n'
        if Error_DataGeographicExtent: Error += u'Data_GeographicExtent: une emprise géographique pour les données doit être renseignée.\n'

        # Data_TemporalExtent
        for i in range(1, 20):
            e, Data_TemporalExtent_Start = _get_xls_value('data_temporalextent'+str(i)+'_start', lst_name, 'date')
            e, Data_TemporalExtent_End = _get_xls_value('data_temporalextent'+str(i)+'_end', lst_name, 'date')

            if Data_TemporalExtent_Start and Data_TemporalExtent_End :
                str19139 += u'<gmd:extent><gmd:EX_Extent>\n'
                str19139 += u'<gmd:temporalElement><gmd:EX_TemporalExtent>\n'
                str19139 += u'<gmd:extent><gml:TimePeriod xsi:type="gml:TimePeriodType">\n'
                str19139 += u'<gml:beginPosition>' + Data_TemporalExtent_Start + '</gml:beginPosition>\n'
                str19139 += u'<gml:endPosition>' + Data_TemporalExtent_End + '</gml:endPosition>\n'
                str19139 += u'</gml:TimePeriod></gmd:extent>\n'
                str19139 += u'</gmd:EX_TemporalExtent></gmd:temporalElement>\n'
                str19139 += u'</gmd:EX_Extent></gmd:extent>\n'

        """
        # Data_VerticalExtent
        for i in range(1, 20):
            e, Data_VerticalExtent_Min = _get_xls_value('data_verticalextent'+str(i)+'_min', lst_name, 'string')
            e, Data_VerticalExtent_Max = _get_xls_value('data_verticalextent'+str(i)+'_max', lst_name, 'string')
            e, Data_VerticalExtent_Unit = _get_xls_value('data_verticalextent'+str(i)+'_unit', lst_name, 'string')
            e, Data_VerticalExtent_Ref = _get_xls_value('data_verticalextent'+str(i)+'_ref', lst_name, 'string')
 
            if Data_VerticalExtent_Min and Data_VerticalExtent_Max and Data_VerticalExtent_Unit and Data_VerticalExtent_Ref:
                str19139 += u'<gmd:extent><gmd:EX_Extent>\n'
                str19139 += u'<gmd:verticalElement><gmd:EX_VerticalExtent>\n'
                str19139 += u'<gmd:minValue><gco:CharacterString>' + Data_VerticalExtent_Min + '</gco:CharacterString></gmd:minValue>\n'
                str19139 += u'<gmd:maxValue><gco:CharacterString>' + Data_VerticalExtent_Max + '</gco:CharacterString></gmd:maxValue>\n'
                str19139 += u'<gmd:uom><gco:CharacterString>' + Data_VerticalExtent_Unit + '</gco:CharacterString></gmd:uom>\n'
                str19139 += u'<gmd:verticalDatum><gco:CharacterString>' + Data_VerticalExtent_Ref + '</gco:CharacterString></gmd:verticalDatum>\n'
                str19139 += u'</gmd:EX_VerticalExtent></gmd:verticalElement>\n'
                str19139 += u'</gmd:EX_Extent></gmd:extent>\n'
        """
        
        str19139 += u'</gmd:MD_DataIdentification></gmd:identificationInfo>\n'

        ### DISTRIBUTION INFO
        str19139 += u'<gmd:distributionInfo><gmd:MD_Distribution>\n'

        # Data_DistFormat
        for i in range(1, 20):
            e, Data_DistFormatName = _get_xls_value('data_distformat'+str(i)+'_name', lst_name, 'string')
            e, Data_DistFormatVersion = _get_xls_value('data_distformat'+str(i)+'_version', lst_name, 'string')
            e, Data_DistFormatSpecification = _get_xls_value('data_distformat'+str(i)+'_specification', lst_name, 'string')
   
            if Data_DistFormatName:
                str19139 += u'<gmd:distributionFormat><gmd:MD_Format>\n'
                str19139 += u'<gmd:name><gco:CharacterString>' + Data_DistFormatName + '</gco:CharacterString></gmd:name>\n'
                str19139 += u'<gmd:version><gco:CharacterString>' + Data_DistFormatVersion + '</gco:CharacterString></gmd:version>\n'
                str19139 += u'<gmd:specification><gco:CharacterString>' + Data_DistFormatSpecification + '</gco:CharacterString></gmd:specification>\n'
                str19139 += u'</gmd:MD_Format></gmd:distributionFormat>\n'

        str19139 += u'<gmd:transferOptions><gmd:MD_DigitalTransferOptions>\n'
        
        # Data_Linkage (url)
        for i in range(1, 20):
            e, Data_LinkageName = _get_xls_value('data_linkage'+str(i)+'_name', lst_name, 'string')
            e, Data_LinkageDescription = _get_xls_value('data_linkage'+str(i)+'_description', lst_name, 'string')
            e, Data_LinkageURL = _get_xls_value('data_linkage'+str(i)+'_url', lst_name, 'string')

            if Data_LinkageURL:
                str19139 += u'<gmd:onLine><gmd:CI_OnlineResource>\n'
                str19139 += u'<gmd:linkage><gmd:URL>' + Data_LinkageURL + '</gmd:URL></gmd:linkage>\n'
                str19139 += u'<gmd:name><gco:CharacterString>' + Data_LinkageName + '</gco:CharacterString></gmd:name>\n'
                str19139 += u'<gmd:description><gco:CharacterString>' + Data_LinkageDescription + '</gco:CharacterString></gmd:description>\n'
                str19139 += u'</gmd:CI_OnlineResource></gmd:onLine>\n'

        str19139 += u'</gmd:MD_DigitalTransferOptions></gmd:transferOptions>\n'
        str19139 += u'</gmd:MD_Distribution></gmd:distributionInfo>\n'

        ### DATA QUALITY INFO
        str19139 += u'<gmd:dataQualityInfo><gmd:DQ_DataQuality>\n'

        # DQ_Level
        e, Data_DQ_Level = _get_xls_value('data_dq_level', lst_name, 'string')
        if Data_DQ_Level: Data_DQ_Level = Data_DQ_Level.split()[0]
        
        if not Data_DQ_Level:
            Data_DQ_Level = MD_HierarchyLevel
        
        if Data_DQ_Level and Data_DQ_Level in ScopeCode:
             str19139 += u'<gmd:scope><gmd:DQ_Scope>\n'
             str19139 += u'<gmd:level><gmd:MD_ScopeCode codeListValue="' + ScopeCode[Data_DQ_Level] + '" codeList="http://standards.iso.org/ittf/PubliclyAvailableStandards/ISO_19139_Schemas/resources/Codelist/ML_gmxCodelists.xml#MD_ScopeCode">' + ScopeCode[Data_DQ_Level] + '</gmd:MD_ScopeCode></gmd:level>\n'
             str19139 += u'</gmd:DQ_Scope></gmd:scope>\n'

        # DQ_InspireConformity
        Error_DQInspireConformity = 1
        for i in range(1, 20):
            e, Data_DQ_InspireConformityTest = _get_xls_value('data_dq_inspireconformity'+str(i)+'_specification', lst_name, 'string')
            if Data_DQ_InspireConformityTest: Data_DQ_InspireConformityTest = Data_DQ_InspireConformityTest.split()[0]
            e, Data_DQ_InspireConformityDateCreation = _get_xls_value('data_dq_inspireconformity'+str(i)+'_datecreation', lst_name, 'date')
            e, Data_DQ_InspireConformityDatePublication = _get_xls_value('data_dq_inspireconformity'+str(i)+'_datepublication', lst_name, 'date')
            e, Data_DQ_InspireConformityDateRevision = _get_xls_value('data_dq_inspireconformity'+str(i)+'_daterevision', lst_name, 'date')
            e, Data_DQ_InspireConformityResult = _get_xls_value('data_dq_inspireconformity'+str(i)+'_explain', lst_name, 'string')
            e, Data_DQ_InspireConformityPass = _get_xls_value('data_dq_inspireconformity'+str(i)+'_pass', lst_name, 'string')
            
            if Data_DQ_InspireConformityTest and Data_DQ_InspireConformityTest in InspireSpecificationCode:
                Error_DQInspireConformity = 0
                str19139 += u'<gmd:report><gmd:DQ_DomainConsistency xsi:type="gmd:DQ_DomainConsistency_Type">\n'
                str19139 += u'<gmd:measureIdentification>\n'
                str19139 += u'<gmd:RS_Identifier>\n'
                str19139 += u'<gmd:code>\n'
                str19139 += u'<gco:CharacterString>InspireConformity_' + str(i) + '</gco:CharacterString>\n'
                str19139 += u'</gmd:code>\n'
                str19139 += u'<gmd:codeSpace>\n'
                str19139 += u'<gco:CharacterString>INSPIRE Conformity</gco:CharacterString>\n'
                str19139 += u'</gmd:codeSpace>\n'
                str19139 += u'</gmd:RS_Identifier>\n'
                str19139 += u'</gmd:measureIdentification>\n'
                str19139 += u'<gmd:result><gmd:DQ_ConformanceResult>\n'
                str19139 += u'<gmd:specification><gmd:CI_Citation>\n'
                str19139 += u'<gmd:title><gco:CharacterString>' + InspireSpecificationCode[Data_DQ_InspireConformityTest] + '</gco:CharacterString></gmd:title>\n'
                if Data_DQ_InspireConformityDateCreation or Data_DQ_InspireConformityDatePublication or Data_DQ_InspireConformityDateRevision :
                    if Data_DQ_InspireConformityDateCreation:
                        str19139 += u'<gmd:date><gmd:CI_Date>\n'
                        str19139 += u'<gmd:date><gco:Date>' + Data_DQ_InspireConformityDateCreation + '</gco:Date></gmd:date>\n'
                        str19139 += u'<gmd:dateType><gmd:CI_DateTypeCode codeList="http://standards.iso.org/ittf/PubliclyAvailableStandards/ISO_19139_Schemas/resources/Codelist/ML_gmxCodelists.xml#CI_DateTypeCode" codeListValue="creation">creation</gmd:CI_DateTypeCode></gmd:dateType>\n'
                        str19139 += u'</gmd:CI_Date></gmd:date>\n'
                    if Data_DQ_InspireConformityDatePublication:
                        str19139 += u'<gmd:date><gmd:CI_Date>\n'
                        str19139 += u'<gmd:date><gco:Date>' + Data_DQ_InspireConformityDatePublication + '</gco:Date></gmd:date>\n'
                        str19139 += u'<gmd:dateType><gmd:CI_DateTypeCode codeList="http://standards.iso.org/ittf/PubliclyAvailableStandards/ISO_19139_Schemas/resources/Codelist/ML_gmxCodelists.xml#CI_DateTypeCode" codeListValue="publication">publication</gmd:CI_DateTypeCode></gmd:dateType>\n'
                        str19139 += u'</gmd:CI_Date></gmd:date>\n'
                    if Data_DQ_InspireConformityDateRevision:
                        str19139 += u'<gmd:date><gmd:CI_Date>\n'
                        str19139 += u'<gmd:date><gco:Date>' + Data_DQ_InspireConformityDateRevision + '</gco:Date></gmd:date>\n'
                        str19139 += u'<gmd:dateType><gmd:CI_DateTypeCode codeList="http://standards.iso.org/ittf/PubliclyAvailableStandards/ISO_19139_Schemas/resources/Codelist/ML_gmxCodelists.xml#CI_DateTypeCode" codeListValue="revision">revision</gmd:CI_DateTypeCode></gmd:dateType>\n'
                        str19139 += u'</gmd:CI_Date></gmd:date>\n'
                else:
                    Error += u'DQ_InspireConformity: une date doit être renseigné pour les tests de conformité INSPIRE.\n'

                str19139 += u'</gmd:CI_Citation></gmd:specification>\n'

                if Data_DQ_InspireConformityResult:
                    str19139 += u'<gmd:explanation><gco:CharacterString>' + Data_DQ_InspireConformityResult + '</gco:CharacterString></gmd:explanation>\n'
                if Data_DQ_InspireConformityPass == "Conforme":
                    str19139 += u'<gmd:pass><gco:Boolean>true</gco:Boolean></gmd:pass>\n'
                elif Data_DQ_InspireConformityPass == "Non conforme":
                    str19139 += u'<gmd:pass><gco:Boolean>false</gco:Boolean></gmd:pass>\n'
                else:
                    str19139 += u'<gmd:pass></gmd:pass>\n'

                str19139 += u'</gmd:DQ_ConformanceResult></gmd:result>\n'
                str19139 += u'</gmd:DQ_DomainConsistency></gmd:report>\n'

        # DQ_Conformity
        Error_DQConformity = 1
        for i in range(1, 20):
            e, Data_DQ_ConformityTest = _get_xls_value('data_dq_conformity'+str(i)+'_specification', lst_name, 'string')
            e, Data_DQ_ConformityDateCreation = _get_xls_value('data_dq_conformity'+str(i)+'_datecreation', lst_name, 'date')
            e, Data_DQ_ConformityDatePublication = _get_xls_value('data_dq_conformity'+str(i)+'_datepublication', lst_name, 'date')
            e, Data_DQ_ConformityDateRevision = _get_xls_value('data_dq_conformity'+str(i)+'_daterevision', lst_name, 'date')
            e, Data_DQ_ConformityResult = _get_xls_value('data_dq_conformity'+str(i)+'_explain', lst_name, 'string')
            e, Data_DQ_ConformityPass = _get_xls_value('data_dq_conformity'+str(i)+'_pass', lst_name, 'string')
            
            if Data_DQ_ConformityTest:
                Error_DQInspireConformity = 0
                str19139 += u'<gmd:report><gmd:DQ_DomainConsistency xsi:type="gmd:DQ_DomainConsistency_Type">\n'
                str19139 += u'<gmd:measureIdentification>\n'
                str19139 += u'<gmd:RS_Identifier>\n'
                str19139 += u'<gmd:code>\n'
                str19139 += u'<gco:CharacterString>Conformity_' + str(i) + '</gco:CharacterString>\n'
                str19139 += u'</gmd:code>\n'
                str19139 += u'<gmd:codeSpace>\n'
                str19139 += u'<gco:CharacterString>Other conformity</gco:CharacterString>\n'
                str19139 += u'</gmd:codeSpace>\n'
                str19139 += u'</gmd:RS_Identifier>\n'
                str19139 += u'</gmd:measureIdentification>\n'
                str19139 += u'<gmd:result><gmd:DQ_ConformanceResult>\n'
                str19139 += u'<gmd:specification><gmd:CI_Citation>\n'
                str19139 += u'<gmd:title><gco:CharacterString>' + Data_DQ_ConformityTest + '</gco:CharacterString></gmd:title>\n'
                if Data_DQ_ConformityDateCreation or Data_DQ_ConformityDatePublication or Data_DQ_ConformityDateRevision :
                    if Data_DQ_ConformityDateCreation:
                        str19139 += u'<gmd:date><gmd:CI_Date>\n'
                        str19139 += u'<gmd:date><gco:Date>' + Data_DQ_ConformityDateCreation + '</gco:Date></gmd:date>\n'
                        str19139 += u'<gmd:dateType><gmd:CI_DateTypeCode codeList="http://standards.iso.org/ittf/PubliclyAvailableStandards/ISO_19139_Schemas/resources/Codelist/ML_gmxCodelists.xml#CI_DateTypeCode" codeListValue="creation">creation</gmd:CI_DateTypeCode></gmd:dateType>\n'
                        str19139 += u'</gmd:CI_Date></gmd:date>\n'
                    if Data_DQ_ConformityDatePublication:
                        str19139 += u'<gmd:date><gmd:CI_Date>\n'
                        str19139 += u'<gmd:date><gco:Date>' + Data_DQ_ConformityDatePublication + '</gco:Date></gmd:date>\n'
                        str19139 += u'<gmd:dateType><gmd:CI_DateTypeCode codeList="http://standards.iso.org/ittf/PubliclyAvailableStandards/ISO_19139_Schemas/resources/Codelist/ML_gmxCodelists.xml#CI_DateTypeCode" codeListValue="publication">publication</gmd:CI_DateTypeCode></gmd:dateType>\n'
                        str19139 += u'</gmd:CI_Date></gmd:date>\n'
                    if Data_DQ_ConformityDateRevision:
                        str19139 += u'<gmd:date><gmd:CI_Date>\n'
                        str19139 += u'<gmd:date><gco:Date>' + Data_DQ_ConformityDateRevision + '</gco:Date></gmd:date>\n'
                        str19139 += u'<gmd:dateType><gmd:CI_DateTypeCode codeList="http://standards.iso.org/ittf/PubliclyAvailableStandards/ISO_19139_Schemas/resources/Codelist/ML_gmxCodelists.xml#CI_DateTypeCode" codeListValue="revision">revision</gmd:CI_DateTypeCode></gmd:dateType>\n'
                        str19139 += u'</gmd:CI_Date></gmd:date>\n'
                else:
                    Error += u'DQ_InspireConformity: une date doit être renseigné pour les tests de conformité INSPIRE.\n'

                str19139 += u'</gmd:CI_Citation></gmd:specification>\n'

                if Data_DQ_ConformityResult:
                    str19139 += u'<gmd:explanation><gco:CharacterString>' + Data_DQ_ConformityResult + '</gco:CharacterString></gmd:explanation>\n'
                if Data_DQ_ConformityPass == "Conforme":
                    str19139 += u'<gmd:pass><gco:Boolean>true</gco:Boolean></gmd:pass>\n'
                elif Data_DQ_ConformityPass == "Non conforme":
                    str19139 += u'<gmd:pass><gco:Boolean>false</gco:Boolean></gmd:pass>\n'
                else:
                    str19139 += u'<gmd:pass></gmd:pass>\n'

                str19139 += u'</gmd:DQ_ConformanceResult></gmd:result>\n'
                str19139 += u'</gmd:DQ_DomainConsistency></gmd:report>\n'
        
        ### DQ_Lineage
        str19139 += u'<gmd:lineage><gmd:LI_Lineage>\n'

        # LI_Statement
        e, LI_Statement = _get_xls_value('li_statement', lst_name, 'string')

        if LI_Statement:
            str19139 += u'<gmd:statement><gco:CharacterString>' + LI_Statement + '</gco:CharacterString></gmd:statement>\n'
        else:
            Error += u'LI_Statement: un texte sur la provenance des données doit être renseigné.\n'

        # LI_ProcessStep
        e, LI_ProcessStep = _get_xls_value('li_processstep', lst_name, 'string')
        
        if LI_ProcessStep:
            str19139 += u'<gmd:processStep><gmd:LI_ProcessStep>\n'
            str19139 += u'<gmd:description><gco:CharacterString>' + LI_ProcessStep + '</gco:CharacterString></gmd:description>\n'
            str19139 += u'</gmd:LI_ProcessStep></gmd:processStep>\n'

        # LI_Source
        e, LI_Source = _get_xls_value('li_source', lst_name, 'string')

        if LI_Source:
            str19139 += u'<gmd:source><gmd:LI_Source>\n'
            str19139 += u'<gmd:description><gco:CharacterString>' + LI_Source + '</gco:CharacterString></gmd:description>\n'
            str19139 += u'</gmd:LI_Source></gmd:source>\n'

        str19139 += u'</gmd:LI_Lineage></gmd:lineage>\n'
        str19139 += u'</gmd:DQ_DataQuality></gmd:dataQualityInfo>\n'

        # Fin du fichier
        str19139 += u'</gmd:MD_Metadata>\n'

        # Définition du nom de fichier XML
        xml_filename = MD_FileIdentifier + '.xml'
        xml_filename = os.path.join(xml_dir, xml_filename)
        
        if not os.path.isfile(xml_filename):             
            xml_file = open(xml_filename, 'w')
            xml_file.write(str19139.encode('utf-8'))
            xml_file.close()
            #Error += u'Le fichier '+xml_filename+' a ete cree avec succes.\n'
            Error_CreateFile = u'Le fichier '+xml_filename+' a ete cree avec succes.\n'
        else:
            Error += u'Le fichier '+xml_filename+' existe deja et ne peut-etre cree.\n'
        
        
        # Définition du nom de fichier de log
        log_filename = MD_FileIdentifier + '_erreur.txt'
        log_filename = os.path.join(xml_dir, log_filename)
        
        if Error != '':
            if os.path.isfile(log_filename):
                os.remove(log_filename)
            log_file = open(log_filename, 'w')
            log_file.write(Error.encode('utf-8'))
            log_file.close()
            print u'Le fichier '+log_filename+' a ete cree.\n'
        else:
            print u'Aucune erreur identifiée.\n'
        
        return Error.encode('utf-8') + Error_CreateFile.encode('utf-8')


# ------------------------------------------------------------------------------
# _GET_VALUE
# ------------------------------------------------------------------------------
def _get_xls_value(cell = '', lst_name = {}, type = 'string'):       
    # print '_get_value'
    error = 1
    result = ''

    if cell == '' or cell not in lst_name:
        error = u'Nom de cellule n\'existe pas (' + cell + ')'
        result = ''
    else:
        value = lst_name[cell][0].cell().value

        # Traiter les valeurs nulles
        if value == '':
            error = 0
            result = ''
        else:
            if type == 'string':
                error = 0
                result = unicode(value)
            elif type == 'integer':
                try: 
                    error = 0
                    result = int(value)
                except: 
                    error = u'La valeur transmise n\'est pas un entier'
                    result = ''
            elif type == 'date':
                try: 
                    error = 0
                    result = datetime.datetime(*xlrd.xldate_as_tuple(value, 0))
                    result = str(result)[:10]
                except: 
                    error = u'La valeur transmise n\'est pas une date'
                    result = ''
            else:
                error = u'Le type indiqué n\'existe pas (' + type + ')'
                result = ''
    
    # Traitement des erreurs
    # if error: print 'Erreur: ', error
    if result != '' : result = unicode(result)
    
    # Transmettre le résultat
    return error, result
        
        
# ------------------------------------------------------------------------------
# CONVERT_XML2XLS
#
# Fonction de conversion d'un fichier XML vers MS Excel
# ------------------------------------------------------------------------------
def convert_xml2xls(xml_file, xls_dir):       
    print 'convert_xml2xls' 


# Fonction "main"
def main():
    ''' Fonction princpal du programme '''
    xls_file = ''
    xml_dir = ''

    # error = xls_file.xls2xml(XML_DIR)
    try:
        error = xls2xml(xls_file, xml_dir)
    except:
        error = u'Erreur de transformation.'
    
    if error:
        print '\n=> ERREUR pour le fichier :', xls_file
        print error
    else:
        print '\n=> OK pour le fichier', xls_file, 'converti en XML.'


if __name__ == '__main__':
    main()
