<?php
/*licence/ 

Module écrit, supporté par la société Alkante SAS <alkante@alkante.com>

Nom du module : Alkanet::Class::Form
Module fournissant les classes d'affichage Alkanet.
Ce module appartient au framework Alkanet.

Ce logiciel est régi par la licence CeCILL-C soumise au droit français et
respectant les principes de diffusion des logiciels libres. Vous pouvez
utiliser, modifier et/ou redistribuer ce programme sous les conditions
de la licence CeCILL-C telle que diffusée par le CEA, le CNRS et l'INRIA
sur le site http://www.cecill.info.

En contrepartie de l'accessibilité au code source et des droits de copie,
de modification et de redistribution accordés par cette licence, il n'est
offert aux utilisateurs qu'une garantie limitée. Pour les mêmes raisons,
seule une responsabilité restreinte pèse sur l'auteur du programme, le
titulaire des droits patrimoniaux et les concédants successifs.

A cet égard l'attention de l'utilisateur est attirée sur les risques
associés au chargement, à l'utilisation, à la modification et/ou au
développement et à la reproduction du logiciel par l'utilisateur étant
donné sa spécificité de logiciel libre, qui peut le rendre complexe à
manipuler et qui le réserve donc à des développeurs et des professionnels
avertis possédant des connaissances informatiques approfondies. Les
utilisateurs sont donc invités à charger et tester l'adéquation du
logiciel à leurs besoins dans des conditions permettant d'assurer la
sécurité de leurs systèmes et ou de leurs données et, plus généralement,
à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.

Le fait que vous puissiez accéder à cet en-tête signifie que vous avez
pris connaissance de la licence CeCILL-C, et que vous en avez accepté les
termes.

/licence*/

require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CLASSE."form/alkhtmlctrl.class.php");

/**
 * @package Alkanet_Class_Form
 * @class alkHtmlCheckBox
 * 
 * classe d'affichage d'un composant groupe de case à cocher
 *        génère : <input type=checkbox>
 */
class AlkHtmlCheckbox extends AlkHtmlCtrl
{
  /** l'attribut $value contient la valeur de la case à cocher si le groupe ne contient qu'une case */

  /** Valeurs d'une case cochée dans le cas où le groupe ne contient qu'une case */
  protected $valueChecked;
  
  /** Valeurs d'une case non cochée dans le cas loù le groupe ne contient qu'une case */
  protected $valueNotChecked;
  
  /** Label (utilisé en mode lecture) d'une case cochée dans le cas où le groupe ne contient qu'une case */
  protected $labelChecked;
  
  /** Label (utilisé en mode lecture) d'une case non cochée dans le cas loù le groupe ne contient qu'une case */
  protected $labelNotChecked;
  
  /** liste des cases à cocher contenant valeur courante, valeur cochée et non cochée, le label et events */
  protected $tabCheckbox;

  /** table indexant les valeurs de checkbox */
  protected $tabIndexValue;

  /** vrai si label à droite de la case à cocher */
  protected $bLabelRight;
  
  /** nombre d'éléments par ligne (si 0 alors affichage sur 1 ligne, si 1 alors affichage sur 1 colonne, autrement iNbItem par ligne) */
  protected $iNbItem;
  
  /** vrai si renommage des checkbox , pas de renommage pour les controles nommés name[] */
  protected $bRename;
  
  /** table des controles ajoutés à chaque checkbox */
  protected $tabCtrlsCheck;

  /**
   *  Constructeur par défaut
   *        Chaque checkbox a pour nom $this->name concaténé avec son index dans la liste
   * @param oForm      référence sur l'objet formulaire
   * @param iMode      Mode du controle de saisie : =0 modif, =1 lecture
   * @param name       Nom du controle de saisie
   * @param value      Valeur du groupe de cases à cocher
   * @param label      Etiquette texte associée au groupe
   * @param valueChecked Valeur de la case à cocher cochée dans le cas 1 checkbox
   * @param valueNotChecked Valeur de la case à cocher non cochée dans le cas 1 checkbox
   * @param labelChecked Label en mode lecture de la case à cocher cochée dans le cas 1 checkbox
   * @param labelNotChecked Label en mode lecture de la case à cocher non cochée dans le cas 1 checkbox
   */
  public function __construct($oForm, $iMode, $name, $value, $label="", $valueChecked="1", $valueNotChecked="0", $labelChecked="oui", $labelNotChecked="non")
  {
    parent::__construct($oForm, $iMode, $name, $value, $label);
    $this->tabCheckbox = array();
    $this->tabCtrlsCheck = array();
    $this->valueChecked = $valueChecked;
    $this->valueNotChecked = $valueNotChecked;
    $this->labelChecked = ( $labelChecked == "oui" ? _t("oui") : $labelChecked ); 
    $this->labelNotChecked = ( $labelNotChecked == "oui" ? _t("oui") : $labelNotChecked );
    $this->tabIndexValue = array();
    $this->bLabelRight = true;
    $this->iNbItem = 0;
    $this->bRename = !mb_ereg("\[.*\]", $name);
    $this->cssFormCtrl = "formCtrlRCB";

  }

  /**
   *  Dans le cas où le groupe ne contient qu'une seule case,
   *        cette méthode est appelée pour initialiser le tableau tabCheckbox
   *        nécessaire à la construction du controle
   */
  protected function initFirstWithProperties()
  {
    $this->tabCheckbox[] = array("value"      => $this->value, 
                                 "checked"    => $this->valueChecked, 
                                 "notchecked" => $this->valueNotChecked, 
                                 "label"      => "",
                                 "labelchecked" => $this->labelChecked,
                                 "labelnotchecked" => $this->labelNotChecked,
                                 "event"      => array(),
                                 "properties"      => array());
    $this->tabIndexValue["_".$this->value] = 0;

    // propage les événements du groupe à la case
    foreach ($this->tabEvents as $strEvent=>$strFunctionJS){
      $this->addEventToCheckBox($this->value, $strEvent, $strFunctionJS);
    }
  }

  /**
   *  Ajoute une case à cocher au groupe
   *        Retourne l'index à laquelle, la case est associée
   * @param strValue   Valeur de la case à cocher si cochée
   * @param strLabel   Etiquette texte associée à la case à cocher
   * @param valueCheck Valeur de la case à cocher cochée
   * @param valueCheck Valeur de la case à cocher non cochée
   * @param bUseIndex  faux par défaut, vrai pour utiliser l'index au lieu de la valeur pour pour indexer les cases
   *                   Cette index est utilisé par la méthode addEventToCheckBox pour associer un event à la case
   * @return int
   */
  public function addCheckBox($value, $strLabel, $valueChecked="1", $valueNotChecked="0", $bUseIndex=false)
  {
    $iIndex = count($this->tabCheckbox);
    if( $bUseIndex )
      $this->tabIndexValue["_".$iIndex] = $iIndex;
    else
      $this->tabIndexValue["_".$value] = $iIndex;
    $this->tabCheckbox[$iIndex] = array("value"      => $value, 
                                        "checked"    => $valueChecked,  
                                        "notchecked" => $valueNotChecked,  
                                        "labelchecked" => $this->labelChecked,
                                        "labelnotchecked" => $this->labelNotChecked,
                                        "label"      => $strLabel,
                                        "event"      => array(),
                                        "properties"      => array());
    return $iIndex;
  }

  /**
   *  Ajoute un attribut HTML à une case à cocher
   * 
   * @param iIndex          Index de la case à cocher
   * @param htmlAttribute   Nom d'un attribut html pouvant aller sur la case à cocher
   * @param attributeValue  Valeur affectée à cet attribut html
   * @param bConcat         Détermine si cette valeur s'ajoute à des valeurs précédemment affectées
   * @return boolean : propriété ajoutée (ie. case à cocher existe)
   */
  public function setCheckBoxProperty($iIndex, $htmlAttribute, $attributeValue, $bConcat=false)
  {
    if ( in_array($iIndex, $this->tabIndexValue) ){
      if ( $bConcat ){
        if ( !array_key_exists($htmlAttribute, $this->tabCheckbox[$iIndex]["properties"]) ){
          $this->tabCheckbox[$iIndex]["properties"][$htmlAttribute] = "";
        }
        $this->tabCheckbox[$iIndex]["properties"][$htmlAttribute] .= $attributeValue;
      }
      else {
        $this->tabCheckbox[$iIndex]["properties"][$htmlAttribute] = $attributeValue;
      }
      return true;
    }
    return false;
  }

  /**
   *  Ajoute une case à cocher au groupe
   * @param oFormData AlkFormData  Objet de donnée contenant la valeur
   * @param strLabel  Etiquette texte associée à la case à cocher
   * @param valueCheck Valeur de la case à cocher cochée
   * @param valueCheck Valeur de la case à cocher non cochée
   */
  public function addDataCheckBox(AlkFormData $oFormData, $strLabel, $valueChecked="1", $valueNotChecked="0")
  {
    $this->addCheckBox($oFormData->value, $strLabel, $valueChecked, $valueNotChecked);
  }

  /**
   *  Ajoute un controle à une case à cocher
   * @param iIndex               Index de la case à cocher
   * @param oCtrl  AlkHtmlCtrl   Controle à ajouter au groupe
   * @param strSeparator         Séparateur entre ce controle et les autres controles du groupe (ceux qui précedent)
   * @param strSeparator         Séparateur entre le label de ce controle et son getHtml
   * @return string html
   */
  public function addCtrlToCheckBox($iIndex, AlkHtmlCtrl &$oCtrl, $strSeparatorLabel="&nbsp;")
  {
    if ( !array_key_exists($iIndex, $this->tabCtrlsCheck) ){
      $this->tabCtrlsCheck[$iIndex] = array();
    }
    $iCtrl = count($this->tabCtrlsCheck[$iIndex]);
    $this->tabCtrlsCheck[$iIndex][$iCtrl]["oCtrl"] = $oCtrl;
    $this->tabCtrlsCheck[$iIndex][$iCtrl]["separator_label"] = $strSeparatorLabel;
    if ( !is_null($this->oBlock) ){
      $oCtrl->setBlock($this->oBlock, true);
    }
  }

  /**
   * Génère le code des controles ajoutés à une case à coché
   * @param iIndex     Index de la case à cocher
   * @param bInTable   =false par défaut, =true pour inclure les ctrls dans une cellule de tableau
   * @return string html
   */
  protected function getHtmlCtrlsByCheckbox($iIndex, $bInTable=false)
  {
    $strHtml = "";
    if ( !array_key_exists($iIndex, $this->tabCtrlsCheck) ){
      return "";
    }
    foreach ($this->tabCtrlsCheck[$iIndex] as $tabCtrl){
      $oCtrl = $tabCtrl["oCtrl"];
      $strSeparatorLabel = $tabCtrl["separator_label"];
      $strLabel = $oCtrl->getLabel();
      $strHtml .= 
        ( !$oCtrl->isTypeOf(self::ALK_CLASS_HTMLBUTTON) 
          ? ( $bInTable 
              ? '<td width="10"></td><td class="'.$this->cssFormLabel.'">' 
              : '' ).
            ( $strLabel!="" 
              ? '<label for="'.$oCtrl->getGUID().'" class="label">'.$strLabel.'</label>'
              : '' ) .
             ( $bInTable 
               ? '</td>' 
               : ( $strLabel!="" ? $strSeparatorLabel : '' ) )
          : '' ).  
        ( $bInTable 
          ? '<td class="'.$this->cssFormCtrl.'">' 
          : '' ).
        $oCtrl->getHtml().
        ( $bInTable 
          ? '</td>' 
          : '' );
    }
    return $strHtml;
  }
  
  /**
   *  Ajoute un evenement sur une case à cocher
   * @param strValue      Valeur de la case à cocher si les cases ont été ajoutées par addCheckBox
   *                      Index de la case à cocher si les cases ont été ajoutées par addCheckBoxByIndex
   * @param strEvent      Nom de l'événement
   * @param strFunctionJS Nom de la fonction javascript appelée (pas nécessaire d'ajouter 'javascript:')
   */
  public function addEventToCheckBox($strValue, $strEvent, $strFunctionJS)
  {
    // recherche l'indice de la valeur
    $iPos = ( array_key_exists("_".$strValue, $this->tabIndexValue) 
              ? $this->tabIndexValue["_".$strValue]
              : -1 );
    $this->addEventToCheckBoxByIndex($iPos, $strEvent, $strFunctionJS);
  }

  /**
   *  Ajoute un evenement sur une case à cocher identifié par son index dans le tableau des checkbox
   * @param iIndex        Index de la case à cocher
   * @param strEvent      Nom de l'événement
   * @param strFunctionJS Nom de la fonction javascript appelée (pas nécessaire d'ajouter 'javascript:')
   */
  public function addEventToCheckBoxByIndex($iIndex, $strEvent, $strFunctionJS)
  {
    if ( array_key_exists($iIndex, $this->tabCheckbox) ){
      if( array_key_exists($strEvent, $this->tabCheckbox[$iIndex]["event"]) ) {
        $this->tabCheckbox[$iIndex]["event"][$strEvent] .= $strFunctionJS;
      } else {
        $this->tabCheckbox[$iIndex]["event"][$strEvent] = $strFunctionJS;
      }
    }
  }
  
  /**
   *  Genere puis retourne tous les evenements liés au controle html
   *
   * @param iPos  Indice dans la liste de cases à cocher
   * @return Retourne les événements concaténés dans une chaine pour être inclus dans
   *         dans un tag html de type controle de saisie
   */
  protected function getHtmlEventCheckBox($iPos)
  {
    $strHtml = "";
    foreach($this->tabCheckbox[$iPos]["event"] as $strEvent => $strFunctionJs)
      $strHtml .= ' '.$strEvent.'="javascript:'.$strFunctionJs.'"';
    return $strHtml;
  }
  
  /**
   *  Genere puis retourne toutes les propriétés attributaires associées à la checkbox
   *
   * @param iPos  Indice dans la liste de cases à cocher
   * @return string : retourne toutes les propriétés attributaires associées à la checkbox
   */
  protected function getHtmlPropertiesCheckBox($iPos)
  {
    $strHtml = "";
    foreach($this->tabCheckbox[$iPos]["properties"] as $attribute => $value)
      $strHtml .= ' '.$attribute.'="'.$value.'"';
    return $strHtml;
  }

  /**
   *  Génère puis retourne le code html associé au controle de saisie
   */
  protected function getCtrlHtml($key=0, $tabLg=array())
  {
    if( empty($this->tabCheckbox) )
      $this->initFirstWithProperties();
    
    $strClassCss = ( $this->cssFormCtrl!=""
                     ? ' class="'.$this->cssFormCtrl.'"'
                     : '' ).
    
    // contruit le controle
    $strHtml = "";
    if( count($this->tabCheckbox)>0 ) {
      if( $this->iMode == "1" ) {
        // mode lecture
        // affiche les labels sélectionnés        
        if (count($this->tabCheckbox) == 1) {
          if( $this->tabCheckbox[0]["value"] == $this->tabCheckbox[0]["checked"] )
            $strHtml .= $this->tabCheckbox[0]["labelchecked"];
          else
            $strHtml .= $this->tabCheckbox[0]["labelnotchecked"];
        } else {
           for($i=0; $i<count($this->tabCheckbox); $i++) {
            if( $this->tabCheckbox[$i]["value"] == $this->tabCheckbox[$i]["checked"] ) {
              $strHtml .= ( $strHtml != "" ? ", " : "" ).$this->tabCheckbox[$i]["label"];
            } 
          }
        }
        if( $strHtml != "" ) {
          $strHtml = '<span '.$strClassCss.'>'.$strHtml.'</span>';
        }
      } 
      else if ( count($this->tabCheckbox)==1 ){
        //mode modif : une seule checkbox dans le groupe
        $strHtml .= $this->getHtmlOne(0, true).$this->getHtmlCtrlsByCheckbox(0);
      } 
      else {        
        // mode modif
        // input type=CheckBox
        if( $this->bRename ) {
          $strHtml .= '<input type="hidden" id="'.$this->guid.'" name="'.$this->name.'" value="'.count($this->tabCheckbox).'"/>';
        }
        for($i=0; $i<count($this->tabCheckbox); $i++) {
          $strHtml .= 
            ( $i==0 
              ? '<table summary="" border="0" cellspacing="0" cellpadding="2" width="*">' 
              : '' ).
            ( $this->iNbItem!=0 && $i%$this->iNbItem==0 
              ? '<tr>' 
              : '' ).
            '<td '.$strClassCss.'>'.$this->getHtmlOne($i).'</td>'.
            $this->getHtmlCtrlsByCheckbox($i, true).
            ( $this->iNbItem!=0 && (($i%$this->iNbItem)-$this->iNbItem==1)
              ? '</tr>'
              : '' ).
            ( $i==count($this->tabCheckbox)-1
              ? '</table>'
              : '' );
        }
      }
    } else {
      $strHtml = '<span '.$strClassCss.'> '._t("Aucun(e)").'</span>';
    }

    return $strHtml;
  }
  
  /**
   *  Retourne le code Html d'une checkbox
   * @param index   Index de la checkbox dans le tableau de checkbox
   * @return string html
   */
  private function getHtmlOne($index, $bOneCheck=false)
  {
    $strHtml = "";
    
    $guid = $this->guid.(!$bOneCheck ? "_".$index : "");
    $name = $this->name. (!$bOneCheck && $this->bRename ? "_".$index : "");

    if( $this->bLabelRight == false && $this->tabCheckbox[$index]["label"] != "" ) {
      $strHtml .= '<label for="'.$guid.'">'.$this->tabCheckbox[$index]["label"].' </label>';
    }

    $strHtml .= '<input type="checkbox"'.
      ( $this->cssFormCtrl!=""
        ? ' class="'.$this->cssFormCtrl.'"'
        : '' ).
      ' id="'.$guid.'"'.
      ' name="'.$name.'"'.
      ' value="'.$this->tabCheckbox[$index]["checked"].'"'.
      ( $this->tabCheckbox[$index]["value"] == $this->tabCheckbox[$index]["checked"] 
        ? ' checked="checked"' 
        : '' ).
      $this->getHtmlEventCheckBox($index).
      $this->getHtmlPropertiesCheckBox($index).
      '/>';

    if( $this->tabCheckbox[$index]["notchecked"] != "null" ) {
      $strHtml .= '<input type="hidden"'.
        ' id="not_'.$guid.'"'.
        ' name="not_'.$name.'"'.
        'value="'.$this->tabCheckbox[$index]["notchecked"].'"'.
        '/>';
    }

    if( $this->bLabelRight == true && $this->tabCheckbox[$index]["label"] != "" ) {
      $strHtml .= '<label for="'.$guid.'"> '.$this->tabCheckbox[$index]["label"].'</label>';
    }

    return $strHtml;  
  }

  /**
   *  Fixe le nombre de case par ligne
   *        =0 toutes les cases sur la même ligne
   * @param iNb  nombre de case par ligne
   */
  public function setItemsPerLine($iNb=0)
  {
    $this->iNbItem = $iNb;
  }
}

?>