<?php
/*licence/ 

Module écrit, supporté par la société Alkante SAS <alkante@alkante.com>

Nom du module : Alkanet::Class::Form
Module fournissant les classes d'affichage Alkanet.
Ce module appartient au framework Alkanet.

Ce logiciel est régi par la licence CeCILL-C soumise au droit français et
respectant les principes de diffusion des logiciels libres. Vous pouvez
utiliser, modifier et/ou redistribuer ce programme sous les conditions
de la licence CeCILL-C telle que diffusée par le CEA, le CNRS et l'INRIA
sur le site http://www.cecill.info.

En contrepartie de l'accessibilité au code source et des droits de copie,
de modification et de redistribution accordés par cette licence, il n'est
offert aux utilisateurs qu'une garantie limitée. Pour les mêmes raisons,
seule une responsabilité restreinte pèse sur l'auteur du programme, le
titulaire des droits patrimoniaux et les concédants successifs.

A cet égard l'attention de l'utilisateur est attirée sur les risques
associés au chargement, à l'utilisation, à la modification et/ou au
développement et à la reproduction du logiciel par l'utilisateur étant
donné sa spécificité de logiciel libre, qui peut le rendre complexe à
manipuler et qui le réserve donc à des développeurs et des professionnels
avertis possédant des connaissances informatiques approfondies. Les
utilisateurs sont donc invités à charger et tester l'adéquation du
logiciel à leurs besoins dans des conditions permettant d'assurer la
sécurité de leurs systèmes et ou de leurs données et, plus généralement,
à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.

Le fait que vous puissiez accéder à cet en-tête signifie que vous avez
pris connaissance de la licence CeCILL-C, et que vous en avez accepté les
termes.

/licence*/

require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CLASSE."form/alkhtmlpanel.class.php");

/**
 * @package Alkanet_Class_Form
 * @class AlkHtmlCtrl
 * 
 * Classe de base pour l'affichage d'un composant html.
 * 
 * Définition des constantes de type controle
 *
 * memo   : champ texte alphanumérique pour un texte long (avec min et max sur la longueur du texte requis)
 * text   : champ texte alphanumérique pour un texte <255 caractères (avec min et max sur la longueur du texte requis)
 * int    : champ texte vérifiant le bon format d'un nombre entier ou réel, négatif ou positif (avec min et max sur la valeur)
 * date10 : champ texte vérifiant le bon format de la date JJ/MM/AAAA (avec min et le max sur la valeur)
 * radio  : pas de vérif
 * check  : pas de vérif
 * select : pas de vérif
 * mail   : champ texte vérifiant le bon format email
 *
 * textnum     : champ texte numérique n'acceptant que des chiffres, exemple : code postal, siret, siren... acceptant
 *               le zéro en premier caractère (avec min et max sur la longueur du texte requis)
 * textalpha   : champ texte alphabétique n'acceptant que les lettres, apostrophes, tirets, 
 *               espace, exemple : Nom, prénom (avec min et max sur la longueur du texte requis)
 * heure245    : champ texte vérifiant le bon format de l'heure HH24:MN (avec min et le max sur la valeur, et 0<=HH24<24)
 * heure5      : champ texte vérifiant le bon format de l'heure HH24:MN (avec min et le max sur la valeur, et 0<=HH)
 * url         : champ texte vérifiant une adresse internet  http, file ou ftp.
 */
abstract class AlkHtmlCtrl extends AlkHtmlPanel
{
  const ALK_CLASS_HTMLHIDDEN    = "alkhtmlhidden";
  const ALK_CLASS_HTMLTEXT      = "alkhtmltext";
  const ALK_CLASS_HTMLRADIO     = "alkhtmlradiogroup";
  const ALK_CLASS_HTMLCHECK     = "alkhtmlcheckboxgroup";
  const ALK_CLASS_HTMLCOLOR     = "alkhtmlcolor";
  const ALK_CLASS_HTMLFILE      = "alkhtmlfile";
  const ALK_CLASS_HTMLUPLOAD    = "alkhtmlupload";
  
  /** référence vers le formulaire conteneur */
  protected $oForm;
  
  /** référence vers le block conteneur */
  protected $oBlock;

  /** Mode du controle de saisie : =1 en lecture, =0 en modification */
  protected $iMode; 

  /** Valeur du controle de saisie par défaut */
  protected $value;

  /** = vrai si en lecture seul, faux par défaut */
  protected $bReadOnly;

  /** =vrai si desactivé */
  protected $bDisabled;

  /** vrai pour afficher la valeur du controle en hidden en mode lecture */
  protected $bWriteHiddenOnReadOnly;

  /** Etiquette texte placée devant le controle dans la cellule du ctrl */
  protected $labelBeforeCtrl;

  /** Etiquette texte placée après le controle dans la cellule du ctrl */
  protected $labelAfterCtrl;

  /** label texte contenant l'aide affiché lorsque le picto aide est actif */
  protected $labelHelp;

  /** Affichage de l'aide sur tous les modes d'affichage du controle si true, sinon seulement en écriture (par défaut) */
  protected $bAllModeHelp;

  /** Si true, affichage du message d'aide sur le onmouseover, sinon sur le onclick (par défaut) */
  protected $bOpenHelpOnMouseOver;

  /** liste des commandes de validation */
  protected $tabValidator;
  
  /** Indique si la valeur gérée est une valeur numérique*/
  protected $isNumeric;
  
  /** tableau de constante liée aux scripts js pour éviter de les charger n fois */
  protected $tabConstJs;

  /** = vrai si le controle est marqué comme obligatoire */
  protected $bStared;

  /** = vrai si le controle est représenté par un hidden */
  protected $bHidden;

  /** liste des controles du groupe */
  protected $tabCtrls;

  /** Message affiché en cas de contenu vide */
  protected $defaultMsg;

  /**
   *  Constructeur par défaut
   *
   * @param oForm Référence du formulaire associé
   * @param iMode Mode du controle de saisie : =0 modif, =1 lecture
   * @param name  Nom du controle de saisie
   * @param value Valeur par défaut du controle de saisie
   * @param label Etiquette texte associée au controle de saisie
   */
  function __construct($oForm, $iMode, $name, $value="", $label="")
  {
    parent::__construct($name, $label);

    $this->oForm        = $oForm;
    if (!is_null($this->oForm))
    $this->oForm->addCtrl($this);
    $this->iMode        = $iMode;
    $this->dataFields   = strtoupper($this->name);
    $this->value        = $value;
    $this->label        = $label;
    $this->tabValidator = array();
    $this->bDisabled    = false;
    $this->bReadOnly    = false;
    $this->oBlock       = null;

    $this->bWriteHiddenOnReadOnly = false;

    $this->labelBeforeCtrl = "";
    $this->labelAfterCtrl  = "";
    $this->labelHelp       = "";
    $this->bAllModeHelp    = false;
    $this->bStared         = false;
    $this->bOpenHelpOnMouseOver   = false;
    
    $this->isNumeric = false;
    $this->bHidden   = false;
    $this->tabCtrls  = array();
    
    $this->cssHelp = "helpMsg";
    
    $this->tabConstJs = array("TXT"       => ALK_ALKANET_ROOT_URL.ALK_ROOT_LIB."lib_formtxt.js",
                              "SELECT"    => ALK_ALKANET_ROOT_URL.ALK_ROOT_LIB."lib_formselect.js",
                              "CHECK"     => ALK_ALKANET_ROOT_URL.ALK_ROOT_LIB."lib_formcheck.js",
                              "DATE"      => ALK_ALKANET_ROOT_URL.ALK_ROOT_LIB."lib_formdate.js",
                              "NUMBER"    => ALK_ALKANET_ROOT_URL.ALK_ROOT_LIB."lib_formnumber.js",
                              "COLOR"     => ALK_ALKANET_ROOT_URL.ALK_ROOT_CLASSE."form/alkhtmlcolor.js",
                              "FORM"      => ALK_ALKANET_ROOT_URL.ALK_ROOT_LIB."lib_form.js",
                              "HELP"      => ALK_ALKANET_ROOT_URL.ALK_ROOT_CLASSE."form/alkhtmlctrl.js",
                              "CALENDAR"  => ALK_ALKANET_ROOT_URL.ALK_ROOT_CLASSE."form/alkhtmldate.js",
                              "SLIDER"    => ALK_ALKANET_ROOT_URL.ALK_ROOT_CLASSE."form/alkhtmlslider.js");
  }
  
  /**
   *  Retourne l'onglet de formulaire dans lequel est contenu le controle ou null sinon
   * 
   * @return AlkHtmlSheet ou null
   */
  public function getSheetOwner()
  {
    if ( !$this->isInBlock() ) return null;
    $oSheet = $this->oBlock;
    while ( !is_null($oSheet) && !$this->oForm->equals($oSheet) && !($bFound = $oSheet->isTypeOf(self::ALK_CLASS_HTMLSHEET)) ){
      $oSheet = $oSheet->getParent();
    } 
    if ( $bFound )
      return $oSheet;
    return null;
  }
  
  /**
   *  Ajoute une vérification sur un champ
   * 
   */
  public function isInBlock()
  {
    return !is_null($this->oBlock) && !is_null($this->oForm) && ( $this->oForm->equals($this->oBlock->getForm()) );
  }
  
  /**
   *  Affecte la propriété de validation "require" (affecte également l'attribut bStared)
   * 
   * @param bRequire    =true si champ obligatoire, =false sinon
   */
  public function setIsRequire($bRequire)
  {
    $this->tabValidator["require"] = $bRequire;

    // place une étoile après le label si obligatoire
    $this->bStared = ( $bRequire == true && $this->iMode=="0" );
  }
  
  /**
   *  Affecte la propriété bHidden : le code généré sera celui d'un hidden
   * 
   * @param bHidden    =true si champ hidden, =false sinon
   */
  public function setHidden($bHidden)
  {
    $this->bHidden = $bHidden;
  }
  
  /**
   *  Ajoute une vérification sur un champ
   * 
   * @param iModeVerif  Utilise les constantes ALK_VERIF_ et si champ obligatoire : ALK_IS_REQUIRED et si vérification des bornes non stricte : ALK_IS_NOTSTRICT
   * @param oMin        valeur minimale requise pour le controle (ou longueur min en fonction de strType)
   * @param oMax        valeur maximale requise pour le controle (ou longueur max en fonction de strType)
   * @param strExcept  valeurx qui font exceptions pour la vérification (séparées par "|" si plusieurs)
   * @param bStrict    true si la comparaison entre min et max est stricte, faux si éventuellement égale (=true par défaut)
   * @param iSheet     indice du layer htmlFormSheet (=-1 par défaut)
   */
  public function addValidator($iModeVerif, $oMin="", $oMax="", $strMsgErr="", $strExcept="", $bStrict=true, $iSheet="-1")
  {
    $tabVerif = array(
                  ALK_VERIF_MEMO        => "memo", 
                  ALK_VERIF_TEXT        => "text", 
                  ALK_VERIF_INT         => "int", 
                  ALK_VERIF_DATE10      => "date10",
                  ALK_VERIF_RADIO       => "radio",
                  ALK_VERIF_SELECT      => "select",
                  ALK_VERIF_MAIL        => "mail",
                  ALK_VERIF_CHECKGROUP  => "checkgroup",
                  ALK_VERIF_TEXTNUM     => "textnum",
                  ALK_VERIF_TEXTALPHA   => "textalpha",
                  ALK_VERIF_HEURE5      => "heure5",
                );
                
    $this->tabValidator["form"]    = $this->oForm->getName();
    
    $bRequire = ( ($iModeVerif & ALK_IS_REQUIRED) == ALK_IS_REQUIRED );
    if ( $bRequire )
      $iModeVerif = $iModeVerif - ALK_IS_REQUIRED;
      
    $bStrict = ( ($iModeVerif & ALK_IS_NOTSTRICT) == ALK_IS_NOTSTRICT ? false : $bStrict );
    if ( ($iModeVerif & ALK_IS_NOTSTRICT) == ALK_IS_NOTSTRICT )
      $iModeVerif = $iModeVerif - ALK_IS_NOTSTRICT;
          
    $strType  = $tabVerif[$iModeVerif];
    
    $this->setIsRequire($bRequire);
    $this->tabValidator["type"]    = $strType;
    $this->tabValidator["min"]     = $oMin;
    $this->tabValidator["max"]     = $oMax;
    $this->tabValidator["msgerr"]  = $strMsgErr;
    $this->tabValidator["except"]  = $strExcept;
    $this->tabValidator["strict"]  = $bStrict;
    $this->tabValidator["iSheet"]  = $iSheet;
    
    
    $this->_setValidator();
  }

  
  /**
   *  Mémorise les types de controle js nécessaire au formulaire
   *        Méthode appelée uniquement le HtmlPanel->addCtrl()
   *
   */
  protected function _setValidator()
  {
    if( array_key_exists("type", $this->tabValidator) ) {
      $bAddForm = true;
      switch( $this->tabValidator["type"] ) {
        case "memo": case "text": case "mail": case "textnum": case "textalpha":
          $this->AddScriptJs($this->tabConstJs["TXT"]); 
        break;
        case "heure5": case "date10": case "datetime16":
          $this->AddScriptJs($this->tabConstJs["DATE"]); 
        break;
        case "int":
          $this->AddScriptJs($this->tabConstJs["NUMBER"]);
        break;
        case "radio": break;
        case "check":
        case "checkgroup":
          $this->AddScriptJs($this->tabConstJs["CHECK"]);
        break;
        case "select":
          $this->AddScriptJs($this->tabConstJs["SELECT"]); 
        break;
        default : 
          $bAddForm = false; 
        break;
      }
      if ( $bAddForm )
        $this->AddScriptJs($this->tabConstJs["FORM"]); 
    }
    
    if( $this->isTypeOf(self::ALK_CLASS_HTMLCOLOR) ) {
      $this->addScriptJs($this->tabConstJs["COLOR"]);
    }

    
  }

  /**
   *  Met à jour iMode et éventuellement le label si ctrl obligatoire
   *
   * @param iMode  Nouveau Mode
   */
  public function setMode($iMode)
  {
    if ( $iMode==-1 ) {
      $this->bStared = false;
      $this->iMode = $iMode;
    }
    $aIMode = $this->iMode;
    $this->iMode = ($aIMode==1 ? $this->iMode : $iMode);
    if( count($this->tabValidator)>0 && $this->tabValidator["require"]==true ) {
      if( $aIMode==0 && $this->iMode==1 && $this->bStared )
        $this->bStared = false;
      elseif( $aIMode==1 && $this->iMode==0 )
        $this->bStared = true;
    }
  }

  /**
   *  Retourne le code html lié au bouton d'aide présent après le controle et le labelBeforeCtrl
   *        Le code js est automatiquement chargé par la classe htmlForm
   *        Retourne un tableau à 2 éléments indicé :
   *         btHelp    => code html du bouton
   *         layerHelp => code html du layer affichant l'aide
   *
   * @return Retourne un tableau
   */
  protected function getTabHtmlHelp()
  {
    $strBtHelpAction = "";
    $strLayerHelp = "";
    if( $this->labelHelp != "" ) {
      $this->addScriptJs($this->tabConstJs["HELP"]);
      $iWidth = ( $this->oBlock != null ? $this->oBlock->getProperty("iWidthCtrl") : 400 );
      $strBtHelpAction = 'HelpShowHideLayer(\''.$this->name.'\', \''.$iWidth.'\')';
      $strLayerHelp = '<div id="help_'.$this->name.'" class="'.$this->cssHelp.'">'.$this->labelHelp.'</div>';
    }
    return array("btHelpAction" => $strBtHelpAction, 
                 "layerHelp"    => $strLayerHelp);
  }

  /**
   *  Genere puis retourne tous les controles de saisie
   *
   * @return Retourne un string
   */
  protected function getHtmlValidator()
  {
    if( !array_key_exists("type", $this->tabValidator) ) return "";
    
    $bWriteValidator = ( !is_null($this->oForm) ? $this->oForm->getWriteJsInner() : false ); 
    
    $this->tabValidator["sheetManager"] = $this->tabValidator['form'];
    if( array_key_exists("iSheet", $this->tabValidator) && $this->tabValidator["iSheet"]=="-1"){
      $oSheetOwner = $this->getSheetOwner();
      if (!is_null($oSheetOwner)){
        $this->tabValidator["iSheet"] = $oSheetOwner->getGUID();
        $this->tabValidator["sheetManager"] = $oSheetOwner->getParent()->getGUID();
      }
    }

    $strHtml = "";
    if( $this->bMultiLanguage == true ) {
      foreach($this->tabLangue as $key=>$tabLg) {
        $strName = $this->name.$tabLg["bdd"];
        $strLabel = html_entity_decode($this->label);
        // suppression des tags html
        //$strLabel = mb_ereg_replace("", "", $strLabel);
        $strLabel = mb_ereg_replace('"', '', $strLabel);

        $strHtml .= ' AlkAddCtrl("'.$this->tabValidator['form'].'"'.
          ', "'.$strName.'"'.
          ', "'.$this->tabValidator['type'].'"'.
          ', '.($this->tabValidator['require']==true ? 'true' : 'false' ).
          ', "'.$strLabel.'"'.
          ', "'.$this->tabValidator['min'].'"'.
          ', "'.$this->tabValidator['max'].'"'.
          ', "'.html_entity_decode($this->tabValidator['msgerr']).'"'.
          ', "'.html_entity_decode($this->tabValidator['except']).'"'.
          ', '.($this->tabValidator['strict']==true ? 'true' : 'false' ).
          ', "'.$this->tabValidator["iSheet"].'"'.
          ', "'.$this->tabValidator["sheetManager"].'"'.
          '); ';
        if( $this->bShowMultiLanguage == false ) 
          break;
      }
    } else {
      $strLabel = html_entity_decode($this->label);
      // suppression des tags html
      //$strLabel = mb_ereg_replace("", "", $strLabel);
      $strLabel = mb_ereg_replace('"', '', $strLabel);

      $strHtml .= ' AlkAddCtrl("'.$this->tabValidator['form'].'"'.
        ', "'.$this->name.'"'.
        ', "'.$this->tabValidator['type'].'"'.
        ', '.($this->tabValidator['require']==true ? "true" : "false").
        ', "'.$strLabel.'"'.
        ', "'.$this->tabValidator['min'].'"'.
        ', "'.$this->tabValidator['max'].'"'.
        ', "'.html_entity_decode($this->tabValidator['msgerr']).'"'.
        ', "'.html_entity_decode($this->tabValidator['except']).'"'.
        ', '.($this->tabValidator['strict']==true ? "true" : "false").
        ', "'.$this->tabValidator["iSheet"].'"'.
        ', "'.$this->tabValidator["sheetManager"].'"'.
        ');';
    }
 
    if( !$bWriteValidator ) {
      $this->addJs($strHtml);
      return "";
    }
 
    return ( $strHtml != "" ? '<script type="text/javascript">'.$strHtml.'</script>' : '' );
  }

  /**
   *  Génère le code des liens ajoutés au controle
   *
   * @return string html
   */
  protected function getHtmlLinks()
  {
    $strHtml = "";
    foreach ($this->tabHtmlLink as $oLink){
      $strHtml .= '&nbsp;'.$oLink->getHtml();
    }
    return $strHtml;
  }

  /**
   *  Génère le code des controles ajoutés au groupe de controles représenté par l'objet
   *
   * @param oCtrl  AlkHtmlCtrl   Controle à ajouter au groupe
   * @param strSeparator         Séparateur entre ce controle et les autres controles du groupe (ceux qui précedent)
   * @param strSeparator         Séparateur entre le label de ce controle et son getHtml
   * @return string html
   */
  public function addCtrl(AlkHtmlCtrl &$oCtrl, $strSeparator="&nbsp;&nbsp;", $strSeparatorLabel="&nbsp;")
  {
    $iCtrl = count($this->tabCtrls);
    $this->tabCtrls[$iCtrl]["oCtrl"] = $oCtrl;
    $this->tabCtrls[$iCtrl]["separator"] = $strSeparator;
    $this->tabCtrls[$iCtrl]["separator_label"] = $strSeparatorLabel;
    if ( !is_null($this->oBlock) ){
      $oCtrl->setBlock($this->oBlock, true);
    }
  }

  /**
   *  Génère le code des controles ajoutés au groupe de controles représenté par l'objet
   *
   * @return string html
   */
  protected function getHtmlCtrls()
  {
    $strHtml = "";
    foreach ($this->tabCtrls as $tabCtrl){
      $oCtrl = $tabCtrl["oCtrl"];
      $strSeparator = $tabCtrl["separator"];
      $strSeparatorLabel = $tabCtrl["separator_label"];
      $strLabel = $oCtrl->getLabel();
      $strHtml .= $strSeparator.
        ( !$oCtrl->isTypeOf(self::ALK_CLASS_HTMLBUTTON) 
          ? ( $strLabel!="" 
              ? '<label for="'.$oCtrl->getGUID().'"' .
                ' id="label_'.$oCtrl->getGUID().'"' .
                ' class="label'.( $oCtrl->isRequired() ? ' required' : '').'">'.
                $strLabel.'</label>' 
              : "") 
          : "").
        $strSeparatorLabel.$oCtrl->getHtml();
    }
    return $strHtml;
  }
  
  /**
   *  Méthode abstraite non implémentée
   *        qui se charge de retourner le code html du ctrl de saisie seul
   *        Méthode appelée par GetHtml qui se chargera d'ajouter le code JS
   *        les textes before et after, les drapeaux, le bouton d'aide
   */
  abstract protected function GetCtrlHtml($key=0, $tabLg=array());

  /**
   *  Retourne le code html du ctrl :
   *        si iMode = 0 : dans un tableau |textBefore |Ctrl|textAfter|imgFlag|btHelp|
   *        si iMode = 1 : retourne le
   *
   * @return Retourne une chaîne vide
   */
  public function getHtml()
  {
    $strHtml = $this->getTabHtmlCtrl();
    return (is_array($strHtml) ? implode("", $strHtml) : $strHtml);
  }
  
  /**
   * retourne le code html des controles sous forme de tableau
   * @return array()
   */
  public function getTabHtmlCtrl()
  {
    if ( !$this->isTypeOf(self::ALK_CLASS_HTMLHIDDEN) ){
      if ( $this->isFieldNotView() ){
        $strHtml = '<input type="hidden" id="'.$this->guid.'" name="'.$this->name.'" value="'.ALK_FIELD_NOT_VIEW.'"/>'; 
        return  array($strHtml);
      }
    
      if ( $this->isHidden() ){
        $strHidden = "";
        if ( is_array($this->value) ){
          foreach ( $this->value as $index=>$value ){
            $strHidden .= '<input type="hidden" id="'.$this->guid.'_'.$index.'" name="'.$this->name.'['.$index.']" value="'.$value.'"/>'; 
          }
        }
        else {
          $strHidden .= '<input type="hidden" id="'.$this->guid.'" name="'.$this->name.'" value="'.$this->value.'"/>';
        }
        return array($strHidden); 
      }
    }
    $strHidden = ""; 
    if ( is_array($this->value) ){
      foreach ( $this->value as $index=>$value ) {
        $strHidden .= '<input type="hidden" id="'.$this->guid.'_'.$index.'" name="'.$this->name.'['.$index.']" value="'.$value.'"/>'; 
      }
    }
    else {
      $strHidden .= '<input type="hidden" id="'.$this->guid.'" name="'.$this->name.'" value="'.$this->value.'"/>';
    }
    $tabRes = $tabHtml = $tabRes2 = array();
    if( $this->bMultiLanguage == false ){
      $tabRes[0] = $this->GetCtrlHtml();
    }
    else {
      foreach($this->tabLangue as $key => $tabLg) {
        $strHtml = $this->GetCtrlHtml($key, $tabLg);  
        if ($this->isTypeOf(self::ALK_CLASS_HTMLTEXT) && $this->bEditor){
          $tabHtml = explode("@@@", $strHtml); 
          $tabRes[$tabLg["rep"]] = $tabHtml[0]; // on ne garde uniquement que le lien drapeau qui permet d'éditer
          $tabRes2[] = (isset($tabHtml[1]) ? $tabHtml[1] : ""); // tabRes2 récupère tous les autres champs de l'éditeur
        }else {
          $tabRes[$tabLg["rep"]] = $strHtml;
        }
      }
      
    if (!empty($tabRes2)){
      $tabRes[$tabLg["rep"]].= implode(" ", $tabRes2); // les autres champs de chaque langue sont concaténés au dernier controle langue
    }
  }
  
  $strCtrlHtml = ($this->bMultiLanguage == false ? $tabRes[0] : $tabRes[$this->tabLangue[0]["rep"]]);
    
  if( $strCtrlHtml=="" || 
      ($this->iMode==1 && ((is_array($this->value) && empty($this->value)) || 
      (is_string($this->value) && $this->value==="") || 
      is_null($this->value))) )
    $strCtrlHtml = $this->defaultMsg;
  $tabHelp = $this->GetTabHtmlHelp();
  $bHelp   = ( $this->iMode == "0" || $this->bAllModeHelp );
  
  if( $this->strTemplateFile == "" ) {
    $strHtml = "";
  
    $bTable = !( $tabHelp["btHelpAction"] == "" && $this->labelBeforeCtrl == "" && $this->labelAfterCtrl == "" );
   
    $strHtml = 
      ( $this->labelBeforeCtrl != ""
    ? '<span id="'.$this->guid.'_labelBefore" class="'.$this->cssFormLabel.'">'.$this->labelBeforeCtrl.'</span>'
    : '' ).
  
    $strCtrlHtml.
    ( $this->bWriteHiddenOnReadOnly && ($this->iMode==1 || $this->bReadOnly)  ? $strHidden : "" ).
    
    ( $this->labelAfterCtrl != ""
      ? '<span id="'.$this->guid.'_labelAfter" class="'.$this->cssText.'">'.$this->labelAfterCtrl.'</span>'
      : '' ).
  
    ( $this->iMode == "0" ? $this->getHtmlLinks() : "").
  
    ( $bHelp && $tabHelp["btHelpAction"] != "" 
      ? '<a class="helpLien" href="#helpMsg_'.$this->guid.'"'.
        ( !$this->bOpenHelpOnMouseOver 
          ? ' onclick="'.$tabHelp["btHelpAction"].'"'
          : ' onmouseover="'.$tabHelp["btHelpAction"].'" onmouseout="'.$tabHelp["btHelpAction"].'"' ) .
        ( !$this->bOpenHelpOnMouseOver 
          ? ' title="'._f("Afficher l'aide pour %s", $this->label).'"'
          : '') .
        '>'.
        '<img border="0" class="helpButton alkimgpic alkimgpicaide" src="'.$this->urlBaseImg.'transp.gif"/>'.
        ( $bHelp && $tabHelp["layerHelp"] != "" 
          ? $tabHelp["layerHelp"] 
          : '' ).  
        '</a>'
      : '' ).
    
    $this->getHtmlCtrls().
    $this->getHtmlValidator();
    
  } 
  else {
    $this->oTemplate->assign("labelBefore",  $this->labelBeforeCtrl);
    $this->oTemplate->assign("labelAfter",  $this->labelAfterCtrl);

    $strName = mb_ereg_replace("\[\]", "", $this->name);
    $this->oTemplate->assign($strName, implode("", $tabRes));

    foreach($this->tabHtmlLink as $oLink) {
      $this->oTemplate->assign($oLink->getName(), $oLink->getHtml());
    }

    foreach($this->tabCtrls as $tabCtrl) {
      $oCtrl = $tabCtrl["oCtrl"];
      $strName = mb_ereg_replace("\[\]", "", $oCtrl->getName());
      $this->oTemplate->assign($strName, $oCtrl->getHtml());
      $this->oTemplate->assign($strName."_separator", $tabCtrl["separator"]);
      $this->oTemplate->assign($strName."_separatorLabel", $tabCtrl["separator_label"]);
    }

    if( $bHelp ) {
      if( array_key_exists("btHelp", $tabHelp) && $tabHelp["btHelp"] != "" ) {
        $this->oTemplate->assign("btHelp", $tabHelp["btHelp"]);
      }
      if( array_key_exists("layerHelp", $tabHelp) && $tabHelp["layerHelp"] != "" ) {
        $this->oTemplate->assign("layerHelp", $tabHelp["layerHelp"]);
      }
    }

    $strHtml = $this->oTemplate->fetch($this->strTemplateFile).
      $this->getHtmlValidator();
    
  } 
  
  if ($this->bMultiLanguage==false || $this->strTemplateFile != "") {
    $tabRes[0] = $strHtml;
  }else {
    $tabRes[$this->tabLangue[0]["rep"]] = $strHtml;
  } 
     
  return $tabRes;
  }
  
  /**
   *  indique si le controle est un controle de type hidden
   * @return boolean  
   */
  public function isFieldNotView()
  {
    return $this->iMode==-1; 
  } 
  
  /**
   *  indique si le controle est un controle de type hidden
   * @return boolean  
   */
  public function isRequired()
  {
    return ( $this->label!="" && $this->bStared ); 
  } 
  
  /**
   *  indique si le controle est un controle de type hidden
   * @return boolean  
   */
  public function isHidden()
  {
    return $this->bHidden; 
  } 
  
  /**
   *  indique si le controle necessite un formulaire de type upload 
   * @return boolean  
   */
  public function isUpload()
  {
    return false; 
  } 

  /**
   *  initialise le panel à partir des requests
   */
  protected function initializeByRequest()
  {
    $this->value = AlkRequest::Request($this->name, $this->dataFields, $this->value, ($this->isNumeric ? "is_numeric" : ""));
  }
  
  /**
   *  initialise le panel à partir d'un datarow
   */
  protected function initializeByDatarow()
  {
    if ( !is_null($this->oData) && $this->dataFields!="" ){
      $this->value = $this->oData->getValueName($this->dataFields, $this->value);
    }
  }
  
  /* --------------------------------
   * accesseurs
   * --------------------------------*/
  
  /**
   *  change le identifiant/nom du panel
   * @param name            Nouvel identifiant
   * @param bTriggerError   boolean : Leve une erreur si non unicité de nouveau nom
   * @return boolean : changement effectué
   */
  public function setName($name, $bTriggerError=false)
  {
    if (parent::setName($name, $bTriggerError)){
      $this->dataFields = strtoupper(preg_replace("!\[[^]]*\]!", "", $this->getName()));
      $this->bIsInit = false;
    }
  }  
  /**
   *  détermine si la valeur du controle est numérique
   * @param isNumeric  Nouvelle valeur
   * @return boolean : changement effectué
   */
  public function setIsNumeric($isNumeric)
  {
    $this->isNumeric = $isNumeric;
    return true; 
  }  
  
  /**
   *  retourne le booléen indiquant que la valeur du controle est numérique
   * @return boolean  
   */
  public function getIsNumeric()
  {
    return $this->isNumeric; 
  } 
  
  /**
   *  retourne le label du controle
   * @return string  
   */
  public function getLabel()
  {
    $strLabel = $this->label;
    if ( $strLabel!="" && $this->bStared )
      $strLabel .= '*';
    return $strLabel; 
  } 
  
  /**
   *  retourne le label du controle
   * @return string  
   */
  public function getLabelFor()
  {
    if ( $this->label=="" )
      return '';
    return '<label for="'.$this->getGUID().'"' .
      ' id="label_'.$this->getGUID().'"' .
      ' class="label'.( $this->isRequired() ? ' required' : '' ).'">'.
      $this->getLabel().
      '</label>'; 
  } 
  
  /**
   *  Affecte la valeur du block
   * 
   * @param oBlock        AlkHtmlBlock   Bloc d'appartenance du controle
   * @param bFromBlock    boolean        Indique si la méthode est appelée depuis la méthode addCtrl du bloc
   */
  public function setBlock(AlkHtmlBlock $oBlock, $bFromBlock=false)
  {
    if ( !$bFromBlock ){
      $oBlock->addCtrl($this);
    }
    else {
      $this->oBlock = $oBlock;
    }   
    if ( is_null($this->oForm) )
      $this->oForm = $this->oBlock->getForm();
       
    foreach ($this->tabCtrls as $tabCtrl){
      $tabCtrl["oCtrl"]->setBlock($oBlock, $bFromBlock);
    }
  }

  /**
   *  Modifie la valeur de $this->labelBeforeCtrl
   */
  public function setLabelBefore($strTxt)
  {
    $this->labelBeforeCtrl = $strTxt;
  }

  /**
   *  Modifie la valeur de $this->labelAfterCtrl
   */
  public function setLabelAfter($strTxt)
  {
    $this->labelAfterCtrl = $strTxt;
  }
  
  /**
   * Définit le message d'aide à associer au controle
   * @param strHelp               Message d'aide
   * @param bAllModeHelp          Affichage de l'aide sur tous les modes d'affichage du controle si true, sinon seulement en écriture (par défaut)
   * @param bOpenHelpOnMouseOver  Si true, affichage du message d'aide sur le onmouseover, sinon sur le onclick (par défaut) 
   * 
   */
  public function setLabelHelp($strHelp, $bAllModeHelp=false, $bOpenHelpOnMouseOver=false)
  {
    $this->labelHelp = $strHelp;
    $this->bAllModeHelp = $bAllModeHelp;
    $this->bOpenHelpOnMouseOver = $bOpenHelpOnMouseOver;
  }
  /**
   *  Fixe l'état d'écriture de la valeur en mode formulaire dans le cas lecture seul
   * @param bWrite true par défaut pour écrire la valeur en hidden en lecture seul, faux aucune écrire formulaire
   */
  public function setWriteHiddenOnReadOnly($bWrite=true)
  {
    $this->bWriteHiddenOnReadOnly = $bWrite;
  }
  
  /**
   *  Fixe l'état de lecture uniquement sur le controle
   * @param bReadOnly  true par défaut pour bloquer la saisie, false sinon
   */
  public function setReadOnly($bReadOnly=true)
  {
    $this->bReadOnly = $bReadOnly;
  }

  /**
   * Fixe l'état du controle : actif ou désactivé
   * @param bDisabled  true par désactiver, false sinon
   */
  public function setDisabled($bDisabled=true)
  {
    $this->bDisabled = $bDisabled;
  }

  public function __toString()
  {
    return "[TYPE] => ".get_class($this)."\n" .
           "[GUID] => ".$this->guid."\n" .
           "[FORM] => ".(!is_null($this->oForm) ? $this->oForm->getGUID() : "!NONE!")."\n" .
           "[FORM] => ".(!is_null($this->oBlock) ? $this->oBlock->getGUID() : "!NONE!")."\n";
  }
}

?>