<?php
/*licence/ 

Module écrit, supporté par la société Alkante SAS <alkante@alkante.com>

Nom du module : Alkanet::Class::Pattern
Module fournissant les classes de base Alkanet.
Ce module appartient au framework Alkanet.

Ce logiciel est régi par la licence CeCILL-C soumise au droit français et
respectant les principes de diffusion des logiciels libres. Vous pouvez
utiliser, modifier et/ou redistribuer ce programme sous les conditions
de la licence CeCILL-C telle que diffusée par le CEA, le CNRS et l'INRIA
sur le site http://www.cecill.info.

En contrepartie de l'accessibilité au code source et des droits de copie,
de modification et de redistribution accordés par cette licence, il n'est
offert aux utilisateurs qu'une garantie limitée. Pour les mêmes raisons,
seule une responsabilité restreinte pèse sur l'auteur du programme, le
titulaire des droits patrimoniaux et les concédants successifs.

A cet égard l'attention de l'utilisateur est attirée sur les risques
associés au chargement, à l'utilisation, à la modification et/ou au
développement et à la reproduction du logiciel par l'utilisateur étant
donné sa spécificité de logiciel libre, qui peut le rendre complexe à
manipuler et qui le réserve donc à des développeurs et des professionnels
avertis possédant des connaissances informatiques approfondies. Les
utilisateurs sont donc invités à charger et tester l'adéquation du
logiciel à leurs besoins dans des conditions permettant d'assurer la
sécurité de leurs systèmes et ou de leurs données et, plus généralement,
à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.

Le fait que vous puissiez accéder à cet en-tête signifie que vous avez
pris connaissance de la licence CeCILL-C, et que vous en avez accepté les
termes.

/licence*/

require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CLASSE."pattern/alkobject.class.php");

/**
 * @package Alkanet_Class_Pattern

 * @class AlkDs
 * @brief Classe de base pour la connexion à une base de données (Dataset)
 */
abstract class AlkDs extends AlkObject implements Iterator
{
  /** Objet ou ressource pointant vers le flux de données */
  protected $oRes;
  
  /** datarow(dr) courant */
  protected $curDr;
  
	/** Indice du datarow(dr) courant */
  protected $iCurDr;

	/** Nombre de datarow (dr) dans le dataset */
  protected $iCountDr;

	/** Nombre total de datarow(dr) dans le dataset paginé */
  protected $iCountTotDr;
  
  /** requete exécutée */
  protected $strSql;
  
  /** Indice début de pagination =0 par défaut */
  protected $iFirst;
  
  /** Indice fin de pagination =-1 par défaut */
  protected $iLast;
  
	/** Booléen indiquant si l'on se trouve à la fin du dataset */
  protected $bEof;

	/** Objet contenant les données du dataset */
  protected $tabDr;

	/** Message de la dernière erreur rencontrée */
	protected $lastErr;
	
	/** true pour afficher les erreurs, false pour les capturer */
	protected $bErr;

	/** tableau d'association entre les types du sgbd et les types sgbd alkanet */
	protected $tabTypeAssoc;
	
  /** tableau des types de champ */
  protected $tabType;
  /** tableau des noms de champ */
  protected $tabFields;
  
  /** encodage utilisé */
  protected $strDbEncoding;

  /** true si le cache interne contient tous les résultats de la requête, false sinon */
  protected $bFetchAll;
  
  /** true pour placer le résultat de la requête dans le cache interne, false sinon */
  protected $bUseInternalCache;
  /** true pour placer le résultat de la requête dans le cache externe, false sinon */
  protected $bUseCache;
  /** délai d'expiration du cache en seconde */
  protected $iCacheExpire;
  /** nom du cache utilisé */
  protected $strCacheName;

  /** 
   * Nom du champ utilisé comme clé pour le tableau placé en mémoire
   * Par défaut, = chaine vide pour utiliser le incrément auto
   * Sinon, $this->tabDr sera un tableau associatif utilisant la valeur du champ identifié par cet attribut  
   */
  protected $strCacheFieldIndex;
  /** tableau d'index permettant d'associer la clé $strCacheFieldIndex à l'index de l'enregistrement */
  protected $tabDrIndex;
  
  /**
   * Constructeur par défaut de la classe
   *
   * @param conn          ressource de la connexion en cours
   * @param idFirst       indice de début pour la pagination (=0 par défaut)
   * @param idLast        indice de fin pour la pagination (=-1 par défaut)
   * @param bErr          true si gestion des erreurs, false pour passer sous silence l'erreur
   * @param strDbEncoding encodage du client sgbd
   */
  public function __construct($conn, $strSql, $idFirst=0, $idLast=-1, $bErr=true, $strDbEncoding=ALK_SGBD_ENCODING)
  {		
    parent::__construct();

    $this->oRes = false;
    $this->conn = $conn;
    $this->bFetchAll = false;
    $this->strSql = $strSql;
    $this->iFirst = $idFirst;
    $this->iLast = $idLast;
    $this->bErr = $bErr;
    $this->curDr = null;
    $this->iCurDr = -1;
    $this->iCountDr = 0;
    $this->iCountTotDr = 0;
    $this->bEof = true;
    $this->tabDr = array();
    $this->tabDrIndex = array();
    $this->lastErr = "";
    $this->tabFields = array();
    $this->tabTypeAssoc = array();
    $this->tabType = array();
    $this->strDbEncoding = $strDbEncoding;
    $this->bUseCache = false;
    $this->bUseInternalCache = false;
    $this->iCacheExpire = 0;
    $this->strCacheName = "alk";
    $this->strCacheFieldIndex = "";
  }

  public function __destruct()
  {
    $this->tabDr = null;
    $this->tabDrIndex = null;
  }

  /**
   * Retourne l'encodage client de la connexion SGBD
   * L'encodage est au format normé : UTF-8 ou ISO-8859-1
   * @return string
   */
  public function getEncoding()
  {
    return $this->strDbEncoding;
  }

	/**
   * Fixe la nouvelle erreur rencontrée
   *
   * @param err false ou tableau associatif contenant l'erreur
   */
	protected function _setError($err)
  {
    $this->lastErr = $err;
  }
  
  /**
   * Fixe le nom du champ à utiliser comme clé pour enregistrer les informations en mémoire
   * $this->tabDr devient un tableau associatif utilisant la valeur du champ identifié par cet attribut 
   * @param strCacheFieldIndex  nom du champ à utiliser
   */
  public function setCacheFieldIndex($strCacheFieldIndex="")
  {
    $this->strCacheFieldIndex = $strCacheFieldIndex;
  }
  
  /**
   * Retourne la valeur de l'attribut strCacheFieldIndex 
   * correspondant au nom du champ utlisé comme clé pour le tableau $this->tabDr
   */
  public function getCacheFieldIndex()
  {
    return $this->strCacheFieldIndex;
  }

  /**
   * Active ou désactive la mise en cache interne et externe avec les paramètres de durée d'expiration et de préfixe pour le cache externe (memcache).
   * Le cache interne est géré par la classe AlkDs
   * L'appel a fetchAll fixe bUseInternalCache à true
   * Le préfixe est utilisé pour le flush du cache afin de ne pas tout effacer.
   * @param bUseInternalCache  false par défaut, true pour activer la mise en cache interne le temps d'exécution du script. Si iExpire>0, ce paramètre est automatiquement mis à true
   * @param iExpire            =0 par défaut, >0 pour mémoriser en cache externe le résutat de la requête avec un délai d'expiration de iExpire secondes
   * @param strCacheName       =alkanet par défaut, permet de regrouper les éléments cachés à l'aide de ce nom afin de gérer en live, la libération du cache 
   */
  public function setCache($bUseInternalCache=false, $iExpire=0, $strCacheName="alkanet")
  {
    $this->bUseCache = ( $iExpire != 0 );
    $this->bUseInternalCache = ( $this->bUseCache || $this->bUseInternalCache );
    $this->iCacheExpire = $iExpire;
    $this->strCacheName = $strCacheName;
  }

  /**
   * accesseur pour consulter l'attribut bUseCache
   * @param bUseCache  true par défaut pour activer le cache, false pour désactiver l'utilisation du cache
   */
  public function isUseCache()
  {
    return $this->bUseCache;
  }
  
  /**
   * Retourne la dernière erreur rencontrée
   *
   * @return Retourne une chaine, vide si pas d'erreur, texte sinon
   */
	public function getLastError()
  {
    if( !is_string($this->lastErr) ) 
      $this->lastErr = "";
    return $this->lastErr;
  }

  /**
   * Méthode qui lit et mémorise tout le flux résultat de la requête
   * Cette méthode doit suivre l'appel à initDataSet() ou est appelée automatiquement par la méthode setTree()
   */
  abstract public function fetchAll();
  
  /**
   * Retourne le datarow (dr) correspondant à l'indice courant ($iCurDr)
   *
   * @return Retourne un dataRow si ok, false sinon
   */
  abstract public function getRow();

  /**
   * Retourne le datarow (dr) correspondant à l'indice courant ($iCurDr) puis passe au suivant
   * Retourne false si EOF est atteint
   *
   * @return Retourne un dataRow si ok, false sinon
   */
  abstract public function getRowIter();

  /**
   *  Retourne un tableau associatif du datarow correspondant à l'indice courant ($iCurDr) puis passe au suivant
   *  Les clés correspondantes au nom des champs avec la casse Majuscule
   *  Retourne false si EOF est atteint
   *
   * @return array : Retourne un tableau associatif si ok, false sinon
   */
  abstract public function getAssocIter();
  
  /**
   * Retourne la valeur du champ $strFieldName correpondant à la valeur de clé primaire identifiée par $strKey
   * en fonction de la valeur de $this->strCacheFieldIndex
   * si strCacheFieldIndex=chaine vide, $strKey correspond à l'indice de l'enregistrement, 
   *    sinon à la valeur de du champ donné par strCacheFieldIndex
   * Cette méthode n'est possible qu'en mode bFetchAll=true
   * @param strKey        valeur de la clé
   * @param strFieldName  nom du champ dont on souhaite avoir la valeur
   * @return string
   */
  public function getValueByKey($strKey, $strFieldName)
  {
    if( !$this->bFetchAll || empty($this->tabDrIndex) ) {
      trigger_error("AlkdsMysql - fonction ".__FUNCTION__." non autorisée en mode FetchItem.", E_USER_ERROR);
    }
    
    $iIndex = 
      ( $this->strCacheFieldIndex=="" && 
        is_numeric($strKey) && 
        $strKey>=0 && 
        $strKey<$this->iCountDr
        ? $strKey
        : ( $this->strCacheFieldIndex != "" && 
            isset($this->tabDrIndex[$this->strCacheFieldIndex]) &&
            is_numeric($this->tabDrIndex[$this->strCacheFieldIndex]) && 
            $this->tabDrIndex[$this->strCacheFieldIndex]>=0 && 
            $this->tabDrIndex[$this->strCacheFieldIndex]<$this->iCountDr
            ? $this->tabDrIndex[$this->strCacheFieldIndex]
            : -1 ));
    
    return ( $iIndex!=-1 && isset($this->tabDr[$iIndex][$strFieldName]) 
             ? $this->tabDr[$iIndex][$strFieldName] 
             : "" );
  }
  
  /**
   * Retourne le datarow (dr) positionné à l'indice id
   * Retourne false si l'indice n'est pas satisfaisant
   * n'agit ni sur bEOF, ni sur le pointeur 
   *
   * @param id Indice du dataRow
   * @return Retourne un dataRow si ok, false sinon
   */
  abstract public function getRowAt($id);

  /**
   * Place le pointeur du dataset sur le premier datarow
   * Met à jour l'indice courant du dataset
   */
  abstract public function moveFirst();

  /**
   * Place le pointeur du dataset sur le datarow précédent
   * Met à jour l'indice courant du dataset
   */
	abstract public function movePrev();

  /**
   * Place le pointeur du dataset sur le datarow suivant
   * Met à jour l'indice courant du dataset
   */
  abstract public function moveNext();

  /**
   * Place le pointeur du dataset sur le dernier datarow
   * Met à jour l'indice courant du dataset
   */
  abstract public function moveLast();

  /**
   * Place le pointeur du dataset sur le (iCurDr+iNb) ième datarow
   * Met à jour l'indice courant du dataset
   *
   * @param iNb Décalage positive ou négatif par rapport au dataRow courant
   */
  abstract public function move($iNb);

  /**
   *  Détruit l'objet dataset()
   */
  /**
   *  Détruit l'objet dataset()
   */
  public function close()
  {
    $this->oRes = false;
    $this->strSql = "";
    $this->tabDr = array();
    $this->tabDrIndex = array();
    $this->tabType = array();
    $this->tabFields = array();
    $this->iCurDr = -1;
    $this->iCountDr = -1;
    $this->iCountTotDr = -1;
    $this->bFetchAll = false;
    $this->iFirst = 0;
    $this->iLast = -1;
    $this->bErr = false;
    $this->bEof = true;
    $this->bUseCache = false;
    $this->iCacheExpire = 0;
    $this->strCacheName = "alk";
    $this->strCacheFieldIndex = "";
  }

  /**
   * Tri le dataSet après lecture dans la source donnée
   * Effectue un tri à plusieurs niveaux
   * Nécessite de connaite les noms de champ : ID et ID_PERE
   * le dataset doit être déjà trié par niv puis par rang
   *
   * @param strFieldId  Nom du champ ID dans la requeête
   * @param strFieldIdp Nom du champ ID_PERE dans la requête
   */
  abstract public function setTree($strFieldId, $strFieldIdp);

  /**
  * mémorise le contenu du dataSet en mémoire pour permettre la mise en cache par la suite
  * 
  */
  abstract protected function saveDataRow($tabDr);
  
  /**
   * Retourne le type du numéro du champ
   * Retourne 0 si chaine, 1 si nombre, 2 si date
   *
   * @return Retourne un nombre
   */
  public function getFieldType($iFieldPosition)
  {
    if( $iFieldPosition >= 0 && $iFieldPosition < count($this->tabType) )
      return $this->tabType[$iFieldPosition];
    return "";
  }

  /**
  * Retourne le type du numéro du champ
  * Retourne 0 si chaine, 1 si nombre, 2 si date
  *
  * @return Retourne un nombre
  */
  public function getFieldName($iFieldPosition)
  {
    if( $iFieldPosition >= 0 && $iFieldPosition < count($this->tabFields) )
      return $this->tabFields[$iFieldPosition];
    return "";
  }
  
  /**
   * Retourne le tableau des champs contenus dans la requête résultat
   */
  public function getFields()
  {
    return $this->tabFields;
  }
  
  /**
   * Retourne le nombre d'élément dans le datarow
   * @return int
   */
  public function getCountDr()
  {
	  return $this->iCountDr;
  }

  /**
   * Retourne le nombre d'élément total du datarow lorsqu'il est paginé
   * @return int
   */
  public function getCountTotDr()
  {
    return ( $this->iCountTotDr == -1 ? $this->iCountDr : $this->iCountTotDr );
  }

  /**
   *  Retourne l'indice courant de lecture dans le dataset (NB: =iCurDr, =indice du dr suivant(faire -1))
   * @return int
   */
  public function getCurrentIndex()
  {
    return $this->iCurDr;
  }

  /**
   *  Retourne le booléen indiquant que la lecture des résultats est terminée
   * @return boolean
   */
  public function isEndOfFile()
  {
    return $this->bEof || ($this->iCurDr==$this->iCountDr);
  }

  /**
  * Retourne le DS résultant en JSON
  *
  * @param strTableName   Le nom de la table dans la base de données
  * @return string
  */
  public function getJson($strTableName="")
  {
    return ( $strTableName==""
             ? json_encode($this->tabDr)
             :  "{".$strTableName.":".json_encode( $this->tabDr)."}" );
  }
  
  /**
   * obsolete
   * Retourne le DS résultant en JSON
   * 
   * @param strTableName Le nom de la table dans la base de données
   * @return string
   */
  public function getJsonFromDs($strTableName="alk")
  {
    $this->getJson($strTableName);
  }
  
  /**
   * @see Iterator::current()
   * 
   * A foreach loop will call Iterator functions in the following order : 
   * rewind -> valid -> current -> key -> next -> valid -> current -> key -> next ... valid
   * End of loop occurs when valid returns false
   * 
   * @return AlkDr the current element
   */
  public function current ()
  {
    // return the current element
    return $this->curDr;
  }
  
  /**
   * @see Iterator::next()
   */
  public function next ()
  {
    // set position to the next element
    // nothing to do, Iterator::valid() calls AlkDs::getRowIter which already calls AlkDs::moveNext()
  }
  
  /**
   * @see Iterator::key()
   * 
   * @return int index key of the current element
   */
  public function key ()
  {
    // return key index of the the current position
    return $this->iCurDr - 1; // zero-based index
  }
  
  /**
   * @see Iterator::valid()
   * 
   * @return boolean true if current element is valid, false when the loop should terminate
   */
  public function valid ()
  {
    // tells if the current element is valid (used to terminate the loop)
    $this->curDr = $this->getRowIter();
    return $this->curDr != false;
  }
  
  /**
   * @see Iterator::rewind()
   */
  public function rewind ()
  {
    // go back to the first position
    $this->moveFirst();
  }
  
}

?>