<?php
/*licence/ 

Module écrit, supporté par la société Alkante SAS <alkante@alkante.com>

Nom du module : Alkanet::Module::Annu
Module annuaire Alkanet.
Ce module appartient au framework Alkanet.

Ce logiciel est régi par la licence CeCILL-C soumise au droit français et
respectant les principes de diffusion des logiciels libres. Vous pouvez
utiliser, modifier et/ou redistribuer ce programme sous les conditions
de la licence CeCILL-C telle que diffusée par le CEA, le CNRS et l'INRIA
sur le site http://www.cecill.info.

En contrepartie de l'accessibilité au code source et des droits de copie,
de modification et de redistribution accordés par cette licence, il n'est
offert aux utilisateurs qu'une garantie limitée. Pour les mêmes raisons,
seule une responsabilité restreinte pèse sur l'auteur du programme, le
titulaire des droits patrimoniaux et les concédants successifs.

A cet égard l'attention de l'utilisateur est attirée sur les risques
associés au chargement, à l'utilisation, à la modification et/ou au
développement et à la reproduction du logiciel par l'utilisateur étant
donné sa spécificité de logiciel libre, qui peut le rendre complexe à
manipuler et qui le réserve donc à des développeurs et des professionnels
avertis possédant des connaissances informatiques approfondies. Les
utilisateurs sont donc invités à charger et tester l'adéquation du
logiciel à leurs besoins dans des conditions permettant d'assurer la
sécurité de leurs systèmes et ou de leurs données et, plus généralement,
à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.

Le fait que vous puissiez accéder à cet en-tête signifie que vous avez
pris connaissance de la licence CeCILL-C, et que vous en avez accepté les
termes.

/licence*/

require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CLASSE."pattern/alkappli.class.php");

/**
 * @package Alkanet_Module_Annu
 *          Module de gestion de l'annuaire
 * 
 * @class AlkAppliAnnu
 * 
 * Classe regroupant des fonctionnalités de l'annuaire
 */
class AlkAppliAnnu extends AlkAppli
{
  const ALK_ANNU_MAIL_ADD      = 1; // création compte actif
  const ALK_ANNU_MAIL_PCUPDATE = 2; // modif param de connexion
  const ALK_ANNU_MAIL_PCINIT   = 3; // réinitialisation des paramètres de connexion
  const ALK_ANNU_MAIL_ACCEPT   = 4; // compte demandé validé
  const ALK_ANNU_MAIL_REATIV   = 5; // compte réactivé
  const ALK_ANNU_MAIL_DESATIV  = 6; // compte désactivé
  const ALK_ANNU_MAIL_REFUS    = 7; // compte refusé
  
  /** vrai pour autoriser l'envoi de mail (par défaut), faux sinon */
  protected $bSendMail;
  
  /**
   *  Constructeur par défaut 
   * @param appli_id  Identifiant de l'application
   */
  public function __construct($appli_id)
  { 	
    parent::__construct(ALK_ATYPE_ID_ANNU, 0);
    $this->bSendMail = true;
  }
  
  /**
   * Fixe l'état de l'envoi de mail de la classe
   * @param bSendMail  true par défaut pour autoriser l'envoi, false sinon
   */
  public function setSendMail($bSendMail=true)
  {
    $this->bSendMail = $bSendMail;
  } 
    
  /**
   *  Fonction qui instancie la bonne interface en fonction de iSheet et iSSheet
   *        puis retourne l'objet panel généré par celle-ci
   */
  public function getPanel(AlkAppli $oAppliFrom=null)
  {	
    $oPanel = null;
    switch( $this->iTypeSheet ) {
  	case ALK_TYPESHEET_CONSULT: // partie consultation
      switch ( $this->iSheet ){
      case ALK_SHEET_RECHERCHE : // onglet recherche
      case ALK_SHEET_RESULTAT : // onglet Resultat
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlformrechercheform.class.php");
        $oPanel = new AlkHtmlFormRechercheForm($this);
        break;

      case ALK_SHEET_ABONNELISTE : // onglet fiche abonné
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmllistabonne.class.php");
        $oPanel = new AlkHtmlListAbonne($this, $this);
        break; 
        
      case ALK_SHEET_FICHE : // onglet ma fiche
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlformagentform.class.php");
        $oPanel = new AlkHtmlFormAgentForm($this);
        break; 
        
      case ALK_SHEET_ARBORESCENCE : // onglet organigramme
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlformorganigform.class.php");
        $oPanel = new AlkHtmlFormOrganigForm($this);
        break; 
        
      case ALK_SHEET_MESLISTES : // onglet mes listes
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlformlsdiflist.class.php");
        $oPanel = new AlkHtmlFormLsDifList($this);
        break; 
        
      case ALK_SHEET_AJOUTER : // onglet nouvelle fiche
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlformagentform.class.php");
        $oPanel = new AlkHtmlFormAgentForm($this);
        break;   
        
      case ALK_SHEET_MODIFIER : // onglet modification fiche
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlformagentform.class.php");
        $oPanel = new AlkHtmlFormAgentForm($this);
        break;

      case ALK_SHEET_ABONNEFORM : // onglet fiche abonné
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlformabonne.class.php");
        $oPanel = new AlkHtmlFormAbonne($this, $this);
        break; 
      
      case ALK_SHEET_USERRIGHT: // gestion des abonnements
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlpopupabonnement.class.php");
        $oPanel = new AlkHtmlPopUpAbonnementFiche($this);
        break;   
      }
      break;
    
    case ALK_TYPESHEET_ADMIN: // partie admin
      switch ( $this->iSheet ){
      case ALK_SHEET_SERVICE : // onglet services
        switch ( $this->iSSheet ){
          case ALK_SHEET_LIST : // liste des services
            require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlformservicelist.class.php");
            $oPanel = new AlkHtmlFormServiceList($this);
          break;
          case ALK_SHEET_FORM : // propriétés des services
            require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlformserviceform.class.php");
            $oPanel = new AlkHtmlFormServiceForm($this);
          break;
        }
        break;
        
      case ALK_SHEET_PROFIL : // onglet profil
        switch ( $this->iSSheet ){
        case ALK_SHEET_LIST : // liste 
          require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlformprofillist.class.php");
          $oPanel = new AlkHtmlFormProfilList($this);
          break;
        case ALK_SHEET_FORM : // propriétés 
          require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlformprofilform.class.php");
          $oPanel = new AlkHtmlFormProfilForm($this);
          break;
        /* case ALK_SHEET_OPEN_DROITS : // ouverture de droits 
          require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/xxxxxxxxxxxxxxxxxclass.php");
          $oPanel = new AlkHtmlFormxxxxxxxxxx($this);
        break;
        case ALK_SHEET_APPLIQUE_DROITS : // propriétés 
          require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/xxxxxxxxxxxxx.class.php");
          $oPanel = new AlkHtmlFormxxxxxxxxxxxxx($this);
        break;*/
        }
        break;
        
      case ALK_SHEET_MISSION : // onglet mission
        switch ( $this->iSSheet ){
        case ALK_SHEET_LIST : // liste 
          require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlformmissionlist.class.php");
          $oPanel = new AlkHtmlFormMissionList($this);
          break;
        case ALK_SHEET_FORM : // propriétés
          require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlformmissionform.class.php");
          $oPanel = new AlkHtmlFormMissionForm($this);
          break;
        }
        break;
        
      case ALK_SHEET_IMPORTEXPORT : // onglet import/export
        switch ( $this->iSSheet ){
        case ALK_SHEET_FORM : // propriétés  
          require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlformimportexportform.class.php");
          $oPanel = new AlkHtmlFormImportExportForm($this);
          break;
        case ALK_SHEET_IMPORT_SERV : // import service  
          require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlformimportserv.class.php");
          $oPanel = new AlkHtmlFormImportService($this);
          break;  
        case ALK_SHEET_IMPORT_MAIL : // import mail  
          require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlformimportmail.php");
          $oPanel = new AlkHtmlFormImportMail($this);
          break; 
        case ALK_SHEET_IMPORT_USER : // import utilisateur  
          require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlformimportuser.class.php");
          $oPanel = new AlkHtmlFormImportUser($this);
          break; 
        }
        break;  
      
      case ALK_SHEET_UTILISATEUR : // onglet synchronisation
        switch ( $this->iSSheet ){
          case ALK_SHEET_NONE : // propriétés  
          require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlsynchronisation.class.php");
          $oPanel = new AlkHtmlSynchronisation($this);
          break;
        }
      break;
      }
      break; // case iType Admin 
      
    case ALK_TYPESHEET_POPUP:
      switch( $this->iSheet) {
      case ALK_SHEET_USERS_LIST: // popup selection multiple utilisateur
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlpopupuserslist.class.php");
        $oPanel = new AlkHtmlPopupUsersList($this);
        break;
        
      case ALK_SHEET_USER_LIST: // popup selection individuel utilisateur
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlpopupuserlist.class.php");
        $oPanel = new AlkHtmlPopupUserList($this);
        break;          
      
      case ALK_SHEET_USER: // popup agent
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlpopupuserfiche.class.php");
        $oPanel = new AlkHtmlPopupUserFiche($this);
        break;
          
      case ALK_SHEET_SERVICE: // popup service
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlpopupservicefiche.class.php");
        $oPanel = new AlkHtmlPopupServiceFiche($this);
        break;             

      case ALK_SHEET_ABONNE: // popup service
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlpopupabonnefiche.class.php");
        $oPanel = new AlkHtmlPopupAbonneFiche($this, $this);
        break;             
      
      case ALK_SHEET_MESLISTES: // popup liste de diffusion
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlpopuplsdif.class.php");
        $oPanel = new AlkHtmlPopupLsDif($this);
        break;          
      
      case ALK_SHEET_PROFIL: // popup liste de diffusion
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlpopupusersprofil.class.php");
        $oPanel = new AlkHtmlPopupUsersProfil($this);
        break;
        
      case ALK_SHEET_IMPORT_USER: // popup import ldap
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlpopupimportuserldap.class.php");
        $oPanel = new AlkHtmlPopupImportUserLdap($this);
        break;         
        
      case ALK_SHEET_USERRIGHT: // gestion des abonnements
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlpopupabonnement.class.php");
        $oPanel = new AlkHtmlPopUpAbonnementFiche($this);
        break;   
      }
      break;
      
      case ALK_TYPESHEET_PROPRIETE:
       switch($this->iSheet){
         case ALK_SHEET_PARAMETRE:
           require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlformparametreform.class.php");
           $oAppliEspace = AlkFactory::getAppli(ALK_ATYPE_ID_ESPACE);
           $oPanel = new AlkHtmlFormParametreForm($this, $oAppliEspace, $this->atype_id);
         break;
       }
       
       break;

    } // switch iTypeSheet 
    return $oPanel;
  }
  
  /**
   *  Initialise les tabsheets
   */
  public function setTabSheets()
  {
    $iTypeSheet = AlkRequest::getToken("iTypeSheet", ALK_TYPESHEET_CONSULT);
    $iSheet = AlkRequest::getToken("iSheet", ALK_SHEET_RECHERCHE);
    $user_id = AlkFactory::getSProperty("user_id", -1);
  	
    if( $iTypeSheet == ALK_TYPESHEET_POPUP ) {
      // pas d'onglet en popup de manière générale (cas particulier à traiter ici)
      return;
    }

    $strUrl = ALK_ALKANET;
    $user_id = AlkFactory::getSProperty("user_id", "-1");
    $agent_id = AlkRequest::getToken("agent_id", "-1");
    $abonne_id = AlkRequest::getToken("abonne_id", "-1");

    // onglets de consultation
    $this->addSheet(ALK_TYPESHEET_CONSULT, ALK_SHEET_RECHERCHE, ALK_SHEET_NONE, 
                    "Recherche", $strUrl, "", "",
                    ALK_APPLI_RIGHT_READ,  ALK_PRIV_SPACE_USER, true);
    if(defined("ALK_B_ANNU_ABONNES") && ALK_B_ANNU_ABONNES){
	    $this->addSheet(ALK_TYPESHEET_CONSULT, ALK_SHEET_ABONNELISTE, ALK_SHEET_NONE, 
	                    "Abonnés", $strUrl, "", "",
	                    ALK_APPLI_RIGHT_READ,  ALK_PRIV_SPACE_USER, true);
    }
    $this->addSheet(ALK_TYPESHEET_CONSULT, ALK_SHEET_FICHE, ALK_SHEET_NONE, 
                    "Ma fiche", $strUrl, "&iMode=".ALK_FORM_MODE_UPDATE."&agent_id=".$user_id, "",
                    ALK_APPLI_RIGHT_READ,  ALK_PRIV_SPACE_USER, true);
                    
    if(defined("ALK_B_ANNU_ABONNES") && ALK_B_ANNU_ABONNES){
	    $this->addSheet(ALK_TYPESHEET_CONSULT, ALK_SHEET_MESLISTES, ALK_SHEET_NONE, 
	                    "Mes listes", $strUrl, "", "",
	                    ALK_APPLI_RIGHT_READ,  ALK_PRIV_SPACE_USER, !defined("ALK_B_ANNU_FILTER_LSDIF") || ALK_B_ANNU_FILTER_LSDIF);
    }
    
    $this->addSheet(ALK_TYPESHEET_CONSULT, ALK_SHEET_ARBORESCENCE, ALK_SHEET_NONE, 
                    ALK_STR_ORGA_SERV4, $strUrl, "", "",
                    ALK_APPLI_RIGHT_READ,  ALK_PRIV_SPACE_USER, true);

    $this->addSheet(ALK_TYPESHEET_CONSULT, ALK_SHEET_AJOUTER, ALK_SHEET_NONE, 
                    "Nouvelle fiche", $strUrl, "&iMode=".ALK_FORM_MODE_ADD, "",
                    ALK_APPLI_RIGHT_READ,  ALK_PRIV_ANNU_SERV+ALK_PRIV_ANNU_ALL, true);

    $this->addSheet(ALK_TYPESHEET_CONSULT, ALK_SHEET_MODIFIER, ALK_SHEET_NONE, 
                    "Modifier fiche", $strUrl, "&iMode=".ALK_FORM_MODE_UPDATE."&agent_id=".$agent_id, "", 
                    ALK_APPLI_RIGHT_READ,  ALK_PRIV_ANNU_SERV+ALK_PRIV_ANNU_ALL, ($iSheet==ALK_SHEET_MODIFIER));
    if(defined("ALK_B_ANNU_ABONNES") && ALK_B_ANNU_ABONNES){      
	    $this->addSheet(ALK_TYPESHEET_CONSULT, ALK_SHEET_ABONNEFORM, ALK_SHEET_NONE, 
	                    "Fiche abonné", $strUrl, "&iMode=".ALK_FORM_MODE_UPDATE."&abonne_id=".$abonne_id, "", 
	                    ALK_APPLI_RIGHT_READ,  ALK_PRIV_ANNU_SERV+ALK_PRIV_ANNU_ALL, ($iSheet==ALK_SHEET_ABONNEFORM));
    }
    // onglets d'administration
    $this->addSheet(ALK_TYPESHEET_ADMIN, ALK_SHEET_SERVICE, ALK_SHEET_LIST, 
                    "Services", $strUrl, "", "",
                    ALK_APPLI_RIGHT_READ, ALK_PRIV_ANNU_SERV+ALK_PRIV_ANNU_ALL, true);
       
    $this->addSheet(ALK_TYPESHEET_ADMIN, ALK_SHEET_PROFIL, ALK_SHEET_LIST, 
                    "Profils généraux", $strUrl, "", "",
                    ALK_APPLI_RIGHT_READ, ALK_PRIV_SPACE_ADMIN, true);
    if(defined("ALK_B_ANNU_MISSION") && ALK_B_ANNU_MISSION){                
	    $this->addSheet(ALK_TYPESHEET_ADMIN, ALK_SHEET_MISSION, ALK_SHEET_LIST, 
	                    "Missions", $strUrl, "", "",
	                    ALK_APPLI_RIGHT_READ, ALK_PRIV_ANNU_SERV+ALK_PRIV_ANNU_ALL, !defined("ALK_B_ANNU_FILTER_MISSION") || ALK_B_ANNU_FILTER_MISSION);
    }
    $this->addSheet(ALK_TYPESHEET_ADMIN, ALK_SHEET_IMPORTEXPORT, ALK_SHEET_FORM, 
                    "Imports Exports", $strUrl, "", "",
                    ALK_APPLI_RIGHT_READ, ALK_PRIV_ANNU_SERV+ALK_PRIV_ANNU_ALL, !defined("ALK_B_ANNU_IMPORTEXPORT") || ALK_B_ANNU_IMPORTEXPORT);
   
    if( defined("ALK_B_LDAP_AD") && ALK_B_LDAP_AD==true ) {
      $this->addSheet(ALK_TYPESHEET_ADMIN, ALK_SHEET_UTILISATEUR, ALK_SHEET_NONE, 
                      "Synchronisation", $strUrl, "", "",
                      ALK_APPLI_RIGHT_READ, ALK_PRIV_SPACE_ADMIN, true);                 
     }
     
   // onglet property
   if ($user_id == ALK_USER_ID_ADMINALK)
     $this->addSheet(ALK_TYPESHEET_PROPRIETE, ALK_SHEET_PARAMETRE, ALK_SHEET_LIST, "Paramètres", $strUrl, "",
                    "Gérer les constantes de l'application", 
                    ALK_APPLI_RIGHT_READ, ALK_PRIV_SPACE_ANIM+ALK_PRIV_SPACE_ADMIN);
   }

  /**
   *  Retourne le droit utilisateur sur l'appli en cours
   * @return int
   */
  public function getUserAppliRight()
  {
    $iRight = ALK_APPLI_RIGHT_NONE;
    $user_priv  = AlkFactory::getSProperty("user_priv", "0");

    $iRight = ( ($user_priv & ALK_PRIV_SPACE_USER) == ALK_PRIV_SPACE_USER 
                ? ALK_APPLI_RIGHT_READ 
                : $iRight ); 

    $iRight = ( ($user_priv & ALK_PRIV_ANNU_SERV) == ALK_PRIV_ANNU_SERV ||
                ($user_priv & ALK_PRIV_ANNU_ALL) == ALK_PRIV_ANNU_ALL 
                ? ALK_APPLI_RIGHT_ADMIN 
                : $iRight ); 

    $iRight = ( ($user_priv & ALK_PRIV_ANNU_ALL) == ALK_PRIV_ANNU_ALL 
                ? ALK_APPLI_RIGHT_PUBLI 
                : $iRight ); 

    return $iRight;
  }

  /**
   *  Ajout d'un utilisateur à l'annuaire.
   *        Retourne un identifiant de l'utilisateur créé
   * @param tabFields tableau associatif des champs nécessaires à la création d'un utilisateur
   * @return int
   */
  public function addUser($tabFields)
  {
    $user_id = AlkFactory::getSProperty("user_id", "-1");
    
    // création de l'utilisateur dans l'annuaire
    $tabInfo = $this->oQueryAction->addUser($user_id, $tabFields);
    $agent_id = $tabInfo["agent_id"];
    
    // traitement de l'ajout au niveau des espaces et des droits
    if( $agent_id != -1 ) {
      $this->oSpace->addUser($agent_id, $tabFields);
    }
    
    // envoi mail pour fiche valide uniquement
    if( $agent_id != -1 && $tabInfo["agent_valide"]==1 ) {
      $this->SendMailToAgent($tabInfo, self::ALK_ANNU_MAIL_ADD);
    }

    return $agent_id;
  }

  /**
   *  Modification d'un utilisateur
   * @param user_id   identifiant d'un utilisateur
   * @param tabFields tableau associatif des champs nécessaires à la modif d'un utilisateur
   */
  public function updateUser($user_id, $tabFields) 
  { 
    $strUpload = ALK_ROOT_UPLOAD.mb_strtolower(constant("ALK_ATYPE_ABREV_".ALK_ATYPE_ID_ANNU))."/";

    // modif de la fiche utilisateur
    $tabInfo = $this->oQueryAction->updateUser($user_id, $tabFields, $strUpload);

    if( $tabInfo["agent_valide"] != $tabInfo["agent_valide_old"] && $tabInfo["agent_valide"] == -1 ) {
      $this->delUser($user_id); 
    } else {
      // traitement lié à la modif de service ou de privilege au niveau de l'espace et des droits
      $this->oSpace->updateUser($user_id, $tabInfo);
    }
    
    $user_id = AlkFactory::getSProperty("user_id", -1);
    // envoi mail, si on ne vient pas de l'import, si les paramètres de connexion ont changé 
    // et que la fiche ne concerne pas l'utilisateur connecté
    if( $tabInfo["agent_id"]!= $user_id && $tabInfo["agent_id"]!=-1 && $tabInfo["alk_b_import"]==false ) {
      $iMode = -1;
      if( $tabInfo["agent_pwdinit"]==true ) {
        $iMode = self::ALK_ANNU_MAIL_PCINIT;
      } elseif( $tabInfo["agent_valide"] != $tabInfo["agent_valide_old"] ) {
        if( $tabInfo["agent_valide"] == 1 && $tabInfo["agent_valide_old"]== -2 ) {
          $iMode = self::ALK_ANNU_MAIL_ACCEPT;
        } elseif( $tabInfo["agent_valide"] == 1 && $tabInfo["agent_valide_old"]== 0 ) {
          $iMode = self::ALK_ANNU_MAIL_REATIV ; 
        } elseif( $tabInfo["agent_valide"] == 0 && $tabInfo["agent_valide_old"]== 1 ) {
          $iMode = self::ALK_ANNU_MAIL_DESATIV ;
        } elseif( $tabInfo["agent_valide"] == -1 ) {
          $iMode = self::ALK_ANNU_MAIL_REFUS;
        }
      } elseif( $tabInfo["agent_login"]!="" || $tabInfo["agent_pwd"]!="" ) {
        $iMode = self::ALK_ANNU_MAIL_PCUPDATE;
      }
      
      if( $iMode != -1 ) {
        $this->SendMailToAgent($tabInfo, $iMode);
      }
    }
  }
  
  /**
   *  Transfère de propriété d'un utilisateur à un autre
   *        Méthode à appeler avant suppression définitive d'un utilisateur
   * @param idUserFrom  identifiant de l'utilisateur perdant la propriété des ses données
   * @param idUserTo    identifiant de l'utilisateur récupérant la propriété des données
   */
  public function replaceUser($idUserFrom, $idUserTo)
  {
    // remplacement sur tous les types d'applis dispo
    $this->oSpace->replaceUser($idUserFrom, $idUserTo);

    // remplacement sur l'annuaire
    $this->oQueryAction->replaceUser($idUserFrom, $idUserTo);
  }

  /**
   *  Suppression d'un utilisateur pour user_id>=10
   *        Les comptes (user_id<10) spéciaux ne sont pas supprimables
   * @param user_id  identifiant d'un utilisateur
   */
  public function delUser($user_id, $bRegion=ALK_B_SIT_REGION)
  {
    // suppression de l'abonné
    list($abonne_id, $abonne_pseudo) = $this->oQuery->getAbonneIdByAgentId($user_id);
    $this->delAbonne($abonne_id);

    // suppression des droits de l'utilisateur et des infos liées aux applis
    $this->oSpace->delUser($user_id);

    // suppression de l'utilisateur dans l'annuaire
    $this->oQueryAction->delUser($user_id, $bRegion);
  }
  
  /**
   * Invitation d'un ou plusieurs utilisateurs à un espace (tous si cont_id=-1, tous les espaces publics si cont_id=-2)
   * @param user_id  identifiant ou liste d'id d'utilisateurs. -1 pour tous les utilisateurs ayant les privilèges priv_id
   * @param cont_id  identifiant de l'espace. -1 pour tous les espaces. -2 tous les espaces publics
   * @param priv_id  ensemble des privilèges
   */
  public function addUserToSpace($user_id, $cont_id, $priv_id)
  {
    // ajoute les utilisateurs à la liste de diffusion de l'espace
    $oQueryEspace = AlkFactory::getQuery(ALK_ATYPE_ID_ESPACE);
    $tabContId = array();
    switch ( $cont_id ) {
      case -1 :
        $oDsEspace = $oQueryEspace->GetDs_listeEspace();
      break;
      case -2 :
        $oDsEspace = $oQueryEspace->GetDs_listeEspace("1");
      break;
      default :
        $oDsEspace = $oQueryEspace->getDs_ficheEspaceById($cont_id);
    }
    while ( $oDrEspace = $oDsEspace->getRowIter() ) {
      $tabContId[$oDrEspace->getValueName("CONT_ID")] = $oDrEspace->getValueName("CONT_INTITULE");
    }
    foreach ( $tabContId as $cont_id => $cont_intitule ) {
      $oDsListe = $this->oQuery->getDsListesForEspace($cont_id, false, 0, -1, "4");
      if ( $oDrListe = $oDsListe->getRowIter() ) {
        // récupère la liste si celle-ci existe
        $liste_id = $oDrListe->getValueName("LISTE_ID");
      } else {
        // crée la liste si celle-ci n'existe pas
        $tabIntitule = array();
        $tabDesc = array();
        foreach($this->tabLangue as $key => $tabLg) {
          $tabIntitule[$key] = "Liste des utilisateurs de l'espace ".$cont_intitule;
          $tabDesc[$key] = "";
        }
        $liste_id = $this->oQueryAction->addListe($tabIntitule, $tabDesc, 0, ALK_USER_ID_ADMINALK, $cont_id, -1, 4, 1);
      }
      if ( $liste_id != -1 ) {
        if ( $user_id != -1 ) {
          // ajout simple
          $this->oQueryAction->addUserToListe($user_id, $liste_id);
        } else {
          // synchronisation
          $this->oQueryAction->syncEspaceListe($cont_id, $liste_id);
        }
      }
    }
  }
  
  /**
   * Suppression de l'accès à un espace (ou tous si cont_id=-1) pour un utilisateur
   * @param user_id  identifiant ou liste d'id d'utilisateurs
   * @param cont_id  identifiant de l'espace. -1 pour tous les espaces.
   */
  public function removeUserFromSpace($user_id, $cont_id)
  {
    // supprime les utilisateurs de la liste de diffusion de l'espace
    $oQueryEspace = AlkFactory::getQuery(ALK_ATYPE_ID_ESPACE);
    $tabContId = array();
    switch ( $cont_id ) {
      case -1 :
        $oDsEspace = $oQueryEspace->GetDs_listeEspace();
      break;
      default :
        $oDsEspace = $oQueryEspace->getDs_ficheEspaceById($cont_id);
    }
    while ( $oDrEspace = $oDsEspace->getRowIter() ) {
      $tabContId[$oDrEspace->getValueName("CONT_ID")] = $oDrEspace->getValueName("CONT_INTITULE");
    }
    foreach ( $tabContId as $cont_id => $cont_intitule ) {
      if ( $cont_id != -1 ) {
        $oDsListe = $this->oQuery->getDsListesForEspace($cont_id, false, 0, -1, "4");
        $liste_id = -1;
        if ( $oDrListe = $oDsListe->getRowIter() ) {
          // récupère la liste si celle-ci existe
          $liste_id = $oDrListe->getValueName("LISTE_ID");
        }
        if ( $user_id != -1 && $liste_id != -1 ) {
          $this->oQueryAction->delUserFromListe($user_id, $liste_id);
        }
      }
    }
  }
  
  /**
   * crée la liste de diffusion d'un espace (tous si cont_id=-1, tous les espaces publics si cont_id=-2) si celle-ci n'existe pas et la synchronise
   * @param cont_id identifiant de l'espace. -1 pour tous les espaces. -2 tous les espaces publics
   */
  public function syncEspaceListe($cont_id){
    $oQueryEspace = AlkFactory::getQuery(ALK_ATYPE_ID_ESPACE);
    $tabContId = array();
    switch ( $cont_id ) {
      case -1 :
        $oDsEspace = $oQueryEspace->GetDs_listeEspace();
      break;
      case -2 :
        $oDsEspace = $oQueryEspace->GetDs_listeEspace("1");
      break;
      default :
        $oDsEspace = $oQueryEspace->getDs_ficheEspaceById($cont_id);
    }
    while ( $oDrEspace = $oDsEspace->getRowIter() ) {
      $tabContId[$oDrEspace->getValueName("CONT_ID")] = $oDrEspace->getValueName("CONT_INTITULE");
    }
    foreach ( $tabContId as $cont_id => $cont_intitule ) {
      $oDsListe = $this->oQuery->getDsListesForEspace($cont_id, false, 0, -1, "4");
      if ( $oDrListe = $oDsListe->getRowIter() ) {
        // récupère la liste si celle-ci existe
        $liste_id = $oDrListe->getValueName("LISTE_ID");
        
      } else {
        // crée la liste si celle-ci n'existe pas
        $tabIntitule = array();
        $tabDesc = array();
        foreach($this->tabLangue as $key => $tabLg) {
          $tabIntitule[$key] = "Liste des utilisateurs de l'espace ".$cont_intitule;
          $tabDesc[$key] = "";
        }
        $liste_id = $this->oQueryAction->addListe($tabIntitule, $tabDesc, 0, ALK_USER_ID_ADMINALK, $cont_id, -1, 4, 1);
      }
    }
    $this->oQueryAction->syncEspaceListe($cont_id, $liste_id);
  }
  
  /**
   * Abonne un abonné à une application
   * Création de l'abonné si il n'existe pas
   * Retourne l'identifiant de l'abonné
   * @param appli_id       identifiant de l'application liée à l'abonné, peut prendre la valeur -1 pour ne pas prendre en compte ce paramètre
   * @param liste_id       identifiant de la liste liée à l'abonné, peut prendre la valeur -1 pour ne pas prendre en compte ce paramètre
   * @param abonne_mail    email de l'abonné
   * @param abonne_valide  =1 pour valide, =0 pour non valide (par défaut)
   * @param abonne_pwd     mot de passe de l'abonné (commentaire / forum)
   * @param abonne_pseudo  pseudo de l'abonné (commentaire / forum)
   * @param abonne_nom     nom de l'abonné (forum)
   * @param abonne_prenom  prenom de l'abonné (forum)
   * @param abonne_adresse adresse
   * @param abonne_cp      code postal
   * @param abonne_ville   ville
   * @param abonne_pays    pays
   * @param abonne_tel     tétéphone
   * @return int
   */
  public function addAbonnement($liste_id, $abonne_mail, $abonne_valide="0", 
                                $abonne_pwd="", $abonne_pseudo="", $abonne_nom="Abonné", $abonne_prenom="Anonyme",
                                $abonne_adresse="", $abonne_cp="", $abonne_ville="", $abonne_pays="", $abonne_tel="")
  {
    $abonne_id = "-1";
    $dsAb = $this->oQuery->getDsAbonneByMail($abonne_mail);
    if( $drAb = $dsAb->getRowIter() ) {
      $abonne_id = $drAb->getValueName("ABONNE_ID"); 
    } 
    if( $abonne_id == "-1" ) { 
      $abonne_id = $this->oQueryAction->addAbonne(ALK_USER_ID_INTERNET, $abonne_nom, $abonne_prenom, $abonne_mail, $abonne_valide,  
                                                  $abonne_adresse, $abonne_cp, $abonne_ville, $abonne_pays, $abonne_tel, $abonne_pwd, $abonne_pseudo);
    }
    if( $abonne_id != "-1" ) {
      $this->oQueryAction->addAbonnement($abonne_id, $liste_id);
    }
    return $abonne_id;
  }

  /**
   * Méthode appelée après suppression d'un abonnement
   * @param abonne_id  identifiant de l'abonné, peut contenir une liste d'identifiant
   * @param appli_id   identifiant de l'appli demandant le désabonnement
   * @param liste_id   identifiant de la liste de laquelle on souhaite désabonné
   */
  public function delAbonnement($abonne_id, $appli_id=-1, $liste_id=-1)
  {
    // remplacement sur tous les types d'applis dispo
    $this->oSpace->delAbonnement($abonne_id, $appli_id, $liste_id);

    // remplacement sur l'annuaire
    $this->oQueryAction->delAbonnement($abonne_id, $liste_id);
  }

  /**
   * Méthode appelée avant suppression définitive d'un abonne
   * Attention, si un abonné est propriétaire d'une info, il ne faut pas la supprimer
   * Implémenter les actions effectuant le ménage interne à l'application
   * Retourne true si l'abonné reste supprimable, faux si il est propriétaire d'info
   * @param abonne_id  identifiant de l'abonné, peut contenir une liste d'identifiant
   * @return boolean
   */
  public function delAbonne($abonne_id)
  {
    // remplacement sur tous les types d'applis dispo
    $this->oSpace->delAbonne($abonne_id);

    // remplacement sur l'annuaire
    $this->oQueryAction->delAbonne($abonne_id);
  }

  /**
   * Méthode appelée avant suppression définitive d'une liste d'abonnés ou des listes associées à un espace
   * @param liste_id  identifiant de la liste, peut contenir une liste d'identifiant
   * @param cont_id   identifiant de l'espae à supprimer (=-1 par défaut pour ne pas prendre en compte ce paramètre)
   */
  public function delListe($liste_id, $cont_id="-1")
  {
    if( $cont_id != "-1" ) {
      $liste_id = "-1";
      // récupération des listes privées de l'espace
      $dsListe = $this->oQuery->getDsListesForEspace($cont_id, false, 0, -1, "0,1,2,3,4");
      while( $drListe = $dsListe->getRowIter() ) {
        $liste_id .= ",".$drListe->getValueName("LISTE_ID");
      }
    }
    // remplacement sur tous les types d'applis dispo
    $this->oSpace->delListe($liste_id);

    // remplacement sur l'annuaire
    $this->oQueryAction->delListe($liste_id);
  }

  /** 
   *  Ajout d'un profil puis retoure l'id du profil généré
   * @param tabFields  tableau contenant les propriétés du profil à créer
   * @param cont_id    identifiant de l'espace associé, =0 pour tous les espaces (par défaut)
   * @return int        Identifiant du profil ajouté
   */
  public function addProfil($tabFields, $cont_id=0)
  {
    $tabFields["field"]["CONT_ID"] = array(ALK_SQL_NUMBER, $cont_id);
    $profil_id = $this->oQueryAction->addProfil($tabFields);
    $this->oSpace->addProfil($profil_id, $cont_id);
    return $profil_id;
  }

  /** 
   *  Suppression du profil sur la gestion des droits interne à l'appli
   * @param profil_id  identifiant du profil à supprimer
   */
  public function delProfil($profil_id)
  {
    $this->oSpace->delProfil($profil_id);
    $this->oQueryAction->delProfil($profil_id);
  }
  
  /**
   * Ajout d'un service sur l'annuaire.
   * Propagation de l'ajout aux applis ayant des données liées au service
   * Retourne l'identifiant du service ajouté
   * @param service_id   identifiant du service créé
   * @param tabFields    tableau associatif contenant les infos modifiées du service
   * @return int
   */
  public function addService($tabFields)
  {
  	$strUpload = ALK_ROOT_UPLOAD.mb_strtolower(constant("ALK_ATYPE_ABREV_".ALK_ATYPE_ID_ANNU))."/";
  	
    $service_id = $this->oQueryAction->addService($tabFields, $strUpload);
    $this->oSpace->addService($service_id, $tabFields);
    return $service_id;
  }
  
  /**
   * Mise à jour d'un service avec propogation des modifications aux applis ayant des données liées au service
   * @param service_id  identifiant du service à modifier
   * @param tabFields   tableau associatif contenant les infos modifiées du service
   */
  public function updateService($service_id, $tabFields)
  {
  	$strUpload = ALK_ROOT_UPLOAD.mb_strtolower(constant("ALK_ATYPE_ABREV_".ALK_ATYPE_ID_ANNU))."/";
  	 
    $this->oSpace->updateService($service_id, $tabFields);
    $this->oQueryAction->updateService($service_id, $tabFields, $strUpload);
  }

  /** 
   * Suppression d'un service de l'annuaire avec propagation de la suppression aux applis ayant des données liées au service
   * @param service_id  identifiant du service à supprimer
   */
  public function delService($service_id)
  {
    $this->oSpace->delService($service_id);
    $this->oQueryAction->delService($service_id);
  }
  
  /**
   * Envoi un mail à un agent
   *
   * @param tabAssoc  tableau associatif contenant les données relatives à l'agent
   * @param iMode     mode d'envoi de mail
   */
  public function SendMailToAgent($tabAssoc, $iMode) 
  {
    if( !$this->bSendMail ||
        defined("ALK_B_ANNU_SEND_MAIL") && ALK_B_ANNU_SEND_MAIL==false ) { 
      return;   
    }
    
    // initialisation de l'objet mail
    setMailAlkLocale($tabAssoc["agent_lg"]);
    
    $oAlkMail = AlkFactory::getMail();
    
    $strDate = date("d/m/Y", AlkFactory::getLocalDate()); 
    $strTime = date("H:i", AlkFactory::getLocalDate());
    
    $strMailFrom = AlkFactory::getSProperty("user_mail", "");
    $strNomFrom  = AlkFactory::getSProperty("user_name", "");
    $user_login  = AlkFactory::getSProperty("user_login", "");
    $oAlkMail->SetFrom($strNomFrom, $strMailFrom);

    $strMailTo = $tabAssoc["agent_mail"];
    $strNomTo  = $tabAssoc["agent_prenom"]." ".$tabAssoc["agent_nom"];
    $oAlkMail->AddTo($strNomTo, $strMailTo);
    
    $strTitle = "";
    $strMsg = "";
    $strMotif = "";
    $strType = "";
    switch( $iMode ) {
    case self::ALK_ANNU_MAIL_ADD : // ajout
      $strType = "ANNU_AJOUT_UTIL";
      $strTitle = _t("Inscription");
      $strMsg = _f("Vous avez été inscrit sur l'annuaire du %s le %s à %s.", 
                   ALK_APP_TITLE, $strDate, $strTime).'<br/>';
      break;
    case self::ALK_ANNU_MAIL_PCUPDATE : // modif de la fiche
      $strType = "ANNU_MODIF_PARAM_CON";
      $strTitle = _t("Modification de vos paramètres de connexion");
      $strMsg = _f("Vos paramètres de connexion au %s ont été modifiés le %s à %s.",
                   ALK_APP_TITLE, $strDate, $strTime).'<br/>';
      break;
    case self::ALK_ANNU_MAIL_PCINIT: // init
      $strType = "ANNU_INIT_PARAM_CON";
      $strTitle = _t("Réinitialisation de vos paramètres de connexion");
      $strMsg = _f("Vos paramètres de connexion au %s ont été réinitialisés le %s à %s.",
                   ALK_APP_TITLE, $strDate, $strTime).'<br/>';
      break;
    case self::ALK_ANNU_MAIL_ACCEPT: // compte demandé validé
      $strType = "ANNU_UTIL_VALIDE";
      $strTitle = _t("Demande de création de compte acceptée");
      $strMsg = _f("Votre demande de compte au %s a été acceptée le %s à %s.",
                   ALK_APP_TITLE, $strDate, $strTime).'<br/>';
      break;
    case self::ALK_ANNU_MAIL_REATIV: // compte réactivé
      $strType = "ANNU_UTIL_ACTIVE";
      $strTitle = _t("Réactivation de votre compte annuaire");
      $strMsg = _f("Votre compte au %s a été réactivé le %s à %s.",
                   ALK_APP_TITLE, $strDate, $strTime).'<br/>';
      break;
    case self::ALK_ANNU_MAIL_DESATIV: // compte désactivé, non implémenté : il manque le motif
      
      break;
    case self::ALK_ANNU_MAIL_REFUS: // compte refusé
      $strType = "ANNU_UTIL_REFUS";
      $strTitle = _t("Demande de création de compte refusée");
      $strMsg = _f("Votre demande de création de compte au %s a été refusée le %s à %s.",
                   ALK_APP_TITLE, $strDate, $strTime).'<br/>';
      $strMotif = _t("Motif du refus :").
        '<ul>'.
        ( $tabAssoc["agent_desc"] != "" 
          ? $tabAssoc["agent_desc"] 
          : _t("non précisé.") ).
        '</ul>';
      break;
    }
    
    $msgParamConn = "";
    if( ($tabAssoc["agent_login"]!="" || $tabAssoc["agent_pwd"]!="") && 
        $iMode!=self::ALK_ANNU_MAIL_REFUS && $strType !="") {
      $strTokenPwd = AlkRequest::getEncodeParam("l=".( $tabAssoc["agent_login"] != "" ? $tabAssoc["agent_login"] : $user_login )."&p=".md5($tabAssoc["agent_pwd"]));
      $strUrlChgPwd = 
        ( defined("ALK_ROOT_URL_FRONTOFFICE") 
          ? ALK_ROOT_URL_FRONTOFFICE.ALK_ROOT_DIR
          : ALK_ALKANET_ROOT_URL ).
        ALK_ROOT_MODULE."alkanet/index_chgpwd.php?token=".$strTokenPwd;
             
      $msgParamConn = _t("Vos paramètres de connexion sont les suivants :").
        '<ul>'.
        '<li>'._f("Identifiant : %s", ( $tabAssoc["agent_login"] != "" ? $tabAssoc["agent_login"] : $user_login )).'</li>'.
        '<li>'.( $tabAssoc["agent_pwd"] != ""  
                 ? ( ALK_B_ANNU_PWD_SENDMAIL
                     ? _f("Mot de passe : [%s] (le mot de passe se situe entre les 2 crochets)", $tabAssoc["agent_pwd"])
                     : '<a href="'.$strUrlChgPwd.'">'._t("Cliquez sur ce lien pour vous connecter").'</a>.' )   
                 : _t("Mot de passe : non modifié") ).'</li>'.
        '</ul>';
    }
    if( $strMsg != "" && $strType != "" ) {
      $tabAssoc["msgTitle"] = $strTitle; 
      $tabAssoc["msg"] = $strMsg.$strMotif.$msgParamConn;
      
      // informations complémentaires 
      $tabAssoc["msgMotif"] = $strMotif;
      $tabAssoc["msgParamConn"] = $msgParamConn;

      AlkMailing::SendMailType($oAlkMail, $strType, $tabAssoc);
    } 
    setCurrentAlkLocale();
  }

  /**
   * Retourne le type de méthode pour mettre à jour les blocs
   *        Retourne un tableau en fonction du type sélectionné.
   * @param typeIdBloc   type de bloc
   * @param atypeIdBloc  type de l'application associée au type de bloc
   * @return array
   */
  public function getGEditBlocTypeUpdate($typeIdBloc, $atypeIdBloc) 
  {
    return array("type"            => ALK_GEDIT_TYPEUPDATE_SELECTEDIT, 
                 "iColumn"         => 0,
                 "newsTitle"       => "",
                 "iFilterSpace"    => 0,
                 "iAssocAppli"     => 0,
                 "iAssocCat"       => 0,
                 "catTableName"    => "", 
                 "dataTableName"   => "",
                 "catTitle"        => "",
                 "dataTitle"       => "Annuaire",
                 "xmltag"          => "annu",
                 "uploadDir"       => "",
                 "iFields"         => 0,
                 "iFieldsPage"     => 0,
                 "tokenBlocPerso"  => "");
  }

  /**
   * Retourne un tableau contenant l'information sélectionnée par cette méthode (appel régulier de getDataContents())
   * @param typeAff         type d'affichage, permet d'établir une conditionnelle dans cette méthode pour gérer plusieurs type d'affichage
   * @param cont_id         Identifiant de l'espace, pris en compte si appli_id vaut -1
   * @param appli_id        Identifiant de l'application, peut valoir =-1 pour toutes les applis de cont_id
   * @param cat_id          Identifiant de la catégorie, peut valoir =-1 pour toutes les infos de appli_id
   * @param data_id         Identifiant de la données, différent de -1 pour afficher la fiche, =-1 sinon par défaut
   * @param lg              langue utilisé, _FR par défaut
   * @param page_typeassoc  Type d'association des données à la page
   * @param page_ordre      Ordre des données dans la page
   * @param page_datedeb    Contient la date de début pour un filtre éventuel en fonction de la valeur de page_typeassoc
   * @param page_datefin    Contient la date de début pour un filtre éventuel en fonction de la valeur de page_typeassoc 
   * @return array
   */
  public function getGEditDataContents($typeAff, $cont_id, $atype_id, $appli_id, $cat_id, $data_id="-1", $lg=ALK_LG_BDD, $page_typeassoc=0, $page_ordre="", $page_datedeb="", $page_datefin="")
  {
    $user_id = AlkFactory::getSProperty("user_id", "-1");
    $tabRes = array();
    switch( $typeAff ) {
    case "AnnuTel": 
    case "Trombi": 
      break;
    }
    return $tabRes;
  }
  
  /**
   * Retourne le libellé de la page d'aide associée à l'application
   * @return string
   */
  public function getHelp(){
  	return _t("Annuaire");
  }
  
}
?>