<?php
/*licence/ 

Module écrit, supporté par la société Alkante SAS <alkante@alkante.com>

Nom du module : Alkanet::Module::Annu
Module annuaire Alkanet.
Ce module appartient au framework Alkanet.

Ce logiciel est régi par la licence CeCILL-C soumise au droit français et
respectant les principes de diffusion des logiciels libres. Vous pouvez
utiliser, modifier et/ou redistribuer ce programme sous les conditions
de la licence CeCILL-C telle que diffusée par le CEA, le CNRS et l'INRIA
sur le site http://www.cecill.info.

En contrepartie de l'accessibilité au code source et des droits de copie,
de modification et de redistribution accordés par cette licence, il n'est
offert aux utilisateurs qu'une garantie limitée. Pour les mêmes raisons,
seule une responsabilité restreinte pèse sur l'auteur du programme, le
titulaire des droits patrimoniaux et les concédants successifs.

A cet égard l'attention de l'utilisateur est attirée sur les risques
associés au chargement, à l'utilisation, à la modification et/ou au
développement et à la reproduction du logiciel par l'utilisateur étant
donné sa spécificité de logiciel libre, qui peut le rendre complexe à
manipuler et qui le réserve donc à des développeurs et des professionnels
avertis possédant des connaissances informatiques approfondies. Les
utilisateurs sont donc invités à charger et tester l'adéquation du
logiciel à leurs besoins dans des conditions permettant d'assurer la
sécurité de leurs systèmes et ou de leurs données et, plus généralement,
à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.

Le fait que vous puissiez accéder à cet en-tête signifie que vous avez
pris connaissance de la licence CeCILL-C, et que vous en avez accepté les
termes.

/licence*/

require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CLASSE."pattern/alkobject.class.php");

/**
 * @brief Classe noeud d'un arbre descendant
 * 
 *
 */
class AlkSvgTreeNodeDesc extends AlkObject
{
  var $spaceHeight = 30;
  var $spaceWidth = 25;
  var $spaceLine = 16;
  var $spaceDelta = 30;
  var $widthWordWrap = 38;
	
  var $limitNiv = 3;

  var $x0;
  var $y0;

  var $iWidth;
  var $iHeight;

  /** base du trapeze */
  var $iBaseWidth;
  /** hauteur du trapeze */
  var $iTrapHeight;

  var $id;
  var $iNiveau;
  var $idPere;
  var $strIntitule;
  var $tabIntitule;
  var $strDesc;
  var $tabDesc;
  var $strPJ;
  var $nbEnf;
  var $type;
  var $iNumEnf;

  /** Référence vers AlkTreeNodeDescSVG pointant vers le pere */
  var $nodePere;
  /** tableau de AlkTreeNodeDescSVG pointant vers les fils */
  var $tabFils;
  
  function svgWidth()  { return $this->iBaseWidth; }
  function svgHeight() { return $this->iTrapHeight; }

  /**
   * @brief Constructeur par défaut
   * 
   * @param oDr  DataRow contenant les colonnes suivantes :
   *             ID       : identifiant du noeud
   *             NIVEAU   : profondeur dans l'arbre
   *             ID_PERE  : identifiant du noeud parent
   *             INTITULE : intitulé du noeud
   *             DESCR    : information complémentaire
   *             PJ       : fichier joint
   */
  public function __construct(&$oDr, $strPathUpload, $strUrlUpload)
  {
    $this->x0 = 0;
    $this->y0 = 0;

    $this->id          = $oDr->getValueName("ID");
    $this->iNiveau     = $oDr->getValueName("NIVEAU");
    $this->idPere      = ( $this->iNiveau > 1 ? $oDr->getValueName("ID_PERE") : "-1" );
    $this->strIntitule = $oDr->getValueName("INTITULE");
    $this->tabIntitule = ( $this->strIntitule != "" 
                           ? explode("|", wordwrap($this->strIntitule, $this->widthWordWrap, "|")) 
                           : array() );
    $this->strDesc     = $oDr->getValueName("DESCR");
    $this->tabDesc     = ( $this->strDesc != "" 
                           ? explode("|", wordwrap($this->strDesc, $this->widthWordWrap, "|")) 
                           : array() );
    $this->strPJ       = $oDr->getValueName("PJ");
    if( $this->strPJ != "" && file_exists($strPathUpload.$this->strPJ) && is_file($strPathUpload.$this->strPJ) ) {
      $this->strPJ = $strUrlUpload.$this->strPJ;
    }
    $this->nbEnf       = 0;
    $this->type        = $oDr->getValueName("TYPE");
    $this->iNumEnf     = -1;

    $this->iHeight = (count($this->tabIntitule)+count($this->tabDesc))*$this->spaceLine+4;
    $this->iWidth = 280;
    $this->iBaseWidth = 280;
    $this->iTrapHeight = $this->iHeight;
    $this->tabFils = null;
    $this->nodePere = null;
  }

  /**
   * Ajoute une ligne au tableau des descriptions
   *
   * @param strDesc  Texte correspondant au descriptif à afficher
   */
  public function addDesc($strDesc, $strPJ="-1")
  {
    if( $strPJ == "-1" )
      $this->tabDesc[count($this->tabDesc)] = $strDesc;
    else
      $this->tabDesc[count($this->tabDesc)] = array($strDesc, $strPJ);
    $this->iHeight = (count($this->tabIntitule)+count($this->tabDesc))*$this->spaceLine+4;
    $this->iTrapHeight = $this->iHeight;
  }

	/**
   * Ajoute un noeud fils à la collection interne
   * @param nodeChild  référence du noeud à ajouter
   * @return Retourne le numéro d'ordre
   */
  public function addChild(&$nodeChild)
  {
    if( $this->tabFils == null )
      $this->tabFils = array();
    $index = count($this->tabFils);
    if( $this->iNiveau == 1 && $nodeChild->type == "fonc" ) {
      $this->AddDesc($nodeChild->strIntitule, $nodeChild->strPJ);
      $index = -1;
    } else {
      $this->tabFils[$index] =& $nodeChild;
      $this->tabFils[$index]->nodePere =& $this;
      $this->tabFils[$index]->iNumEnf = $index;
      $this->nbEnf++;
    }
    return $index;
  }

  /**
   * Calcul la largeur et la hauteur du sous-arbre
   * formant un trapèze. Méthode recursive sur les fils
   * La largeur haute : $this->iWidth
   * Calcul la hauteur $this->iTrapHeight et la base $this->iBaseWidth
   * @param nodeParent  référence sur le noeud parent
   */
  public function setFrame(&$nodeParent)
  {
    if( $this->iNiveau < $this->limitNiv-1 ) {
      // fils disposés sur la largeur
      if( $this->nbEnf > 0 ) {
        // parcours les fils
        $this->iBaseWidth = 0;
        $iSubTreeHeightMax = 0;
        for($i=0; $i<$this->nbEnf; $i++) {
          $tnThis =& $this;
          $tn =& $this->tabFils[$i];
          // appel récursif sur le fils
          $tn->setFrame($tnThis);
          // largeur du sous arbre de this += largeur du sous arbre du fils tn + espace à droite
          $this->iBaseWidth += $tn->iBaseWidth + $this->spaceWidth; 
          // hauteur du trapeze de this : max avec la hauteur du sous arbre du fils tn
          $iSubTreeHeightMax = max($iSubTreeHeightMax, $tn->iTrapHeight);
        }

        // retire le dernier espace placé à droite
        $this->iBaseWidth -= $this->spaceWidth;

        // ajoute la hauteur du sous arbre + la connexion
        if( $this->iNiveau > 1 ) {
          $this->iTrapHeight += $this->spaceHeight + $iSubTreeHeightMax;
        } else {
          $this->iTrapHeight += $this->spaceHeight/2 + $iSubTreeHeightMax;
        }
      } else {
        if( $nodeParent != $this ) {
          // ajoute la hauteur du trait vertical remontant vers le père
          $this->iTrapHeight += $this->spaceHeight/2;
        }
      }
    } else {
      // fils disposés sur la hauteur sous le pere
      if( $this->nbEnf > 0 ) {
        // parcours les fils
        $iSubTreeWidthMax = 0;
        for($i=0; $i<$this->nbEnf; $i++) {
          $tnThis =& $this;
          $tn =& $this->tabFils[$i];
          $tn->setFrame($tnThis);

          // largeur du sous arbre de this += largeur du sous arbre du fils tn + espace à droite
          $iSubTreeWidthMax = max($iSubTreeWidthMax, $tn->iBaseWidth);
          // hauteur du trapeze de this : max avec la hauteur du sous arbre du fils tn
          $this->iTrapHeight += $tn->iTrapHeight + ($this->spaceHeight/2);
        }

        // ajoute la hauteur du sous arbre + la connexion
        $this->iBaseWidth = $this->spaceDelta + $iSubTreeWidthMax;

      } else {
        // $this->iTrapHeight 
        //$this->iBaseWidth 
      }
    }
  }

  /**
   * Calcul la position de chaque sous-arbre
   * @param x0 abscisse du coin haut gauche du sous-arbre
   * @param y0 ordonnée du coin haut gauche du sous-arbre
   */
  public function setPosition($x0, $y0)
  {
    // positionne le cadre du sous-arbre
    $this->x0 = $x0;
    $this->y0 = $y0;

    if( $this->iNiveau < $this->limitNiv-1 ) {
      // les fils sont placés sur la largeur
      if( $this->idPere == -1 ) {
        $y0 = $this->y0 + $this->iHeight + ($this->spaceHeight/2);
      } else {
        $y0 = $this->y0 + $this->iHeight + $this->spaceHeight;
      }
      
      for($i=0; $i<$this->nbEnf; $i++) {
        $tn =& $this->tabFils[$i];
        $tn->SetPosition($x0, $y0);
        if( $this->iNiveau < $this->limitNiv-1 ) {
          $x0 += $tn->iBaseWidth + $this->spaceWidth;
        }
      }
    } else {
      // les fils sont placés sur la hauteur
      $x0 += $this->spaceDelta;
      $y0 += $this->iHeight+($this->spaceHeight/2);

      for($i=0; $i<$this->nbEnf; $i++) {
        $tn =& $this->tabFils[$i];
        $tn->SetPosition($x0, $y0);
        $y0 +=  $tn->iTrapHeight+($this->spaceHeight/2);
      }      
    }
  }

  /**
   * @brief Retourne le texte formaté pour le xml
   *
   * @param strText  Texte à convertir
   * @return Retourne un string
   */
  protected function getXmlText($strText)
  {
    $strText = str_replace("&", "&amp;", $strText);
    return $strText;
  }

  /**
   * @brief Retourne le code SVG du nom du service et de la fonction principale dirigeant ce service
   *
   * @param x0  abscisse du début de texte
   * @param y0  ordonnée du début de texte
   * @return Retourne un string
   */
  protected function getTextType($x0, $y0, $type)
  {
    $strSvg = "";
    for($i = 0; $i<count($this->tabIntitule); $i++) {
      $strText = $this->getXmlText($this->tabIntitule[$i]);
      $strSvg .= "<text class='txt".$type."'".
          " x='".$x0."'".
          " y='".$y0."'>".
          //"<a xlink:href=\"javascript:openTrombi('".$this->id."')\">".
          $strText. //"</a>".
          "</text>\n";
      $y0 += $this->spaceLine;
    }

    for($i = 0; $i<count($this->tabDesc); $i++) {
      if( is_string($this->tabDesc[$i]) ) {
        // Nom de la fonction principale
        $strText = $this->getXmlText($this->tabDesc[$i]);
        $strSvg .= "<text class='txt".$type."Desc'".
          " x='".$x0."'".
          " y='".$y0."'".
          /*( $this->strPJ != "" 
            ? " text-decoration='underline'"
            : "" ).*/
          ">".
          /*( $this->strPJ != "" 
            ? "<a xlink:href=\"".$this->strPJ."\" target=\"_blank\">".$strText."</a>"
            : $strText ).*/
          $strText.
          "</text>\n";
        $y0 += $this->spaceLine;
      } else {
        // fonction secondaire
        $strText = $this->getXmlText($this->tabDesc[$i][0]);
        $strPJ = $this->tabDesc[$i][1];
        $strSvg .= "<text class='txtFoncServ'".
          " x='".$x0."'".
          " y='".$y0."'".
          /*( $strPJ != "" 
            ? " text-decoration='underline'"
            : "" ).*/
          ">".
          /*( $strPJ != "" 
            ? "<a xlink:href=\"".$strPJ."\" target=\"_blank\">".$strText."</a>"
            : $strText ).*/
          $strText.
          "</text>\n";
        $y0 += $this->spaceLine;        
      }
    }
    return $strSvg;
  }

  /**
   * retourne le sous-arbre
   * @return string
   */
  public function getSvg()
  {
    $strtext = $this->getXmlText($this->strIntitule);

    $strSvg = "";
    if( $this->iNiveau <  $this->limitNiv ) {
      if( $this->iNiveau <  $this->limitNiv-1 ) {
        $x0 = $this->x0 + (($this->iBaseWidth-$this->iWidth)/2);
        $this->x0 = $x0;
      } else {
        $x0 = $this->x0;
      }
      $y0 = ($this->idPere == -1 ? $this->y0 : $this->y0+($this->spaceHeight/2));
			
      $x1 = $x0 + 5;
      $y1 = $y0 + 12;
      
      $strSvg = $this->getTextType($x1, $y1, $this->type);
      
      $nbLigne = 1;
      $x1 += 10;

      $nbFoncServ = 0;
      if( count($this->tabDesc) > 0 ) {
        for($i = 0; $i<count($this->tabDesc); $i++) {
          if( is_array($this->tabDesc[$i]) ) $nbFoncServ++;
        }
      }
      
      if( $nbFoncServ > 0 ) {
        $iHeightFonc = ($nbFoncServ*$this->spaceLine)+4;
        $iHeightServ = $this->iHeight - $iHeightFonc;

        $strSvg = "<rect".
          " x='".$x0."'".
          " y='".$y0."'".
          " width='".$this->iWidth."'".
          " height='".$iHeightServ."'".
          " class='cadreServ'".
          "/><rect".
          " x='".$x0."'".
          " y='".($y0+$iHeightServ+2)."'".
          " width='".$this->iWidth."'".
          " height='".($iHeightFonc-2)."'".
          " class='cadreFonc'".
          "/>\n".$strSvg;
        
      } else {
        $strSvg = "<rect".
          " x='".$x0."'".
          " y='".$y0."'".
          " width='".$this->iWidth."'".
          " height='".$this->iHeight."'".
          " class='cadre".$this->type."'".
          "/>\n".$strSvg;
      }

      if( $this->iNiveau >1 ) {
        // Ajoute une connexion sur le haut
        $x1 = $x0 + $this->iWidth/2;
        $y1 = $y0 - ($this->spaceHeight/2);
        $strSvg .= "<line".
          " x1='".$x1."'".
          " y1='".$y0."'".
          " x2='".$x1."'".
          " y2='".$y1."'".
          " class='lien'/>\n";
      }

      if( $this->nbEnf > 0 && $this->iNiveau <  $this->limitNiv-1 ) {
        // ajoute la connexion vers les enfants
        $x1 = $x0 + $this->iWidth/2;
        $y1 = $y0 + $this->iHeight;
        $y2 = $y1 + ($this->spaceHeight/2);
        $strSvg .= "<line".
          " x1='".$x1."'".
          " y1='".$y1."'".
          " x2='".$x1."'".
          " y2='".$y2."'".
          " class='lien'/>\n";
        
        // barre de rattachement horizontale
        // x1 : milieu du cadre du sous-arbre du premier fils
        // x2 : milieu du cadre du sous-arbre du dernier fils
        $tn =& $this->tabFils[0];
        $x1 = $tn->x0+($tn->iWidth/2);
        
        $tn =& $this->tabFils[$this->nbEnf-1];
        $x2 = $tn->x0+($tn->iWidth/2);
        
        $strSvg .= "<line".
          " x1='".$x1."'".
          " y1='".$y2."'".
          " x2='".$x2."'".
          " y2='".$y2."'".
          " class='lien'/>\n";
      } else {
        $strSvg .= "\n";
      }
    } else {
      // coin haut à gauche
      $x0 = $this->x0;
      $y0 = $this->y0+($this->spaceHeight/2);
			
      $x1 = $x0 + 5;
      $y1 = $y0 + 12;
      
      $strSvg = $this->getTextType($x1, $y1, $this->type); 
      
      $nbLigne = 1;
      $x1 += 10;

      $strSvg = "<rect".
        " x='".$x0."'".
        " y='".$y0."'".
        " width='".$this->iWidth."'".
        " height='".$this->iHeight."'".
        " class='cadre".$this->type."'".
        "/>\n".$strSvg;
          
      if( $this->iNiveau > 1 ) {
        // Ajoute une connexion sur le haut
        if( $this->iNiveau < $this->limitNiv ) {
          // trait remontant du milieu de la bordure haute
          $x1 = $x0 + $this->iWidth/2;
          $y1 = $y0 - ($this->spaceHeight/2);
          $strSvg .= "<line".
            " x1='".$x1."'".
            " y1='".$y0."'".
            " x2='".$x1."'".
            " y2='".$y1."'".
            " class='lien'/>\n";
        } else {
          // trait partant du côté gauche puis remonte vers le haut
          $x1 = $x0 - ($this->spaceDelta/2);
          $y1 = $y0 + ($this->iHeight/2);
          $strSvg .= "<line".
            " x1='".$x0."'".
            " y1='".$y1."'".
            " x2='".$x1."'".
            " y2='".$y1."'".
            " class='lien'/>\n";
          if( $this->iNumEnf == 0 ) {
            $y2 = $y1 - ($this->iHeight/2) - $this->spaceLine;
          } else {
            // récupère la hauteur du sous arbre du frere précédent
            $ht = $this->nodePere->tabFils[$this->iNumEnf-1]->iTrapHeight;
            $ht2 = $this->nodePere->tabFils[$this->iNumEnf-1]->iHeight/2;
            $y2 = $y1 - $ht - $this->spaceLine - ($this->iHeight/2) + $ht2;
          } 
          $strSvg .= "<line".
            " x1='".$x1."'".
            " y1='".$y1."'".
            " x2='".$x1."'".
            " y2='".$y2."'".
            " class='lien'/>\n";          
        }
      }
    }
    return $strSvg;
  }

}

?>