<?php
/*licence/ 

Module écrit, supporté par la société Alkante SAS <alkante@alkante.com>

Nom du module : Alkanet::Module::Espace
Module gestionnaire d'espaces.
Ce module appartient au framework Alkanet.

Ce logiciel est régi par la licence CeCILL-C soumise au droit français et
respectant les principes de diffusion des logiciels libres. Vous pouvez
utiliser, modifier et/ou redistribuer ce programme sous les conditions
de la licence CeCILL-C telle que diffusée par le CEA, le CNRS et l'INRIA
sur le site http://www.cecill.info.

En contrepartie de l'accessibilité au code source et des droits de copie,
de modification et de redistribution accordés par cette licence, il n'est
offert aux utilisateurs qu'une garantie limitée. Pour les mêmes raisons,
seule une responsabilité restreinte pèse sur l'auteur du programme, le
titulaire des droits patrimoniaux et les concédants successifs.

A cet égard l'attention de l'utilisateur est attirée sur les risques
associés au chargement, à l'utilisation, à la modification et/ou au
développement et à la reproduction du logiciel par l'utilisateur étant
donné sa spécificité de logiciel libre, qui peut le rendre complexe à
manipuler et qui le réserve donc à des développeurs et des professionnels
avertis possédant des connaissances informatiques approfondies. Les
utilisateurs sont donc invités à charger et tester l'adéquation du
logiciel à leurs besoins dans des conditions permettant d'assurer la
sécurité de leurs systèmes et ou de leurs données et, plus généralement,
à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.

Le fait que vous puissiez accéder à cet en-tête signifie que vous avez
pris connaissance de la licence CeCILL-C, et que vous en avez accepté les
termes.

/licence*/


require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CLASSE."pattern/alkquery.class.php");

/**
 * @package Alkanet_Module_Espace
 * @class AlkQueryEspace
 * Ensemble des requetes de type sélection liées à la gestion des espaces de travail
 */
class AlkQueryEspace extends AlkQuery
{

  /**
   * Constructeur par défaut
   */ 
 
  public function __construct(&$oDb) 
  {
    parent::__construct($oDb);
  }
  
  /**
   * Destructeur par défaut
   */
  public function __destruct()
  {  
  }

  /**
   * Retourne la liste de toutes les constantes génériques (appli_id=-1)
   * ainsi que les constantes spécifiques pour l'espace et l'appli courants.
   * Le résultat contient certainement des doublons, il sera nécessaire de tester l'existance avant de déclarer une constante
   * Le résultat est ordonné de telle sorte que les constantes spécifiques sortent avant les génériques.
   * Les constantes de type expression sont placées à la fin du dataset.
   * @param tableau des cont_id   identifiants des espaces
   * @param tableau des appli_id  identifiant du type d'application (à utiliser avec cont_id=-1: toutes appli ou cont_id=appli_id recherché)
   * @param tableau des atype_id  identifiants atype_id
   * 
   * * @return AlkDs
   */
  public function getDsSqlConstant($tabCont=array(), $tabAppli=array(), $tabAtype=array())
  {
    $strSql = "select *".
        " from SIT_CONSTANTE";

    $bWithContId = ( is_array($tabCont) && count($tabCont)==1 && $tabCont[0]=="-1" ? false : true );
    
    $strSql.=  " where ATYPE_ID in (".implode(",", $tabAtype).")".
              " and APPLI_ID in (".implode(",", $tabAppli).")".
              ( $bWithContId ? " and CONT_ID in (".implode(",", $tabCont).")" : "" );
              
    /*if ( $atype_id!=-2 ){
      $strSql .= " ATYPE_ID=".$atype_id.
                 " and APPLI_ID=".( in_array($atype_id, array(ALK_ATYPE_ID_ESPACE, ALK_ATYPE_ID_ANNU)) 
                                    ? $cont_id 
                                    : $appli_id );
    } else {
      $strSql .= "APPLI_ID=-1" .
                 " or (ATYPE_ID in (".ALK_ATYPE_ID_ESPACE.", ".ALK_ATYPE_ID_ANNU.") and APPLI_ID in (".$cont_id."))".
                 " or (ATYPE_ID>0 and APPLI_ID=".$appli_id.")";
    }*/
   $strSql .= " order by CONT_ID desc, APPLI_ID desc, CONST_LEVEL desc, CONST_TYPE desc";  
    return $this->dbConn->InitDataset( $strSql );
  }
  
  /**
   * retourne la valeur d'une constante
   * @param tableau des cont_id   identifiants des espaces
   * @param tableau des appli_id  identifiant du type d'application (à utiliser avec cont_id=-1: toutes appli ou cont_id=appli_id recherché)
   * @param tableau des atype_id  identifiants atype_id
   * 
   * @return valeur de la constante en fonction du type, null si constante non trouvée
   */
  public function getConstantValue($contIntitule="", $tabCont=array(), $tabAppli=array(), $tabAtype=array())
  {
    $value = null;
    
    $strSql = "select CONST_VALEUR, CONST_TYPE".
        " from SIT_CONSTANTE" .
        " where CONST_INTITULE='".$contIntitule."'";
        
    $strSql.=  " and ATYPE_ID in (".implode(",", $tabAtype).")".
               " and APPLI_ID in (".implode(",", $tabAppli).")".
               " and CONT_ID in (".implode(",", $tabCont).")";
    
    $strSql .= " order by CONT_ID desc, APPLI_ID desc, CONST_LEVEL desc, CONST_TYPE desc";  
    
    $oDs = $this->dbConn->InitDataset( $strSql );
    if ( $oDr = $oDs->getRowIter() ) {
      $const_valeur = $drConst->getValueName("CONST_VALEUR");
      $const_type   = $drConst->getValueName("CONST_TYPE");
      
      $bEval = false;
      switch( $const_type ) {
      case "0": // string
      case "2": // date
        $value = $const_valeur;
        break;
    
      case "1": // int
        $value = (int)$const_valeur;
      break;
      case "3": // bool
        $value = ( $const_valeur == "true" ? true : false );
        break;
    
      case "4": // expression php
        $value = eval($const_valeur);
        break;
      }
    }
    
    return $value;
  }
  
  /**
   * retourne les valeurs de constantes sans les évaluer
   * @param appli identifiant de l'appli 
   * @param const_intitule liste des constantes
   */
  public function getDsConstantes($appli_id=-1, $const_intitule=array()){
    $strSql = " select CONST_INTITULE, CONST_VALEUR ".
              " from SIT_CONSTANTE ".
              (!empty($const_intitule)
                ? " where CONST_INTITULE in (".implode(",", $const_intitule).")"
                : "").
              ($appli_id != -1 
                ? (!empty($const_intitule) ? " and " : " where ").
                  " APPLI_ID=". $appli_id
                : ""); 
     return $this->dbConn->InitDataset($strSql);          
  }
  /**
   * Destructeur par défaut
   */
  public function getATypeID($appli_id)
  {
    $atype_id = "-1";
    $strSql = "select ATYPE_ID from SIT_APPLI where APPLI_ID=".$appli_id;
    $dsAppli = $this->dbConn->InitDataSet($strSql);
    if( $drAppli = $dsAppli->getRowIter() )
      $atype_id = $drAppli->getValueName("ATYPE_ID");
    
    return $atype_id;      
  }
  
  /**
   * Retourne un tableau des applications (appli_id) de type atype_id de tous les espaces si cont_id=-1
   * ou d'un espace si cont_id>0
   *
   * @param atype_id identifiant de type d'application
   * @param cont_id  identifiant de l'espace, =-1 par défaut pour toutes les applis de l'extranet
   * @return array
   */
  public function getApplisId($atype_id, $cont_id="-1")
  {
    $appli_id = array();
    $strSql = "select a.APPLI_ID".
      " from SIT_APPLI a".
      ( $cont_id != "-1" ? " inner join SIT_APPLI_CONT ac on a.APPLI_ID=ac.APPLI_ID" : "" ). 
      " WHERE a.ATYPE_ID=".$atype_id.
      ( $cont_id != "-1" ? " and ac.CONT_ID=".$cont_id : "" ).
      " order by a.APPLI_ID"; 
    $dsAppli = $this->dbConn->InitDataset( $strSql );
    while($dsAppli && ($drAppli = $dsAppli->GetRowIter()) ) {
      $appli_id[] = $drAppli->GetValueName("APPLI_ID");
    }
    return $appli_id;
  }

  /**
   * Retourne l'identifiant de l'espace contenant l'appli_id
   * @param appli_id  identifiant de l'application 
   * @return int 
   */
  public function getContId($appli_id)
  {
    $cont_id = -1;
    $strSql = "select CONT_ID from SIT_APPLI_CONT where APPLI_ID=".$appli_id; 
    $dsCont = $this->dbConn->InitDataset( $strSql );
    if( $drCont = $dsCont->GetRowIter() ) {
      $cont_id = $drCont->GetValueName("CONT_ID");
    }
    return $cont_id;
  }

  /**
   * Retourne une instance d'appli de type atype_id contenu dans l'espace si cont_id>0
   * ou de tous les espaces si cont_id=-1 (par défaut).
   * Si plusieurs applis sont trouvées, la première est retournée dans l'ordre de création
   *
   * @param atype_id identifiant de type d'application
   * @param cont_id  identifiant de l'espace, =-1 par défaut pour toutes les applis de l'extranet
   * @return AlkAppli
   */
  public function getAppliBySpace($atype_id, $cont_id="-1")
  {
    $oAppli = null;
    $tabAppliId = $this->getApplisId($atype_id, $cont_id);
    if( count($tabAppliId) > 0 ) { 
      $oAppli = AlkFactory::getAppli($atype_id, $tabAppliId[0]);
    }
    return $oAppli;
  }

  /**
   * Retourne le nombre de département visible
   * @return int
   */
  public function getNbDeptVisible()
  {
    $iNb = 0;
    $strSql = "select count(DEPT_ID) as nb from SIT_DEPT where DEPT_VISIBLE=1";
    $dsNb = $this->dbConn->initDataset($strSql);
    if( $drNb = $dsNb->getRowIter() ) { 
      $iNb = $drNb->getValueName("NB");
      if( $iNb == "" ) $iNb = 0; 
      
    }
    return $iNb;
  }
  
  /**
   * Retourne la liste des identifiants d'utilisateurs valide ou non étant animateur de tous les espaces
   * @param strlistIdUserExcept   liste d'identifiant à exclure
   * @return AlkDs
   */
  public function getDsListUserIdAdminAll($cont_id)
  {
    $currentUser_id = AlkFactory::getSProperty("user_id", "-1");
    $bDept = false;
    $cont_public = "0";

    if( ALK_B_SPACE_ADMIN_ALL_DEPT == true ) {
      $strSql = "select count(DEPT_ID) as nb from SIT_DEPT where DEPT_VISIBLE=1";
      $dsNb = $this->dbConn->initDataset($strSql);
      $iNb = $dsNb->getCountTotDr();
      $bDept = ( $iNb == 0 ? false : true );
        
      $strSql = "select CONT_PUBLIC from SIT_CONTENEUR where CONT_ID=".$cont_id;
      $dsCont = $this->dbConn->initDataset($strSql);
      if( $drCont = $dsCont->getRowIter() ) {
        $cont_public = $drCont->getValueName("CONT_PUBLIC");
      }
    }    
    
    if( !$bDept ) { 
      // sélection de tous les animateurs de tous les espaces
      $strSql = "select AGENT_ID".
        " from SIT_AGENT".
        " where AGENT_VALIDE>=0 and (AGENT_PRIV & ".ALK_PRIV_SPACE_ADMIN."=".ALK_PRIV_SPACE_ADMIN.")";
    } else {
      if( $cont_public == "0" ) {
        // espace privé, on ne sélectionne aucun animateur de tous les espaces
        $strSql = "select AGENT_ID from SIT_AGENT where AGENT_ID=-99";
      } else {
        // espace public : on sélectionne
        $strSql = "select distinct a.AGENT_ID".
          " from SIT_AGENT a".
          " where a.AGENT_ID=".ALK_USER_ID_ADMINALK.        
          " union".
          " select distinct a.AGENT_ID".
          " from SIT_AGENT a".
          "   inner join SIT_AGENT_SERVICE ase on a.AGENT_ID=ase.AGENT_ID".
          "   inner join SIT_SERVICE s on (ase.SERVICE_ID=s.SERVICE_ID and s.SERVICE_NIVEAU=1 and s.SERVICE_ETAT=1)".
          "   inner join SIT_DEPT_CONT dc on s.DEPT_ID=dc.DEPT_ID".
          "   inner join SIT_CONTENEUR c on (dc.CONT_ID=c.CONT_ID and c.CONT_PUBLIC=1 and c.CONT_ID=".$cont_id.")".
          " where a.AGENT_VALIDE>-1 and a.AGENT_ID<>".ALK_USER_ID_ADMINALK." and (a.AGENT_PRIV & ".ALK_PRIV_SPACE_ADMIN.")>0";
      }        
    }
    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   * Retourne la liste des idUsers valide (ou non) 
   * appartenant à un service de type etat lié aux departements associé à l'espace cont_id 
   * et n'appartenant pas déjà à l'espace cont_id
   * @param cont_id  identifiant de l'espace
   * @return AlkDs
   */
  public function getDsListUserIdServEtat($cont_id)
  {
    $bDept = false;
    if( ALK_B_SPACE_ADMIN_ALL_DEPT == true ) {
      $strSql = "select count(DEPT_ID) as nb from SIT_DEPT where DEPT_VISIBLE=1";
      $dsNb = $this->dbConn->initDataset($strSql);
      $iNb = $dsNb->getCountTotDr();
      $bDept = ( $iNb == 0 ? false : true );
    }   
     
    // si $dept  : selection des utils de service de type état liés aux depts de cont_id et n'appartenant pas à cont_id
    // si !$dept : selection des utils de service de type état n'appartenant pas à cont_id
    $strSql = "select a.AGENT_ID".
      " from SIT_AGENT a".
      "   inner join SIT_AGENT_SERVICE ase ON a.AGENT_ID = ase.AGENT_ID".
      "   inner join SIT_SERVICE s ON ase.SERVICE_ID = s.SERVICE_ID".
      ( $bDept
        ? "   inner join SIT_DEPT_CONT dc ON ( s.DEPT_ID = dc.DEPT_ID AND dc.CONT_ID=".$cont_id.")"
        : "" ).
      "   left join SIT_AGENT_CONT ac on (a.AGENT_ID=ac.AGENT_ID and ac.CONT_ID=".$cont_id.")".
      " where a.AGENT_VALIDE >=0".
      "   and s.SERVICE_ETAT=1 and s.SERVICE_NIVEAU=1".
      "   and ac.CONT_ID is null".
      " group by a.AGENT_ID";
    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   * Retourne la liste des profils associés à une appli
   * @param appli_id  identifiant de l'application, =vide si pas d'appli
   * @return AlkDs
   */
  public function getDsProfilListByAppli($appli_id="-1", $cont_id="-1")
  {
    if( $appli_id == "-1" && $cont_id!="-1" ) {
      $strSql = "select ".(defined("ALK_ESPACE_APPLIDEFAULTRIGHT") ? ALK_ESPACE_APPLIDEFAULTRIGHT : "1" )." as DROIT_ID, p.*".
        " from SIT_PROFIL p".
        " where p.CONT_ID=0 or p.CONT_ID=".$cont_id.
        " order by p.PROFIL_INTITULE";

    } else {
      $strSql = "select pa.APPLI_DROIT_ID as DROIT_ID, p.*".
        " from SIT_PROFIL p".
        "   left join SIT_PROFIL_APPLI pa on p.PROFIL_ID=pa.PROFIL_ID".
        " where pa.APPLI_ID=".$appli_id.
        ( $cont_id!="-1" ? " and (p.CONT_ID=0 or p.CONT_ID=".$cont_id.")" : "" ).
        " order by p.PROFIL_INTITULE";
    }

    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   * Retourne la liste des applications de l'espace avec pour chaque :
   *        les droits de chaque profil de l'utilisateur
   * @param nbGroupPerUser retourne le nombre de groupe associé à user_id
   * @param cont_id        identifiant de l'espace
   * @param agent_id       identifiant de l'utilisateur sélectionné
   * @return AlkDs
   */ 
  function getDsUserRightOnSpace(&$nbGroupPerUser, $cont_id, $user_id)
  {
    $nbGroupPerUser = 0;
    $strSql = "select count(ap.PROFIL_ID) as NB_GROUP".
      " from SIT_AGENT_PROFIL ap" .
      "   inner join SIT_PROFIL p on ap.PROFIL_ID=p.PROFIL_ID".
      " where (p.CONT_ID=0 or p.CONT_ID=".$cont_id.") and ap.AGENT_ID=".$user_id;
    $dsNb = $this->dbConn->InitDataSet($strSql);
    if( $drNb = $dsNb->getRowIter() ) {
      $nbGroupPerUser = $drNb->getValueName("NB_GROUP");
    }

    $strSql = "select a.APPLI_ID, a.APPLI_INTITULE, aa.APPLI_DROIT_ID as DROIT_ID".
      " from SIT_APPLI_CONT ac".
      "  inner join SIT_APPLI a on a.APPLI_ID=ac.APPLI_ID".
      "  left join SIT_AGENT_APPLI aa on (aa.APPLI_ID=a.APPLI_ID and aa.AGENT_ID=".$user_id.")".
      " where ac.CONT_ID=".$cont_id.
      " order by a.APPLI_INTITULE";

    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   * Retourne la liste des droits d'un profil sur toutes les applis d'un espace
   * @param cont_id    identifiant de l'espace
   * @param profil_id  identifiant du profil sélectionné      
   * @return AlkDs
   */
  public function getProfilAppliRight($cont_id, $profil_id)
  {
    $strSql = "select *".
       " from V_PROFIL_RIGHT_SPACE".
       " where CONT_ID=".$cont_id." and PROFIL_ID=".$profil_id. 
       " order by PROFIL_INTITULE, APPLI_INTITULE";
     return $this->dbConn->InitDataSet($strSql);
  }

  /**
   * Retourne le droit utilisateur sur l'appli en cours
   * @param appli_id  identifiant de l'application
   * @param user_id   identifiant de l'utilisateur
   */
  public function getUserAppliRight($appli_id, $user_id)
  {
    $iRight = ALK_APPLI_RIGHT_NONE;
    $strSql = "select * from V_USER_RIGHT where APPLI_ID=".$appli_id." and USER_ID=".$user_id;
    $dsRight = $this->dbConn->InitDataSet($strSql);
    if( $drRight = $dsRight->getRowIter() ) {
      $right_id   = $drRight->getValueName("RIGHT_ID"); 
      $cont_admin = $drRight->getValueName("CONT_ADMIN");
      $iRight = ( $cont_admin == "1"
                  ? ALK_APPLI_RIGHT_PUBLI
                  : $right_id );
    }
    return $iRight;
  }

  /**
   * Retourne la liste des utilisateurs de profil profil_id ayant accès l'espace cont_id
   * @param cont_id   identifiant de l'espace
   * @param profil_id identifiant du profil
   * @return AlkDs
   */
  public function getDsUserListByProfilCont($cont_id, $profil_id)
  {
    $strSql = "select * from V_USER_SPACE_PROFIL where CONT_ID=".$cont_id." and PROFIL_ID=".$profil_id;
    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   * Retourne le nom de l'application
   *
   * @param appli_id Identifiant de l'application
   * @return Retourne un string : Le nom de l'application
   */
  function GetStr_applicationIntitule($appli_id)
  {
    $strRes = "";
    $strSql = "select APPLI_INTITULE from SIT_APPLI where APPLI_ID=".$appli_id;
    $dsAppli = $this->dbConn->InitDataSet($strSql);
    if( $drAppli = $dsAppli->getRowIter() )
      $strRes = $drAppli->getValueName("APPLI_INTITULE");
    
    return $strRes;      
  }
  
  /**
   * retourne l'appli_id d'une appli à partir de son intitulé
   * @param $appli_intitule
   * @param $cont_id
   */
  public function GetAppliIdByIntitule($appli_intitule,$cont_id){
     $appli_id = 0;
     $strSql = "SELECT * ".
     " FROM SIT_APPLI s inner join SIT_APPLI_CONT si on s.APPLI_ID = si.APPLI_ID".
     " WHERE s.APPLI_INTITULE = '".$appli_intitule."'".
     " AND si.CONT_ID=".$cont_id;
     
     $dsAppli = $this->dbConn->InitDataSet($strSql);
    if( $drAppli = $dsAppli->getRowIter() )
      $appli_id = $drAppli->getValueName("APPLI_ID");
      
    return $appli_id;
  }
  
 /**
   * Retourne les propriétés de l'application distante appli_id
   *
   * @param appli_id Identifiant de l'application distante
   * @return Retourne un dataSet
   */
  function GetDs_ficheAppliDistById($appli_id)
  {
    $strSql = "select a.*, at.ATYPE_ID_DIST".
      " from SIT_APPLI a, SIT_APPLI_TYPE at".
      " where a.APPLI_ID=".$appli_id." and a.ATYPE_ID=at.ATYPE_ID";    
    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   * Retourne les propriétés de l'application appli_id
   *
   * @param appli_id   Identifiant de l'application
   * @param bWithAgent vrai pour retourne le nom des agents créateur et modificateur, faux sinon par défaut
   * @return Retourne un dataSet
   */
  public function GetDs_ficheAppliById($appli_id, $bWithAgent=false)
  {
    $strSql = "select a.*,".
      $this->dbConn->GetDateFormat("DD/MM/YYYY, HH:MI", "a.APPLI_DATECREA", false)." as appli_date_crea,".
      $this->dbConn->GetDateFormat("DD/MM/YYYY, HH:MI", "a.APPLI_DATEMAJ", false)." as appli_date_maj".
      ( $bWithAgent
        ? ", ".$this->dbConn->getConcat("ac.AGENT_PRENOM", "' '", "ac.AGENT_NOM")." as agent_nomcrea".
          ", ".$this->dbConn->getConcat("am.AGENT_PRENOM", "' '", "am.AGENT_NOM")." as agent_nommaj"
        : "" ).
      ",".$this->dbConn->compareSql("c.CONT_APPLI_DEFAUT", "=", "a.APPLI_ID", "1", "0")." as APPLI_DEFAUT".
      ",".$this->dbConn->compareSql("at.ATYPE_URL", "=", "''", "1", $this->dbConn->compareSql("at.ATYPE_URL", "is", "null", "1", "0"))." as CAN_DEFAULT".
      " from SIT_CONTENEUR c".
      "   inner join SIT_APPLI_CONT aco on c.CONT_ID=aco.CONT_ID".
      "   inner join SIT_APPLI a on aco.APPLI_ID=a.APPLI_ID".
      "   inner join SIT_APPLI_TYPE at on a.ATYPE_ID=at.ATYPE_ID".
      ( $bWithAgent
        ? "   inner join SIT_AGENT ac on (a.AGENT_IDCREA=ac.AGENT_ID)".
          "   inner join SIT_AGENT am on (a.AGENT_IDMAJ=am.AGENT_ID)"
        : "" ).
      " where a.APPLI_ID=".$appli_id;    
    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   *
   *
   */
  function GetStr_nextRangAppli($cont_id)
  {
    $iRang = 0;
    $strSql = "select max(a.APPLI_RANG) as max_rg".
      " from".
      "   SIT_APPLI a,".
      "   SIT_APPLI_CONT ac".
      " where a.APPLI_ID=ac.APPLI_ID and ac.CONT_ID=".$cont_id;

    $dsRg = $this->dbConn->initDataset($strSql);
    if( $drRg = $dsRg->getRow() )
      $iRang = $drRg->getValueName("MAX_RG");

    $iRang++;
    return $iRang;
  }
  
  /**
   * Retourne les propriétés de l'espace cont_id
   *        actu/01_sommaire.php classes/alkWorkspace.class.php
   *
   * @param cont_id    Identifiant de l'espace
   * @param strDefaut  valeur de l'alias defaut dans le select, vide par défaut
   * @param bWithAgent false par défaut, true pour avoir le nom et prénom des agents créateurs et modificateurs, si user_id!=-1, bWithAgent=false
   * @param user_id    identifiant de l'agent connecté pour retourner son privilege sur cont_id (CONT_ADMIN), non présent si user_id=-1
   * @return Retourne un dataSet
   */
  public function getDs_ficheEspaceById($cont_id, $strDefaut="", $bWithAgent=false, $user_id=-1)
  {
    if ($strDefaut != "")
      $strDefaut = ", ".$strDefaut." as defaut";
    $strSelect = "";
    $strFrom   = "";
    if( $user_id != -1 ) {
      $strSelect = $this->dbConn->compareSql("agc.CONT_ADMIN","is", "null", "-1", "agc.CONT_ADMIN")." as CONT_ADMIN,";
      $strFrom  = " left join SIT_AGENT_CONT agc on (c.CONT_ID=agc.CONT_ID and agc.AGENT_ID=".$user_id.")";
      $bWithAgent = false;
    }
      
    $strSql = "select c.*, ".$strSelect.
      $this->dbConn->GetDateFormat("DD/MM/YYYY, HH:MI", "c.CONT_DATE_CREA", false)." as cont_date_crea,".
      $this->dbConn->GetDateFormat("DD/MM/YYYY, HH:MI", "c.CONT_DATE_MAJ", false)." as cont_date_maj".
      $strDefaut.
      ( $bWithAgent
        ? ", ".$this->dbConn->getConcat("ac.AGENT_PRENOM", "' '", "ac.AGENT_NOM")." as agent_nomcrea".
          ", ".$this->dbConn->getConcat("am.AGENT_PRENOM", "' '", "am.AGENT_NOM")." as agent_nommaj"
        : "" ).
      " from SIT_CONTENEUR c".
      ( $bWithAgent
        ? "   inner join SIT_AGENT ac on (c.AGENT_ID=ac.AGENT_ID)".
          "   inner join SIT_AGENT am on (c.AGENT_ID_MAJ=am.AGENT_ID)"
        : "" ).
      $strFrom.
      " where c.CONT_ID=".$cont_id;
    return $this->dbConn->InitDataSet($strSql);
  }
  
  /**
   * Retourne la liste des espaces accessibles pour l'agent_id pour un combobox
   * @param agent_id identifiant de l'agent
   * @return AlkDs
   */
  public function getDsListSpaceByAgentForComboMaj($agent_id)
  {
    $strSql = "select ".
      $this->dbConn->GetSubstring("CONT_INTITULE", 0, 55).", c.CONT_ID".
      " from SIT_AGENT_CONT ac".
      "   inner join SIT_CONTENEUR c on c.CONT_ID=ac.CONT_ID".
      " where ac.AGENT_ID=".$agent_id." and c.CONT_NIVEAU>0".
      " order by c.CONT_INTITULE";

    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   * Retourne la liste des espaces pour un démarrage par défaut aux agents d'un service
   *        annu/07_service_form.php
   *
   * @return Retourne un dataSet pour un comboBox
   */
  function GetDs_listeEspaceByServiceForComboMaj()
  {
    $strSql = "select ".
      $this->dbConn->GetSubstring("CONT_INTITULE", 0, 55).", CONT_ID".
      " from SIT_CONTENEUR".
      " where CONT_NIVEAU>0".
      " order by CONT_INTITULE";

    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   * Retourne la liste des applis de l'espace cont_id
   * @param cont_id Identifiant de l'espace
   * @param idFirst Indice de pagination : début
   * @param idLast  Indice de pagination : fin
   * @param bWithStaticAppli  true pour afficher les applis statiques, false sinon (par défaut)
   * @param bWithAppliLiee    true pour afficher les applis liées (par défaut), false sinon
   * @param atype_id          Identifiant de type d'appli, =-1 pour tous (par défaut)
   * @return AlkDs
   */ 
  public function getDs_listeAppliByEspace($cont_id, $idFirst=0, $idLast=-1, $bWithStaticAppli=false, $bWithAppliLiee=true, $atype_id=-1)
  {
    $strSql = "select a.APPLI_ID, a.APPLI_INTITULE, a.APPLI_RANG, a.ATYPE_ID, a.APPLI_LIEE,".
      " at.ATYPE_INTITULE, at.ATYPE_COMPATIBLE".
      " from SIT_APPLI_CONT ac".
      "   inner join SIT_APPLI a on ac.APPLI_ID=a.APPLI_ID".
      "   inner join SIT_APPLI_TYPE at on a.ATYPE_ID=at.ATYPE_ID".
      " where ac.CONT_ID=".$cont_id." and a.APPLI_ID>0 ".
      ($atype_id != -1  ? " and a.ATYPE_ID=".$atype_id : -1).
      ( !$bWithAppliLiee
	    ? " and a.APPLI_LIEE is null"
		: "" ).
      ( $bWithStaticAppli
        ? " or a.APPLI_ID<-3"
        : "" ).
      " order by a.APPLI_RANG, a.APPLI_INTITULE";

    //echo $strSql."<br/>";
    return $this->dbConn->InitDataSet($strSql, $idFirst, $idLast);
  }

  /**
   * Retourne la liste des types d'application
   * @param strListAtypeId  Liste des identifiants de type d'appli (séparés par une virgule)
   * @param strDefaut       nom de la colonne ou valeur pour l'alias default, ="" par défaut
   * @param user_priv       Permet de filtrer les applis compatibles si =-1 (par défaut)
   *                        affiche tous les types si =ALK_PRIV_SPACE_ADMIN
   * @param bDispoOnly      true pour n'afficher que les applis dispos, false pour toute
   * @return Retourne un dataSet
   */ 
  public function getDs_listeTypeAppliByListId($strListAtypeId, $strDefaut="", $user_priv="-1", $bDispoOnly=false)
  {
    if( $strDefaut != "" )
      $strDefaut = ", ".$strDefaut." as DEFAUT";
    
    $strWhere = " where 1=1";
    if( $bDispoOnly == true)
      $strWhere .= " and ATYPE_DISPO=1";
    if( $user_priv != "-1" && ($user_priv & ALK_PRIV_SPACE_ADMIN) != ALK_PRIV_SPACE_ADMIN )
      $strWhere .= " and ATYPE_COMPATIBLE=1";

    if( $strListAtypeId != "" )
      $strWhere .= " and ATYPE_ID in (".$strListAtypeId.")";

    $strSql = "select ATYPE_INTITULE, ATYPE_ID, ATYPE_URL,".
      " ATYPE_LOGO, ATYPE_URL_ADMIN, ATYPE_URL_TARGET, ATYPE_ADMIN_LOGO".$strDefaut.
      " from SIT_APPLI_TYPE ".$strWhere.
      " order by ATYPE_INTITULE";
    
    return $this->dbConn->InitDataSet($strSql);
  }
  
  /**
   * Retourne la liste des actions de tâche planifiée en fonction du type d'application
   * @param atype_id    identifiant de l'application (defaut -1 = Toutes)
   * @return dataset contenant les actions
   */
  public function getDs_listeCronAction($atype_id=-1)
  {
    $strSql = "select * from SIT_CRON_ACTION";
    
    if ( $atype_id != -1 ) {
      $strSql.= " where ATYPE_ID=".$atype_id;
    }
    
    return $this->dbConn->initDataSet($strSql);
  }
  
  /**
   * Retourne les propriétés d'une tâche planifiée
   * @param task_id   identifiant de la tâche planifiée
   * @return dataset
   */
  public function getDs_ficheCronTaskById($task_id)
  {
    $strSql = "select *" .
        ", ".$this->dbConn->getDateFormat("DD/MM/YYYY", "TASK_DATE", false)." as TASK_DATE" .
        " from SIT_CRON_TASK sct" .
        " where TASK_ID=".$task_id;
    
    return $this->dbConn->initDataSet($strSql);
  }
  
  /**
   * Retourne la liste des tâches en fonction des paramètres
   * @param task_id           identifiant d'une tâche en particulier (defaut -1 = Toutes)
   * @param cont_id           identifiant de l'espace (defaut -1 = Tous)
   * @param atype_id          identifiant du type d'application (defaut -1 = Tous)
   * @param appli_id          identifiant de l'application (defaut -1 = Toutes)
   * @param action_id         identifiant de l'action (defaut -1 = Toutes)
   * @param task_typetask     type de tâche
   *                            defaut -1 : Tout type de tâche
   *                            0 : tâches périodiques
   *                            1 : tâches ponctuelles
   * @return dataset contenant la liste des tâches
   *         null si atype_id non fourni
   */
  public function getDs_listeCronTask($cont_id=-1, $atype_id=-1, $appli_id=-1, $action_id=-1, $task_typetask=-1, $iFirst=0, $iLast=-1)
  {
    if ( !$atype_id ) return null;
    
    $strSql = "select sct.*, sca.*" .
        ", ".$this->dbConn->getDateFormat("DD/MM/YYYY HH:MI", "TASK_DATECREA", false)." as TASK_DATECREA" .
        ", ".$this->dbConn->getDateFormat("DD/MM/YYYY", "TASK_DATE", false)." as TASK_DATE" .
        ", ".$this->dbConn->getDateFormat("DD/MM/YYYY HH:MI", "TASK_DATELASTEXEC", false)." as TASK_DATELASTEXEC" .
        " from SIT_CRON_TASK sct" .
        " inner join SIT_CRON_ACTION sca on sca.ACTION_ID=sct.ACTION_ID" .
        " where 1=1" .
        ( $atype_id != -1     ? " and sct.ATYPE_ID=".$atype_id            : "" ) .
        ( $cont_id > -1      ? " and (sct.CONT_ID=".$cont_id." or sct.CONT_ID=0 )"              : "" ) .
        ( $appli_id > -1      ? " and sct.APPLI_ID=".$appli_id            : "" ) .
        ( $action_id > -1     ? " and sct.ACTION_ID=".$action_id          : "" ) .
        ( $task_typetask > -1 ? " and sct.task_TYPETASK=".$task_typetask  : "" ) .
        " order by sct.TASK_DATECREA";
    
    return $this->dbConn->initDataSet($strSql, $iFirst, $iLast);
  }
  
  /**
   * Retourne la fiche du type d'application dont l'abréviation est passée en paramètre
   *        classes/alkserviceappli.php
   *
   * @param strAbrev Abreviation d'une application
   * @return Retourne un dataSet
   */
  function GetDs_ficheTypeAppliByAbrev($strAbrev)
  {
    $strSql = "select ATYPE_INTITULE, ATYPE_ID".
      " from SIT_APPLI_TYPE ".
      " where ATYPE_ABREV = '".$strAbrev."'";

    return $this->dbConn->InitDataSet($strSql);
  }
  
  /**
   * Retourne le droit sur le profil d'une appli
   *        annu/09_profil_droit.php, 07_appli_droit.php
   *
   * @param appli_id  Identifiant de l'application
   * @param profil_id Identifiant du profil
   * @return Retourne un string : le droit du profil sur une application
   */
  function GetStr_idDroitByAppliProfil($appli_id, $profil_id)
  {
    $strDroit = "0";
    $strSql = "select APPLI_DROIT_ID".
      " from SIT_PROFIL_APPLI ".
      " where PROFIL_ID=".$profil_id." and APPLI_ID=".$appli_id;
    $dsDroit = $this->dbConn->InitDataSet($strSql);
    if( $drDroit = $dsDroit->getRowIter() )
      $strDroit = $drDroit->getValueName("APPLI_DROIT_ID");
    
    return $strDroit;
  }

  /**
   * Retourne la liste des agents ayant accès l'espace cont_id
   *        annu/01_user_form.php, espace/05_agent_list.php
   *
   * @param fagent_nom Filtre sur le nom de l'agent ou sur le nom du service
   * @param cont_id    Identifiant de l'espace
   * @param idFirst    Indice de pagination : début
   * @param idLast     Indice de pagination : fin
   * @param fservice   =1 : filtre sur service, = 0 : filtre sur le nom
   * @return Retourne un dataSet
   */
  function GetDs_listeAgentByEspace($fagent_nom, $cont_id="", $iLgMax=55, $idFirst=0, $idLast=-1, 
                                    $fService="0", $profil_id=-1, $appli_id=-1)
  {
    // utilisé pour un combo : 1er champ nom de l'agent, 2nd champ agent_id, puis le reste
    $strSelectAgent = $this->dbConn->GetConcat($this->dbConn->GetUpperCase("a.AGENT_NOM"),"' '", 
                                               "a.AGENT_PRENOM");
    if( $iLgMax != "-1" ) {
      $strSelect = $this->dbConn->GetConcat($strSelectAgent, "' - '", "s.SERVICE_INTITULE_COURT");
      $strSelect = $this->dbConn->GetSubstring($strSelect,0, $iLgMax);
    } else {
      $strSelect = $strSelectAgent;
    }

    $strFrom = "";
    $strWhere = " a.AGENT_VALIDE>-1 and s.SERVICE_NIVEAU=1";
    $strSelectCont = "";
    if( $cont_id != "" ) {
      $strFrom = " inner join SIT_AGENT_CONT ac on a.AGENT_ID=ac.AGENT_ID";
      $strWhere .= " and ac.CONT_ID=".$cont_id;
      $strSelectCont = ", ac.CONT_ADMIN, ac.CONT_DROIT";
    }

    /*if( $appli_id != "-1" ) {
      // filtre sur appli et affichage des droits
      $strFrom = " left join SIT_AGENT_APPLI aa on a.AGENT_ID=aa.AGENT_ID".
        " left join SIT_AGENT_PROFIL ap on a.AGENT_ID=ap.AGENT_ID".
        " left join SIT_PROFIL_APPLI pa on pa.PROFIL_ID=ap.PROFIL_ID".
      $strWhere .= " and aa.APPLI_ID=".$appli_id;
      $strSelectCont = ", aa.APPLI_DROIT_ID as USER_RIGHT";
    }
    
    if( $profil_id != "-1" ) {
      $strFrom .= " inner join SIT_AGENT_PROFIL ap on a.AGENT_ID=ap.AGENT_ID".
        " inner join SIT_PROFIL p on ap.PROFIL_ID=p.PROFIL_ID";
      $strWhere .= " and ".$this->dbConn->GetLowerCase("p.PROFIL_INTITULE")." like ".
        $this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($fagent_nom)."%'");
    }*/

    if( $fagent_nom != "" ) {
      if( $fService == "0" ) {
        // filtre sur non agent
        $strWhere .= " and ".$this->dbConn->GetLowerCase("a.AGENT_NOM")." like ".
          $this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($fagent_nom)."%'");
      } else {
        // filtre sur nom service
        $strWhere .= " and ".$this->dbConn->GetLowerCase("s.SERVICE_INTITULE_COURT")." like ".
        $this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($fagent_nom)."%'");        
      }
    }
    
    $strSql = "select ".$strSelectAgent." as NOM_PRENOM, a.AGENT_ID, a.AGENT_NOM, a.AGENT_PRENOM, a.PROFIL_ID,".
      " s.SERVICE_ID, s.SERVICE_INTITULE_COURT, ".$strSelect." as NOM".$strSelectCont.
      " from SIT_AGENT a".
      "   inner join SIT_AGENT_SERVICE ase on a.AGENT_ID=ase.AGENT_ID".
      "   inner join SIT_SERVICE s on ase.SERVICE_ID=s.SERVICE_ID".
      $strFrom.
      " where ".$strWhere.
      " order by a.AGENT_NOM, a.AGENT_PRENOM";
    return $this->dbConn->InitDataSet($strSql, $idFirst, $idLast);
  }
  
  /**
   * Retourne la liste des agents ayant accès l'espace cont_id
   *        annu/01_user_form.php, espace/05_agent_list.php
   *
   * @param fagent_nom Filtre sur le nom de l'agent ou sur le nom du service
   * @param cont_id    Identifiant de l'espace
   * @param idFirst    Indice de pagination : début
   * @param idLast     Indice de pagination : fin
   * @param fservice   =1 : filtre sur service, = 0 : filtre sur le nom
   * @return Retourne un dataSet
   */
  function GetDs_listeAgentByService($service_id, $cont_id="")
  {
    // utilisé pour un combo : 1er champ nom de l'agent, 2nd champ agent_id, puis le reste
    $strCivil = $this->dbConn->CompareSql("a.AGENT_CIVILITE", "is", "null", "' '", "a.AGENT_CIVILITE");
    $strSelectAgent = $this->dbConn->GetConcat($strCivil, "' '",
                                               $this->dbConn->GetUpperCase("a.AGENT_NOM"),"' '", 
                                               "a.AGENT_PRENOM");
 
    $strSelect = $strSelectAgent;

    $strFrom = "";
    $strWhere = " a.AGENT_VALIDE>-1 and s.SERVICE_NIVEAU=1";
    $strSelectCont = "";
    if( $cont_id != "" ) {
      $strFrom = ", SIT_AGENT_CONT ac";
      $strWhere .= " and a.AGENT_ID=ac.AGENT_ID and ac.CONT_ID=".$cont_id;
      $strSelectCont = ", ac.CONT_ADMIN";
    }

    if( $service_id == "-1" ) {
      $strWhere .= "";
    } else {
      $strWhere .= " and s.SERVICE_ID = ".$service_id;
    }
    
    $strSql = "select ".$strSelectAgent." as NOM_PRENOM, a.AGENT_ID, a.AGENT_NOM, a.AGENT_PRENOM, a.PROFIL_ID,".
      " s.SERVICE_ID, s.SERVICE_INTITULE_COURT, ".$strSelect." as NOM".$strSelectCont.
      " from".
      "   SIT_AGENT a,".
      "   SIT_SERVICE s,".
      "   SIT_AGENT_SERVICE ase".
      $strFrom.
      " where ".$strWhere." and a.AGENT_ID=ase.AGENT_ID and ase.SERVICE_ID=s.SERVICE_ID".
      " order by a.AGENT_NOM, a.AGENT_PRENOM";
    return $this->dbConn->InitDataSet($strSql);
  }
  
  
  /**
   * Retourne le dataset de l'acces à un espace pour un agent
   *        sit_menu/login_verif.php
   *        lib/lib_secu.php
   *
   * @param cont_id Identifiant de l'espace
   * @param agent_id Identifiant de l'agent
   * @param strCond Condition complémentaire
   * @return Retourne un dataSet
   */
  public function getDs_accesEspaceAgentById($cont_id, $agent_id, $strCond="")
  {
    $strSql = "select ac.AGENT_ID, ac.CONT_ADMIN".
      " from SIT_AGENT_CONT ac".
      ( $strCond != ""
        ? " inner join SIT_CONTENEUR c on ac.CONT_ID=c.CONT_ID"
        : "" ).
      " where ac.AGENT_ID=".$agent_id." and ac.CONT_ID=".$cont_id.
      ( $strCond != "" ? " and ".$strCond : "" );
    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   * Retourne le dataset de l'acces à une application pour un agent
   *        sit_menu/login_verif.php
   *        lib/lib_secu.php
   *
   * @param cont_id Identifiant de l'espace
   * @param agent_id Identifiant de l'agent
   * @param strCond Condition complémentaire
   * @return Retourne un dataSet
   */
  function GetDs_accesAppliAgentById($appli_id, $agent_id, $strCond="")
  {
    $strSql = "select aa.AGENT_ID, aa.APPLI_DROIT_ID from SIT_AGENT_APPLI aa, SIT_APPLI a ".
      "where aa.AGENT_ID=".$agent_id." and aa.APPLI_ID=".$appli_id.
      " and aa.APPLI_ID=a.APPLI_ID ".$strCond;
              
    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   * Retourne la fiche d'un conteneur
   * sit_menu/login_verif.php
   *
   * @param $agent_id        identifiant de l'agent
   * @param $nc              intitule
   * @param $strCondProfil   condition complémentaire liée au profil
   * @return Retourne un dataSet
   */
  function GetDs_ficheEspaceByAgentIntitule($agent_id, $nc, $strCondProfil="")
  {
    $strWhere = "";
    if( $nc != "" )
      $strWhere = " and ".$this->dbConn->GetLowerCase("c.CONT_ALIAS").
        " like ".$this->dbConn->GetLowerCase("'".$this->dbConn->analyseSQL($nc)."'");

    $strSql = "select c.CONT_ID".
      " from".
      "   SIT_CONTENEUR c,".
      "   SIT_AGENT_CONT ac".
      " where c0.CONT_ID=ac.CONT_ID and ac.AGENT_ID=".$agent_id.$strWhere;

    return $this->dbConn->initDataset($strSql) ;    
  }

  /*
   * Retourne les propriétés de l'espace, de l'appli et de son type pour le tableau tabProperty de AlkAppli
   *
   * @param appli_id        identifiant de l'application
   * @param user_id         Retourne le champ cont_admin si user_id>-1
   * @return Retourne un dataSet
   */
  function getDsFicheProperty($appli_id, $user_id=-1)
  {
    $strJoin   = ( $user_id != -1 ? " left" : " inner" );
    $strSelect = ( $user_id != -1 ? $this->dbConn->compareSql("agc.CONT_ADMIN","is", "null", "-1", "agc.CONT_ADMIN")." as CONT_ADMIN," : "" );
    $strFrom   = ( $user_id != -1 ? " left join SIT_AGENT_CONT agc on (c.CONT_ID=agc.CONT_ID and agc.AGENT_ID=".$user_id.")" : "" );

    $strSql = "select c.*, a.*, ".$strSelect.
      " at.ATYPE_INTITULE, at.ATYPE_URL, at.ATYPE_URL_ADMIN, at.ATYPE_LOGO, at.ATYPE_ADMIN_LOGO, at.ATYPE_URL_TARGET,".
      $this->dbConn->compareSql("a.APPLI_ID", "=", "c.CONT_APPLI_DEFAUT", "1", "0")." as DEFAUT".
      " from SIT_APPLI a".
      $strJoin." join SIT_APPLI_TYPE at on a.ATYPE_ID=at.ATYPE_ID".
      $strJoin." join SIT_APPLI_CONT ac on ac.APPLI_ID=a.APPLI_ID".
      $strJoin." join SIT_CONTENEUR c on c.CONT_ID=ac.CONT_ID".
      $strFrom.
      " where".
      "  a.APPLI_ID=".$appli_id;
    return $this->dbConn->initDataset($strSql) ;    
  }

  /**
   * Retourne les propriétés du conteneur et de l'appli donnés
   *        lib/lib_sqlmenu.php
   *
   * @param $cont_id          identifiant du conteneur
   * @param $appli_id         identifiant de l'application
   * @param $strCondProfil    condition complémentaire liée au profil
   * @return Retourne un dataSet
   */
  function GetDs_ficheEspaceAppliById($cont_id, $appli_id, $strCondProfil="")
  {
    $strSql = "select c.CONT_ID, c.CONT_APPLI_DEFAUT, c.CONT_INTITULE, c.CONT_INTITULE_COURT, c.CONT_LOGO, c.CONT_PUBLIC,".
      " a.APPLI_ID, a.APPLI_INTITULE,".
      " at.ATYPE_ID, at.ATYPE_URL, at.ATYPE_URL_ADMIN, at.ATYPE_LOGO, at.ATYPE_ADMIN_LOGO, at.ATYPE_URL_TARGET,".
      $this->dbConn->compareSql("a.APPLI_ID", "=", "c.CONT_APPLI_DEFAUT", "1", "0")." as DEFAUT".
      " from ".
      "   SIT_APPLI a, ".
      "   SIT_APPLI_TYPE at, ".
      "   SIT_APPLI_CONT ac,".
      "   SIT_CONTENEUR c".
      " where".
      "  a.APPLI_ID=".$appli_id." and ac.CONT_ID=".$cont_id." and ac.APPLI_ID=".$appli_id." and".
      "  c.CONT_ID=".$cont_id.$strCondProfil." and".
      "  c.CONT_ID=ac.CONT_ID and ac.APPLI_ID=a.APPLI_ID and a.ATYPE_ID=at.ATYPE_ID";

    return $this->dbConn->initDataset($strSql) ;
  }
      
  /**
   * Retourne la liste des espaces accessibles par l'agent donné
   * @param agent_id      Identifiant de l'agent
   * @param bAdminOnly    =false par défaut, =true pour filtrer les animateurs de l'espace
   * @param cont_id_root  identifiant de l'espace racine de l'arbre, =-1 par défaut pour tout l'arbre
   * @param bForceVisible =false par défaut, true pour forcer l'affichage même si pas invité
   * @return AlkDs
   */
  public function getDs_listeEspaceByAgentForArbo($agent_id, $bAdminOnly=false, $cont_id_root="-1", $bForceVisible=false)
  {
    $strWhere = ($bAdminOnly==false ? "" : " and ac.CONT_ADMIN=1");
    $iNiv = "0";

    if( $cont_id_root != "-1" ) {
      $strSql = "select CONT_ARBRE, CONT_NIVEAU from SIT_CONTENEUR where CONT_ID=".$cont_id_root;
      $dsCont = $this->dbConn->initDataset($strSql);
      if( $drCont = $dsCont->getRowIter() ) {
        $strArbre = $drCont->getValueName("CONT_ARBRE");
        $iNiv     = $drCont->getValueName("CONT_NIVEAU");
        $strWhere .= " and c.CONT_ARBRE like '".$strArbre."%'";
      }
    }
    $strJoin = "";
    if( !$bForceVisible ) {
      $strWhere .= " and ac.AGENT_ID=".$agent_id;
    } else {
      $strJoin = " and ac.AGENT_ID=".$agent_id;
    }
    $strSql = "Select c.CONT_ID as ID, c.CONT_INTITULE as NOM, c.CONT_PERE as ID_PERE,".
      " c.CONT_RANG, c.CONT_NIVEAU as NIV, c.CONT_ARBRE, ".
      " ac.CONT_VISIBLE, ac.CONT_ADMIN as droit, '' as url, c.CONT_VISIBLE as forceVisible".
      " from SIT_CONTENEUR c".
      ( $bForceVisible ? " left" : " inner" )." join SIT_AGENT_CONT ac on c.CONT_ID=ac.CONT_ID".$strJoin.
      " where c.CONT_NIVEAU>=".$iNiv.$strWhere.
      " order by NIV, NOM";  
    return $this->dbConn->initDataset($strSql) ;
  }
              
  /**
   * Retourne le droit sur l'application pour l'agent donné
   *        espace/05_agent_droit.php
   *
   * @param agent_id        identifiant de l'agent
   * @param appli_id        identifiant de l'application
   * @return Retourne un entier
   */
  function GetStr_idDroitByAgentAppli($agent_id, $appli_id)
  {
    $strDroit = "0";    
    $strSql = "select APPLI_DROIT_ID".
      " from SIT_AGENT_APPLI ".
      "where AGENT_ID=".$agent_id." and APPLI_ID=".$appli_id;
    $dsDroit = $this->dbConn->InitDataSet($strSql);
    if( $drDroit = $dsDroit->getRowIter() )
      $strDroit = $drDroit->getValueName("APPLI_DROIT_ID");
    
    return $strDroit;
  }
    
  /**
   * Retourne la liste des applications de type Liste de diff
   * annu/01_recherche_form.php
   *
   * @param cont_id        identifiant de l'espace, =-1 pour toutes les applis de type atype_id
   * @param atype_id       identifiant du type d'appli
   * @param user_id        id de l'utilisateur avec lequel, il faut vérifier les droits, =-1 par défaut sans vérif droit
   * @param iRightMin      droit minimum requis par l'utilisateur, =ALK_APPLI_RIGHT_NONE par défaut, (utilisé uniquement si user_id!=-1)
   * @return Retourne un dataSet
   */
  public function getDs_listeAppliByEspaceType($cont_id, $atype_id, $user_id="-1", $iRightMin=ALK_APPLI_RIGHT_NONE)
  {
    $strSql = "select a.APPLI_ID, a.APPLI_INTITULE, at.ATYPE_URL ".
      ( $user_id != "-1" ? ", r.RIGHT_ID" : "" ).
      " from SIT_APPLI a".
      "   inner join SIT_APPLI_CONT ac on ac.APPLI_ID=a.APPLI_ID".
      "   inner join SIT_APPLI_TYPE at on a.ATYPE_ID=at.ATYPE_ID".
      ( $user_id != "-1" 
        ? " inner join V_USER_RIGHT r on a.APPLI_ID=r.APPLI_ID" 
        : "" ).
      " where a.ATYPE_ID=".$atype_id.
      ( $cont_id != "-1" ? " and ac.CONT_ID=".$cont_id : "" ).
      ( $user_id != "-1" ? " and r.USER_ID=".$user_id." and r.RIGHT_ID>=".$iRightMin : "" );
    return $this->dbConn->initDataset($strSql) ;
  }
    
  /**
   * Retourne la liste des applications du conteneur auxquelles l'utilisateur à droit de connexion
   *
   * @param agent_id         identifiant de l'agent
   * @param cont_id          identifiant de l'espace
   * @param strCondProfil    condition complémentaire liée au profil
   * @return Retourne un dataSet
   */
  public function getDs_listeAppliAgentByEspace($agent_id, $cont_id, $strCondProfil="")
  {
    $strFromRestreint = "";
    $strWhereRestreint = " and a.ATYPE_ID NOT IN (".ALK_ATYPE_ID_ACL.") ";

    if( $strCondProfil != "" ) {
      $strFromRestreint .= " inner join SIT_CONTENEUR c on ac.CONT_ID=c.CONT_ID";
      $strWhereRestreint .= " ".$strCondProfil;
    }

    // liste des applications conteneur accessible à l'utilisateur
    $strSql = "select a.APPLI_ID, a.ATYPE_ID, a.APPLI_LOGO, a.APPLI_RANG, a.APPLI_INTITULE, a.APPLI_LIEE, ".
      " at.ATYPE_ID, at.ATYPE_INTITULE, at.ATYPE_URL, at.ATYPE_LOGO, at.ATYPE_URL_TARGET, ur.RIGHT_ID,".
      $this->dbConn->compareSql("at.ATYPE_URL_SITE", " is ", "NULL", "''", "at.ATYPE_URL_SITE")." as ATYPE_URL_SITE".
      " from SIT_APPLI a".
      "  inner join SIT_APPLI_TYPE at on a.ATYPE_ID=at.ATYPE_ID".
      "  inner join SIT_APPLI_CONT ac on ac.APPLI_ID=a.APPLI_ID".
      "  inner join V_USER_RIGHT ur on (ur.APPLI_ID=ac.APPLI_ID and ur.CONT_ID=ac.CONT_ID)".
      $strFromRestreint.
      " where ac.CONT_ID=".$cont_id." and ur.USER_ID=".$agent_id.
      " and (ur.CONT_ADMIN=1 or (ur.RIGHT_ID>0 and at.ATYPE_DROIT=0 or ur.RIGHT_ID>1 and at.ATYPE_DROIT=1))".
      $strWhereRestreint.
      " order by a.APPLI_RANG, a.APPLI_INTITULE";

    return $this->dbConn->initDataset($strSql) ;
  }
    
  /**
   * Retourne la liste des applications de l'espace accessible par agent_id
   *        espace/05_agent_droit.php
   *
   * @param agent_id  Identifiant de l'agent
   * @param cont_id   Identifiant de l'espace
   * @return Retourne un dataSet
   */
  public function getDs_listeAppliByAgentEspace($agent_id, $cont_id=-1)
  {
    if( $cont_id != -1 ) {
      $strSql = "select a.APPLI_ID, a.APPLI_INTITULE, a.ATYPE_ID ".
        " from  SIT_APPLI_CONT ac".
        "   inner join SIT_AGENT_CONT agc on ac.CONT_ID=agc.CONT_ID".
        "   inner join SIT_APPLI a on ac.APPLI_ID=a.APPLI_ID".
        " where".
        "   ac.CONT_ID=".$cont_id." and agc.AGENT_ID=".$agent_id.
        " order by a.APPLI_INTITULE";
    } else {
      $strSql = "select a.APPLI_ID, a.APPLI_INTITULE, a.ATYPE_ID, at.ATYPE_LOGO, c.CONT_ID, c.CONT_INTITULE,".
        " c.CONT_PUBLIC, c.CONT_PERE, c.CONT_NIVEAU, c.CONT_ARCHIVE, agc.CONT_ADMIN, aa.RIGHT_ID as DROIT_ID".
        " from  SIT_CONTENEUR c".
        "   left join SIT_APPLI_CONT ac on c.CONT_ID=ac.CONT_ID".
        "   left join SIT_AGENT_CONT agc on ac.CONT_ID=agc.CONT_ID".
        "   left join SIT_APPLI a on ac.APPLI_ID=a.APPLI_ID".
        "   left join SIT_APPLI_TYPE at on a.ATYPE_ID=at.ATYPE_ID".
        "   left join V_USER_RIGHT aa on (aa.CONT_ID=agc.CONT_ID and aa.APPLI_ID=a.APPLI_ID and aa.USER_ID=agc.AGENT_ID)".
        " where".
        "   agc.AGENT_ID=".$agent_id.
        " order by c.CONT_NIVEAU, c.CONT_INTITULE, a.APPLI_INTITULE";
    }

    return $this->dbConn->initDataset($strSql) ;
  }
    
  /**
   * Retourne la liste des statistiques de l'agent
   * lib_sqlmenu.php
   *
   * @param agent_id        identifiant de l'agent
   * @param type_conn       type de connexion
   * @param type_conn_id    identifiant du type de connexion
   * @return Retourne un dataSet
   */
  function GetDs_listeStatsByAgent($agent_id, $type_conn, $type_conn_id)
  {
    $strSql = "select * from SIT_STATISTIQUE".
      " where AGENT_ID=".$agent_id.
      " and MOIS_CONNEXION=".$this->dbConn->GetDateFormat("MM", $this->dbConn->GetDateCur(), false, true).
      " and ANNEE_CONNEXION=".$this->dbConn->GetDateFormat("YYYY", $this->dbConn->GetDateCur(), false, true).
      " and TYPE_CONNEXION=".$type_conn.
      " and ID_TYPE_CONNEXION=".$type_conn_id;
    return $this->dbConn->initDataset($strSql);
  }
  
  /**
   * Retourne la liste des statistiques de l'agent sur l'application
   * lib_sqlmenu.php
   *
   * @param agent_id        identifiant de l'agent
   * @param data_id         identifiant de la donnée
   * @param atype_id        identifiant du type d'application
   * @param datatype_id     identifiant du type de donnée
   * @param appli_id        identifiant de l'application
   * @param stat_id         identifiant de l'action réalisée
   * @return Retourne un dataSet
   */
  function GetDs_listeStatsAppliByAgent($agent_id, $data_id, $atype_id, $datatypeId, $appli_id, $stat_id)
  {
    $strSql = "select * from SIT_STAT_APPLI".
      " where AGENT_ID=".$agent_id.
      " and STAT_MOIS=".$this->dbConn->GetDateFormat("MM", $this->dbConn->GetDateCur(), false, true).
      " and STAT_ANNEE=".$this->dbConn->GetDateFormat("YYYY", $this->dbConn->GetDateCur(), false, true).
      " and DATA_ID=".$data_id.
      " and ATYPE_ID=".$atype_id.
      " and DATATYPE_ID=".$datatypeId.
      " and APPLI_ID=".$appli_id.
      " and STAT_ACTION_ID=".$stat_id;
    return $this->dbConn->initDataset($strSql) ;
  }

  /**
   * Retourne la liste des espaces du SIT
   * fdoc_01/07_doclink_liste.sql, workspace
   *
   * @return Retourne un dataSet
   */
  function GetDs_listeEspace($cont_public="-1", $cont_archive=-1)
  {
    $strWhere = "";
    if( $cont_public != "-1" ) {
      $strWhere = " and CONT_PUBLIC=".$cont_public;
    }
    
    $strWhere.= ( $cont_archive != -1 
                 ? " and CONT_ARCHIVE=".$cont_archive 
                 : "");

    $strSql = "select CONT_ID, CONT_INTITULE, CONT_INTITULE_COURT from SIT_CONTENEUR ".
              " where CONT_NIVEAU>0 ".$strWhere." order by CONT_INTITULE";
    return $this->dbConn->initDataset($strSql) ;
  }
  
    /**
   * retourne la liste des espaces collab
   * @param user_id identifiant de la personne connectée
   * @param iArchive filtre sur les espaces (0= non archivé , 1= archivé)
   * @param cont_public 
   */
  function GetDs_listeEspaceCollab($user_id, $iArchive=0, $cont_public="-1")
  {
    $strSql = "select c.CONT_ID, c.CONT_INTITULE" .
        " from V_USER_RIGHT r" .
        " inner join SIT_CONTENEUR c on c.CONT_ID=r.CONT_ID" .
        " inner join COLLAB_APPLI ca on ca.APPLI_IDL=r.APPLI_ID" .
        " where r.USER_ID=".$user_id .
        " and r.RIGHT_ID>0".
        ($iArchive != -1 ? " and c.CONT_ARCHIVE =".$iArchive : "").
        ($cont_public != -1 ? " and CONT_PUBLIC=".$cont_public : "");
    
    return $this->dbConn->initDataset($strSql) ;
  }
  
  /**
   * Fontion qui permet de savoir si l'espace passé en paramètre est ou non un espace Collab
   * @param  integer $cont_id : Identifiant de l'espace
   * @return boolean          : Retourne Vrai si l'espace est un espace Collaboratif
   */
  function Is_CollabSpace($cont_id){
    $strSql = " SELECT COUNT( c.CONT_ID ) ".
              " FROM SIT_CONTENEUR c ".
              "   INNER JOIN SIT_APPLI a ON ( c.CONT_APPLI_DEFAUT = a.APPLI_ID ) ".
              " WHERE c.CONT_ID = ".$cont_id." ".
              " AND a.ATYPE_ID = ".ALK_ATYPE_ID_COLLAB;
    $IsCollabSpace = $this->dbConn->getScalarSql($strSql, 0);
    return ($IsCollabSpace == 1 ? true : false);
  }
 
  /**
   * Retourne la liste des sous-espaces (bfils=true)
   *        ou l'espace parent (bFils=false) de cont_id
   *        dont agent_id est animateur
   *        espace/02_admin_list.php
   *
   * @param cont_id  Identifiant de l'espace courant
   * @param agent_id Identifiant de l'agent
   * @param bFils    
   * @return Retourne un dataSet
   */
  function GetDs_listeEspaceByAnimateur($cont_id, $agent_id, $bFils=true)
  {
    $strChamp = "c.CONT_PERE";
    if( $bFils == false ) 
      $strChamp = "c.CONT_ID";

    $strSql = "select c.CONT_ID, c.CONT_INTITULE, c.CONT_RANG ".
      " from".
      "   SIT_CONTENEUR c,".
      "   SIT_AGENT_CONT ac ".
      " where c.CONT_ID=ac.CONT_ID and ".$strChamp."=".$cont_id.
      "   and ac.AGENT_ID=".$agent_id." and ac.CONT_ADMIN=1".
      " order by c.CONT_INTITULE";
    return $this->dbConn->initDataset($strSql);
  }

  /**
   * Retourne vrai si l'espace est vide, faux sinon
   *        espace/02_admin_list.php
   *
   * @param cont_id  Identifiant de l'espace
   * @return Retourne un booleen
   */
  function GetBool_espaceVide($cont_id)
  {
    $bRes = false;
    $strSql = "select APPLI_ID from SIT_APPLI_CONT where CONT_ID=".$cont_id.
    " union select CONT_ID from SIT_CONTENEUR where CONT_PERE=".$cont_id;
    $dsId = $this->dbConn->initDataset($strSql);
    $nb = $dsId->getCountTotDr();
    return ( $nb == 0 ? true : false );
  }

  /**
   * Retourne la liste de tous les animateurs
   *
   * @param idFirst    Indice de pagination : début
   * @param idLast     Indice de pagination : fin
   * @return Retourne un dataSet
   */
  function getDs_listeAnimateur($idFirst=0, $idLast=-1)
  {
    $strSql = "select distinct c.CONT_ID, c.CONT_INTITULE,".
      " a.AGENT_ID, a.AGENT_NOM, a.AGENT_PRENOM, a.PROFIL_ID,".
      " s.SERVICE_ID, s.SERVICE_INTITULE_COURT".
      " from".
      "   SIT_AGENT a,".
      "   SIT_AGENT_SERVICE ase,".
      "   SIT_SERVICE s,".
      "   SIT_AGENT_CONT ac,".
      "   SIT_CONTENEUR c".
      " where".
      "   a.AGENT_VALIDE>=0 and".
      "   a.AGENT_ID=ase.AGENT_ID and ase.SERVICE_ID=s.SERVICE_ID and s.SERVICE_NIVEAU=1 and".
      "   a.AGENT_ID=ac.AGENT_ID and ac.CONT_ID=c.CONT_ID and ac.CONT_ADMIN=1".
      " order by a.AGENT_NOM, a.AGENT_PRENOM, c.CONT_INTITULE";

     return $this->dbConn->initDataset($strSql, $idFirst, $idLast);   
  }

  /**
   * Retourne la liste des animateurs d'un espace pour un combo
   * @param cont_id Identifiant de l'espace
   * @return AlkDs
   */
  public function getDs_listeAnimateurByEspaceForCombo($cont_id, $bWithServNiv1=false, $bWithStar=true)
  {
    if( $bWithStar ) {
      $strSelect = $this->dbConn->getConcat("a.AGENT_NOM", "' '", "a.AGENT_PRENOM", 
                                            $this->dbConn->compareSql("(a.AGENT_PRIV & ".ALK_PRIV_SPACE_ADMIN.")", "=", "".ALK_PRIV_SPACE_ADMIN, "'*'", "''"));
      
      $strSelectServ = $this->dbConn->getConcat("a.AGENT_NOM", "' '", "a.AGENT_PRENOM", "' - '", "s.SERVICE_INTITULE_COURT",
                                                $this->dbConn->compareSql("(a.AGENT_PRIV & ".ALK_PRIV_SPACE_ADMIN.")", "=", "".ALK_PRIV_SPACE_ADMIN, "'*'", "''"));
    } else {
      $strSelect = $this->dbConn->getConcat("a.AGENT_NOM", "' '", "a.AGENT_PRENOM"); 
      $strSelectServ = $this->dbConn->getConcat("a.AGENT_NOM", "' '", "a.AGENT_PRENOM", "' - '", "s.SERVICE_INTITULE_COURT");
    }
    
    $strSql = "SELECT ".$strSelect." as nom, a.AGENT_ID as id, a.AGENT_MAIL as mail, a.AGENT_LG,".
      $this->dbConn->getConcat("a.AGENT_PRENOM", "' '", "a.AGENT_NOM")." as prenom_nom".
      ( $bWithServNiv1
        ? ", ".$strSelectServ." as prenom_nom_serv"
        : "" ).
      " from SIT_AGENT_CONT ac".
      "   inner join SIT_AGENT a on ac.AGENT_ID=a.AGENT_ID".
      ( $bWithServNiv1
        ? " inner join SIT_AGENT_SERVICE ase on ase.AGENT_ID=a.AGENT_ID".
          " inner join SIT_SERVICE s on (ase.SERVICE_ID=s.SERVICE_ID and s.SERVICE_NIVEAU=1)"
        : "" ).
      " where ac.CONT_ADMIN=1 and ac.CONT_ID=".$cont_id." and a.AGENT_VALIDE>-1".
      " order by nom";
    return $this->dbConn->initDataset($strSql);
  }
  
  /**
   * Lit puis retourne dans un tableau, la liste des paramètres d'une application
   *        Retourne tableau vide si pas de paramètre
   *
   * @param appli_id  Identifiant de l'application
   * @return Retourne un array
   */
  function getTab_appliParam($appli_id)
  {
    $tabParam = array();
    $strSql = "select * from SIT_APPLI_PARAM where APPLI_ID=".$appli_id;
    $dsParam = $this->dbConn->InitDataSet($strSql);
    $i = 0;
    while( $drParam = $dsParam->getRowIter() ) {
      $tabParam[$i]["PARAM_ID"]    = $drParam->getValueName("PARAM_ID"); 
      $tabParam[$i]["PARAM_FIELD"] = $drParam->getValueName("PARAM_FIELD"); 
      $tabParam[$i]["PARAM_TYPE"]  = $drParam->getValueName("PARAM_TYPE");
      switch( $tabParam[$i]["PARAM_TYPE"] ) {
      case "0": $tabParam[$i]["PARAM_VALUE"] = $drParam->getValueName("PARAM_VTEXT"); break;
      case "1": $tabParam[$i]["PARAM_VALUE"] = $drParam->getValueName("PARAM_VINT"); break;
      case "2": $tabParam[$i]["PARAM_VALUE"] = $drParam->getValueName("PARAM_VDATE"); break;
      }
      $i++;
    }

    return $tabParam;
  }
   
  /**
   * Retourne l'appli par defaut de l'espace donné pour l'agent donné
   * sit_menu/login_verif.php
   *
   * @param $agent_id        identifiant de l'agent
   * @param $cont_id         identifiant du conteneur
   * @param $strCondProfil   condition complémentaire liée au profil
   * @return Retourne un dataSet
   */
  public function getTab_appliDefautByAgent($agent_id, $cont_id, $strCondProfil="")
  {
    $strContId = "";
    if( $cont_id != "" && is_numeric($cont_id) && $cont_id>0 ) {
      $strContId = "c.CONT_ID=".$cont_id." and ";
    } else {
      $cont_id = "-1"; 
    }
    $appli_id = "-1";
    
    // recherche le premier espace accessible
    $strSql = "select c.CONT_ID, c.CONT_APPLI_DEFAUT".
      " from SIT_CONTENEUR c".
      "   inner join SIT_AGENT_CONT ac on c.CONT_ID=ac.CONT_ID".
      " where ".$strContId." c.CONT_ID>0 ".$strCondProfil.
      "   and ac.AGENT_ID=".$agent_id.
      " order by CONT_NIVEAU, CONT_RANG, CONT_INTITULE";
    $dsCont = $this->dbConn->initDataset($strSql);
    if( $drCont = $dsCont->getRowIter() ) {
      $cont_id  = $drCont->getValueName("CONT_ID");
      $appli_id = $drCont->getValueName("CONT_APPLI_DEFAUT"); 
    }

    // vérifie l'accès à l'application par défaut si elle existe.
    $strSql = "select * from V_USER_RIGHT".
      " where USER_ID=".$agent_id." and CONT_ID=".$cont_id." and APPLI_ID=".$appli_id;
    $dsAppli = $this->dbConn->initDataset($strSql) ;
    if( $drAppli = $dsAppli->getRowIter() ) {
      $right_id   = $drAppli->getValueName("RIGHT_ID"); 
      $cont_admin = $drAppli->getValueName("CONT_ADMIN");

      if( !($cont_admin == "1" || $right_id>ALK_APPLI_RIGHT_NONE) ) {
         // pas de droit sur l'appli par défaut, redirige vers la page de recherche
         $appli_id = -1;
      } 
    } else {
      $appli_id = -1;
    }

    return array("cont_id" => $cont_id, "appli_id" => $appli_id);
  }
    
  /**
   * Retourne la liste des conteneurs fils accessibles à l'utilisateur avec détermination de l'appli par defaut
   * lib_sqlmenu.php
   *
   * @param $agent_id        identifiant de l'agent
   * @param $cont_id        identifiant de l'espace
   * @param $strCondProfil    condition complémentaire liée au profil
   * @return Retourne un dataSet
   */
  public function getDs_listeSousEspaceByAgent($agent_id, $cont_id, $strCondProfil="")
  {
    $strSqlAllow = "select c0.*, ac0.AGENT_ID as AGENT_IDC".
      " from SIT_CONTENEUR c0".
      "   inner join SIT_AGENT_CONT ac0 on c0.CONT_ID=ac0.CONT_ID".
      " where ac0.AGENT_ID=".$agent_id." and ac0.CONT_VISIBLE=1";

    // selection des conteneurs fils accessibles à l'utilisateur avec
    // détermination de l'appli par defaut
    $strSql = "select distinct c.CONT_ID, c.CONT_APPLI_DEFAUT, c.CONT_PERE, c.CONT_NIVEAU,".
      " c.CONT_RANG, c.CONT_INTITULE, c.CONT_INTITULE_COURT, c.CONT_PUBLIC, c.CONT_RESERVE, c.CONT_ARBRE, ".
      " a.APPLI_ID, at1.ATYPE_URL, at1.ATYPE_URL_SITE, ".
      $this->dbConn->compareSql("a.APPLI_ID", " is ", "NULL", "0", "1")." as DEFAUT, ".
      " c.AGENT_IDC".
      " from ".
      "   (".$strSqlAllow.") c".
      "   left join SIT_APPLI_CONT aco on c.CONT_ID=aco.CONT_ID ".
      "   left join  SIT_APPLI a on c.CONT_APPLI_DEFAUT=a.APPLI_ID".
      "   left join SIT_APPLI_TYPE at1 on a.ATYPE_ID=at1.ATYPE_ID".
      /*"       left join SIT_AGENT_APPLI aa on a.APPLI_ID=aa.APPLI_ID".
      "     where  aa.AGENT_ID=".$agent_id." and aa.APPLI_DROIT_ID>0) a1 on c.CONT_APPLI_DEFAUT=a1.APPLI_ID".*/

      /*      "   left join (select a.APPLI_ID, at.ATYPE_URL, at.ATYPE_URL_SITE".
      "    from V_USER_RIGHT ur
      "   ) a1 on c.CONT_APPLI_DEFAUT=a1.APPLI_ID".*/

      " where c.CONT_PERE=".$cont_id.
      $strCondProfil;
      
    if( defined("ALK_ESPACE_PRIVE_VISIBLE_SPACE") && ALK_ESPACE_PRIVE_VISIBLE_SPACE==true ) {
      // sélection des espaces fils privés visibles pour lesquels l'utilisateur ne possède pas de droit
      $strSql .= " union".
        " select c.CONT_ID, c.CONT_APPLI_DEFAUT, c.CONT_PERE, c.CONT_NIVEAU,".
        "   c.CONT_RANG, c.CONT_INTITULE, c.CONT_INTITULE_COURT, c.CONT_PUBLIC, c.CONT_RESERVE, c.CONT_ARBRE, ".
        "   -1 as APPLI_ID, ".$this->dbConn->getConcat("'javascript:OpenWindEspaceInvite('", "c.CONT_ID", "')'")." as ATYPE_URL,".
        "   '' as ATYPE_URL_SITE, 1 as DEFAUT, ac.AGENT_ID as AGENT_IDC".
        " from ".
        "  SIT_CONTENEUR c".
        "    left join SIT_AGENT_CONT ac on (c.CONT_ID=ac.CONT_ID and ac.AGENT_ID=".$agent_id." and ac.CONT_VISIBLE=1)".
        " where c.CONT_PERE=".$cont_id." and c.CONT_PUBLIC=2 and ac.AGENT_ID is null".
        $strCondProfil;
    }
      
    $strSql .= " order by CONT_NIVEAU, CONT_INTITULE";
    return $this->dbConn->initDataset($strSql) ;
  }
  
    /**
   * Retourne la liste des conteneurs accessibles à l'utilisateur avec détermination de l'appli par defaut et dont le pere n'est pas accessible
   *        la liste ne contient pas les espaces privés visibles
   * lib_sqlmenu.php
   *
   * @param $agent_id        identifiant de l'agent
   * @param $cont_id        identifiant de l'espace
   * @param $strCondProfil    condition complémentaire liée au profil
   * @return Retourne un dataSet
   */
  public function getDsListeSousEspaceOrphelinsByAgent($agent_id, $cont_id, $strCondProfil="")
  {
    // liste des espaces dont le pere n'est pas accessible
    $strSql = "select ca0.CONT_ID, ca0.CONT_INTITULE, ca0.CONT_INTITULE_COURT, ca0.CONT_PUBLIC, ca0.CONT_NIVEAU, ca0.CONT_APPLI_DEFAUT".
      "   from ".
      "    (select cont1.CONT_ID".
      "     from (select CONT_ID from SIT_CONTENEUR where CONT_ID>0) cont1".
      "      left join ".
      "        (select CONT_ID from SIT_AGENT_CONT where AGENT_ID=".$agent_id.") cont2 on cont1.cont_id=cont2.cont_id where cont2.CONT_ID is NULL) ci0,".
      "    (select c0.CONT_ID, c0.CONT_APPLI_DEFAUT, c0.CONT_PERE, c0.CONT_NIVEAU, c0.CONT_RANG, c0.CONT_INTITULE, c0.CONT_INTITULE_COURT, c0.CONT_PUBLIC, c0.CONT_RESERVE".
      "     from SIT_CONTENEUR c0, (select CONT_ID from SIT_AGENT_CONT where AGENT_ID=".$agent_id.") ac0".
      "     where c0.CONT_ID<>".$cont_id." and c0.CONT_ID=ac0.CONT_ID) ca0".
      "   where ca0.CONT_PERE=ci0.CONT_ID";

    // jointure pour obtenir les infos sur les applis de démarrage
    $strSql = "select distinct c.CONT_ID, c.CONT_INTITULE, c.CONT_INTITULE_COURT, c.CONT_PUBLIC, c.CONT_NIVEAU,".
      " a1.APPLI_ID, a1.ATYPE_URL, a1.ATYPE_URL_SITE, ".
      $this->dbConn->compareSql("a1.APPLI_ID", " is ", "NULL", "0", "1")." as DEFAUT ".
      " from ".
      "  (".$strSql.") c".
      "  left join SIT_AGENT_CONT ac on ac.CONT_ID=c.CONT_ID".
      "  left join".
      "   (select a.APPLI_ID, at.ATYPE_URL, at.ATYPE_URL_SITE".
      "    from SIT_APPLI a".
      "      left join SIT_APPLI_TYPE at on a.ATYPE_ID=at.ATYPE_ID".
      "      left join SIT_AGENT_APPLI aa on a.APPLI_ID=aa.APPLI_ID".
      "    where aa.AGENT_ID=".$agent_id." and aa.APPLI_DROIT_ID>0) a1 on c.CONT_APPLI_DEFAUT=a1.APPLI_ID".
      " where ac.AGENT_ID=".$agent_id." and ac.CONT_VISIBLE=1".$strCondProfil.
      " order by c.CONT_NIVEAU, c.CONT_INTITULE";    

    return $this->dbConn->initDataset($strSql) ;
  }
   
  /**
   * Retourne la liste des conteneurs accessibles à l'utilisateur avec détermination de l'appli par defaut et dont le pere n'est pas accessible
   *        la liste ne contient pas les espaces privés visibles
   * lib_sqlmenu.php
   *
   * @param $agent_id        identifiant de l'agent
   * @param $cont_id        identifiant de l'espace
   * @param $strCondProfil    condition complémentaire liée au profil
   * @return Retourne un dataSet
   */
  function GetDs_listeSousEspaceOrphelinsByAgent($agent_id, $cont_id, $strCondProfil="")
  {
    // liste des espaces dont le pere n'est pas accessible
    $strSql = "select ca0.*".
      "   from ".
      "    (select cont1.CONT_ID".
      "     from (select CONT_ID from SIT_CONTENEUR where CONT_ID>0) cont1".
      "      left join ".
      "        (select CONT_ID from SIT_AGENT_CONT where AGENT_ID=".$agent_id.") cont2 on cont1.cont_id=cont2.cont_id where cont2.CONT_ID is NULL) ci0,".
      "    (select c0.CONT_ID, c0.CONT_APPLI_DEFAUT, c0.CONT_PERE, c0.CONT_NIVEAU, c0.CONT_RANG, c0.CONT_INTITULE, c0.CONT_PUBLIC, c0.CONT_RESERVE".
      "     from SIT_CONTENEUR c0, (select CONT_ID from SIT_AGENT_CONT where AGENT_ID=".$agent_id.") ac0".
      "     where c0.CONT_ID<>".$cont_id." and c0.CONT_ID=ac0.CONT_ID) ca0".
      "   where ca0.CONT_PERE=ci0.CONT_ID";

    // jointure pour obtenir les infos sur les applis de démarrage
    $strSql = "select distinct c.*,".
      " a1.APPLI_ID, a1.ATYPE_URL, a1.ATYPE_URL_SITE, ".
      $this->dbConn->compareSql("a1.APPLI_ID", " is ", "NULL", "0", "1")." as DEFAUT ".
      " from ".
      "  (".$strSql.") c".
      "  left join SIT_AGENT_CONT ac on ac.CONT_ID=c.CONT_ID".
      "  left join".
      "   (select a.APPLI_ID, at.ATYPE_URL, at.ATYPE_URL_SITE".
      "    from SIT_APPLI a".
      "      left join SIT_APPLI_TYPE at on a.ATYPE_ID=at.ATYPE_ID".
      "      left join SIT_AGENT_APPLI aa on a.APPLI_ID=aa.APPLI_ID".
      "    where aa.AGENT_ID=".$agent_id." and aa.APPLI_DROIT_ID>0) a1 on c.CONT_APPLI_DEFAUT=a1.APPLI_ID".
      " where ac.AGENT_ID=".$agent_id." and ac.CONT_VISIBLE=1".$strCondProfil.
      " order by c.CONT_NIVEAU, c.CONT_INTITULE";    

    return $this->dbConn->initDataset($strSql) ;
  }

  /**
   * Retourne la liste des espaces administrable par admin_id
   *        et accessible à agent_id
   *        espace/05_agent_droit.php
   *
   * @param cont_id   Identifiant de l'espace
   * @param agent_id  Identifiant de l'agent sélectionné
   * @param admin_id  Identifiant de l'agent connecté
   * @return Retourne un dataSet
   */
  function GetDs_listeEspaceByAgentAdminForCombo($cont_id, $agent_id, $admin_id)
  {
    $strSql = "select c.CONT_INTITULE, c.CONT_ID".
      " from SIT_CONTENEUR c".
      "   inner join SIT_AGENT_CONT ac0 on c.CONT_ID=ac0.CONT_ID".
      "   inner join SIT_AGENT_CONT ac1 on c.CONT_ID=ac1.CONT_ID".
      " where ac0.AGENT_ID=".$agent_id." and ac1.AGENT_ID=".$admin_id.
      " and ac1.CONT_ADMIN=1".
      " order by CONT_INTITULE";
    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   * Retourne la liste des agents ayant accès à une appli selon un certain droit
   *
   * @param cont_id    Identifiant de l'espace
   * @param cont_admin =0 utilisateur de l'espace, =1 animateur de l'espace
   * @param appli_id   Identifiant de l'application
   * @param droit_id   Droit recherché (=0 aucun, =1 lecture, =2 écriture)
   * @return Retourne un string : La requete sélection retournant l'alias AGENT_ID
   */
  function GetStr_listeAgent($cont_id, $cont_admin, $appli_id, $droit_id)  
  {
    if( $cont_id=="-1" && $appli_id!="-1" )
      // retourne la liste des agents ayant accès à une appli
      $strSql = "select distinct AGENT_ID from SIT_AGENT_APPLI".
        " where APPLI_ID=".$appli_id.
        ( $droit_id != "-1" ? " and APPLI_DROIT_ID=".$droit_id : "");
    
    elseif( $cont_id!="-1" && $appli_id=="-1" )
      // retourne la liste des agents, animateur ou non, ayant accès à un espace
      $strSql = "select distinct AGENT_ID from SIT_AGENT_CONT".
        " where CONT_ID=".$cont_id.
        ( $cont_admin != "-1"  ? " and CONT_ADMIN=".$cont_admin : "");
    
    elseif( $cont_id!="-1" && $appli_id!="-1" && $cont_admin!="-1" )
      // retourne la liste des agents, animateurs ou non, ayant accès à une application
      $strSql = "select distinct aa.AGENT_ID from SIT_AGENT_CONT ac, SIT_AGENT_APPLI aa".
        " where ac.CONT_ID=".$cont_id.
        ( $cont_admin != "-1"  ? " and ac.CONT_ADMIN=".$cont_admin : "").
        " and aa.APPLI_ID=".$appli_id.
        ( $droit_id != "-1" ? " and aa.APPLI_DROIT_ID=".$droit_id : "").
        " and ac.AGENT_ID=aa.AGENT_ID";

    if( ALK_BDD_TYPE == ALK_SGBD_MYSQL ) {
      $strListeAgentId = "-1";
      $dsAgent = $this->dbConn->InitDataSet($strSql);
      while( $drAgent = $dsAgent->getRowIter() ) {
        $strListeAgentId .= ",".$drAgent->getValueName("AGENT_ID");
      }
    } else {
      $strListeAgentId = $strSql;
    }

    return $strListeAgentId;
  }

  /**
   * Retourne la liste des espaces accessibles par l'agent donné
   * lib_sqlmenu.php
   *
   * @param $agent_id        identifiant de l'agent
   * @param $liste_cont_id   liste des identifiants de l'arbre du conteneur courant (séparée par des ,)
   * @param $iNiveauMin      niveau minimal des conteneurs à obtenir
   * @return Retourne un dataSet
   */
  function GetDs_listeEspaceNavByAgent($agent_id, $liste_cont_id, $iNiveauMin=1)
  {
    $strSql = "select c.CONT_ID, c.CONT_INTITULE, c.CONT_NIVEAU, c.CONT_APPLI_DEFAUT, at.ATYPE_URL,".
      " ".$this->dbConn->compareSql("a.APPLI_ID", " is ", "NULL", "0", "1")." as DEFAUT,".
      " ".$this->dbConn->compareSql("ac.AGENT_ID", " is ", "NULL", "0", "1")." as DROIT".
      " from ".
      "   SIT_CONTENEUR c".
      "   left join SIT_AGENT_CONT ac on c.CONT_ID=ac.CONT_ID".
      "   left join SIT_APPLI a on c.CONT_APPLI_DEFAUT=a.APPLI_ID".
      "   left join SIT_APPLI_TYPE at on a.ATYPE_ID=at.ATYPE_ID".
      " where c.CONT_ID in (".$liste_cont_id.") and c.CONT_NIVEAU>".$iNiveauMin.
      "  and ac.AGENT_ID=".$agent_id.
      " order by c.CONT_NIVEAU, c.CONT_INTITULE";

    return $this->dbConn->initDataset($strSql) ;
  }
  
  /**
   * Retourne la liste des agents pour la sélection multiple
   * annu/01_users_form.php
   *
   * @param FILTRE_NOM        filtre sur le nom d'agent
   * @param FILTRE_SERVICE_ID filtre sur l'identifiant du service de niveau 1
   * @param FILTRE_MISSION_ID filtre sur l'identifiant de la mission
   * @param cont_id_filtre    identifiant de l'espace sur lequel on effectue le filtre
   * @param cont_id           identifiant de l'espace en cours
   * @param FILTRE_ESPACE     =1 si on effecture le filtre sur cont_id_filtre, =0 sinon
   * @param bFiltreEspace     =true si on effectue le filtre sur cont_id, false sinon
   * @param cont_id_exclude   identifiant de l'espace pour ne pas retourné les agents qui appartiennent dèjà à cet espace
   * @param appli_id_filtre   identifiant de l'application sur laquelle on effectue le filtre
   * @param iAffAdmin         =1 affiche les administrateurs, =0 sinon
   * @param FILTRE_LISTE_ID   =-1 par défaut, >0 pour filtrer les utilisateurs de liste_id
   * @param FILTRE_PRIV_ID    filtre sur le privilège utilisateur
   * @param profil_id_exclude =-1 par défaut, >0 pour exclure les utilisateurs ayant ce profil_id_exclude
   * @param abonne_id_filtre  =0 pour filtrer les agents et retourner les agent_id, 
   *                          =1 pour filtrer les abonnnés et retourner les abonne_id
   *                          =2 pour filtrer les agents et retourner les abonne_id
   *                          =3 pour filtrer les abonnés et retourner les abonne_id et agent_id
   * @return Retourne un dataSet
   */
  public function getDsListUsers($FILTRE_NOM, $FILTRE_PROFIL_ID, $FILTRE_SERVICE_ID, $FILTRE_MISSION_ID, 
                                 $cont_id_filtre, $cont_id, $FILTRE_ESPACE, $bFiltreEspace, 
                                 $cont_id_exclude, $appli_id_filtre, $iAffAdmin, $FILTRE_LISTE_ID="-1", $FILTRE_PRIV_ID="-1",
                                 $profil_id_exclude="-1", $abonne_id_filtre="0")
  {
    $strFrom = "";
    $strWhere = " ag.AGENT_VALIDE=1 and s.SERVICE_NIVEAU=1";
    if( $FILTRE_NOM != "" ) {
      $strField = ( $abonne_id_filtre=="0" ? "ag.AGENT_NOM" : "ab.ABONNE_NOM" );
      $strWhere .= " and ".$this->dbConn->GetLowerCase($strField)." like ".
        $this->dbConn->GetLowerCase("'".$this->dbConn->AnalyseSql($FILTRE_NOM)."%'");
    }

    if( $FILTRE_SERVICE_ID != "-1" ) {
      $strWhere .= " and s.SERVICE_ID=".$FILTRE_SERVICE_ID;
    }
    
    if( $FILTRE_MISSION_ID != "0" ) {
      $strFrom .= " inner join SIT_AGENT_MISSION am on ag.AGENT_ID=am.AGENT_ID";
      $strWhere .= " and am.MISSION_ID=".$FILTRE_MISSION_ID;
    }

    if( $cont_id_filtre != "" ) {
      $strFrom .= " inner join SIT_AGENT_CONT ac on ag.AGENT_ID=ac.AGENT_ID";
      $strWhere .= " and ac.CONT_ID=".$cont_id_filtre;
    }
    
    if( $cont_id!="" && $bFiltreEspace==true && $FILTRE_ESPACE!="0" ) {
      $strFrom .= " inner join SIT_AGENT_CONT ac1 on ag.AGENT_ID=ac1.AGENT_ID";
      $strWhere .= " and ac1.CONT_ID=".$cont_id;
    } 

    if( $appli_id_filtre != "" ) {
      $strFrom .= " inner join V_USER_RIGHT ap on ag.AGENT_ID=ap.USER_ID";
      $strWhere .= " and ap.APPLI_ID=".$appli_id_filtre." and ap.RIGHT_ID>0";
    }

    if( $FILTRE_PROFIL_ID != "0" || $profil_id_exclude != "-1" ) {
      $strFrom .= " inner join SIT_AGENT_PROFIL apr on ag.AGENT_ID=apr.AGENT_ID";
      if( $profil_id_exclude != "-1" ) { 
        $strWhere .= " and p.PROFIL_ID<>".$profil_id_exclude;
      }
      if( $FILTRE_PROFIL_ID != "0" ) {
        $strWhere .= " and apr.PROFIL_ID=".$FILTRE_PROFIL_ID;
      }
    }

    $strListeDifSql = "";
    if( $FILTRE_LISTE_ID != "-1" ) {
      $strFrom .= ( $abonne_id_filtre == "0"
                    ? " inner join SIT_ABONNE ab on ag.AGENT_ID=ab.AGENT_ID"
                    : "" ).
        " inner join SIT_ABONNE_LISTE al on ab.ABONNE_ID=al.ABONNE_ID";
      $strWhere .= " and al.LISTE_ID=".$FILTRE_LISTE_ID;
    }

    if( $iAffAdmin != 1 ) {
      // n'affiche pas les admin généraux
      $strWhere .= " and (ag.AGENT_PRIV & ".ALK_PRIV_SPACE_ADMIN.")<>".ALK_PRIV_SPACE_ADMIN;
    } elseif( $FILTRE_PRIV_ID != -1 ) {
      
      $strWhere .= ( $FILTRE_PRIV_ID != ALK_PRIV_SPACE_USER
                     ? " and (ag.AGENT_PRIV & ".$FILTRE_PRIV_ID.")=".$FILTRE_PRIV_ID
                     : " and ag.AGENT_PRIV=0" );
    }

    if( $cont_id_exclude != "" ) {
      // on retire les agents appartenant à l'espace à exclure de la selection
      /*$strSelect = ", ac2.CONT_ID, ac2.AGENT_ID";
      $strFrom .= "  ) left join SIT_AGENT_CONT ac2 on ag.AGENT_ID=ac2.AGENT_ID";
      $strWhere .= " and ac2.CONT_ID=".$cont_id_exclude;*/
      $strWhere .= " and ag.AGENT_ID not in (select AGENT_ID from SIT_AGENT_CONT where CONT_ID=".$cont_id_exclude.")";
    }
    
    if( $abonne_id_filtre == "2" ) {
      $strWhere .= " and ab.AGENT_ID<>".ALK_USER_ID_INTERNET;
    }

    $strSql = "select distinct " .
      ( $abonne_id_filtre == "0"
        ? "ag.AGENT_ID as id, ".
          $this->dbConn->GetConcat("ag.AGENT_NOM", "' '", "ag.AGENT_PRENOM")." as nom, ag.AGENT_ID,"
        : ( $abonne_id_filtre == "3"
            ? $this->dbConn->GetConcat("ab.ABONNE_ID", "'_'", "ag.AGENT_ID")." as id, " 
            : "ab.ABONNE_ID as id, " ).
          $this->dbConn->GetConcat("ab.ABONNE_NOM", "' '", "ab.ABONNE_PRENOM", 
                                   $this->dbConn->compareSql("ab.AGENT_ID", "=", ALK_USER_ID_INTERNET."", "''", "'*'"))." as nom,".
          " ab.AGENT_ID," ).
      "   p.PROFIL_INTITULE as profil,".
      "   s.SERVICE_INTITULE_COURT as service, s.SERVICE_ID as idservice".
      " from " .
      ( $abonne_id_filtre == "0"
        ? "SIT_AGENT ag"
        : "SIT_ABONNE ab".
          " inner join SIT_AGENT ag on ab.AGENT_ID=ag.AGENT_ID"  ).
      "  inner join SIT_PROFIL p on ag.PROFIL_ID=p.PROFIL_ID".
      "  inner join SIT_AGENT_SERVICE ase on ag.AGENT_ID=ase.AGENT_ID".
      "  inner join SIT_SERVICE s on ase.SERVICE_ID=s.SERVICE_ID".
      $strFrom.
      " where ".$strWhere.
      " order by nom, service";
    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   * Retourne la liste des espaces contenant des applications type atype_id
   * accessibles à l'utilisateur
   * @param user_id   Identifiant de l'agent connecté
   * @param atype_id  Type d'application
   * @return AlkDs
   */
  function getDsListSpaceByATypeIdForCombo($user_id, $atype_id)
  {
    $strSql = "select distinct c.CONT_INTITULE_COURT, c.CONT_ID ".
      " from SIT_CONTENEUR c".
      "   inner join V_USER_RIGHT r on c.CONT_ID=r.CONT_ID".
      "   inner join SIT_APPLI a on a.APPLI_ID=r.APPLI_ID".
      " where r.USER_ID=".$user_id." and a.ATYPE_ID in (".$atype_id.") and r.RIGHT_ID>0".
      " order by c.CONT_INTITULE_COURT";
    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   * Retourne la liste des applications type atype_id de l'espace cont_id
   * accessibles à l'utilisateur
   * @param user_id             Identifiant de l'agent connecté
   * @param cont_id             Identifiant de l'espace sélectionné
   * @param atype_id            Type d'application
   * @return AlkDs
   */
  function getDsListAppliBySpaceByATypeIdForCombo($user_id, $cont_id, $atype_id)
  {
    $strSql = "select distinct a.APPLI_INTITULE, a.APPLI_ID, ".
      $this->dbConn->getConcat("a.APPLI_ID", "'_'", "a.ATYPE_ID")." as appliid_atypeid".
      " from SIT_APPLI a".
      "   inner join V_USER_RIGHT r on a.APPLI_ID=r.APPLI_ID".
      " where r.CONT_ID=".$cont_id." and r.USER_ID=".$user_id." and a.ATYPE_ID in (".$atype_id.") and r.RIGHT_ID>0".
      " order by a.APPLI_INTITULE";
    return $this->dbConn->InitDataSet($strSql);
  }
  
  /**
   * Retourne la liste des applications de l'espace cont_id
   * accessibles à l'utilisateur
   * @param user_id   Identifiant de l'agent connecté
   * @param cont_id   Identifiant de l'espace sélectionné
   * @return AlkDs
   */
  function getDsListAppliBySpaceForCombo($user_id, $cont_id)
  {
    $strSql = "select distinct a.APPLI_INTITULE, a.APPLI_ID ".
      " from SIT_APPLI a".
      "   inner join V_USER_RIGHT r on a.APPLI_ID=r.APPLI_ID".
      " where r.CONT_ID=".$cont_id." and r.USER_ID=".$user_id." and r.RIGHT_ID>0".
      " order by a.APPLI_INTITULE";
    return $this->dbConn->InitDataSet($strSql);
  }
        
  /**
   * Retourne la liste des départements visibles
   * @param bAffDept  true pour afficher le nom département, false pour un tiret
   * @return AlkDs
   */
  public function getDsListeDptCont($cont_id)
  {
    $strRegion = "R&eacute;gion";
    $strSql = "select ".
      $this->dbConn->getConcat(
        $this->dbConn->compareSql("d.DEPT_PERE", "=", "0", 
                                  $this->dbConn->compareSql("d.DEPT_ID", "=", "0", "''", "'".$strRegion." '"), 
                                  "'&nbsp;&nbsp;- '"),
        "d.DEPT_INTITULE")." as DEPT_LABEL, d.DEPT_ID, d.DEPT_PERE, d.DEPT_INTITULE, dc.CONT_ID".
      " from SIT_DEPT d".
      "   left join SIT_DEPT_CONT dc on (dc.DEPT_ID=d.DEPT_ID and dc.CONT_ID=".$cont_id.")".
      " where d.DEPT_VISIBLE=1".
      " order by d.DEPT_PERE, DEPT_LABEL";
    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   * Retourne un tableau contenant les clés : cont_id, dir
   * la clé dir vaut / si l'alias correspond à Internet
   * @param strAlias  nom de l'alias recherché correspondant à un répertoire d'url
   * @return array 
   */
  public function getContIdByAlias($strAlias)
  {
    $tabRes = array("cont_id" => -1, "dir" => "/");
    
    $strSql = "select CONT_ID, CONT_INTITULE_COURT".
      " from SIT_CONTENEUR".
      " where CONT_INTITULE_COURT='".$this->dbConn->analyseSql($strAlias)."'";
    $dsCont = $this->dbConn->initDataSet($strSql);
    if( $drCont = $dsCont->getRowIter() ) {
      $tabRes["cont_id"] = $drCont->getValueName("CONT_ID");
      $cont_alias        = $drCont->getValueName("CONT_INTITULE_COURT");
      if( mb_strtolower($cont_alias) != "internet" ) {
        $tabRes["dir"] .= $cont_alias;
      }
    }
    
    return $tabRes;
  }
  
  /**
   * Retourne un tableau contenant les clés suivantes :
   *   - date      => 02/09/2009
   *   - dateshort => 2 sept. 2009
   *   - datelong  => lundi 2 septembre 2009
   *   - time      => 15:02 (heure courante)
   *   - saint     => saint du jour
   * @param iDay    numéro du jour, vide par défaut pour le jour courant
   * @param iMonth  numéro du jour, vide par défaut pour le mois courant
   * @param iYear   année, vide par défaut pour l'année courante
   * @param iHour   heure au format 0..24, vide pour l'heure courante
   * @param iMin    minute, vide pour l'heure courante
   * @return array
   */
  public function getEphemiride($iDay="", $iMonth="", $iYear="", $iHour="", $iMin="")
  {
    $tabDay        = array("Dimanche", "Lundi", "Mardi", "Mercredi", "Jeudi", "Vendredi", "Samedi");
    $tabShortMonth = array("", "jan.", "fév.", "mar.", "avr.", "mai", "jun.", "jul.", "aoû.", "sep.", "oct.", "nov.", "déc.");
    $tabMonth      = array("", "janvier", "février", "mars", "avril", "mai", "juin", "juillet", "août", "septembre", "octobre", "novembre", "décembre"); 
    
    if( !($iDay != "" && $iMonth!="" && $iYear!="") ) {
      $iDay   = date("j");
      $iMonth = date("n");
      $iYear  = date("Y"); 
    }
    
    $iTime = "time()";
    if( !($iHour != "" && $iMin != "") ) {
      $iHour = date("G");
      $iMin  = date("i")*1;  
    } 
    $iTime = mktime($iHour, $iMin, 0, $iMonth, $iDay, $iYear);
    
    $tabRes = array("date"      => date("d/m/Y", $iTime),
                    "time"      => date("H:i", $iTime),
                    "saint"     => "",
                    "dateshort" => date("j", $iTime)." ".$tabShortMonth[date("n", $iTime)]." ".$iYear,
                    "datelong"  => $tabDay[date("w", $iTime)]." ".date("j", $iTime).( date("j", $iTime)==1 ? "er" : "")." ".$tabShortMonth[date("n", $iTime)]." ".$iYear);
    $strSql = "select * from SIT_EPHEMERIDE where EPH_MOIS=".$iMonth." and EPH_JOUR=".$iDay;
    $dsEph = $this->dbConn->initDataSet($strSql);
    while( $drEph = $dsEph->getRowIter() ) {
      $tabRes["saint"] .= ( $tabRes["saint"] != "" ? ", " : "" ).$drEph->getValueName("EPH_SAINT");
    } 
    
    return $tabRes;
  }
  
  public function getDsEspaceDupliquablesForCombo($user_id){
    $strSql = "select CONT_INTITULE, sc.CONT_ID from ".
              " SIT_CONTENEUR sc".
              " inner join SIT_AGENT_CONT sag on (sag.CONT_ID=sc.CONT_ID".
              " and sag.CONT_ADMIN=1 and sag.AGENT_ID=".$user_id.") order by CONT_INTITULE";
    
    return $this->dbConn->InitDataSet($strSql);
              
  }
  
  /**
   * retourne la liste des constantes triée par label en excluant celles qui ne peuvent pas 
   * être administrées
   * @param atype_id
   * @param appli_id identfiant de l'appli courante
   * @param cont_id identifiant de l'espace courant
   * @return Array
   */
  public function getTableauListeSitConstantes($atype_id, $appli_id, $cont_id){
    $strSql = " select * from SIT_CONSTANTE".
               " where APPLI_ID in(-1,".$appli_id.")".
               " and CONT_ID in (-1,".$cont_id.")".
               " and CONST_ADMIN <> 0".
               " and ATYPE_ID=".$atype_id.
               " order by CONST_BLOC, CONST_LABEL, CONST_LEVEL";
    $dsConst = $this->dbConn->InitDataSet($strSql);
    
    $tabRes = array();
    while($drConst = $dsConst->getRowIter()){
      $const_intitule   = $drConst->getValueName("CONST_INTITULE");
      $const_valeur     = $drConst->getValueName("CONST_VALEUR");
      $const_type       = $drConst->getValueName("CONST_TYPE");
      $const_label      = $drConst->getValueName("CONST_LABEL");
      $const_admin      = $drConst->getValueName("CONST_ADMIN");
      $appli_id         = $drConst->getValueName("APPLI_ID");
      $strTitle         = $drConst->getValueName("CONST_BLOC");
      $cont_id          = $drConst->getValueName("CONT_ID");
      
      if (!array_key_exists($const_intitule, $tabRes)){
        $tabRes[$const_intitule] = array("CONST_VALEUR"=>$const_valeur,
                                         "CONST_TYPE"=>$const_type,
                                         "CONST_LABEL"=>$const_label,
                                         "CONST_ADMIN"=>$const_admin,
                                         "APPLI_ID"=>$appli_id,
                                         "CONT_ID" =>$cont_id,
                                         "CONST_BLOC"=>$strTitle);
      }else {
        
        if ($appli_id != -1 && $cont_id != -1){
          $tabRes[$const_intitule]["APPLI_ID"] = $appli_id;
          $tabRes[$const_intitule]["CONT_ID"] = $cont_id;
          $tabRes[$const_intitule]["CONST_VALEUR"] = $const_valeur;
        }
      }
    }
    return $tabRes;
  }
  
  /**
   * retourne la liste des connexions
   * @param bDay indique s'il faut retourner uniquement les connexions de la journée
   */
  public function getDsListConnexionUser($bDay=true){
    
    $date_debut = $this->dbConn->getDateFormat("DD/MM/YYYY HH:MI", "'".date("d/m/Y")." 00:00"."'", true);
    $date_fin   = $this->dbConn->getDateFormat("DD/MM/YYYY HH:MI", "'".date("d/m/Y")." 23:59"."'", true);

    $strSql = " select ag.AGENT_ID, ".$this->dbConn->GetConcat("ag.AGENT_PRENOM", "' '", "ag.AGENT_NOM")." as NOM, ".
              " CONT_INTITULE, APPLI_INTITULE, SESSION_ID".
              " from SIT_AGENT_CONN conn".
              " inner join SIT_CONTENEUR cont on cont.CONT_ID=conn.CONT_ID".
              " inner join SIT_APPLI appl on appl.APPLI_ID=conn.APPLI_ID".
              " inner join SIT_AGENT ag on ag.AGENT_ID=conn.AGENT_ID".
             ($bDay
              ? " where DATE_CONN <=".$date_fin." and DATE_CONN >=".$date_debut
              : "").
             " order by NOM";
    return $this->dbConn->InitDataSet($strSql);
    
  }
  
   /**
    * retourne la liste des thématiques associées  au groupe de travail courant 
    * @param cont_id : id du groupe de travail (ou espace_fils) courant
    */
  
   public function getDsListTypeByGrpTravailForCombo($cont_id){
   
        $strSql = "select TYPE_INTITULE, TYPE_ID from ".
              " SIT_CONT_TYPE ct ".
              " left join SIT_CONTENEUR st on (st.CONT_TYPE_ID = ct.TYPE_ID and st.CONT_ID = ".$cont_id.") ".
              " order by TYPE_INTITULE"; 
        
    return $this->dbConn->InitDataSet($strSql);
              
 }
  
 /**
  * Retourne les propriétés de la categorie type_id
  *
  * @param type_id   Identifiant de la categorie
  * @return Retourne un dataSet
  */
 public function GetDs_ficheCategorieById($type_id)
 {
   $strSql = "select TYPE_INTITULE from SIT_CONT_TYPE WHERE TYPE_ID=".$type_id;
   return $this->dbConn->InitDataSet($strSql);
 }
 
 /**
  * Retourne true si l'agent est animateur de l'espace
  * @param user_id   Id de l'agent
  * @param cont_id   Id de l'espace
  * @return Retourne un booleen, vrai si l'agent est animateur de l'espace
  */
  public function userIsAnimateurOfSpace($user_id, $cont_id)
  {
    $strSql = "SELECT CONT_ADMIN ".
      " FROM SIT_AGENT_CONT ".
      " WHERE AGENT_ID = ".$user_id.
      " AND CONT_ID = ".$cont_id;
    return ( $this->dbConn->getScalarSql($strSql,0) == 1 ? true : false );
  }

  /**
   * Retourne la liste des espaces correspondant au nom de domaine passé en paramètre
   * @param string   $Root_Cont_Url  Url du domaine dont on recherche l'espace
   * @param boolean  $bMainOnly      Si vrai renvoi juste le premier résultat
   * @return AlkDs Liste des Espaces correspondant à l'url du domaine recherché
   */
  public function getDs_ListEspaceByRootContUrl($Root_Cont_Url, $bMainOnly=false){
    
    $strSql = "SELECT DISTINCT * ".
      " FROM SIT_CONTENEUR ".
      " WHERE ";
    $bFirst=true;
    foreach($this->tabLangue as $key=>$tabLg) {
      $strSql .= ($bFirst ? "" : " OR ")." CONT_ROOT_FRONT_URL".$tabLg["bdd"]." LIKE '".$this->dbConn->analyseSql( substr($Root_Cont_Url, -1)=="/" ? substr($Root_Cont_Url, 0, -1) : $Root_Cont_Url )."' ";
      $strSql .= " OR CONT_ROOT_FRONT_URL".$tabLg["bdd"]." LIKE '".$this->dbConn->analyseSql( substr($Root_Cont_Url, -1)=="/" ? $Root_Cont_Url : $Root_Cont_Url."/" )."' ";
      $strSql .= " OR ".$this->dbConn->getConcat("CONT_ROOT_FRONT_URL".$tabLg["bdd"], "'/'")." LIKE '".$this->dbConn->analyseSql( substr($Root_Cont_Url, -1)=="/" ? $Root_Cont_Url : $Root_Cont_Url."/" )."' ";
      $strSql .= " OR CONT_ROOT_ADMIN_URL".$tabLg["bdd"]." LIKE '".$this->dbConn->analyseSql( substr($Root_Cont_Url, -1)=="/" ? substr($Root_Cont_Url, 0, -1) : $Root_Cont_Url )."' ";
      $strSql .= " OR CONT_ROOT_ADMIN_URL".$tabLg["bdd"]." LIKE '".$this->dbConn->analyseSql( substr($Root_Cont_Url, -1)=="/" ? $Root_Cont_Url : $Root_Cont_Url."/" )."' ";
      $strSql .= " OR ".$this->dbConn->getConcat("CONT_ROOT_ADMIN_URL".$tabLg["bdd"], "'/'")." LIKE '".$this->dbConn->analyseSql( substr($Root_Cont_Url, -1)=="/" ? $Root_Cont_Url : $Root_Cont_Url."/" )."' ";
      $bFirst = false;
    }
    $strSql .= " ORDER BY CONT_NIVEAU DESC ";
    
    if($bMainOnly) {
      return $this->dbConn->InitDataSet($strSql,0 ,1);
    }else{
      return $this->dbConn->InitDataSet($strSql);
    }
  }
  
  /**
   * Retourne la liste des espaces correspondant au nom de domaine passé en paramètre
   * @param string  $Root_Cont_Url  Url du domaine dont on recherche l'espace
   * @return int Retourne le cont_id de l'espace correspondant à l'url du domaine recherché
   */
  public function getEspaceIDByRootContUrl($Root_Cont_Url){
    $strSql = "SELECT DISTINCT CONT_ID ".
      " FROM SIT_CONTENEUR ".
      " WHERE ";
    $bFirst=true;
    foreach($this->tabLangue as $key=>$tabLg) {
      $strSql .= ($bFirst ? "" : " OR ")." CONT_ROOT_FRONT_URL".$tabLg["bdd"]." LIKE '".$this->dbConn->analyseSql( substr($Root_Cont_Url, -1)=="/" ? substr($Root_Cont_Url, 0, -1) : $Root_Cont_Url )."' ";
      $strSql .= " OR CONT_ROOT_FRONT_URL".$tabLg["bdd"]." LIKE '".$this->dbConn->analyseSql( substr($Root_Cont_Url, -1)=="/" ? $Root_Cont_Url : $Root_Cont_Url."/" )."' ";
      $strSql .= " OR ".$this->dbConn->getConcat("CONT_ROOT_FRONT_URL".$tabLg["bdd"], "'/'")." LIKE '".$this->dbConn->analyseSql( substr($Root_Cont_Url, -1)=="/" ? $Root_Cont_Url : $Root_Cont_Url."/" )."' ";
      $strSql .= " OR CONT_ROOT_ADMIN_URL".$tabLg["bdd"]." LIKE '".$this->dbConn->analyseSql( substr($Root_Cont_Url, -1)=="/" ? substr($Root_Cont_Url, 0, -1) : $Root_Cont_Url )."' ";
      $strSql .= " OR CONT_ROOT_ADMIN_URL".$tabLg["bdd"]." LIKE '".$this->dbConn->analyseSql( substr($Root_Cont_Url, -1)=="/" ? $Root_Cont_Url : $Root_Cont_Url."/" )."' ";
      $strSql .= " OR ".$this->dbConn->getConcat("CONT_ROOT_ADMIN_URL".$tabLg["bdd"], "'/'")." LIKE '".$this->dbConn->analyseSql( substr($Root_Cont_Url, -1)=="/" ? $Root_Cont_Url : $Root_Cont_Url."/" )."' ";
      $bFirst = false;
    }
    return $this->dbConn->getScalarSql($strSql, -1);
  }

  /**
   * Retourne l'url du domaine de l'espace pour coté site ou administration, si aucune url n'est renseigné pour l'administration, celle du site est renvoyé à la place
   * @param int     $cont_id         Id de l'espace
   * @param boolean $bRootUrlAdmin   Si Vrai, la fonction renvoi l'url domaine d'administration de l'espace
   * @return string Url domaine de l'espace
   */
  public function getEspaceRootContUrlByID($cont_id, $bRootUrlAdmin=false, $lg="_FR"){
     $strSql = "SELECT DISTINCT CONT_ROOT_".($bRootUrlAdmin ? "ADMIN" : "FRONT")."_URL".$lg." ".
      " FROM SIT_CONTENEUR ".
      " WHERE CONT_ID = ".$cont_id;
     $Res = $this->dbConn->getScalarSql($strSql, "");
     if($Res != "" || $bRootUrlAdmin === false) return $Res;
     else{
       $strSql = "SELECT DISTINCT CONT_ROOT_FRONT_URL".$lg." ".
      " FROM SIT_CONTENEUR ".
      " WHERE CONT_ID = ".$cont_id;
       return $this->dbConn->getScalarSql($strSql, "");
     }
  }
  
  /**
   * Retourne la liste des options d'une application
   * @param int $appli_id
   * @return AlkDs
   */
  public function getDsOptions($appli_id) {
    $strSql = "SELECT * FROM sit_appli_option WHERE appli_id = ".$appli_id;
    return $this->dbConn->initDataset($strSql);
  }
    
   /**
   * retourne les informations d'une option d'application donnée
   * @param option identifiant de l'option
   * @return AlkDs
   */
  public function getDsOptionById($option_id){
    $strSql = " select * from SIT_APPLI_OPTION where OPTION_ID=".$option_id;
    return $this->dbConn->InitDataSet($strSql);     
  }
  
}

?>