<?php
/*licence/ 

Module écrit, supporté par la société Alkante SAS <alkante@alkante.com>

Nom du module : Alkanet::Module::GEdit
Module gestion éditoriale.
Ce module appartient au framework Alkanet.

Ce logiciel est régi par la licence CeCILL-C soumise au droit français et
respectant les principes de diffusion des logiciels libres. Vous pouvez
utiliser, modifier et/ou redistribuer ce programme sous les conditions
de la licence CeCILL-C telle que diffusée par le CEA, le CNRS et l'INRIA
sur le site http://www.cecill.info.

En contrepartie de l'accessibilité au code source et des droits de copie,
de modification et de redistribution accordés par cette licence, il n'est
offert aux utilisateurs qu'une garantie limitée. Pour les mêmes raisons,
seule une responsabilité restreinte pèse sur l'auteur du programme, le
titulaire des droits patrimoniaux et les concédants successifs.

A cet égard l'attention de l'utilisateur est attirée sur les risques
associés au chargement, à l'utilisation, à la modification et/ou au
développement et à la reproduction du logiciel par l'utilisateur étant
donné sa spécificité de logiciel libre, qui peut le rendre complexe à
manipuler et qui le réserve donc à des développeurs et des professionnels
avertis possédant des connaissances informatiques approfondies. Les
utilisateurs sont donc invités à charger et tester l'adéquation du
logiciel à leurs besoins dans des conditions permettant d'assurer la
sécurité de leurs systèmes et ou de leurs données et, plus généralement,
à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.

Le fait que vous puissiez accéder à cet en-tête signifie que vous avez
pris connaissance de la licence CeCILL-C, et que vous en avez accepté les
termes.

/licence*/

require_once ALK_ALKANET_ROOT_PATH.ALK_ROOT_CLASSE."pattern/alkappli.class.php";

$iCpt = 0;
define("ALK_LTYPE_ID_PAGE", $iCpt++);
foreach($GLOBALS["_LG_tab_langue"] as $key => $tabLg) {
  define("ALK_LTYPE_ID_PAGE".$tabLg["bdd"], $iCpt++);
}
$iCpt = 100;

/**
 * @package Alkanet_Module_GEdit
 * Module gestion éditoriale
 * @class AlkAppliGEdit
 * Classe de l'application Gestion éditoriale  
 */
class AlkAppliGEdit extends AlkAppli implements AlkIntAppli, AlkIntRights, AlkIntGedit, AlkIntWorkflow, AlkIntAbonne
{

  /**
   * Constructeur par défaut
   * @param appli_id  Identifiant de l'application associée
   */
  public function __construct($appli_id)
  {
    parent::__construct(ALK_ATYPE_ID_GEDIT, $appli_id);
    
    if ( !defined("ALK_GEDIT_NB_VISUEL") ){
      define("ALK_GEDIT_NB_VISUEL", 0);
    }
    if ( !defined("ALK_GEDIT_NB_BANDEAU") ){
      define("ALK_GEDIT_NB_BANDEAU", 0);
    }
    if($this->cont_id != -1){
      $strPathUpload = AlkFactory::getUploadPath(ALK_ATYPE_ID_GEDIT).$this->cont_id."/visuel/";
      if (!is_dir($strPathUpload."declinaisons/")) {
        if (!is_dir($strPathUpload."")) {
          @mkdir($strPathUpload."", 0770);
        }
        @mkdir($strPathUpload."declinaisons/", 0770);
      }
    }
    
    $this->strXML = "";
    $this->strHTML = "";
    $this->strXSL = "";
    $this->bError = "";

    $this->iTypeSheet = ( defined("ALK_B_GEDIT_CONSULT") && ALK_B_GEDIT_CONSULT==TRUE
                          ? ALK_TYPESHEET_CONSULT
                          : ( strpos($_SERVER["PHP_SELF"], "alkanet_site.php")===false 
                              ? ALK_TYPESHEET_ADMIN 
                              : ALK_TYPESHEET_CONSULT ) );
    $this->iSheet     = ALK_SHEET_FORM;
    $this->iSSheet    = ALK_SHEET_NONE;
  }

  /**
   * Destructeur par défaut
   */
  public function __destruct() { }

  /**
   *  Vérifie la sécurité. Retourne vrai si l'utilisateur peut afficher cette page, faux sinon.
   * @return boolean
   */
  public function verifSecu()
  {
    // ajoute le privilège ALK_PRIV_SPACE_ANIM si utilisateur animateur de l'espace en cours
    // ajoute le privilège ALK_PRIV_SPACE_USER si utilisateur valide
    $this->oSpace->verifIfUserIsAnim();
    $user_priv = AlkFactory::getSProperty("user_priv", ALK_PRIV_SPACE_NONE);
    
    $iTypeSheet = AlkRequest::getToken("iTypeSheet", ALK_TYPESHEET_CONSULT);

    // récupération du droit utilisateur sur l'application en cours
    $iRight = $this->getUserAppliRight();
    AlkFactory::setSProperty("user_right", $iRight);

    $bRes = 
      ($user_priv & ALK_PRIV_SPACE_USER) == ALK_PRIV_SPACE_USER 
      &&  ( 
          ($user_priv & ALK_PRIV_SPACE_ANIM) == ALK_PRIV_SPACE_ANIM 
      ||  ( $iTypeSheet == ALK_TYPESHEET_CONSULT 
            &&  ( $iRight>=ALK_APPLI_RIGHT_READ 
                  || ($user_priv & ALK_PRIV_SPACE_VIEWER) == ALK_PRIV_SPACE_VIEWER ) 
          ) 
      ||  ( $iTypeSheet == ALK_TYPESHEET_ADMIN      && $iRight>=ALK_APPLI_RIGHT_READ ) 
      ||  ( $iTypeSheet == ALK_TYPESHEET_PROPRIETE  && $iRight==ALK_APPLI_RIGHT_PUBLI ) 
      ||  ( $iTypeSheet == ALK_TYPESHEET_POPUP ) 
      );

    // droit spécifique à l'application
    $bRes = $bRes && $this->verifSecuAppli();

    return $bRes;
  }

  /**
   * Vérifie les droits spécifiques à l'application
   *        Retourne vrai si l'utilisateur connecté possède les droits, faux sinon
   * @return booleen
   */
  public function verifSecuAppli()
  { 
  	return true;
  }

  /** interface Appli */

  /**
   * Ajout d'une application à un espace. Retourne l'identifiant de l'appli créée
   * @param tabFields tableau associatif des champs nécessaires à la création d'une appli
   * @return int
   */
  public function addAppli($tabFields) { /** rien à faire */ }

  /**
   * Modification d'une application
   * @param tabFields tableau associatif des champs nécessaires à la création d'une appli
   */
  public function updateAppli($tabFields) { /** rien à faire */ }

  /**
   * Suppression d'une application d'un espace
   */
  public function delAppli()
  {
    /* suppression de tous les blocs appartenant aux autres applications associées aux pages de cette appli */
    $tabBlocId = array();
    $dsBlocListe = $this->oQuery->getBlocByAppliId($this->appli_id);
    while( $drBlocListe = $dsBlocListe->getRowIter() ) {
      $bloc_id  = $drBlocListe->getValueName("BLOC_ID");
      $atype_id = $drBlocListe->getValueName("ATYPE_ID");
      $tabBlocId[$atype_id] = ( array_key_exists($atype_id, $tabBlocId)
                                 ? $tabBlocId[$atype_id].",".$bloc_id
                                 : $bloc_id );
    }
    
    foreach($tabBlocId as $atype_id => $strListBlocId) {
      $oAppli = AlkFactory::getAppli($atype_id);
      if( $atype_id!=ALK_ATYPE_ID_GEDIT && !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "delGEditBloc") )
        $oAppli->delGEditBloc(-1, $strListBlocId, -1, -1);
    }
    
    $this->delDataAssoc($this->cont_id);
    $this->oQueryAction->delAppli($this->appli_id, $this->strPathUpload);
  }

  /**
   * Déplacement d'une application d'un espace vers un autre
   *        Cette méthode se charge d'appelé la recopie des droits
   * @param $appli_id  identifiant de l'appli
   * @param $idContTo  identifiant de l'espace recevant l'appli
   */
  public function moveAppli($idContTo) { /** rien à faire */ }

  /**
   * Duplication de la structure et du paramètrage de l'application src à la courante
   * @param appli_id_src  identifiant de l'application source
   * @param iTypeDup      type de duplication : 0 = application vierge (ne fait rien) 
   *                                            1 = structure et paramétrage
   *                                            2 = structure, paramétrage et données  
   */
  public function dupAppli($appli_id_src, $iTypeDup) 
  {  
    if( $iTypeDup == 0 ) return;
    
    $this->oQueryAction->dupAppli($appli_id_src, $this->appli_id, $iTypeDup);
  }

  /** interface Abonne */
  
  /**
   * Méthode appelée pour créer une liste d'abonnés associée à la donnée
   */
  public function createListeAbonnes($data_id=-1, $mode=-1, $iAbonnement=0)
  {
    if (defined("ALK_B_GEDIT_ABONNEMENT") && ALK_B_GEDIT_ABONNEMENT==true) {
      $oQueryActionAnnu = AlkFactory::getQueryAction(ALK_ATYPE_ID_ANNU);
      $oQueryActionLsDif = AlkFactory::getQueryAction(ALK_ATYPE_ID_LSDIF);
      $user_id = AlkFactory::getSProperty("user_id", -1);
      $tabIntitule = array();
      $tabDesc = array();
      foreach($this->tabLangue as $key => $tabLg) {
        $tabIntitule[$key] = "Abonnés de la page id=".$data_id;
        $tabDesc[$key] = "";
      }
      $liste_id = $oQueryActionAnnu->addListe($tabIntitule, $tabDesc, 3, $user_id, 0, -1, 2);
      if (defined("ALK_ATYPE_ID_LSDIF") && ALK_ATYPE_ID_LSDIF==true){
        $msg_id = $oQueryActionLsDif->saveMsg(-1, $this->appli_id, -1, -1, "", "", 0, 0, 0, -1, -1, array());
        $oQueryActionLsDif->setInfoAbonnement("GEDIT_01_PAGE_LISTE", "PAGE_ID", $data_id, $liste_id, $msg_id, $iAbonnement);
      }
    }
  }
  
  /**
   * Méthode appelée pour autoriser l'abonnement
   */
  public function enableAbonnement($data_id=-1, $mode=-1)
  {
    $this->oQueryAction->enablePageAbonnement($data_id);
  }
  
  /**
   * Méthode appelée pour interdire l'abonnement
   */
  public function disableAbonnement($data_id=-1, $mode=-1)
  {
    $this->oQueryAction->disablePageAbonnement($data_id);
  }
  
  /**
   * Méthode appelée après suppression d'un abonnement
   * @param abonne_id  identifiant de l'abonné, peut contenir une liste d'identifiant
   * @param liste_id   identifiant de la liste de laquelle on souhaite désabonné
   */
  public function delAbonnement($abonne_id, $liste_id=-1) { /** rien à faire */ }

  /**
   * Méthode appelée avant suppression définitive d'un abonne
   * Attention, si un abonné est propriétaire d'une info, il ne faut pas la supprimer
   * Implémenter les actions effectuant le ménage interne à l'application
   * Retourne true si l'abonné reste supprimable, faux si il est propriétaire d'info
   * @param abonne_id  identifiant de l'abonné, peut contenir une liste d'identifiant
   * @return boolean
   */
  public function delAbonne($abonne_id) { /** rien à faire */ }

  /**
   * Méthode appelée avant suppression définitive d'une liste d'abonnés
   * @param liste_id  identifiant de la liste, peut contenir une liste d'identifiant
   */
  public function delListe($liste_id) 
  {
    $this->oQueryAction->delListe($liste_id);
  }


  /** interface User */

  /**
   * Transfère de propriété d'un utilisateur à un autre
   *        Méthode à appeler avant suppression définitive d'un utilisateur
   * @param idUserFrom  identifiant de l'utilisateur perdant la propriété des ses données
   * @param idUserTo    identifiant de l'utilisateur récupérant la propriété des données
   */
  public function replaceUser($idUserFrom, $idUserTo)
  {
    $this->oQueryAction->replaceUser($idUserFrom, $idUserTo);
  }

  /**
   * Invitation d'un utilisateur à un espace (tous si cont_id=-1, tous les espaces publics si cont_id=-2)
   * @param user_id  identifiant d'un utilisateur
   * @param cont_id  identifiant de l'espace. -1 pour tous les espaces. -2 tous les espaces publics
   * @param priv_id  identifiant du privilège
   */
  public function addUserToSpace($user_id, $cont_id, $priv_id) 
  {
    $this->oQueryAction->addUserToSpace($user_id, $cont_id, $this->appli_id);
  }

  /**
   * Modification du privilège espace pour l'utilisateur à un espace (tous si cont_id=-1)
   * @param user_id  identifiant d'un utilisateur
   * @param cont_id  identifiant de l'espace. -1 pour tous les espaces.
   * @param priv_id  identifiant du privilège
   */
  public function updateUserPrivToSpace($user_id, $cont_id, $priv_id) { /** rien à faire */ }

  /**
   * Suppression de l'accès à un espace (ou tous si cont_id=-1) pour un utilisateur
   * @param user_id  identifiant d'un utilisateur
   * @param cont_id  identifiant de l'espace. -1 pour tous les espaces.
   */
  public function removeUserFromSpace($user_id, $cont_id)
  {
    $this->oQueryAction->removeUserFromSpace($user_id, $cont_id);
  }

  /**
   * Personnalisation des droits d'un utilisateur
   * @note Appelé en admin. annuaire avec gestion des droits sur un profil sur l'espace courant
   * @param cont_id       identifiant de l'espace
   * @param user_id       identifiant de l'utilisateur sélectionné
   * @param iRight        droit fixé sur l'appli gedit
   * @param iDefaultRight non utilisé
   */
  public function setUserRightsOnSpace($cont_id, $user_id, $iRight=-1, $iDefaultRight=-1)
  {
    $this->oQueryAction->setUserRightsOnSpace($this->appli_id, $user_id, $iRight, $iDefaultRight);
  }
  
  /**
   * Enregistre les droits des profils sur les applications d'un espace
   * @param cont_id       identifiant d'utilisateur
   * @param tabRight      non utilisé dans le cas espace, 
   *                      sinon tableau de type "_".idProfil => droid_id pour les autres applis
   */
  public function setProfilRightsOnSpace($cont_id, $tabRight=array())
  {
    $this->oQueryAction->setProfilRightsOnSpace($this->appli_id, $tabRight);
  }
    
  /**
   * Application de droits sur les profils pour l'application
   * @note Appelé en config. espace avec gestion des droits sur les profils pour l'appli
   * @param atype_id  type d'application
   * @param appli_id  identifiant de l'application
   */
  public function setRightsOnAppli($atype_id, $appli_id)
  {
    $this->oQueryAction->setRightsOnAppli($appli_id);
  }
  
    /**
   * Applique les droits d'un utilisateur par rapport à ses profils
   * @param user_id  identifiant de l'utilisateur 
   */
  public function setProfilRightToUser($user_id)
  {
    $this->oQueryAction->setProfilRightToUser($user_id);
  }
    
  /** 
   * Ajout d'un profil sur l'extranet.
   *        Si l'application gère des droits, il faut y associer le droit NONE
   * @param profil_id  identifiant du profil à supprimer
   * @param cont_id    identifiant de l'espace associé, =0 pour tous les espaces (par défaut)
   */
  public function addProfil($profil_id, $cont_id=0) 
  {
    $this->oQueryAction->addProfil($profil_id, $cont_id);
  }

  /** 
   * Suppression du profil sur la gestion des droits interne à l'appli
   * @param profil_id  identifiant du profil à supprimer
   */
  public function delProfil($profil_id) 
  {
    $this->oQueryAction->delProfil($profil_id);
  }

  /** interface gedit */

  /**
   * Suppression d'une page gedit et de toute sa sous arbo
   * @param page_id   Identifiant de la page
   */
  public function delPage($page_id)
  {
    /* suppression de tous les blocs appartenant à page_id et à toute son arbo */
    $tabBlocId = array();
    $dsBlocListe = $this->oQuery->getBlocByPageId($page_id, 0, true, true);
    while( $drBlocListe = $dsBlocListe->getRowIter() ) {
      $bloc_id  = $drBlocListe->getValueName("BLOC_ID");
      $atype_id = $drBlocListe->getValueName("ATYPE_ID");
      $tabBlocId[$atype_id] = ( array_key_exists($atype_id, $tabBlocId)
                                 ? $tabBlocId[$atype_id].",".$bloc_id
                                 : $bloc_id );
    }
     
    foreach($tabBlocId as $atype_id => $strListBlocId) {
      if( $atype_id != ALK_ATYPE_ID_EDITEUR ) {
        $oAppli = AlkFactory::getAppli($atype_id);
        if( $atype_id!=ALK_ATYPE_ID_GEDIT && !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "delGEditBloc") ) {
          $oAppli->delGEditBloc($page_id, $strListBlocId, -1, -1);
        }
      }
    }
    
    $this->oQueryAction->delPage($page_id, $this->appli_id);
  }

  /**
   * Ajoute un bloc éditorial d'un certain type d'application
   *        Retourne l'identifiant du bloc créé
   * @param page_id     identifiant de la page
   * @param bloc_id     identifiant du bloc
   * @param typeIdBloc  type de bloc 
   * @param atypeIdBloc type de l'application associé au type de bloc
   * @param iColonne    type de représentation : normal=0, colonne=1
   * @return int
   */
  public function addGEditBlocInGEdit($page_id, $typeIdBloc, $atypeIdBloc, $iColonne, $bRssUser=false)
  {
    $user_id = AlkFactory::getSProperty("user_id", "-1");
    $bloc_id = $this->oQueryAction->addGEditBloc($page_id, $user_id, $typeIdBloc, $iColonne, $bRssUser);
    $this->addGEditBloc($bloc_id, $typeIdBloc, $atypeIdBloc);
    return $bloc_id;
  }
  
  /**
   * Méthode appelée à l'ajout d'un bloc gedit
   *        Implémenter les actions effectuant l'ajout de ce bloc sur ce type d'application
   * @param bloc_id     identifiant du bloc
   * @param typeIdBloc  type de bloc 
   * @param atypeIdBloc type de l'application associé au type de bloc
   */
  public function addGEditBloc($bloc_id, $typeIdBloc, $atypeIdBloc)
  {
    $oAppli = AlkFactory::getAppli($atypeIdBloc);
    if( $atypeIdBloc!=ALK_ATYPE_ID_GEDIT && !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "addGEditBloc") )
      $oAppli->addGEditBloc($bloc_id, $typeIdBloc, $atypeIdBloc);
  }
  
  /**
   * Méthode appelée avant suppression définitive d'une application de type gedit
   *        ou avant suppression d'une page gedit
   *        ou avant suppression d'un bloc gedit
   *        Implémenter les actions effectuant le ménage interne à l'application
   * @param page_id     identifiant de la page
   * @param bloc_id     identifiant du bloc
   * @param typeIdBloc  type de bloc 
   * @param atypeIdBloc type de l'application associé au type de bloc
   */
  public function delGEditBloc($page_id, $bloc_id, $typeIdBloc, $atypeIdBloc)
  {
    $user_id = AlkFactory::GetSProperty("user_id", "-1");

    $oAppli = AlkFactory::getAppli($atypeIdBloc);
    if( $atypeIdBloc!=ALK_ATYPE_ID_GEDIT && !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "delGEditBloc") )
      $oAppli->delGEditBloc($page_id, $bloc_id, $typeIdBloc, $atypeIdBloc);
      
    $this->oQueryAction->delGEditBloc($page_id, $bloc_id, $user_id);
  }

  /**
   * Recopie les associations d'un bloc à un autre bloc
   * 
   * @param bloc_id_src  identifiant du bloc source
   * @param bloc_id_dest identifiant du bloc destination
   */
  public function dupGEditBloc($bloc_id_src, $bloc_id_dest) { /** rien à faire */ }

  /**
   * Publie ou restaure la page de brouillon en fonction de la valeur de bCopy
   * 
   * @param uploadDir chemin d'acces au répertoire d'upload
   * @param page_id  identifiant de la page brouillon
   * @param bPublish =true par défaut pour copier le contenu du brouillon sur le contenu publié, false pour l'inverse
   */
  public function publishRestoreDraftPage($uploadDir, $page_id, $bPublish=true)
  {
    return $this->oQueryAction->publishRestoreDraftPage($uploadDir, $page_id, $bPublish);
  }

  /**
   * Associe une application de type atypeIdBloc au bloc
   * @param bloc_id     Identifiant du bloc
   * @param atypeIdBloc Type de l'application associée 
   * @param appli_id    Identifiant de l'application associée
   */
  public function assocAppliToBloc($bloc_id, $atypeIdBloc, $appli_id) 
  {
    $oAppli = AlkFactory::getAppli($atypeIdBloc, $appli_id);
    if( $atypeIdBloc!=ALK_ATYPE_ID_GEDIT && !is_null($oAppli) && is_object($oAppli) && 
        method_exists($oAppli, "removeCatFromBloc") && method_exists($oAppli, "removeDataFromBloc") ) { 
      $oAppli->removeCatFromBloc($bloc_id, "-1");
      $oAppli->removeDataFromBloc($bloc_id, "-1");
    }
    $this->oQueryAction->assocAppliToBloc($bloc_id, $appli_id);
  }
  
  /**
   * Associe une catégorie appartenant à l'application de type atypeIdBloc 
   * et d'identifiant appliIdBloc au bloc
   * @param bloc_id     Identifiant du bloc
   * @param cat_id      Identifiant de la catégorie
   * @param atypeIdBloc type de l'application contenant la catégorie, =-1 par défaut si non utile
   * @param appliIdBloc Identifiant de l'application contenant la catégorie, =-1 par défaut si non utile
   */
  public function assocCatToBloc($bloc_id, $cat_id, $atypeIdBloc=-1, $appliIdBloc=-1)
  {
    $this->removeAppliFromBloc($bloc_id, -1);
    
    $oAppli = AlkFactory::getAppli($atypeIdBloc, $appliIdBloc);
    if( $atypeIdBloc!=ALK_ATYPE_ID_GEDIT && !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "assocCatToBloc") )
      $oAppli->assocCatToBloc($bloc_id, $cat_id, $atypeIdBloc, $appliIdBloc);
  }

  /**
   * Associe un ensemble de données appartenant à l'application de type atypeIdBloc 
   * et d'identifiant appliIdBloc au bloc
   * @param bloc_id     Identifiant du bloc
   * @param tabDataId   Tableau contenant les identifiants des données à associer
   * @param atypeIdBloc type de l'application contenant la catégorie, =-1 par défaut si non utile
   * @param appliIdBloc Identifiant de l'application contenant la catégorie, =-1 par défaut si non utile
   */
  public function assocDataToBloc($bloc_id, $tabDataId, $atypeIdBloc=-1, $appliIdBloc=-1)
  {
    $this->removeAppliFromBloc($bloc_id, -1);
    $this->removeCatFromBloc($bloc_id, -1);
    
    $oAppli = AlkFactory::getAppli($atypeIdBloc, $appliIdBloc);
    if( $atypeIdBloc!=ALK_ATYPE_ID_GEDIT && !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "assocDataToBloc") )
      $oAppli->assocDataToBloc($bloc_id, $tabDataId, $atypeIdBloc, $appliIdBloc);
  }
  
  /**
   * Supprime une association entre une application de type atypeIdBloc et le bloc
   * @param bloc_id     Identifiant du bloc
   * @param appli_id    Identifiant de l'application
   * @param atypeIdBloc type de l'application contenant la catégorie, =-1 par défaut si non utile
   */
  public function removeAppliFromBloc($bloc_id, $appli_id)
  {
    $this->oQueryAction->removeAppliFromBloc($bloc_id, $appli_id); 
  }
  
  /**
   * Supprime une association entre une catégorie appartenant à l'application de type atypeIdBloc 
   * et d'identifiant appliIdBloc et le bloc
   * @param bloc_id     Identifiant du bloc
   * @param cat_id      Identifiant de la catégorie, =-1 pour supprimer toutes les catégories du bloc
   * @param atypeIdBloc type de l'application contenant la catégorie, =-1 par défaut si non utile
   * @param appliIdBloc Identifiant de l'application contenant la catégorie, =-1 par défaut si non utile
   */
  public function removeCatFromBloc($bloc_id, $cat_id, $atypeIdBloc=-1, $appliIdBloc=-1)
  {
    $oAppli = AlkFactory::getAppli($atypeIdBloc, $appliIdBloc);
    if( $atypeIdBloc!=ALK_ATYPE_ID_GEDIT && !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "removeCatFromBloc") )
      $oAppli->removeCatFromBloc($bloc_id, $cat_id, $atypeIdBloc, $appliIdBloc);
  }
  
  /**
   * Supprime une association entre une donnée appartenant à l'application de type atypeIdBloc 
   * et d'identifiant appliIdBloc et le bloc
   * @param bloc_id     Identifiant du bloc
   * @param data_id     Identifiant de la données à associer, =-1 pour supprimer toutes les données du bloc
   * @param atypeIdBloc type de l'application contenant la catégorie, =-1 par défaut si non utile
   * @param appliIdBloc Identifiant de l'application contenant la catégorie, =-1 par défaut si non utile
   */
  public function removeDataFromBloc($bloc_id, $data_id, $atypeIdBloc=-1, $appliIdBloc=-1)
  {
    $oAppli = AlkFactory::getAppli($atypeIdBloc, $appliIdBloc);
    if( $atypeIdBloc!=ALK_ATYPE_ID_GEDIT && !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "removeDataFromBloc") )
      $oAppli->removeDataFromBloc($bloc_id, $data_id, $atypeIdBloc, $appliIdBloc);
  }
  
  /**
   * Met à jour le rang d'une donnée dans un bloc
   * @param bloc_id     Identifiant du bloc
   * @param data_id     Identifiant de la donnée
   * @param iRank       Rang de la donnée courante
   * @param iDelta      Entier : 1 pour descendre d'un rang, -1 pour monter d'un rang
   * @param atypeIdBloc Type de l'application contenant la catégorie, =-1 par défaut si non utile
   * @param appliIdBloc Identifiant de l'application contenant la catégorie, =-1 par défaut si non utile
   */
  public function updateDataRankInBloc($bloc_id, $data_id, $iRank, $iDelta, $atypeIdBloc=-1, $appliIdBloc=-1)
  {
    $oAppli = AlkFactory::getAppli($atypeIdBloc, $appliIdBloc);
    if( $atypeIdBloc!=ALK_ATYPE_ID_GEDIT && !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "updateDataRankInBloc") )
      $oAppli->updateDataRankInBloc($bloc_id, $data_id, $iRank, $iDelta, $atypeIdBloc, $appliIdBloc);
  }
 
  /**
   * Retourne le contenu raccourci d'un bloc éditorial
   * pour un affichage de gestion ou d'aperçu
   * @apram bloc_id        Identifiant du bloc
   * @param bloc_typeassoc type d'association : 0=aucune, 1=appli, 2=cat, 3=data
   *                       si bit 2 on =4 : filtre publication : news uniquement
   *                       si bit 3 on =8 : filtre publication : infos publiés passés (archivés)
   *                       si bit 4 on =16: filtre publication : infos publiés présents
   *                       si bit 5 on =32: filtre publication : infos non publiés
   *                       bit 6 au bit 12 : anciennes versions de tris prédéfinis 
   *                       si bit 13 on : ordre : champ nouveauté
   *                       si bit 14 on : ordre : champ catégorie
   *                       si bit 15 on : ordre : champ data
   *                       si bit 16 on : ordre : champ date pub décroissant
   *                       si bit 17 on : ordre : champ date pub croissant
   *                       si bit 18 on : ordre : champ date info décroissant
   *                       si bit 19 on : ordre : champ date info croissant
   *                       si bit 20 on : ordre : champ date maj décroissant
   *                       si bit 21 on : ordre : champ date maj croissant
   *                       si bit 22 on : filtre calendaire : date début-fin pub 
   *                       si bit 23 on : filtre calendaire : date début-fin info 
   *                       si bit 24 on : filtre calendaire : date début-fin màj
   *                       si bit 25 on : filtre publication : infos synf 
   *                       si bit 26 on : filtre publication : en cours de validation
   *                       si bit 27 on : filtre publication : 30 derniers jours date pub
   *                       si bit 28 on : filtre publication : 30 derniers jours date info
   *                       si bit 29 on : filtre publication : 30 derniers jours date màj
   *                       si bit 30 on : ordre : champ rang des données décroissant
   *                       si bit 31 on : ordre : champ rang des données croissant
   * @param atypeIdBloc    type de l'application liée, =-1 si non utilisé
   * @param bloc_ordre     liste de nombres séparés par une virgule. 1 nombre correspond à une puissance de deux. 
   *                       cette liste correspond à l'ordre des champs d'après les champs de bit de bloc_typeassoc
   * @param bloc_datedeb   Contient la date de début pour un filtre éventuel en fonction de la valeur de bloc_typeassoc
   * @param bloc_datefin   Contient la date de début pour un filtre éventuel en fonction de la valeur de bloc_typeassoc 
   * @param bloc_limit     contient le nbre de résultats à afficher
   * @return array
   */
  public function getGEditBlocShortContents($bloc_id, $bloc_typeassoc, $atypeIdBloc=-1, $bloc_ordre="", $bloc_datedeb="", $bloc_datefin="", $bloc_limit="-1") 
  {  
    $user_id = AlkFactory::getSProperty("user_id", "-1");
    $tabContent = array();
    $i = 0;
    $dsDoc = $this->oQuery->getDsListRssByBlocId($bloc_id, $bloc_typeassoc, $user_id, true);
    while( $drDoc = $dsDoc->getRowIter() ) {
      $tabContent[$i]["data_id"]     = $drDoc->getValueName("DATA_ID");
      $tabContent[$i]["data_rank"]   = $drDoc->getValueName("DATA_RANG");
      $tabContent[$i]["data_title"]  = $drDoc->getValueName(AlkFactory::getDBCurrentLanguageField("DATA_TITRE"));
      $tabContent[$i]["data_desc"]   = $drDoc->getValueName(AlkFactory::getDBCurrentLanguageField("DATA_DESC"));
      $tabContent[$i]["data_type"]   = ALK_ATYPE_ID_GEDIT;
      $i++;
    }
    
    return $tabContent;
  }

  /**
   * Retourne le contenu complet d'un bloc éditorial dans la langue sélectionnée
   * pour un affichage de consultation
   * @param bloc_id   Identifiant du bloc
   * @param lg        clé de la langue sélectionnée
   * @param bloc_typeassoc type d'association : 0=aucune, 1=appli, 2=cat, 3=data
   *                       si bit 2 on =4 : filtre publication : news uniquement
   *                       si bit 3 on =8 : filtre publication : infos publiés passés (archivés)
   *                       si bit 4 on =16: filtre publication : infos publiés présents
   *                       si bit 5 on =32: filtre publication : infos non publiés
   *                       bit 6 au bit 12 : anciennes versions de tris prédéfinis 
   *                       si bit 13 on : ordre : champ nouveauté
   *                       si bit 14 on : ordre : champ catégorie
   *                       si bit 15 on : ordre : champ data
   *                       si bit 16 on : ordre : champ date pub décroissant
   *                       si bit 17 on : ordre : champ date pub croissant
   *                       si bit 18 on : ordre : champ date info décroissant
   *                       si bit 19 on : ordre : champ date info croissant
   *                       si bit 20 on : ordre : champ date maj décroissant
   *                       si bit 21 on : ordre : champ date maj croissant
   *                       si bit 22 on : filtre calendaire : date début-fin pub 
   *                       si bit 23 on : filtre calendaire : date début-fin info 
   *                       si bit 24 on : filtre calendaire : date début-fin màj
   *                       si bit 25 on : filtre publication : infos synf 
   *                       si bit 26 on : filtre publication : en cours de validation
   *                       si bit 27 on : filtre publication : 30 derniers jours date pub
   *                       si bit 28 on : filtre publication : 30 derniers jours date info
   *                       si bit 29 on : filtre publication : 30 derniers jours date màj
   *                       si bit 30 on : ordre : champ rang des données décroissant
   *                       si bit 31 on : ordre : champ rang des données croissant
   * @param atypeIdBloc    type de l'application liée, =-1 si non utilisé
   * @param bloc_ordre     liste de nombres séparés par une virgule. 1 nombre correspond à une puissance de deux. 
   *                       cette liste correspond à l'ordre des champs d'après les champs de bit de bloc_typeassoc
   * @param bloc_datedeb   Contient la date de début pour un filtre éventuel en fonction de la valeur de bloc_typeassoc
   * @param bloc_datefin   Contient la date de début pour un filtre éventuel en fonction de la valeur de bloc_typeassoc 
   * @param bloc_limit     contient le nbre de résultats à afficher
   * @return array
   */
  public function getGEditBlocContents($bloc_id, $lg, $bloc_typeassoc, $atypeIdBloc=-1, $bloc_ordre="", $bloc_datedeb="", $bloc_datefin="", $bloc_limit="-1") 
  {  
    $user_id = AlkFactory::getSProperty("user_id", "-1");
    $tabContent = array();
    $i = 0;
    $dsDoc = $this->oQuery->getDsListRssByBlocId($bloc_id, $bloc_typeassoc, $user_id, true);
    while( $drDoc = $dsDoc->getRowIter() ) {
      $tabContent[$i]["cont_id"]     = $this->cont_id;
      $tabContent[$i]["appli_id"]    = $this->appli_id;
      $tabContent[$i]["data_id"]     = $drDoc->getValueName("DATA_ID");
      $tabContent[$i]["data_rank"]   = $drDoc->getValueName("DATA_RANG");
      $tabContent[$i]["data_title"]  = $drDoc->getValueName("DATA_TITRE".$lg);
      $tabContent[$i]["data_type"]   = ALK_ATYPE_ID_GEDIT;
      $tabContent[$i]["data_rss_url"]= $drDoc->getValueName("DATA_URL");
      
      $rss_url          = $drDoc->getValueName("DATA_URL");
      $rss_nb_elt_aff   = $drDoc->getValueName("DATA_NB_ELT_AFF");
      $rss_aff_detail   = $drDoc->getValueName("DATA_AFF_DETAIL");
      $rss_aff_date     = $drDoc->getValueName("DATA_AFF_DATE");
      $page_id          = $drDoc->getValueName("PAGE_ID");
      $agent_id_crea    = $drDoc->getValueName("AGENT_ID");
      
      // lien Editer - Supprimer
      $strUrlSupp=""; 
      $strUrlEdit="";
      
      $strMsg = "<div id='bloc-rss-".$bloc_id."'></div>".
        "<script type='text/javascript' src='".ALK_RSS_READER_URL."?name=".$bloc_id."&amp;urlFlowx=".base64_encode($rss_url)."&amp;nbItems=".$rss_nb_elt_aff."'></script>".
        (file_exists(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CONF."lib/lib_rss.js") ? 
        "<script type='text/javascript' src='".ALK_ALKANET_ROOT_URL.ALK_ROOT_CONF."lib/lib_rss.js'></script>" :
        "<script type='text/javascript' src='".ALK_ALKANET_ROOT_URL.ALK_ROOT_LIB."lib_rss.js'></script>" ).
        "<script type='text/javascript'>".
        "  document.body.onLoad = ".
        "  loadRss('".$bloc_id."', 'bloc-rss-".$bloc_id."', '".addslashes($tabContent[$i]["data_title"])."', ".$rss_aff_detail.", ".$rss_aff_date.", '".addslashes($strUrlEdit)."','".addslashes($strUrlSupp)."');".
        "</script>";
      
      if ($agent_id_crea !="") {
        $param1 = "cont_id=".$this->cont_id."&appli_id=".$this->appli_id."&iTypeSheet=".ALK_TYPESHEET_POPUP."&iSheet=".ALK_SHEET_RSS.
       "&iSSheet=".ALK_SHEET_NONE;
        $strToken = AlkRequest::getEncodeParam($param1);
        $param="&close=ok&bloc_id=";
        // 1. Lien Editer
        $strUrlEdit = "&nbsp;[<a href=\"#\" onclick=\"javascript:OpenFormBlocPerso('".$strToken.$param.$bloc_id."'".")\">Editer</a>]&nbsp;";
        //2. Lien Supprimer
        $strUrlSupp = "&nbsp;[<a href=\"#\" onclick=\"javascript:DelBloc(16,".ALK_ATYPE_ID_GEDIT.",0,".$page_id.",".$this->appli_id.",".$this->cont_id.",".$bloc_id.")\">Supprimer</a>]";
        
        if ($rss_url==""){
          $strMsg = "<div id='noflux'><img class=\"alkimgicon alkimgicondocrss\" src=\"".ALK_ALKANET_ROOT_URL."media/images/transp.gif\"/><i>Flux rss non renseigné</i>".$strUrlEdit.$strUrlSupp."</div>";
        } else {
            $strMsg = "<div id='bloc-rss-".$bloc_id."'></div>".
          "<script type='text/javascript' src='".ALK_RSS_READER_URL."?name=".$bloc_id."&amp;urlFlowx=".base64_encode($rss_url)."&amp;nbItems=".$rss_nb_elt_aff."'></script>".
          (file_exists(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CONF."lib/lib_rss.js") ? 
          "<script type='text/javascript' src='".ALK_ALKANET_ROOT_URL.ALK_ROOT_CONF."lib/lib_rss.js'></script>" :
          "<script type='text/javascript' src='".ALK_ALKANET_ROOT_URL.ALK_ROOT_LIB."lib_rss.js'></script>" ).
          "<script type='text/javascript'>".
          "  document.body.onLoad = ".
          "  loadRss('".$bloc_id."', 'bloc-rss-".$bloc_id."', '".addslashes($tabContent[$i]["data_title"])."', ".$rss_aff_detail.", ".$rss_aff_date.", '".addslashes($strUrlEdit)."','".addslashes($strUrlSupp)."');".
          "</script>";
        }
      }
      $tabContent[$i]["data_title"] = ""; // pas de titre, on prend celui du flux
      $tabContent[$i]["data_desc"]  = $strMsg;
      
      $i++;
    }
    
    return $tabContent;
  }

  /**
   * Retourne le type de méthode pour mettre à jour les blocs
   *        Retourne un tableau en fonction du type sélectionné.
   * @param typeIdBloc   type de bloc
   * @param atypeIdBloc  type de l'application associée au type de bloc
   * @return array
   */
  public function getGEditBlocTypeUpdate($typeIdBloc, $atypeIdBloc) 
  {
    return array("type"            => ALK_GEDIT_TYPEUPDATE_SELECTEDIT, 
                 "iColumn"         => ( defined("ALK_B_GEDIT_BLOC_COLONNE") && ALK_B_GEDIT_BLOC_COLONNE==false ? 0 : 1),
                 "newsTitle"       => "",
                 "iFilterSpace"    => 1,
                 "iAssocAppli"     => 0,
                 "iAssocCat"       => 0,
                 "catTableName"    => "", 
                 "dataTableName"   => "",
                 "catTitle"        => "",
                 "dataTitle"       => "rss",
                 "xmltag"          => "rss",
                 "uploadDir"       => "",
                 "iFields"         => 262143-1-2-4-8-32, // = 2^18-1
                 "iFieldsPage"     => 262143-1-2-4-8-32, // = 2^18-1
                 "tokenBlocPerso"  => $this->getToken(ALK_TYPESHEET_POPUP, ALK_SHEET_RSS, ALK_SHEET_NONE));
  }
  
  /**
   * Retourne un tableau contenant l'information sélectionnée par cette méthode (appel régulier de getDataContents())
   * @param typeAff         type d'affichage, permet d'établir une conditionnelle dans cette méthode pour gérer plusieurs type d'affichage
   * @param cont_id         Identifiant de l'espace, pris en compte si appli_id vaut -1
   * @param appli_id        Identifiant de l'application, peut valoir =-1 pour toutes les applis de cont_id
   * @param cat_id          Identifiant de la catégorie, peut valoir =-1 pour toutes les infos de appli_id
   * @param data_id         Identifiant de la données, différent de -1 pour afficher la fiche, =-1 sinon par défaut
   * @param lg              langue utilisé, _FR par défaut
   * @param page_typeassoc  Type d'association des données à la page
   * @param page_ordre      Ordre des données dans la page
   * @param page_datedeb    Contient la date de début pour un filtre éventuel en fonction de la valeur de page_typeassoc
   * @param page_datefin    Contient la date de début pour un filtre éventuel en fonction de la valeur de page_typeassoc 
   * @return array
   */
  public function getGEditDataContents($typeAff, $cont_id, $atype_id, $appli_id, $cat_id, $data_id="-1", $lg=ALK_LG_BDD, $page_typeassoc=0, $page_ordre="", $page_datedeb="", $page_datefin="")
  {
    $user_id = AlkFactory::getSProperty("user_id", "-1");
    $tabRes = array();
    switch( $typeAff ) {
    case "Page": 
    case "Contact": 
    case "Credit":
      break;
       
    case "Map": 
      break;
    }
    return $tabRes;
  }

  /**
   * Retourne le contenu html de la page 
   * @param page_id     Identifiant de la page
   * @param typePubli   Entier =8  pour les docs publiés passés
   *                           =16 pour les docs publiés présents
   *                           =0 par défaut pour tous les docs publiés : présents, passés, futurs
   * @param lg          suffixe de la langue utitlisée, =_FR par défaut
   * @param bAddTitle   true par défaut pour afficher le titre de la page dans le contenu, false pour ne pas l'afficher
   * @param bVerifRight true par défaut, false pour ne pas vérifier les droits de l'utilisateur
   * @return string
   */
  public function getGEditHtml($page_id, $typePubli=0, $lg=ALK_LG_BDD, $bAddTitle=true, $bVerifRight=true)
  {
    $strXml = $this->getGEditXml($page_id, $typePubli, $lg, $bAddTitle, $bVerifRight);
    $oCleaner = AlkFactory::getHtmlCleaner();
    $oCleaner->cleanWithTidy($strXml);
    $strXsl = AlkXmlGEdit::getXsl();
    
    /*$hf = fopen(ALK_ALKANET_ROOT_PATH.ALK_ROOT_UPLOAD."alkanet.xml", "w");
    fwrite($hf, $strXml);
    fclose($hf);
    $hf = fopen(ALK_ALKANET_ROOT_PATH.ALK_ROOT_UPLOAD."alkanet.xsl", "w");
    fwrite($hf, $strXsl);
    fclose($hf);*/
    
    $oXml = AlkFactory::getXml();
    $oXml->setXml($strXml);
    $oXml->setXsl($strXsl);
    
    $oXml->setTransformation();
   
    return $oXml->getHtml();
  }
  
  /**
   * Retourne l'information sous forme XML
   * @param page_id   Identifiant de la page
   * @param typePubli Entier =8  pour les docs publiés passés
   *                         =16 pour les docs publiés présents
   *                         =0 par défaut pour tous les docs publiés : présents, passés, futurs
   * @param lg        suffixe de la langue utitlisée, =_FR par défaut
   * @param bVerifRight true par défaut, false pour ne pas vérifier les droits de l'utilisateur
   * @return string
   */
  public function getGEditArray($page_id, $typePubli=0, $lg=ALK_LG_BDD, $bVerifRight=true)
  {
    $oXml = $this->getGEditData($page_id, $typePubli, $lg, true, $bVerifRight);
    
    return $oXml->getArray();
  }
 
 
  /**
   * Retourne l'information sous forme XML
   * @param page_id     Identifiant de la page
   * @param typePubli   Entier =8  pour les docs publiés passés
   *                           =16 pour les docs publiés présents
   *                           =0 par défaut pour tous les docs publiés : présents, passés, futurs
   * @param lg          suffixe de la langue utitlisée, =_FR par défaut
   * @param bAddTitle   true par défaut pour afficher le titre de la page dans le contenu, false pour ne pas l'afficher
   * @param bVerifRight true par défaut, false pour ne pas vérifier les droits de l'utilisateur
   * @return string
   */
  private function getGEditXml($page_id, $typePubli=0, $lg=ALK_LG_BDD, $bAddTitle=true, $bVerifRight=true)
  {
    $oXml = $this->getGEditData($page_id, $typePubli, $lg, $bAddTitle, $bVerifRight);
    
    return $oXml->getXml();
  }
  
  /**
   * Retourne l'objet de type AlkXmlGedit correspondant à la page
   * @param page_id     Identifiant de la page
   * @param typePubli   Entier =8  pour les pages publiées passées
   *                           =16 pour les pages publiées présentes
   *                           =32 pour les pages à valider
   *                           =0 par défaut pour tous les docs publiés : présents, passés, futurs
   * @param lg          suffixe de la langue utitlisée, =_FR par défaut
   * @param bAddTitle   true par défaut pour afficher le titre de la page dans le contenu, false pour ne pas l'afficher
   * @param bVerifRight true par défaut, false pour ne pas vérifier les droits de l'utilisateur
   * @return AlkXmlGedit
   */
  private function getGEditData($page_id, $typePubli=0, $lg=ALK_LG_BDD, $bAddTitle=true, $bVerifRight=true)
  {
    $user_id = AlkFactory::getSProperty("user_id", "-1");
    $page_title = mb_strtoupper($this->oQuery->getPageTitle($page_id, $lg));

    $oXml = AlkFactory::getXmlGEdit($page_id, ( $bAddTitle ? $page_title : "" ), $lg);  
    
    // lecture des informations liées à une application
    $dsPage = $this->oQuery->getPageById($page_id, ( $bVerifRight ? $user_id : -1 ));
    if( $drPage = $dsPage->getRowIter() ) {
      $atype_id         = $drPage->getValueName("ATYPE_ID");  
      $page_appliid     = $drPage->getValueName("PAGE_APPLIID");  
      $page_catid       = $drPage->getValueName("PAGE_CATID");
      $ptype_servicename = $drPage->getValueName("PTYPE_SERVICENAME");
      $page_typeassoc   = $drPage->getValueName("PAGE_TYPEASSOC");
      $page_ordre       = $drPage->getValueName("PAGE_ORDRE");
      $page_datedeb     = $drPage->getValueName("PAGE_DATEDEB");
      $page_datefin     = $drPage->getValueName("PAGE_DATEFIN");
      list($serviceName, $iTypeAff) = explode("|", $ptype_servicename);
      
      $oAppliPage = AlkFactory::getAppli($atype_id, $page_appliid);
      if( !is_null($oAppliPage) && is_object($oAppliPage) && method_exists($oAppliPage, "getGedit".$serviceName."Contents") ) {
        $cont_id = ( $page_appliid == "-1" ? $this->cont_id : "-1" );
        $page_catid = ( $page_catid=="-1" ? AlkRequest::getToken("cat_id", -1) : $page_catid);
        $data_id = AlkRequest::getToken("data_id", -1);
        $tabContents = call_user_func(array($oAppliPage, "getGedit".$serviceName."Contents"), $iTypeAff, $cont_id, $atype_id, $page_appliid, $page_catid, $data_id, $lg, $page_typeassoc, $page_ordre, $page_datedeb, $page_datefin);
        $tabTypeUpdate = $oAppliPage->getGEditBlocTypeUpdate("-1", $atype_id);
        
        $oXml->addBloc("-1", $tabTypeUpdate, $tabContents, 0, 0, $typePubli);
      }
    } 
    
    // lecture des blocs
    $dsBloc = $this->oQuery->getBlocByPageId($page_id);
    $nbBloc = $dsBloc->getCountTotDr(); 
    $iCpt = 0;
    while( $drBloc = $dsBloc->getRowIter() ) {
      $bloc_id        = $drBloc->getValueName("BLOC_ID");
      $strBlocNom     = $drBloc->getValueName("BLOC_NOM");
      $typeIdBloc     = $drBloc->getValueName("TB_ID");
      $atypeIdBloc    = $drBloc->getValueName("ATYPE_ID");
      $iColonne       = ( $drBloc->getValueName("BLOC_COLONNE")==1 ? 1 : 0);
      $bloc_typeassoc = $drBloc->getValueName("BLOC_TYPEASSOC");
      $bloc_ordre     = $drBloc->getValueName("BLOC_ORDRE");
      $bloc_datedeb   = $drBloc->getValueName("BLOC_DATE_DEB");
      $bloc_datefin   = $drBloc->getValueName("BLOC_DATE_FIN");
      $bloc_limit     = $drBloc->getValueName("BLOC_LIMIT");
      $blocUne        = $drBloc->getValueName("BLOC_UNE");
      $bloc_nom       = $drBloc->getValueName("BLOC_NOM".$lg);
      
      $bloc_typeassoc_publi = $bloc_typeassoc + ( $blocUne=="1" ? 4 : 0 );

      if( isset($GLOBALS["ALK_GEDIT_TYPEASSOC"]) ) {  
         $bloc_typeassoc_publi = $bloc_typeassoc_publi | $GLOBALS["ALK_GEDIT_TYPEASSOC"];
      }
      
      $oAppliBloc    = AlkFactory::getAppli($atypeIdBloc, $this->appli_id);
      $tabContents   = $oAppliBloc->getGEditBlocContents($bloc_id, $lg, $bloc_typeassoc_publi, $atypeIdBloc, $bloc_ordre, $bloc_datedeb, $bloc_datefin, $bloc_limit);
      $tabTypeUpdate = $oAppliBloc->getGEditBlocTypeUpdate($typeIdBloc, $atypeIdBloc);

      $oXml->addBloc($bloc_id, $tabTypeUpdate, $tabContents, $iColonne, $blocUne, $bloc_typeassoc, $bloc_nom);
    }    
    
    return $oXml;   
  }

  /**
   * Retourne dans un tableau contenant une sélection d'informations de l'application
   * 
   * @param object_id   identifiant de la donnée : 
   *                      cont_id si typeassoc3&=0, 
   *                      appli_id si typeassoc&1=1, 
   *                      cat_id si typeassoc&2=2, 
   *                      data_id si typeassoc&3=3
   * @param typeAssoc   type d'association : 0=cont, 1=appli, 2=cat, 3=data
   *                    si bit 2 on : présente les news uniquement
   *                    si bit 3 on : présente les docs publiés passés
   *                    si bit 4 on : présente les docs publiés présents
   *                    si bit 5 on : présente les docs à valider
   *                    si bit 6 on : ordre les éléments d'abord par intitulé de catégorie (non utilisé)
   *                    si bit 7 on : présente les docs syndiqués
   *                    si bit 8 on : non utilisé
   *                    si bit 9 on : présente les  infos en cours de validation 
   *                    si <=3, présente tous les docs publiés : passés, présent, à venir
   * @param user_id     identifiant de l'utilisateur pour vérifier les droits, =-1 pour non vérification
   * @param lg          langue utilisé, _FR par défaut
   * @param atypeIdBloc type de l'application liée, =-1 si non utilisé
   * @return array
   */
  public function getDataContents($object_id, $typeAssoc, $user_id, $lg=ALK_LG_BDD, $atypeIdBloc=-1)
  {
    $tabContent = array();
    $i = 0;
    $dsDoc = $this->oQuery->getDsListPageById($object_id, $typeAssoc, $user_id, $lg);
    while( $drDoc = $dsDoc->getRowIter() ) {
      $tabContent[$i]["cont_id"]     = $drDoc->getValueName("CONT_ID");
      $tabContent[$i]["appli_id"]    = $drDoc->getValueName("APPLI_ID");
      $tabContent[$i]["cat_name"]    = "";
      $tabContent[$i]["data_id"]     = $drDoc->getValueName("PAGE_ID");
      $tabContent[$i]["data_rank"]   = $drDoc->getValueName("PAGE_RANG");
      $tabContent[$i]["data_title"]  = $drDoc->getValueName("PAGE_TITRE".$lg);
      $tabContent[$i]["data_title_court"]  = $drDoc->getValueName("PAGE_TITRE_COURT".$lg);
      $tabContent[$i]["data_date"]   = $drDoc->getValueName("PAGE_DATE_PDEB");
      $tabContent[$i]["data_datemaj"]   = $drDoc->getValueName("PAGE_DATE_MAJ");
      $tabContent[$i]["data_descl"]  = $this->getGEditHtml($tabContent[$i]["data_id"], 0, $lg, false);
      $tabContent[$i]["data_img"]    = "";
      $tabContent[$i]["data_url"]    = $drDoc->getValueName("PAGE_URL_REDIRECTION");
      $tabContent[$i]["data_type"]   = ALK_ATYPE_ID_GEDIT;
      
      $i++;
    }
    return $tabContent;
  }
  
  /** interface workflow */
  
  /**
   * Méthode appelée par le workflow à partir du tableau de bord pour valider l'information à publier
   * @param data_id  identifiant de la page à valider et à publier
   */
  public function valideData($data_id)
  {
    $this->oQueryAction->publishRestoreDraftPage(AlkFactory::getUploadPath(ALK_ATYPE_ID_GEDIT).$this->cont_id."/", $data_id);
  }

  /**
   * Méthode appelée par le workflow à partir du tableau de bord pour invalider l'information à publier
   * @param data_id  identifiant de la page à valider et à publier
   */
  public function invalideData($data_id)
  {
    $this->oQueryAction->invalideData($data_id);
  }
   
  /** méthodes de la classe */
    
  /**
   * Retourne l'objet Panel correspondant au iSheet et iSSheet sélectionné
   * 
   * @param oAppliFrom    AlkAppli   Application appelant cette fonction (principe call services par dérivation)
   * 
   * @return AlkHtmlPanel
   */
  public function getPanel(AlkAppli $oAppliFrom=null)
  {  
    $iMode = AlkRequest::getToken("iMode", ALK_FORM_MODE_UPDATE);
    $oPanel = null;

    switch( $this->iTypeSheet ) {
    case ALK_TYPESHEET_CONSULT: // partie consultation
      switch( $this->iSheet ) {        
      case ALK_SHEET_FORM: // consultation editorial externet 
      default:
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."gedit/classes/alkhtmlceditpage.class.php");
        $oPanel = new AlkHtmlCEditPage($this);
        break;
      }      
      break;
    
    case ALK_TYPESHEET_ADMIN: // partie administration
      switch ( $this->iSheet ){
      case ALK_SHEET_PONDERATION : // onglet nuage mots clés
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."gedit/classes/alkhtmltagcloudform.class.php");
        $oPanel = new AlkHtmlTagCloudForm($this);
        break;
      case ALK_SHEET_FORM : // onglet formulaire
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."gedit/classes/alkhtmlformgeditform.class.php");
        $oPanel = new AlkHtmlFormGEditForm($this);
        
        // pour les cas 4, 5 et 6 : appel ajax sur le doSql avec réaffichage de la liste des blocs en retour
        switch( $iMode ) {
        case "4": // ajout de bloc
        case "5": // suppression de bloc
        case "6": // modif rang bloc
        case "8": // modif rang d'une donnée d'un bloc
        case "9": // suppression donnée d'un bloc
        case "10":// mise à jour d'un bloc
        case "14":// test l'unicite de l'intitule url (page_alias_[fr|uk])
        case "15":// mise à jour des templates
        case "16":// mise à jour de l'association avec la liste d'abonnement
          $oPanel = $oPanel->doSql();
          break;
        }
        break;
      case ALK_SHEET_FILEMANAGER : //onglet file manager
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."gedit/classes/alkhtmlfilemanager.class.php");
        $oPanel = new AlkHtmlFileManager($this);
      break;
      }
    break; 
    case ALK_TYPESHEET_POPUP: // partie popup
      switch ( $this->iSheet ){
      case ALK_SHEET_FORM : // fenêtre sélection d'une information
        if(file_exists(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CONF."libconf/alkhtmlpopupdataselect.class.php") && 
           is_file(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CONF."classes/alkhtmlpopupdataselect.class.php")){
          require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CONF."classes/alkhtmlpopupdataselect.class.php");
          $oPanel = new AlkHtmlPopupDataSelectGed($this);
        } else {
          require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."gedit/classes/alkhtmlpopupdataselect.class.php");
          $oPanel = new AlkHtmlPopupDataSelect($this);
        }
        // pour les cas 4, 5 et 6 : appel ajax sur le doSql avec réaffichage de la liste des blocs en retour
        switch( $iMode ) {
          case "4": // modif epace
          case "5": // modif appli
          case "6": // modif catégorie
          case "7": // association appli - bloc
          case "8": // association cat - bloc
          case "9": // association data - bloc
          case "10":// modif collection
          case "11": // modif catégorie
          case "12": // bouton rechercher
          case "13": // ajout de la sélection
          case "14": // associer le filtre au bloc
          case "15": // affichage des classifs liés à l'espace
          case "16": // affichage des sous-thèmes 
          case "17": // affichage des collections en fonction des sous-catégories
          $oPanel = $oPanel->doSql();
          break;
        }
        break;
      case ALK_SHEET_APERCU: // apercu html de la page et la langue sélectionnées
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."gedit/classes/alkhtmlpopupapercu.class.php");
        $oPanel = new AlkHtmlPopupApercu($this);
        break;
      case ALK_SHEET_RSS: // popup de configuration d'un flux RSS
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."gedit/classes/alkhtmlpopuprss.class.php");
        $oPanel = new AlkHtmlPopupRss($this);
        
        // 
        switch( $iMode ) {
        case "4": // modif epace
        case "5": // modif appli
          $oPanel = $oPanel->getPanelAjax($iMode);
          break;
        }
        break;
      case ALK_SHEET_INTERNET:
        // cas de l'envoyer à un ami
        $this->writeEnvoyerAmi();
        break; 
        
      case ALK_SHEET_LIST:
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."gedit/classes/alkhtmllistpage.class.php");
        $oPanel = new AlkPopupListPage($this);
      break;
      
      case ALK_SHEET_FICHE:
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."gedit/classes/alkhtmllistdef.class.php");
        $oPanel = new AlkPopupListDef($this);
        switch( $iMode ) {
          case "4":
          case "5": 
          $oPanel = $oPanel->doSql();
        break;   
         }
       break;
       
       case ALK_SHEET_NONE: // paramétrage du bloc Editeur
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."gedit/classes/alkhtmlpopupediteur.class.php");
        $oPanel = new AlkhtmlPopupEditeur($this);
         break;
      }
            
     case ALK_TYPESHEET_PROPRIETE:
       switch($this->iSheet){
         case ALK_SHEET_PARAMETRE:
           require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlformparametreform.class.php");
           $oAppliEspace = AlkFactory::getAppli(ALK_ATYPE_ID_ESPACE);
           $oPanel = new AlkHtmlFormParametreForm($this, $oAppliEspace, $this->atype_id);
         break;
       }
       
       break;

    }
    return $oPanel;
  }

  /**
   * Permet d'initialiser le tableau this->tabSheets
   */
  public function setTabSheets() 
  {
    if( $this->iTypeSheet == ALK_TYPESHEET_POPUP ) return;

    $strUrl = ALK_ALKANET;
    $user_id = AlkFactory::getSProperty("user_id", -1);
    
    if( defined("ALK_B_GEDIT_CONSULT") && ALK_B_GEDIT_CONSULT==TRUE && !isset($_SERVER["REDIRECT_URL"]) ) {
      $this->addSheet(ALK_TYPESHEET_CONSULT, ALK_SHEET_FORM, ALK_SHEET_NONE, 
                      $this->getAppliProperty("APPLI_INTITULE"), $strUrl, "", "",
                      ALK_APPLI_RIGHT_READ, ALK_PRIV_SPACE_USER, true);
    } else {
      if (isset($_SERVER["REDIRECT_URL"]) && strpos($_SERVER["PHP_SELF"], "alkanet_site.php")!==false){
        return;
      }
    }
    
    $this->addSheet(ALK_TYPESHEET_ADMIN, ALK_SHEET_FORM, ALK_SHEET_NONE, 
                    "Gestion des pages", $strUrl, "", "",
                    ALK_APPLI_RIGHT_ADMIN, ALK_PRIV_SPACE_USER, true);
                    
    if( defined("ALK_B_GEDIT_TAGCLOUD") && ALK_B_GEDIT_TAGCLOUD==true ) { 
      $this->addSheet(ALK_TYPESHEET_ADMIN, ALK_SHEET_PONDERATION, ALK_SHEET_NONE, 
                     "Nuage de mots clés", $strUrl, "", "",
                     ALK_APPLI_RIGHT_ADMIN, ALK_PRIV_SPACE_USER, true);
    }
    
    // onglet property (restriction temporaire au user alkante)
    if ($user_id == ALK_USER_ID_ADMINALK)
      $this->addSheet(ALK_TYPESHEET_PROPRIETE, ALK_SHEET_PARAMETRE, ALK_SHEET_LIST, "Paramètres", $strUrl, "",
      								"Gérer les constantes de l'application", 
                      ALK_APPLI_RIGHT_PUBLI, ALK_PRIV_SPACE_ANIM+ALK_PRIV_SPACE_ADMIN);
     
  	/* Onglet du gestionnaire de fichiers */
		/* cet onglet n'est visible que si le partage des médias est désactivé */
		if ( !(defined("ALK_B_MEDIA_SHARED") && ALK_B_MEDIA_SHARED==true) ){                      
	    if ( defined("ALK_B_EDITEUR_PDW_FILEBROWSER") && ALK_B_EDITEUR_PDW_FILEBROWSER==true ){ 
	      $strTokenFileManager = ALK_ALKANET_ROOT_URL.ALK_ROOT_MODULE."editeur/editor/pdw_file_browser/index.php?editor=ckeditor&ServerPath=".ALK_VIRTUAL_DIR.AlkFactory::getUploadPath(ALK_ATYPE_ID_GEDIT, true).$this->cont_id."/".
	      	"&UrlRoot=".ALK_ROOT_URL;
	    } else {
	      $strTokenFileManager = ALK_ALKANET_ROOT_URL.ALK_ROOT_MODULE."editeur/editor/filemanager2/index.html?&ServerPath=".ALK_VIRTUAL_DIR.AlkFactory::getUploadPath(ALK_ATYPE_ID_GEDIT, true).$this->cont_id."/".
	        "&UrlRoot=".ALK_ROOT_URL.
	        "&CKEditor=TextHtml" .
	        "&CKEditorFuncNum=1" .
	        "&langCode=fr";
	    }
	    $strUrlFileManager = "javascript:OpenFileManager('".$strTokenFileManager."')";
	    $this->addSheet(ALK_TYPESHEET_ADMIN, ALK_SHEET_FILEMANAGER, ALK_SHEET_NONE, 
	                   "Gestionnaire de Fichiers", $strUrlFileManager, "", "",
	                     ALK_APPLI_RIGHT_ADMIN, ALK_PRIV_SPACE_USER, true);
		}
  }

  /**
   * Ecrit sur la sortie standard, le flux rss correspondant au token passé
   * Info du token : cont_id, appli_id
   *   - page_id       >0 pour une page sélectionnée
   *                   =-1 pour la liste des pages de l'application sélectionnée
   *                   =-2 pour la liste des pages de l'espace sélectionné
   *                   =-3 pour la liste des pages de tous les espaces
   *   - pub           =1 (par défaut) pour obtenir les pages publiées, =0 pour obtenir les pages à valider
   *   - user_id       identifiant de l'utilisateur, nécessaire si pub=0 pour vérifier les droits, =-1 par défaut pour non vérif des droits
   *   - lg            =_FR par défaut
   */
  public function writeRss()
  {
    $cont_id  = AlkRequest::getToken("cont_id", "-1");
    $appli_id = AlkRequest::getToken("appli_id", "-1");
    $page_id  = AlkRequest::getToken("page_id", "-1");
    $pub      = AlkRequest::getToken("pub", "1"); 
    $user_id  = AlkRequest::getToken("user_id", "-1");
    $lg       = AlkRequest::getToken("lg", ALK_LG_BDD);

    $page_titre = $this->oQuery->getPageTitle($page_id);
    $bOk = true;
    
    // page publiée
    $iFiltre = 16;
    if( $user_id != "-1" && $pub == "0" ) {
      // vérifie que l'utilisateur a le droit de publication sur l'application
      $iRight = $this->getUserAppliRight();
      $bOk = ( $iRight == ALK_APPLI_RIGHT_PUBLI );
      // pages à valider
      $iFiltre = 32;
      if( $page_id > 0 ) $page_id = -1; 
    } else {
      // filtre sur page syndiquée uniquement
      $iFiltre += 128; 
    }
    
    $oFSyndXml = AlkFactory::getFSyndXmlWriter(ALK_RSS2);
    $oFSyndXml->setTitle(( $page_id < 0
                            ? "Toutes les pages éditoriales à valider" 
                            : "Page éditoriale : ".$page_titre ));
    $oFSyndXml->setLink(ALK_ROOT_URL.$_SERVER["REQUEST_URI"]);
    $oFSyndXml->setDescription("Flux RSS 2.0 provenant de ".ALK_APP_TITLE);
    
    if( $bOk ) {
      $typeAssoc = TASSOC_BYDATA; 
      $object_id=$page_id;
      switch( $page_id ) {
      case -3: $typeAssoc = TASSOC_EMPTY; $object_id=-1;        break;
      case -2: $typeAssoc = TASSOC_EMPTY; $object_id=$cont_id;  break;
      case -1: $typeAssoc = TASSOC_BYAPPLI; $object_id=$appli_id; break;
      }
      $typeAssoc += $iFiltre;
      
      // recherche les infos à syndiquer
      $tabData = $this->getDataContents($object_id, $typeAssoc, $user_id, $lg);
      foreach($tabData as $tabContent ) {
        $newItem = $oFSyndXml->createNewItem();
        
        $newItem->setTitle($tabContent["data_title"]);
        if( $page_id > 0 ) {
          $newItem->setDescription($tabContent["data_descl"]);
        } else {
          $newItem->setDescription(""); 
        }
        if( $tabContent["data_url"] != "" ) {
          $newItem->setLink($tabContent["data_url"]);
        } else {
          $newItem->setLink("#");
        }
        if( $tabContent["data_date"] != "" ) {
          $newItem->setDate($tabContent["data_date"], "%d/%m/%Y");
        } else {
          $newItem->setDate($tabContent["data_datemaj"], "%d/%m/%Y");
        }
        
        $oFSyndXml->addItem($newItem);
      }
    }

    $oFSyndXml->genarate(); 
  }
  
  /**
   * Envoi le mail au destinataire
   * Ecrit sur la sortie standard le message de confirmation
   * 
   */
  public function writeEnvoyerAmi()
  {
    $urlAEnvoyer = AlkRequest::_GET("url");
    $strNomExp   = AlkRequest::_GET("nom_exp");
    $strMailExp  = AlkRequest::_GET("mail_exp");
    $strMailDest = AlkRequest::_GET("mail_dest");
    $strSujet    = AlkRequest::_GET("sujet");
    $strMessage  = AlkRequest::_GET("message");
    
    $oAlkMail = AlkFactory::getMail();
    $oAlkMail->SetFrom($strNomExp, $strMailExp);
    $oAlkMail->AddTo($strMailDest, $strMailDest);
    
    $tabAssoc = array("page_url"      => $urlAEnvoyer,
                      "nom_exp"       => $strNomExp,
                      "mail_exp"      => $strMailExp,
                      "mail_dest"     => $strMailDest,
                      "mail_sujet"    => $strSujet,
                      "mail_message"  => $strMessage
                      );
    
    if ( AlkMailing::SendMailType($oAlkMail, "GEDIT_ENVOYER_AMI", $tabAssoc) ){ // le message a été envoyé
      echo "<p>".$strNomExp.", un mail vient d'&ecirc;tre envoy&eacute; &agrave; <strong>".$strMailDest."</strong>.<p>";
    } else { // problème au moment de l'envoi du message
      echo "<p>Un probl&egrave;me est survenu lors de l'envoi du message.<br/> Veuillez essayer &agrave; nouveau.<p>";
      echo "<p><a href=\"javascript:annulerEnvoyerAmi();\">Retour</a></p>";
    }
    
    echo "<p><a href=\"javascript:cache('envoyer_ami_message');\">fermer</a></p>";
  }

  /**
   * Cette méthode traduit l'url réécrite passée en paramètre en un tableau
   * Retourne un tableau contenant les clés suivantes : cont_id, appli_id, page_id, lg, iMode, cat_id, data_id 
   * @param strRewriteUrl  url réécrite sans le nom de domaine et sans slash en premier caractère
   * @param mode           mode d'accès à la page (= brouillon pour accèder àa la page brouillon)
   * @param agent_id       =-1 par défaut, >0 pour vérifier les droits à la page
   * @return array
   */
  public function rewriteUrl($strRewriteUrl, $mode="", $agent_id=-1)
  {
    // supprime le ou les premiers slashes au début de l'url
    while( mb_substr($strRewriteUrl, 0, 1) == "/" ) {
      $strRewriteUrl = mb_substr($strRewriteUrl, 1);
    }

    // découpage de l'url
    $tabPages = explode("/", mb_strtolower($strRewriteUrl));
    
    /** strRewriteUrl : [espace_alias]0..1 [page_alias]1..n [data_ident]0..1 [data_name|cat_name]0..1 */
   
    $oQueryEspace = AlkFactory::getQuery(ALK_ATYPE_ID_ESPACE);
    $tabCont = $oQueryEspace->getContIdByAlias((count($tabPages)>0 ? $tabPages[0] : "")); 
    $cont_id = $tabCont["cont_id"]; 
    
    // recherche des pages successibes
    $tabInfoId = array("cat_id" => -1, "data_id" => -1, "iMode" => -1);
		$strPattern = "/^([0-9]*)_([0-9]*)$|^([0-9]*)_([0-9]*)_([0-9]*)$/";
    $index = -1;
    $bFound = false;
    foreach($tabPages as $strDir) {
      $tabRes = array();
      $index++;
      if( preg_match_all($strPattern, $strDir, $tabRes) ) {
        $bFound = true;
        if ($tabRes[3][0]==""){
          $tabInfoId["cat_id"] = ($tabRes[1][0]!="" ? $tabRes[1][0] : -1);
          $tabInfoId["data_id"] = ($tabRes[2][0]!="" ? $tabRes[2][0] : -1);
        } else {
          $tabInfoId["cat_id"] = ($tabRes[3][0]!="" ? $tabRes[3][0] : -1);
          $tabInfoId["data_id"] = ($tabRes[4][0]!="" ? $tabRes[4][0] : -1);
          $tabInfoId["iMode"] = ($tabRes[5][0]!="" ? $tabRes[5][0] : -1);;
        }
        break;
      }   
    }

    // calcul de l'indice de fin pour la reconstruction de l'url 
    $iIndiceFin = ( $bFound 
                     ? ( $index>0 ? $index-1 : 0 ) 
                     : ( count($tabPages)>0 ? count($tabPages)-1 : 0 ));
    $strPageUrl = "";
    if ($iIndiceFin>=0) {
      for($i=0; $i<=$iIndiceFin; $i++){
        $strPageUrl .= "/".$tabPages[$i];        
      }
    }
    $bUseCache = ( $mode!="brouillon" ? true : false );
    $tabInfoPage = $this->oQuery->getPrimaryTokenData($strPageUrl, $cont_id, ($mode=="brouillon" ? true: false), $agent_id, $bUseCache);
    
    foreach($tabInfoPage as $strChamp =>$strValue){
      $tabInfoId[$strChamp]  = $strValue;
    }
    
    return $tabInfoId;
  }
  
 /**
   * vérifie les liens gérés par l'application et retourne le résultat sous forme de tableau
   * format du tableau retourné : []["data_id", "token", "url", "desc"]
   * @return array
   * 
   */
  public function verifLinks(){
    $dsLinks = $this->oQuery->getLinksPages();
    $tabLink = array();
    while ($drLink = $dsLinks->getRowIter()){
      $tabTemp = array();
      $page_id = $drLink->getValueName("PAGE_ID");
      $tb_id      = $drLink->getValueName("TB_ID");
      $bloc_id    = $drLink->getValueName("BLOC_ID");
      $cont_id    = $drLink->getValueName("CONT_ID");
      $appli_id   = $drLink->getValueName("APPLI_ID");

      $tabUrl = array();
      switch($tb_id){
        case "1": // bloc éditeur
        foreach($this->tabLangue as $key => $tabLg) {
          $champ = "CONTENU_".$tabLg["rep"];
          $contenu = $drLink->getValueName($champ);
          $desc = "Contenu du bloc éditorial";
          if ($contenu != ""){
            $tab = VerifTextLink($contenu);
            if(!empty($tab))
              $tabUrl = array_merge($tab,$tabUrl);
            // vérification des src
            $tab = verifTextImage($contenu);
            if (!empty($tab))
              $tabUrl = array_merge($tab,$tabUrl);
          }
        } 
         break;
        case "16": // bloc rss
         $desc = "lien rss";
         $tabUrl = (verifLink($drLink->getValueName("CONTENU_FR"))==false ? array($drLink->getValueName("CONTENU_FR")): array());
      }
      
      if (!empty($tabUrl)){
        $tabTemp["url"]= $tabUrl;
        $tabTemp["desc"] = $desc;
        $tabTemp["data_id"]=$bloc_id; // bloc id dans lequel se trouve le lien cassé

      
        $strToken = AlkRequest::getEncodeParam("cont_id=".$cont_id."&appli_id=".$appli_id."&iTypeSheet=".
        ALK_TYPESHEET_ADMIN."&iSheet=".ALK_SHEET_FORM."&iSSheet=".ALK_SHEET_NONE)."&page_id=".$page_id;
      
        $tabTemp["token"] = $strToken;
        
        array_push($tabLink, $tabTemp);
      }      
    }
    
    return $tabLink;
  }
  
  /**
   * Supprime les informations annexes associées à cette application
   * Si cat_id =-1 et data_id=-1, supprime toutes les informations annexes associées à cette application
   * Si cat_id!=-1 et data_id=-1, supprime toutes les informations annexes associées à cette classification cat_id
   * Si data_id!=-1, supprime toutes les informations annexes associées à cet donnée data_id 
   * 
   * @param cont_id  identifiant de l'espace contenant cette application
   * @param cat_id   identifiant de la classification, =-1 par défaut
   * @param data_id  identifiant de l'information, =-1 par défaut
   */
  public function delDataAssoc($cont_id, $cat_id=-1, $data_id=-1)
  {
    $this->oQueryAction->delDataAssoc($this->appli_id, $data_id);
    parent::delDataAssoc($cont_id, $cat_id, $data_id);
  }
  
  /**
   * Retourne le libellé de la page d'aide associée à l'application
   * @return string
   */
  public function getHelp(){
    return _t("Gestion éditoriale");
  }
  
  /**
   * initialise la gestion des liens et construit les associations des liens existants
   * @param $bSimul booléen à true pour simuler, false sinon par défaut
   */
  public function initLinks($bSimul=false)
  {
    $oAppliLink = AlkFactory::getAppli(ALK_ATYPE_ID_LINK);
    if ( $oAppliLink ) {
      $oQueryEspace = AlkFactory::getQuery(ALK_ATYPE_ID_ESPACE);
      if ( $oQueryEspace ) {
        
        $tabReplacement = array(ALK_ALKANET_ROOT_URL => "");
        
        // traite les applis GEDIT de chaque espace séparément
        $oDsEspace = $oQueryEspace->GetDs_listeEspace();
        while ( $oDrEspace = $oDsEspace->getRowIter() ) {
          
          // récupère les propriétés de l'espace et les applis GEDIT qu'il contient
          $cont_id = $oDrEspace->getValueName("CONT_ID");
          $oDsEspaceDetail = $oQueryEspace->getDs_ficheEspaceById($cont_id, "", false, -1);
          $oDrEspace = $oDsEspaceDetail->getRowIter();
          $oDsAppliGedit = $oQueryEspace->getDs_listeAppliByEspace($cont_id, 0, -1, false, true, ALK_ATYPE_ID_GEDIT);
          foreach ( $this->tabLangue as $key => $tabLg ) {
            $tabReplacement[$oDrEspace->getValueName("CONT_ROOT_FRONT_URL".$tabLg["bdd"])] = "";
          }
          
          // contruit un tableau contenant les identifiants des pages publiées de l'espace indexés par alias de page
          $tabPagePubliee = array();
          while ( $oDrAppliGedit = $oDsAppliGedit->getRowIter() ) {
            $appli_id = $oDrAppliGedit->getValueName("APPLI_ID");
            $oDsPagePubliee = $this->oQuery->getArboAutorise(ALK_USER_ID_ADMINALK, $appli_id, true, "-1", true, "-1", "-1", -1, true);
            while ( $oDrPagePublie = $oDsPagePubliee->getRowIter() ) {
              foreach ( $this->tabLangue as $key => $tabLg ) {
                $tabPagePubliee[$oDrPagePublie->getValueName("PAGE_URL".$tabLg["bdd"])] = $oDrPagePublie->getValueName("PAGE_ID");
              } 
            }
          }
          
          // génère les liens
          $oDsAppliGedit->moveFirst();
          while ( $oDrAppliGedit = $oDsAppliGedit->getRowIter() ) {
            $appli_id = $oDrAppliGedit->getValueName("APPLI_ID");
            $oDsPageBrouillon = $this->oQuery->getArboAutorise(ALK_USER_ID_ADMINALK, $appli_id, false, "-1", true, "-1", "-1", -1, true);
            $oDsPagePubliee = $this->oQuery->getArboAutorise(ALK_USER_ID_ADMINALK, $appli_id, true, "-1", true, "-1", "-1", -1, true);
            $this->makeLinks($oDsPageBrouillon, $tabReplacement, $tabPagePubliee, $bSimul);
            $this->makeLinks($oDsPagePubliee, $tabReplacement, $tabPagePubliee, $bSimul);
          }
        }
      }
    }
  }
  
  /**
   * initialise la gestion des liens et construit les associations des liens existants
   * @param $oDsPage          dataset de pages
   * @param $tabReplacement   tableau des remplacements à effectuer sur les liens
   * @param $tabPagePubliee   tableau des identifiants des pages publiées indexées par alias
   * @param $bSimul           booléen à true pour simuler, false sinon par défaut
   */
  protected function makeLinks($oDsPage, $tabReplacement=array(), $tabPagePubliee=array(), $bSimul=false)
  {
    $oDsPage->moveFirst();
    while ( $oDrPage = $oDsPage->getRowIter() ) {
      
      $page_id = $oDrPage->getValueName("PAGE_ID");
      $page_url = $oDrPage->getValueName("PAGE_URL".ALK_LG_BDD);
      
      // url de redirection
      $oAppliLink = AlkFactory::getAppli(ALK_ATYPE_ID_LINK);
      if ( $oAppliLink ) {
        $page_url_redirection = $oDrPage->getValueName("PAGE_URL_REDIRECTION");
        foreach ( $tabReplacement as $search => $replace ) {
          $page_url_redirection = str_replace($search, $replace, $page_url_redirection);
        }
        if ( trim($page_url_redirection) != "" ) {
          $page_url_redirection = ( substr($page_url_redirection, 0, 1) != "/" ? "/" : "" ).$page_url_redirection;
          if ( array_key_exists($page_url_redirection, $tabPagePubliee) ) {
            echo "Création d'un lien depuis la page page_id:".$page_id."|page_alias:".$page_url." vers la page page_id:".$tabPagePubliee[$page_url_redirection]."|page_alias:".$page_url_redirection."\n";
            if ( !$bSimul ) {
              $oAppliLink->setLink(ALK_ATYPE_ID_GEDIT, ALK_LTYPE_ID_PAGE, $page_id, $tabPagePubliee[$page_url_redirection]);
            }
          }
        }
      }
      
      // contenu éditorial
      $oAppliEditeur = AlkFactory::getAppli(ALK_ATYPE_ID_EDITEUR);
      if ( $oAppliEditeur ) {
        $oDsBloc = $this->oQuery->getBlocByPageId($page_id);
        while ( $oDrBloc = $oDsBloc->getRowIter() ) {
          $bloc_id = $oDrBloc->getValueName("BLOC_ID");
          $oDsContenu = $oAppliEditeur->oQuery->getContenuByBlocId($bloc_id);
          while ( $oDrContenu = $oDsContenu->getRowIter() ) {
            $contenu_id = $oDrContenu->getValueName("CONTENU_ID");
            foreach ( $this->tabLangue as $key => $tabLg ) {
              $strContenu = $oDrContenu->getValueName("CONTENU".$tabLg["bdd"]);
              $strContenuUpdated = $oAppliEditeur->makeLinks($strContenu, $tabReplacement, $tabPagePubliee, $bSimul);
              if ( strcmp($strContenu, $strContenuUpdated) != 0 ) {
                if ( !$bSimul ) {
                  $oDbConn = AlkFactory::getDbConn();
                  $oDbConn->executeSql("update GEDIT_01_CONTENU set CONTENU".$tabLg["bdd"]."='".$oDbConn->analyseSql($strContenuUpdated, false)."' where CONTENU_ID=".$contenu_id);
                }
              }
            }
          }
        }
      }
    }
  }
  
  /**
   * retourne un lien vers une ressource
   * @param $ltype_id     identifiant du type de lien
   * @param $data_from_id identifiant d'un objet
   * @param()data_to_id   identifiant de l'objet associé
   * @param $value        valeur de lien optionnelle
   * @param $tabParams    tableau de paramètres supplémentaires
   * @return string lien regénéré
   */
  public function getLink($ltype_id=null, $data_from_id=null, $data_to_id=null, $value=null, $tabParams=array())
  {
    switch ( $ltype_id ) {
      case ALK_LTYPE_ID_PAGE :
        return ( !array_key_exists("bWithRootUrl", $tabParams) || array_key_exists("bWithRootUrl", $tabParams) && $tabParams["bWithRootUrl"] == true ? "" : "/" ).$this->oQuery->getPageUrl($data_to_id, ( array_key_exists("strParam", $tabParams) ? $tabParams["strParam"] : "" ), ( array_key_exists("bWithRootUrl", $tabParams) ? $tabParams["bWithRootUrl"] : true ), ( array_key_exists("lg", $tabParams) ? $tabParams["lg"] : ALK_LG_BDD ));
      break;
    }
  }
}

?>