<?php
/*licence/ 

Module écrit, supporté par la société Alkante SAS <alkante@alkante.com>

Nom du module : Alkanet::Class::Form
Module fournissant les classes d'affichage Alkanet.
Ce module appartient au framework Alkanet.

Ce logiciel est régi par la licence CeCILL-C soumise au droit français et
respectant les principes de diffusion des logiciels libres. Vous pouvez
utiliser, modifier et/ou redistribuer ce programme sous les conditions
de la licence CeCILL-C telle que diffusée par le CEA, le CNRS et l'INRIA
sur le site http://www.cecill.info.

En contrepartie de l'accessibilité au code source et des droits de copie,
de modification et de redistribution accordés par cette licence, il n'est
offert aux utilisateurs qu'une garantie limitée. Pour les mêmes raisons,
seule une responsabilité restreinte pèse sur l'auteur du programme, le
titulaire des droits patrimoniaux et les concédants successifs.

A cet égard l'attention de l'utilisateur est attirée sur les risques
associés au chargement, à l'utilisation, à la modification et/ou au
développement et à la reproduction du logiciel par l'utilisateur étant
donné sa spécificité de logiciel libre, qui peut le rendre complexe à
manipuler et qui le réserve donc à des développeurs et des professionnels
avertis possédant des connaissances informatiques approfondies. Les
utilisateurs sont donc invités à charger et tester l'adéquation du
logiciel à leurs besoins dans des conditions permettant d'assurer la
sécurité de leurs systèmes et ou de leurs données et, plus généralement,
à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.

Le fait que vous puissiez accéder à cet en-tête signifie que vous avez
pris connaissance de la licence CeCILL-C, et que vous en avez accepté les
termes.

/licence*/

require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CLASSE."form/alkhtmlctrl.class.php");

if( !defined("ALK_CAL_MODE_DAY") )   define("ALK_CAL_MODE_DAY",   1);
if( !defined("ALK_CAL_MODE_WEEK") )  define("ALK_CAL_MODE_WEEK",  2);
if( !defined("ALK_CAL_MODE_MONTH") ) define("ALK_CAL_MODE_MONTH",   3);
if( !defined("ALK_CAL_MODE_YEAR") )  define("ALK_CAL_MODE_YEAR", 4);

/**
 * @package Alkanet_Class_Form
 * @class AlkHtmlCalendar
 * 
 * Classe d'affichage d'un calendier
 */
class AlkHtmlCalendar extends AlkHtmlCtrl
{
  /** mode du calendrier 
   *  ALK_CAL_MODE_DAY  : journalier     
   *  ALK_CAL_MODE_WEEK : hebdomadaire
   *  ALK_CAL_MODE_MONTH  : mensuel
   *  ALK_CAL_MODE_YEAR: annuel          */
  protected $iMode;
  
  /** paramètrage du calendrier */
  protected $iTimeStart;              // heure de début pour affichage journalier ou hebdomadaire 0 à 24 heure 
  protected $iTimeEnd;                // heure de fin pour affichage journalier ou hebdomadaire 0 à 24 heure
  protected $iPrecision;              // découpage de l'heure en décimal : 25, 50 (par défaut) ou 100
  protected $iPrecisionCell;          // découpage de l'heure pour l'affichage en décimal : 25, 50 (par défaut) ou 100
  protected $iNbDayVisible;           // nombre de jours visibles (à partir du lundi)
  protected $iNbMonthVisible;         // nombre de mois visibles (à partir du mois courant) pour affichage annuel
  protected $bTimeVisible;            // affichage de la colonne gauche des heures pour affichage journalier ou hebdomadaire 0 à 24 heure
  protected $iCalWidth;               // largeur du calendrier
  protected $iCalCellHeight;          // hauteur d'une cellule
  protected $iCalCellSpacing;         // cellspacing des cellules
  protected $iCalCellHeightPaddingBorder; // hauteur total de border-top+border-bottom+padding-top+padding-bottom
  protected $bAffSelectMode;          // affichage de la sélection de mode d'affichage
  protected $strCalTitle;             // Titre à ajouter au calendrier
  protected $bAffTitleAuto;           // affichage du titre automatique
  protected $bAffMonthAnnee;          // affichage de l'annee dans l'entete du calendrier
  protected $bAffDayMonth;            // affichage du mois dans l'entete du calendrier
  protected $bAffMenuBar;             // affichage de la barre de menu permettant à l'utilisateur de chosir son mode d'affichage
  protected $strCalAlign;             // alignement du calendrier
  protected $strUrlMenuImage;         // url des images de la barre de menu
  protected $tabClosedDay;            // tableau associatif contenant les jours fériés (clé = JJ_MM_AAAA) => true si jour férié
  
  /** paramètrage des évenements */
  protected $bDesccInLayer;           // affichage du dec court dans un layer
  protected $bDesccVisible;           // affichage du dec court
  protected $bDesclInLayer;           // affichage du dec long dans un layer
  protected $bDesclVisible;           // affichage du dec long
  protected $bImgVisibleInLayer;      // affichage de l'image sur le layer
  protected $bTimeEvtVisible;         // affichage des heures de l'évenement
  protected $iWidthDescLayer;         // largeur des layers des descriptifs
  protected $iHeightDescLayer;        // hauteur des layers des descriptifs
  protected $strBgColorDescLayer;     // couleur de fond des layers des descriptifs
  
  protected $iCurrentDay;             // jour courant
  protected $iCurrentMonth;           // mois courant
  protected $iCurrentYear;            // année courante
  
  protected $tabCalEvt;               // tableau contenant les évenements à afficher
  
  /** classe css sur les cellules du calendrier */
  protected $cssCalendar;              // style du calendrier
  protected $cssLayerCalendar;         // style infobulle du calendrier
  protected $cssCellCalendar;          // style cellule d'un event du calendrier
  
  /** url de pagination */
  protected $strUrlPagine;            
  
  /** vrai si la cellule du calendrier est cliquable : 
   *  si oui, implémenter la fonction js onClickCalCell[appli->name]()
   *  avec les paramètres (day, month, year, hour, minute)
   */
  protected $bOnClickCell;
  
  /** vrai si seules les cellules postérieures à la date courante sont cliquables
   */
  protected $bOnClickOnlyAfter;

  /** info bulle sur le lien onClickCalCell[appli->name]() */
  protected $titleOnClickCell;

  /**
   *  Constructeur par défaut
   *
   * @param iMode     Mode du calendrier : ALK_CAL_MODE_DAY, ALK_CAL_MODE_WEEK, ALK_CAL_MODE_MONTH, ALK_CAL_MODE_YEAR
   * @param name      Nom du calendrier
   * @param iCurrentDay  jour courant (1à 31)
   * @param iCurrentMonth  mois courant 
   * @param iCurrentYear année courante
   * @param strUrlPagine 
   * @param iCalWidth
   * @param iCalCellHeight
   */
  public function __construct($oForm, $iMode, $strName, $iCurrentDay, $iCurrentMonth, 
                              $iCurrentYear, $strUrlPagine, $iCalWidth=650, $iCalCellHeight=50)
  {
    parent::__construct($oForm, $iMode, $strName);
    
    $this->iTimeStart      = 800; 
    $this->iTimeEnd        = 2000;
    $this->iPrecision      = 50;
    $this->iPrecisionCell  = 50;
    $this->iNbDayVisible   = 7;
    $this->iNbMonthVisible = 3;
    $this->iCalWidth       = $iCalWidth;
    $this->iCalCellHeight  = $iCalCellHeight; 
    $this->iCalCellSpacing = 2;
    $this->iCalCellHeightPaddingBorder = 2; // border-top et bottom=1 et padding-top et bottom=0)
    $this->bAffSelectMode  = false;
    $this->strCalTitle     = "";
    $this->bTimeVisible    = true;
    $this->bAffMenuBar     = false;
    $this->strCalAlign     = "center";
    $this->bAffTitleAuto   = true;
    $this->bAffMonthAnnee  = true;
    $this->bAffDayMonth    = true;
    $this->strUrlMenuImage = ALK_MEDIA_URL."images/";
    $this->tabClosedDay    = array();
    
    $this->bTimeEvtVisible    = false;
    $this->bDesccInLayer      = true;
    $this->bDesccVisible      = true;
    $this->bDesclInLayer      = false;
    $this->bDesclVisible      = false;
    $this->bImgVisibleInLayer = false;
    $this->iWidthDescLayer    = 250;
    $this->iHeightDescLayer   = 100;
    
    $this->iCurrentDay   = $iCurrentDay;
    $this->iCurrentMonth = $iCurrentMonth;
    $this->iCurrentYear  = $iCurrentYear;
    
    $this->strUrlPagine  = $strUrlPagine;

    $this->bOnClickCell      = false;
    $this->bOnClickOnlyAfter = false;
    $this->titleOnClickCell  = "";

    $this->bTemplate = false;

    $this->tabCalEvt = array();
    
    $this->addVarJs("tabEvt", "new Array();");
    $this->addScriptJs(ALK_ALKANET_ROOT_URL.ALK_ROOT_LIB."lib_layer.js");
    $this->addScriptJs(ALK_ALKANET_ROOT_URL.ALK_ROOT_LIB."lib_layerdesc.js");

    $this->cssCalendar          = "tabCalendar";
    $this->cssLayerCalendar     = "layerCalendar";
    $this->cssCellCalendar      = "cellCalendar";
    $this->setDefaultCss("alkhtmlcalendar.css");
    //$this->strBgColorDescLayer  = "#ffffe0";
  }

  /**
   *  Retourne les dates de début et de fin affichés sur le calendrier
   *        Permet de filtrer les évenements avant de les ajouter au calendrier
   *        Retourne le résultat dans un tableau de type ("start" => timestamp, "end" => timestamp)
   *
   * @return Retourne un array
   */
  public function getLimitCalendar()
  {
    $hStart = floor($this->iTimeStart/100);
    $mStart = ($this->iTimeStart-($hStart*100))*60/100;
    $hEnd   = floor($this->iTimeEnd/100);
    $mEnd   = ($this->iTimeEnd-($hEnd*100))*60/100;

    // ALK_CAL_MODE_DAY par défaut
    $dStart = mktime($hStart, $mStart, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear);
    $dEnd = mktime($hEnd, $mEnd, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear);
    switch( $this->iMode ) {
    case ALK_CAL_MODE_WEEK:
      $iNumDay = date("w", mktime(0, 0, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear));
      if( $iNumDay == 0 ) $iNumDay = 7;
      $iFirstDay = $this->iCurrentDay - ($iNumDay-1);
      $dStart = mktime($hStart, $mStart, 0, $this->iCurrentMonth, $iFirstDay, $this->iCurrentYear);
      $dEnd = mktime($hEnd, $mEnd, 0, $this->iCurrentMonth, $iFirstDay+6, $this->iCurrentYear);
      break;
      
    case ALK_CAL_MODE_MONTH:
      $dStart = mktime($hStart, $mStart, 0, $this->iCurrentMonth, 1, $this->iCurrentYear);
      $dEnd = mktime($hEnd, $mEnd, 0, $this->iCurrentMonth, 
                     date("t", mktime(0, 0, 0, $this->iCurrentMonth, 1, $this->iCurrentYear)), $this->iCurrentYear);
      break;
      
    case  ALK_CAL_MODE_YEAR:
      $mStart = $this->iCurrentMonth;
      $mEnd = $this->iCurrentMonth+$this->iNbMonthVisible-1;

      $dStart = mktime($hStart, $mStart, 0, $this->iCurrentMonth, 1, $this->iCurrentYear);
      $dEnd = mktime($hEnd, $mEnd, 0, $mEnd, date("t", mktime(0, 0, 0, $mEnd, 1, $this->iCurrentYear)), $this->iCurrentYear);
      break;
    }
    return array("start" => $dStart, "end" => $dEnd);
  }

  /**
   *  Affecte la précision du calendrier : 15mn, 30mn ou 1h
   *
   * @param iPrecision  intervalle de temps en minutes 
   */
  public function setPrecision($iPrecision=30)
  {
    $this->iPrecision = 50;

    if( $iPrecision == 15 )
      $this->iPrecision = 25;
    elseif( $iPrecision == 60 )
      $this->iPrecision = 100;
  }

  /**
   *  Affecte l'heure de début de journée
   *
   * @param strTime  Heure au format hh:mn
   */
  public function setTimeStartDay($strTime)
  {
    $this->iTimeStart = $this->time2Int($strTime); 
  }

  /**
   *  Affecte l'heure de fin de journée
   *
   * @param strTime  Heure au format hh:mn
   */
  public function setTimeEndDay($strTime)
  {
    $this->iTimeEnd = $this->time2Int($strTime);
  }

  /**
   *  Convertit puis retourne une heure au format entier = hh*100+(mn*100/60) interne à l'objet
   *
   * @param strTime   heure au format hh:mn
   * @return Retourne un entier
   */
  protected function time2Int($strTime)
  {
    $iTime = 0;
    $tabTime = explode(":", $strTime);
    if( count($tabTime) == 2 ) {
      if( $tabTime[1] % ($this->iPrecision*60/100) !=0 ) {
        $iMinute = round($tabTime[1]/15)*15;
        if( $iMinute == 60 ) {
          $tabTime[1] = 0;
          $tabTime[0]++;
        } else {
          $tabTime[1] = $iMinute;
        }
      }
      $iTime = intval($tabTime[0])*100+(intval($tabTime[1])*100/60);
    }
    return $iTime;
  }

  /**
   *  Convertit puis retourne une heure au format hh:mn à partir d'un entier interne à l'objet
   *
   * @param iTime  entier représentant une heure = hh*100+(mn*100/60)
   * @param bAll   =true pour affichage hh:mn, =false pour affichage [h]h:mn (n'affiche pas le premier caractère zéro de l'heure)
   * @return Retourne un string
   */
  protected function int2time($iTime, $bAll=false)
  {
    $hh = floor($iTime/100);
    $mn = ($iTime-($hh*100))*60/100;
    
    if( $bAll==true && $hh<10 )
      $hh = "0".$hh;

    if( $mn<10 )
      $mn = "0".$mn;

    $strTime = $hh.":".$mn;
    return $strTime;
  }

  /**
   *  Retourne le timestamp début d'un événement
   *
   * @param evt        Tableau contenant les infos d'un événement
   * @param strType    type d'heure = deb ou fin
   * @return Retourne un timestamp
   */
  protected function getEvtTimestamp($evt, $strType)
  {
    $d = mb_substr($evt["date".$strType], 0, 2);
    $m = mb_substr($evt["date".$strType], 3, 2);
    $y = mb_substr($evt["date".$strType], 6, 4);

    // l'heure est décimale
    $h = floor($evt["heure".$strType]/100);
    $mn = ($evt["heure".$strType]-($h*100))*60/100;
    return mktime((int)$h, (int)$mn, 0, (int)$m, (int)$d, (int)$y);
  }

  /**
   *  Ajoute un jour férié au tableau
   *
   * @param strDate  date fériée au format dd/mm/yyyy
   */
  public function addClosedDay($strDate)
  {
    $strKey = mb_ereg_replace("/", "_", $strDate);
    $this->tabClosedDay[$strKey] = true;
  }

  /**
   *  Retourne vrai si le jour est férié, faux sinon
   *
   * @param iDay   numéro du jour
   * @param iMonth numéro du mois
   * @param iYear  année sur 4 chiffres
   * @return Retourne un bool
   */
  protected function isClosedDay($iDay, $iMonth, $iYear)
  {
    if( strlen($iDay)<2 ) $iDay = "0".$iDay;
    if( strlen($iMonth)<2 ) $iMonth = "0".$iMonth;

    return array_key_exists($iDay."_".$iMonth."_".$iYear, $this->tabClosedDay);
  }

  /**
   *  Recupère les variables postées nécessaires au ctrl calendar
   *
   * @param reqMethod  Méthode de récupération (REQ_POST, REQ_GET, REQ_POST_GET (par défaut), REQ_GET_POST)
   */
  public function getRequest($reqMethod=REQ_GET)
  {
    $this->iCurrentDay     = AlkRequest::_REQUESTint("j", $this->iCurrentDay);
    $this->iCurrentMonth   = AlkRequest::_REQUESTint("m", $this->iCurrentMonth);
    $this->iCurrentYear    = AlkRequest::_REQUESTint("a", $this->iCurrentYear);
    $this->iMode           = AlkRequest::_REQUESTint("iModeAff", $this->iMode);
    $this->iNbMonthVisible = AlkRequest::_REQUESTint("nbmois", $this->iNbMonthVisible);
  }

  /**
   *  Retourne les paramètres nécessaires à une url pour réafficher le calendrier
   *        dans le même état. Ne retourne pas un & en premier caractère.
   *
   * @return Retourne un string
   */
  public function getRequestParam()
  {
    return "iModAff=".$this->iMode."&j=".$this->iCurrentDay."&m=".$this->iCurrentMonth."&a=".$this->iCurrentYear;
  }

  /**
   *  Ajoute un évenement au calendrier
   *
   * @param strLien     lien de l'evt
   * @param idEnt       identifiant de l'evt
   * @param strTitle    titre
   * @param dDateStart    date de début (jj/mm/aaaa)
   * @param dDateEnd    date de fin   (jj/mm/aaaa)
   * @param dTimeStart   heure de début (11=11:00, hh:mn)
   * @param dTimeStart   heure de fin  (11=11:00, hh:mn)
   * @param strDescc    desc courte (= chaine vide par défaut)     
   * @param strDescl    desc longue (= chaine vide par défaut)
   * @param pj          pièce jointe (lien complet)
   * @param strcssTxt   style du titre
   * @param strBgColor  couleur de fond de l'evt
   * @param strImg      url de l'image correspondant à l'événement (ratio 4x3)
   */
  public function addCalEvt($strLien, $idEnt, $strTitle, $dDateStart, $dDateEnd, $dTimeStart, 
                            $dTimeEnd, $strDescc="", $strDescl="", $pj="", $strCssTxt="", $strBgColor="", $strImg="")
  {
    if( mb_strlen($dTimeStart)<=2 )
      $dTimeStart .= ":00";
    if( mb_strlen($dTimeEnd)<=2 )
      $dTimeEnd .= ":00";

    if( $dTimeStart != ":00" )
      $dTimeStart = $this->time2Int($dTimeStart);
    else 
      $dTimeStart = $this->iTimeStart;
      
    if( $dTimeEnd != ":00" )
      $dTimeEnd = $this->time2Int($dTimeEnd);
    else
      $dTimeEnd = $this->iTimeEnd;

    $this->tabCalEvt[count($this->tabCalEvt)] = 
      array("lien"     => $strLien, 
            "id"       => $idEnt,
            "titre"    => $strTitle,
            "datedeb"  => $dDateStart,
            "datefin"  => $dDateEnd,
            "heuredeb" => $dTimeStart, /* décimal */
            "heurefin" => $dTimeEnd, /* décimal */
            "descc"    => $strDescc,
            "descl"    => $strDescl,
            "pj"       => $pj,
            "csstxt"   => $strCssTxt,
            "bgcolor"  => $strBgColor,
            "img"      => $strImg);
  }

  /**
   *  Retourne le nombre de jour dans un mois
   * 
   * @param iMonth  numéro du mois (de 1 à 12)
   * @param iYear   année sur 4 chiffres
   * @return Retourne un integer
   */
  protected function getDaysInMonth($iMonth, $iYear) 
  {
    $tabDays = array(31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31);
    $iRes = ( $iMonth>0 && $iMonth<13 ? $tabDays[$iMonth - 1] : -1 );
    if( $iMonth == 2 && checkdate(2, 29, $iYear) ) 
      $iRes = 29;
    return $iRes;
  }
  
  /**
   *  Convertit un jour ou un mois en français
   * 
   * @param iMode   0->jour   1->mois
   * @param iValue  Valeur à traduire
   * @param bUpperCase vrai si texte en majuscule, faux texte normal (par défaut)
   * 
   * @return la valeur traduite
   */
  protected function convertDateToFrench($iMode, $iValue, $bUpperCase=false, $bAbbrev=false)
  {
    $tabDay = array(0 => "Dimanche",
                     1 => "Lundi",
                     2 => "Mardi",
                     3 => "Mercredi",
                     4 => "Jeudi",
                     5 => "Vendredi",
                     6 => "Samedi",
                     7 => "Dimanche");

     $tabMonth = array(1 => "janvier",
                      2 => "février",
                      3 => "mars",
                      4 => "avril",
                      5 => "mai",
                      6 => "juin",
                      7 => "juillet",
                      8 => "août",
                      9 => "septembre",
                      10 => "octobre",
                      11 => "novembre",
                      12 => "décembre");
     // afin d'éviter les jours ou mois en 0x pour x<=9
     $iValue = $iValue*1;
     if( $iMode == 1 ) {
       if( $iValue>0 && $iValue<13 ) {
         $strMonth = $tabMonth[$iValue];
         if( $bAbbrev == true ) {
           $strMonth = mb_substr($strMonth, 0, 3);
           if( $iValue == 6 ) $strMonth = "jun";
           if( $iValue == 7 ) $strMonth = "jul";
           $strMonth .= ".";
         }
         if( $bUpperCase == true ) {
           $strMonth = mb_ereg_replace("é", "e", $strMonth);
           $strMonth = mb_ereg_replace("û", "u", $strMonth);
           $strMonth = mb_strtoupper($strMonth);
         }
         return $strMonth;
       }
     } else {
       if( $iValue>=0 && $iValue<8 ) {
         $strJr = ( $bAbbrev == true ? mb_substr($tabDay[$iValue], 0, 3) : $tabDay[$iValue] );
         if( $bUpperCase == true ) {
           $strJr = mb_strtoupper($strJr);
         }
         return $strJr;
       }
     }
     return "";
  }

  /**
   *  Retourne le code html du layer affiché
   * 
   * @return Retourne 
   */
  protected function getHtmlCalEvtLayer($tabCalEvt)
  {
    $strDescc = mb_ereg_replace("\r", "", mb_ereg_replace("\n", "<br/>", htmlentities($tabCalEvt["descc"], ENT_COMPAT, ALK_HTML_ENCODING)));
    $strDescl = mb_ereg_replace("\r", "", mb_ereg_replace("\n", "<br/>", htmlentities($tabCalEvt["descl"], ENT_COMPAT, ALK_HTML_ENCODING)));
    $strImg = $tabCalEvt["img"];

    $strHoraires = "";
    if ( !$this->bTimeEvtVisible ){
      $strDeb = trim(
        ( $tabCalEvt["datedeb"]!=$tabCalEvt["datefin"] ? "Du ".$tabCalEvt["datedeb"]." " : "" )
      );
      $strDeb .= trim(
        ( $tabCalEvt["heuredeb"]!="00:00" 
        ? ( $strDeb=="" ? ( $tabCalEvt["heuredeb"]!="00:00" ? "De" : "A partir de")." " : ", " ).
          $this->int2Time($tabCalEvt["heuredeb"]) 
        : "")
      );
      $strFin = trim(
        ( $tabCalEvt["datedeb"]!=$tabCalEvt["datefin"] ? " au ".$tabCalEvt["datefin"]." " : "" )
      );
      $strFin .= trim(
        ( $tabCalEvt["heurefin"]!="0" 
        ? ( $strFin=="" ? ( $tabCalEvt["heuredeb"]!="00:00" ? " à" : "Jusqu'à")." " : ", " ).
          $this->int2Time($tabCalEvt["heurefin"]) 
        : "")
      );
      $strHoraires = trim($strDeb." ".$strFin);
    }

    $strHtml = "<div class='layerDescCalendar'>".
      ( $strHoraires!=""
        ? "<div class='title'>".$strHoraires."</div>"
        : "" ).
      ( $this->bImgVisibleInLayer==true && $strImg!=""
        ? "<img border='0' src='".$strImg."' width='60' vspace='1' hspace='2' align='left'/>"
        : "" ).
      ( $this->bDesccInLayer==true && $strDescc!=""
        ? "<div class='desc'>".$strDescc."</div>"
        : "" ).
      ( $this->bDesclInLayer==true && $strDescl!=""
        ? "<div class='descl'>".$strDescl."</div>"
        : "" ).
      "</div>";
    
    $strHtml = addslashes($strHtml);

    return $strHtml;
  }
  
  /**
   *  Retourne le code html d'un évenement d'une cellule
   *
   * @param tabCalEvt tableau de l'évenement
   * @return Retourne un string
   */
  protected function getHtmlCalEvt($tabCalEvt, $strDivParam="")
  {
    /*$strDivParam .= ( $tabCalEvt["bgcolor"] != ""
                      ? " style='background-color: ".$tabCalEvt["bgcolor"]."' "
                      : "" );*/
    
    $strHtml = "<a "; 
    
    if( ($this->bDesccVisible==true && $this->bDesccInLayer==true) || 
        ($this->bDesclVisible==true && $this->bDesclInLayer==true) ||
        $this->bImgVisibleInLayer==true 
    ){
      $strDivParam .= " onMouseOver=\"ShowDesc('".$this->getHtmlCalEvtLayer($tabCalEvt)."', '', '', '".$this->cssLayerCalendar."')\"".
        " onMouseOut=\"HideDesc()\"";
    }
    
    $strHtml .= " href=\"".$tabCalEvt["lien"]."\">".
      ( $tabCalEvt["heuredeb"]<>"00:00" && $this->bTimeEvtVisible==true
        ? ( $tabCalEvt["datedeb"]!=$tabCalEvt["datefin"] ? $tabCalEvt["datedeb"]." " : "" ).$this->int2Time($tabCalEvt["heuredeb"]).
          ( $tabCalEvt["heurefin"]!="0" 
            ? " - ".( $tabCalEvt["datedeb"]!=$tabCalEvt["datefin"] ? $tabCalEvt["datefin"]." " : "" ).$this->int2Time($tabCalEvt["heurefin"])
            : "")."<br/>"
        : "" ).
      $tabCalEvt["titre"].
      ( $this->bDesccVisible==true ? "<br/>".$tabCalEvt["descc"] : "").
      ( $this->bDesclVisible==true ? "<br/>".$tabCalEvt["descl"] : "").
      "</a>".
      ( $tabCalEvt["pj"] != ""
        ? "<a href=\"".$tabCalEvt["pj"]."\" target=\"_blank\">".
          "<img align='absmiddle' class='alkimgicon alkimgiconpiecesjointes' src='".$this->strUrlMenuImage."transp.gif' border='0'/></a>"
        : "");

    // lien vers l'event
    if( $tabCalEvt["lien"] != "" ) {
      $strDivParam .= " onclick=\"".$tabCalEvt["lien"]."\" title=".(self::$strNavigator == "IE" ? "\"\"" : "\" \"");
    }

    // ajoute le div autour du tableau
    $strHtml = "<div ".$strDivParam.">".$strHtml."</div>";

    return $strHtml;
  }
  
  /**
   *  Retourne le code html de l'entête du calendrier : 
   *        mois, semaine ou année suivi des liens pour afficher les différents modes
   *
   * @return Retourne un string
   */
  protected function getHtmlCalMenu()
  {
    $strUrl = $this->strUrlPagine."&j=".$this->iCurrentDay."&m=".$this->iCurrentMonth."&a=".$this->iCurrentYear."&iModeAff=";

    $strHtml = "<table summary='ToolBar' cellpadding='0' cellspacing='1' border='0' class='toolBar' align='right'><tr>".
      "<td><a title='Journalier' href=\"".$strUrl.ALK_CAL_MODE_DAY."\">".
      "<img class='alkimgicon alkimgiconcalendard".( $this->iMode==1 ? "rol" : "")."' src='".$this->strUrlMenuImage."transp.gif' border='0' alt='Journalier'/></a></td>" .
      "<td><a title='Hebdomadaire' href=\"".$strUrl.ALK_CAL_MODE_WEEK."\">" .
      "<img class='alkimgicon alkimgiconcalendarw".( $this->iMode==2 ? "rol" : "")."' src='".$this->strUrlMenuImage."transp.gif' border='0' alt='Hebdomadaire'/></a></td>" .
      "<td><a title='Mensuel' href=\"".$strUrl.ALK_CAL_MODE_MONTH."\">".
      "<img class='alkimgicon alkimgiconcalendarm".( $this->iMode==3 ? "rol" : "")."' src='".$this->strUrlMenuImage."transp.gif' border='0' alt='Mensuel'/></a></td>" .
      "<td><a title='Trimestriel' href=\"".$strUrl.ALK_CAL_MODE_YEAR."\">".
      "<img class='alkimgicon alkimgiconcalendar3m".( $this->iMode==4 && $this->iNbMonthVisible==3 ? "rol": "")."' src='".$this->strUrlMenuImage."transp.gif' border='0' alt='Trimestriel'/></a></td>" .
      "<td><a title='Semestriel' href=\"".$strUrl.ALK_CAL_MODE_YEAR."&nbmois=6\">".
      "<img class='alkimgicon alkimgiconcalendar6m".( $this->iMode==4 && $this->iNbMonthVisible==6 ? "rol": "")."' src='".$this->strUrlMenuImage."transp.gif' border='0' alt='Semestriel'/></a></td>" .
      "</tr></table>";    
    return $strHtml;
  }

  /**
   *  Retourne le code html du layer affichant l'info bulle
   *
   * @return Retourne un string
   */
  protected function getHtmlLayerDesc()
  {
    return "<div id=\"layerDesc\" class='".$this->cssLayerCalendar."'".
      " style=\"width:".$this->iWidthDescLayer."px; height:".$this->iHeightDescLayer."px; \">".
      "&nbsp;</div>";
  }

  /**
   *  Retourne le code html du calendrier journalier
   *
   * @return Retourne un string
   */
  protected function getHtmlDailyCalendar()
  {
    $tabIdEvt = array();
    $tabH = array();
    $tabNbH = array();

    $iBorderWidth = ( self::$strNavigator == "IE" ? 0 : 4 );
    
    $iLargeurCell = ( $this->bTimeVisible==true 
                      ? $this->iCalWidth-40-5*$this->iCalCellSpacing
                      : $this->iCalWidth-4*$this->iCalCellSpacing );
    $iLargeurCellCal = $iLargeurCell-44-2*$this->iCalCellSpacing;
    
    $iNumDay = date("w", mktime(0, 0, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear));
    if( $iNumDay == 0 ) 
      $iNumDay = 7;
    
    $strHtmlJs = "";
    $strHtmlLayer = "";
    
    // entete titre
    $strHtml = "<table class='".$this->cssCalendar."' summary='Calendar' cellpadding='0' cellspacing='".$this->iCalCellSpacing."' border='0' align='".$this->strCalAlign."'>".
      "<thead>".
      "<tr>".
      ( $this->bTimeVisible==true ? "<td class='rowDimension' style='width:40px'></td>" : "").
      "<td class='rowDimension' style='width:22px'></td>".
      "<td class='rowDimension' style='width:".$iLargeurCellCal."px'></td>".
      "<td class='rowDimension' style='width:22px'></td>".
      "</tr>".
      "<tr>".

      ( $this->bTimeVisible==true ? "<td class='none'></td>" : "").
      "<td colspan='3'><div>".
      ( $this->strCalTitle!="" 
        ? $this->strCalTitle.( $this->bAffTitleAuto==true ? " - " : "" )
        : "").
      ( $this->bAffTitleAuto==true 
        ? $this->ConvertDateToFrench(1, $this->iCurrentMonth, true)." / ".$this->iCurrentYear
        : "").
      ( $this->bAffMenuBar==true 
        ? "</div><div class='toolBar'>".$this->getHtmlCalMenu()
        : "" ).
      "</div>".
      "</td></tr></thead>";
    
    $strHtml .= "<tbody><tr>";
         
    $iDayPrev = date("j", strtotime("-1 day", mktime(0, 0, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear)));
    $iMonthPrev = date("n", strtotime("-1 day", mktime(0, 0, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear)));
    $iAnneePrev = date("Y", strtotime("-1 day", mktime(0, 0, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear)));
    
    $iDayNext = date("j", strtotime("+1 day", mktime(0, 0, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear)));
    $iMonthNext = date("n", strtotime("+1 day", mktime(0, 0, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear)));
    $iAnneeNext = date("Y", strtotime("+1 day", mktime(0, 0, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear)));
         
    $iCurrentDay = ( $this->iCurrentDay[0] == "0" ? mb_substr($this->iCurrentDay, 1) : $this->iCurrentDay);

    $bJrFerie = $this->isClosedDay($this->iCurrentDay, $this->iCurrentMonth, $this->iCurrentYear);

    $strFerie = "";
    /*    if( $bJrFerie == true && $iCurrentDay>0 && $iCurrentDay<6 ) {
      $strFerie = " - <b>Férié</b>";
    }*/

    if( $this->bTimeVisible==true )
      $strHtml .= "<td class='none'></td>";
      
    $strUrlPrev = $this->strUrlPagine."&j=".$iDayPrev."&m=".$iMonthPrev."&a=".$iAnneePrev."&iModeAff=".$this->iMode;
    $strUrlNext = $this->strUrlPagine."&j=".$iDayNext."&m=".$iMonthNext."&a=".$iAnneeNext."&iModeAff=".$this->iMode;

    // entete calendrier
    $strHtml .= "<td class='header left'>".
      "<a onclick=\"AlkGoUrl('".$strUrlPrev."')\" title='Jour précédent'><<</a></td>".
      "<td class='header'>".
      $this->ConvertDateToFrench(0, $iNumDay)." ".$iCurrentDay.( $iCurrentDay==1 ? "er" : "" ).
      ( $this->bAffDayMonth==true 
        ? " ".$this->ConvertDateToFrench(1, date("n", mktime(0, 0, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear)))
        : "" ).
      ( $this->bAffMonthAnnee==true 
        ? " ".$this->iCurrentYear
        : "").$strFerie."</td>".
      "<td class='header right'>".
      "<a title='Jour suivant' onclick=\"AlkGoUrl('".$strUrlNext."')\">>></a></td>".
      "</tr>";
         
    // Ajuste l'heure de début et de fin de journée en fonction de la précision d'affichage.
    // Exemple : Afficher le cadre 8h pour une journée qui démarre à 8h15 avec 30mn de précision d'affichage.
    $hStart = ( $this->iTimeStart % $this->iPrecisionCell == 0
                ? $this->iTimeStart 
                :  $this->iTimeStart - ($this->iTimeStart % $this->iPrecisionCell) );
    $hEnd  = ( $this->iTimeEnd % $this->iPrecisionCell == 0
               ? $this->iTimeEnd 
               : $this->iTimeEnd + ($this->iTimeEnd % $this->iPrecisionCell) );
    $iRowSpanFerie = ($hEnd-$hStart)/$this->iPrecisionCell;

    for($h = $hStart; $h < $hEnd; $h += $this->iPrecision) {
      $tabNbH[$h] = 0;
      $tabH[$h] = 0;
    }
    
    // mémorise le nombre d'évenements par tranche horaire
    if( !$bJrFerie ) {
      for($h = $hStart; $h < $hEnd; $h += $this->iPrecision) {
        $hh = floor($h/100);
        $mn = (($h-($hh*100))*60)/100;
        $dateDayCal = mktime($hh, $mn, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear);
        
        foreach($this->tabCalEvt as $evt) {
          if( $evt["datedeb"] != "" )
            $dateStartEvt = $this->getEvtTimestamp($evt, "deb");
          else
            $dateStartEvt = 0;
          
          if( $evt["datefin"] != "" )
            $dateEndEvt = $this->getEvtTimestamp($evt, "fin");
          else if ($evt["heurefin"] != "" && $evt["datedeb"]!=""){
          	$evt["datefin"]=$evt["datedeb"];
          	$dateEndEvt = $this->getEvtTimestamp($evt, "fin");
          } 
          else
            $dateEndEvt = $dateStartEvt;
          if( ($dateStartEvt<=$dateDayCal && $dateDayCal<$dateEndEvt) ||
              ($dateStartEvt==$dateDayCal && $dateEndEvt=="") ) {
            $tabNbH[$h] = $tabNbH[$h]+1;
            $tabH[$h] = $tabH[$h]+1;
          }
        }
      }
    }
    
    // création des plages horaires et calcul du positionnement
    $bRowSpan = false;
    $cptLayer = 0;
    for($h = $hStart; $h < $hEnd; $h += $this->iPrecision) {
      $hh = floor($h/100);
      $mn = ($h-($hh*100))*60/100;
      $dateDayCal = mktime($hh, $mn, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear);
      $strTime = $this->int2Time($h);
      
      if( $h % $this->iPrecisionCell == 0 ) {
        $strHtml .= "<tr>";
      
        if( $this->bTimeVisible==true )
          $strHtml .= "<td class='header'>".
            ($mn == 0 ? $strTime : "&nbsp;" ).
            "</td>";
        
        $bOnClickCell = $this->bOnClickCell == true && ($this->bOnClickOnlyAfter == true ? $dateDayCal >= AlkFactory::getLocalDate() : true);
      
        if( !$bJrFerie ) {
          $strHtml .= "<td id='cell".$h."' valign='top' style='height:".$this->iCalCellHeight."px' colspan='3'".
            " onmouseover=\"onMouseOverCalCell(this, 'roll', ".($bOnClickCell == true ? "true" : "false").")\"".
            " onmouseout=\"onMouseOutCalCell(this)\"".
            ( $bOnClickCell == true
              ? " onclick=\"onClickCalCell".$this->name."(".$this->iCurrentDay.",".$this->iCurrentMonth.",".$this->iCurrentYear.",".$hh.",".$mn.")\""
              : "" ).
            ( $bOnClickCell == true
              ? " title=\"".$this->titleOnClickCell."\""
              : "" ).
            ">&nbsp;";
        } else {
          if( !$bRowSpan ) {
            $bRowSpan = true;
            $strHtml .= "<td class='closed' id='cell".$h."' rowspan='".$iRowSpanFerie.
              "' align='center' colspan='3'>Fermé";
          }
        }
      }

      if( !$bJrFerie ) {
        foreach($this->tabCalEvt as $evt) {
          $dateStartEvt = $this->getEvtTimestamp($evt, "deb");
          if( $evt["datefin"] != "" )
            $dateEndEvt = $this->getEvtTimestamp($evt, "fin");
          else
            $dateEndEvt = $dateStartEvt;
        
          if( ( ($dateStartEvt<=$dateDayCal && $dateDayCal<=$dateEndEvt) || 
                ($dateStartEvt==$dateDayCal && $dateEndEvt=="") ) && in_array($evt["id"], $tabIdEvt)==false ) {

            $dateHier = strtotime("-1 day", mktime(23, 59, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear));
            $dateDemain = strtotime("+1 day", mktime(0, 0, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear));
          
            if( $dateStartEvt <= $dateHier ) {
              $hStartEvt = $this->iTimeStart;
            } else {
              $hStartEvt = ( $evt["heuredeb"] < $this->iTimeStart ? $this->iTimeStart : $evt["heuredeb"] );
            }
            
            if( $dateEndEvt >= $dateDemain ) {
              $hEndEvt = $this->iTimeEnd;
            } else {
              $hEndEvt = ( $evt["heurefin"] > $this->iTimeEnd ? $this->iTimeEnd : $evt["heurefin"] );
            }

            $iMaxNbEvt = max(array_slice($tabNbH, ($hStartEvt-$hStart)/$this->iPrecision, 
                                         max(1, ($hEndEvt-$hStartEvt)/$this->iPrecision)));
            $iWidthEvt = floor($iLargeurCell / $iMaxNbEvt) - $iBorderWidth;
            $iHeightEvt = ( $hEndEvt-$hStartEvt > 0 
                            ? floor($this->iCalCellHeight*(($hEndEvt-$hStartEvt)/$this->iPrecisionCell) + 
                                    ($this->iCalCellSpacing+( ALK_NAV!=ALK_NAV_IE7 && ALK_NAV!=ALK_NAV_IE6 ? 0 : $this->iCalCellHeightPaddingBorder))*(($hEndEvt-$hStartEvt-1)/$this->iPrecisionCell))
                            : $this->iCalCellHeight );
            $iTopEvt = floor($this->iCalCellHeight*(($hStartEvt-$hStart)/$this->iPrecisionCell) + 
                             ($this->iCalCellSpacing+( ALK_NAV!=ALK_NAV_IE7 && ALK_NAV!=ALK_NAV_IE6 ? 0 : $this->iCalCellHeightPaddingBorder))*(($hStartEvt-$hStart)/$this->iPrecisionCell));
            $iLeftEvt = ($tabNbH[$h]-$tabH[$h]) * $iWidthEvt;
          
            $strParamDiv = "id=\"layer".$evt["id"]."\" ".
              " class=\"".$this->cssCellCalendar." floatCalendar\"".
              " style=\" background-color:".$evt["bgcolor"]."; width:".$iWidthEvt."px; height:".$iHeightEvt."px;".$evt["csstxt"]."\" ";

            $strHtmlLayer .= $this->getHtmlCalEvt($evt, $strParamDiv);
            $strHtmlJs .= " var layerEvt = new LayerEvt('layer".$evt["id"]."', 'cell".$hStart."', ".$iLeftEvt.", ".$iTopEvt.");".
              " tabEvt.push(layerEvt);";
            array_push($tabIdEvt, $evt["id"]);     
          
            for($t=$hStartEvt; $t<$hEndEvt; $t++) {
              if( array_key_exists($t, $tabH)==true ) {
                $tabH[$t] = $tabH[$t]-1;
              } else {
                $tabH[$t] = 0;
              }
            }
          }
        }
      }
        
      if( $cptLayer==0 ) {
        $strHtmlLayer .= $this->getHtmlLayerDesc();
        $cptLayer++;
      }
      
      if( $h % $this->iPrecisionCell == 0 ) {
        $strHtml .= "</td></tr>";
      }
    }
    
    $strHtml .= "</tbody><tfoot><tr>";
    if( $this->bTimeVisible==true )
      $strHtml .= "<td class='none'></td>";
      
    $strHtml .= "<td class='right'>".
      "<a onclick=\"AlkGoUrl('".$strUrlPrev."')\" title='Jour précédent'><<</a></td>".
      "<td>&nbsp;</td>".
      "<td class='right'>".
      "<a onclick=\"AlkGoUrl('".$strUrlNext."')\" title='Jour suivant'>>></a></td>".
      "</tr></tfoot></table>";
          
    $strHtmlJs .= " window.onresize=winReload;";

    $this->AddJs($strHtmlJs);
    $this->addOnLoad(" winReload(); ");
    $this->AddScriptJs(ALK_ALKANET_ROOT_URL.ALK_ROOT_CLASSE."form/alkhtmlcalendar.js");

    return $strHtml.$strHtmlLayer;
  }
  
  /**
   *  Retourne le code html du calendrier hebdomadaire
   *
   * @return Retourne un string
   */
  protected function getHtmlWeeklyCalendar()
  {
    $tabIdEvt = array();
    $tabH = array();
    $tabNbH = array();

    $iBorderWidth = ( self::$strNavigator == "IE" ? 0 : 4 );
    
    $strHtmlJs = "";
    $strHtmlLayer = "";
    
    $iLargeurCell = ( $this->bTimeVisible==true 
                      ? floor(($this->iCalWidth-40-(2+$this->iNbDayVisible)*$this->iCalCellSpacing)/$this->iNbDayVisible)
                      : floor(($this->iCalWidth-(1+$this->iNbDayVisible)*$this->iCalCellSpacing)/$this->iNbDayVisible) );
    
    $iNumWeek = date("W", mktime(0, 0, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear));
    $iNumDay = date("w", mktime(0, 0, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear));
    if( $iNumDay==0 ) $iNumDay=7;
    
    // entete titre
    $strHtml = "<table summary='' class='".$this->cssCalendar."' cellpadding='0' cellspacing='".$this->iCalCellSpacing."' border='0' align='".$this->strCalAlign."'>".
      "<thead>".
      "<tr>".
      ( $this->bTimeVisible==true ? "<td class='rowDimension' style='width:40px'></td>" : "");
    for($i=0; $i<$this->iNbDayVisible; $i++) {
      $strHtml .= "<td  class='rowDimension' style='width:".$iLargeurCell."px'></td>";
    }
    $strHtml .= "</tr>".
      "<tr>".
      ( $this->bTimeVisible==true
        ? "<td class='none'></td>"
        : "" ).
      "<td colspan='".$this->iNbDayVisible."'>".
      "<div>".
      ( $this->strCalTitle != "" 
        ? $this->strCalTitle.( $this->bAffTitleAuto==true ? " - " : "" ) 
        : "").
      ( $this->bAffTitleAuto==true
        ? "Semaine ".$iNumWeek." ".$this->iCurrentYear
        : "" ).
      ( $this->bAffMenuBar==true 
        ? "</div><div class='toolBar'>".$this->getHtmlCalMenu()
        : "" ).
      "</div>".

      "</td></tr>";
    
    $strHtml .= "</thead><tbody><tr>";
    
    if ($this->bTimeVisible==true)
      $strHtml .= "<td class='none'></td>";
      
    $iDayPrev  = date("j", strtotime("-7 day", mktime(0, 0, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear)));
    $iMonthPrev  = date("n", strtotime("-7 day", mktime(0, 0, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear)));
    $iAnneePrev = date("Y", strtotime("-7 day", mktime(0, 0, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear)));
    
    $iDayNext  = date("j", strtotime("+7 day", mktime(0, 0, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear)));
    $iMonthNext  = date("n", strtotime("+7 day", mktime(0, 0, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear)));
    $iAnneeNext = date("Y", strtotime("+7 day", mktime(0, 0, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear)));

    $strUrlPrev = $this->strUrlPagine."&j=".$iDayPrev."&m=".$iMonthPrev."&a=".$iAnneePrev."&iModeAff=".$this->iMode;
    $strUrlNext = $this->strUrlPagine."&j=".$iDayNext."&m=".$iMonthNext."&a=".$iAnneeNext."&iModeAff=".$this->iMode;
    $strUrlDay = $this->strUrlPagine."&iModeAff=".ALK_CAL_MODE_DAY;

    for($k=1; $k<=$this->iNbDayVisible; $k++) {
      $iDelta = $k - $iNumDay;
      $dateDayCal = strtotime(($iDelta < 0 ? $iDelta : "+".$iDelta)." day", 
                               mktime(0, 0, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear));
      $iNumJr = date("j", $dateDayCal);
      
      $strHtml .= "<td class='header'>".
        ( $k == 1
          ? "<a title='Semaine précédente' onclick=\"AlkGoUrl('".$strUrlPrev."')\"><<</a>&nbsp;&nbsp;"
          : "").
        "<a style='text-decoration:none; border:none; padding:none' title='Afficher le calendrier de ce jour'".
        "   onclick=\"AlkGoUrl('".$strUrlDay."&j=".date("d", $dateDayCal)."&m=".date("m", $dateDayCal)."&a=".date("Y", $dateDayCal)."')\">".
        $this->ConvertDateToFrench(0, $k, false, true)." ".     
        $iNumJr.( $iNumJr=="1" ? "er" : "" ).
        ( $this->bAffDayMonth == true 
          ? " ".$this->ConvertDateToFrench(1, date("n", $dateDayCal), false, true)
          : "" ).
        ( $this->bAffMonthAnnee == true 
          ? " ".date("y", $dateDayCal)
          : "" ).
        "</a>".
        ( $k == $this->iNbDayVisible
          ? "&nbsp;&nbsp;<a title='Semaine suivante' onclick=\"AlkGoUrl('".$strUrlNext."')\">>></a>"
          : "" ).
        "</td>";
    }
    $strHtml .= "</tr>";
         
    $cptLayer = 0;
    
    // Ajuste l'heure de début et de fin de journée en fonction de la précision d'affichage.
    // Exemple : Afficher le cadre 8h pour une journée qui démarre à 8h15 avec 30mn de précision d'affichage.
    $hStart = ( $this->iTimeStart % $this->iPrecisionCell == 0
                ? $this->iTimeStart 
                :  $this->iTimeStart - ($this->iTimeStart % $this->iPrecisionCell) );
    $hEnd  = ( $this->iTimeEnd % $this->iPrecisionCell == 0
               ? $this->iTimeEnd 
               : $this->iTimeEnd + ($this->iTimeEnd % $this->iPrecisionCell) );
    $iRowSpanFerie = ($hEnd-$hStart)/$this->iPrecisionCell;

    $tabJrFerie = array_fill(1, $this->iNbDayVisible, false);
    $tabIdEvt = array_fill(1, $this->iNbDayVisible, array());
    for($i=0; $i<=$this->iNbDayVisible; $i++) {
      for($h = $hStart; $h < $hEnd; $h += $this->iPrecision) {
        $tabNbH[$i][$h] = 0;
        $tabH[$i][$h] = 0;
      }
    }
    
    // mémorise le nombre d'évenements par tranche horaire et par jour de la semaine
    for($i=1; $i<=$this->iNbDayVisible; $i++) {
      $iDelta = $i - $iNumDay;
      $dateDayCal = strtotime(($iDelta < 0 ? $iDelta : "+".$iDelta)." day", 
                               mktime(0, 0, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear));
      $tabJrFerie[$i] = $this->isClosedDay(date("d", $dateDayCal), date("m", $dateDayCal), date("Y", $dateDayCal));
      if( !$tabJrFerie[$i] ) {
        for($h = $hStart; $h < $hEnd; $h += $this->iPrecision) {
          $hh = floor($h/100);
          $mn = (($h-($hh*100))*60)/100;
          $dateDayCal = strtotime(($iDelta < 0 ? $iDelta : "+".$iDelta)." day", 
                                   mktime($hh, $mn, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear));
        
          foreach($this->tabCalEvt as $evt) {
            if( $evt["datedeb"] != "" )
              $dateStartEvt = $this->getEvtTimestamp($evt, "deb");
            else
              $dateStartEvt = 0;
            
            if( $evt["datefin"] != "" )
	            $dateEndEvt = $this->getEvtTimestamp($evt, "fin");
	          else if ($evt["heurefin"] != "" && $evt["datedeb"]!=""){
	          	$evt["datefin"]=$evt["datedeb"];
	          	$dateEndEvt = $this->getEvtTimestamp($evt, "fin");
	          } 
	          else
	            $dateEndEvt = $dateStartEvt;
            
            if( ($dateStartEvt<=$dateDayCal && $dateDayCal<$dateEndEvt) ||
                ($dateStartEvt==$dateDayCal && $dateEndEvt=="") ) {
              $tabNbH[$i][$h] = $tabNbH[$i][$h]+1;
              $tabH[$i][$h] = $tabH[$i][$h]+1;
            }
          }
        }
      }
    }
    $dateToday = date("d/m/Y", AlkFactory::getLocalDate());

    // création des plages horaires et calcul du positionnement
    $tabRowSpan = array_fill(1, $this->iNbDayVisible, array(false, false));
    $cptLayer = 0;
    for($h = $hStart; $h < $hEnd; $h += $this->iPrecision) {
      $hh = floor($h/100);
      $mn = ($h-($hh*100))*60/100;
      $strTime = $this->int2Time($h);

      if( $h % $this->iPrecisionCell == 0 ) {
        $strHtml .= "<tr>";
      
        if( $this->bTimeVisible==true )
          $strHtml .= "<td class='header'>".
            ($mn == 0 ? $strTime : "&nbsp;" ).
            "</td>";
      }
        
      for($i=1; $i<=$this->iNbDayVisible; $i++) {
        $iDelta = $i - $iNumDay;
        $dateDayCal = strtotime(($iDelta < 0 ? $iDelta : "+".$iDelta)." day", mktime($hh, $mn, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear));

        if( $h % $this->iPrecisionCell == 0 ) {
          if( $tabJrFerie[$i] == true ) {
            if( !$tabRowSpan[$i][0] ) {
              $tabRowSpan[$i][0] = true;
              $strHtml .= "<td class='closed' ".( $h==$hStart ? "id='cell".$h.$i : "" ).
                "' rowspan='".$iRowSpanFerie."'>Fermé";
            }
          } else {
            $strClass = "";
            if( $dateToday == date("d/m/Y", $dateDayCal) ) {
              $strClass = "current";
            }
            
            $bOnClickCell = $this->bOnClickCell == true && ($this->bOnClickOnlyAfter == true ? $dateDayCal >= AlkFactory::getLocalDate() : true);
      
            $strHtml .= "<td ".( $h==$hStart ? " id='cell".$h.$i."'" : "")."\"".
              " class='".$strClass."' valign='top' style='height:".$this->iCalCellHeight."px'".
              " onmouseover=\"onMouseOverCalCell(this, 'roll', ".($bOnClickCell == true ? "true" : "false").")\"".
              " onmouseout=\"onMouseOutCalCell(this)\"".
              ( $bOnClickCell == true
                ? " onclick=\"onClickCalCell".$this->name."(".date("d", $dateDayCal).", ".
                  date("m", $dateDayCal).", ".date("Y", $dateDayCal).",".$hh.",".$mn.")\""
                : "" ).
              ( $bOnClickCell == true
                ? " title=\"".$this->titleOnClickCell."\""
                : "" ).
              ">&nbsp;";
          }
        }
        
        if( !$tabJrFerie[$i]) {
          foreach($this->tabCalEvt as $evt) {
            $dateStartEvt = $this->getEvtTimestamp($evt, "deb");
            if( $evt["datefin"] != "" )
              $dateEndEvt = $this->getEvtTimestamp($evt, "fin");
            else
              $dateEndEvt = $dateStartEvt;
        
            if( ( ($dateStartEvt<=$dateDayCal && $dateDayCal<=$dateEndEvt) || 
                  ($dateStartEvt==$dateDayCal && $dateEndEvt=="") ) && in_array($evt["id"], $tabIdEvt[$i])==false ) {

              $dateHier = strtotime(($iDelta-1 < 0 ? $iDelta-1 : "+".($iDelta-1))." day", mktime(23, 59, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear));
              $dateDemain = strtotime(($iDelta+1 < 0 ? $iDelta+1 : "+".($iDelta+1))." day", mktime(0, 0, 0, $this->iCurrentMonth, $this->iCurrentDay, $this->iCurrentYear));
          
              if( $dateStartEvt <= $dateHier ) {
                $hStartEvt = $this->iTimeStart;
              } else {
                $hStartEvt = ( $evt["heuredeb"] < $this->iTimeStart ? $this->iTimeStart : $evt["heuredeb"] );
              }
            
              if( $dateEndEvt >= $dateDemain ) {
                $hEndEvt = $this->iTimeEnd;
              } else {
                $hEndEvt = ( $evt["heurefin"] > $this->iTimeEnd ? $this->iTimeEnd : $evt["heurefin"] );
              }

              $iMaxNbEvt = max(array_slice($tabNbH[$i], ($hStartEvt-$hStart)/$this->iPrecision, 
                                           max(1, ($hEndEvt-$hStartEvt)/$this->iPrecision)));
              $iWidthEvt = floor($iLargeurCell / $iMaxNbEvt)-$iBorderWidth;
              $iHeightEvt = ( $hEndEvt-$hStartEvt > 0 
                              ? floor($this->iCalCellHeight*(($hEndEvt-$hStartEvt)/$this->iPrecisionCell) + 
                                      ($this->iCalCellSpacing+( ALK_NAV!=ALK_NAV_IE7 && ALK_NAV!=ALK_NAV_IE6 ? 0 : $this->iCalCellHeightPaddingBorder))*(($hEndEvt-$hStartEvt-1)/$this->iPrecisionCell))
                              : $this->iCalCellHeight );
              $iTopEvt = floor($this->iCalCellHeight*(($hStartEvt-$hStart)/$this->iPrecisionCell) + 
                               ($this->iCalCellSpacing+( ALK_NAV!=ALK_NAV_IE7 && ALK_NAV!=ALK_NAV_IE6 ? 0 : $this->iCalCellHeightPaddingBorder))*(($hStartEvt-$hStart)/$this->iPrecisionCell));
              $iLeftEvt = ($tabNbH[$i][$h]-$tabH[$i][$h]) * ($iWidthEvt + $this->iCalCellSpacing);

              $strParamDiv = "id=\"layer".$evt["id"].$i."\"".
                " class=\"".$this->cssCellCalendar."  floatCalendar\"".
                " style=\" background-color:".$evt["bgcolor"]."; width:".$iWidthEvt."px; height:".$iHeightEvt."px;".$evt["csstxt"]."\" ";

              $strHtmlLayer .= $this->getHtmlCalEvt($evt, $strParamDiv);
              $strHtmlJs .= " var layerEvt = new LayerEvt('layer".$evt["id"].$i."', 'cell".$hStart.$i."', ".$iLeftEvt.", ".$iTopEvt.");".
                " tabEvt.push(layerEvt);";
              array_push($tabIdEvt[$i], $evt["id"]);
            
              for($t=$hStartEvt; $t<$hEndEvt; $t++) {
                if( array_key_exists($t, $tabH[$i])==true ) {
                  $tabH[$i][$t] = $tabH[$i][$t]-1;
                } else {
                  $tabH[$i][$t] = 0;
                }
              }
            }
          }
        }

        if( $h % $this->iPrecisionCell == 0 ) {
          if( $tabJrFerie[$i] == true ) {
            if( !$tabRowSpan[$i][1] ) {
              $tabRowSpan[$i][1] = true;
              $strHtml .= "</td>";
            }
          } else {
            $strHtml .= "</td>";
          }
        }
      }
      
      if( $cptLayer==0 ) {
        $strHtmlLayer .= $this->getHtmlLayerDesc();
        $cptLayer++;
      }

      $strHtml .= "</tr>";
    }

    $strHtml .= "</tbody><tfoot><tr>";
    if( $this->bTimeVisible == true )
      $strHtml .= "<td class='none'></td>";
    
    for($k=1; $k<=$this->iNbDayVisible; $k++)
      $strHtml .= "<td>".
        ( $k==1 
          ? "<a title='Semaine précédente' onclick=\"AlkGoUrl('".$strUrlPrev."')\"><< précédente</a>"
          : "&nbsp;").
        ( $k==$this->iNbDayVisible
          ? "<a title='Semaine suivante' onclick=\"AlkGoUrl('".$strUrlNext."')\">suivante >></a>"
          : "&nbsp;").
        "</td>";
     
    $strHtml .= "</tr></table>";

    $strHtmlJs .= " window.onresize=winReload;";

    $this->AddJs($strHtmlJs);
    $this->addOnLoad(" winReload(); ");
    $this->AddScriptJs(ALK_ALKANET_ROOT_URL.ALK_ROOT_CLASSE."form/alkhtmlcalendar.js");

    return $strHtml.$strHtmlLayer;
  }
  
  /**
   *  Retourne le code html du calendrier mensuel
   *
   * @return Retourne un string
   */
  protected function getHtmlMonthlyCalendar()
  {
    $iBorderWidth = 9;

    $iCalWidth = ( $this->bTimeVisible==true 
                   ? $this->iCalWidth-42
                   : $this->iCalWidth );
    
    $iLargeurCellCal = floor(($iCalWidth-($this->iNbDayVisible-1)*2) / $this->iNbDayVisible);

    $dayone = date("w", mktime(0, 0, 0, $this->iCurrentMonth, 1, $this->iCurrentYear));
    if( $dayone == 0 )
      $dayone = 7;
    
    $nbDayPrev = date("t", strtotime("-1 month", mktime(0, 0, 0, $this->iCurrentMonth, 1, $this->iCurrentYear)));
    $iDateCurPrev = mktime(0, 0, 0, $this->iCurrentMonth, min($this->iCurrentDay, $nbDayPrev), $this->iCurrentYear);
    $nbDayNext = date("t", strtotime("+1 month", mktime(0, 0, 0, $this->iCurrentMonth, 1, $this->iCurrentYear)));
    $iDateCurNext = mktime(0, 0, 0, $this->iCurrentMonth, min($this->iCurrentDay, $nbDayNext), $this->iCurrentYear);
    
    $iDayPrev  = date("j", strtotime("-1 month", $iDateCurPrev));
    $iMonthPrev  = date("n", strtotime("-1 month", $iDateCurPrev));
    $iAnneePrev = date("Y", strtotime("-1 month", $iDateCurPrev));
    
    $iDayNext  = date("j", strtotime("+1 month", $iDateCurNext));
    $iMonthNext  = date("n", strtotime("+1 month", $iDateCurNext));
    $iAnneeNext = date("Y", strtotime("+1 month", $iDateCurNext));
    
    $strHtml = "<table summary='' class='".$this->cssCalendar."' cellpadding='0' cellspacing='".$this->iCalCellSpacing."' border='0' align='".$this->strCalAlign."'>".
      "<thead>".
      "<tr>".
      ( $this->bTimeVisible == true ? "<td class='rowDimension' style='width:40px'></td>" : "" );
    
    for($k=0; $k<$this->iNbDayVisible; $k++) {
      $strHtml .= "<td class='rowDimension' style='width:".$iLargeurCellCal."px'></td>";
    }
    
    $strHtml .= "</tr>".
      "<tr>".
      ( $this->bTimeVisible == true ? "<td class='none'></td>" : "" ).
      "<td colspan='".$this->iNbDayVisible."'>".
      "<div>".
      ( $this->strCalTitle!=""
        ? $this->strCalTitle.( $this->bAffTitleAuto==true ? " - " : "" )
        : "" ).
      ( $this->bAffTitleAuto==true
        ? $this->ConvertDateToFrench(1, $this->iCurrentMonth, true)." ".$this->iCurrentYear
        : "" ).
      ( $this->bAffMenuBar==true
        ? "</div><div class='toolBar'>".$this->getHtmlCalMenu()
        : "" ).
      "</div>".
      "</td></tr>".
      "</thead>".
      "<tbody>".
      "<tr>".
      ( $this->bTimeVisible == true ? "<td class='none'></td>" : "" );
  
    $strUrlPrev = $this->strUrlPagine."&j=".$iDayPrev."&m=".$iMonthPrev."&a=".$iAnneePrev."&iModeAff=".$this->iMode;
    $strUrlNext = $this->strUrlPagine."&j=".$iDayNext."&m=".$iMonthNext."&a=".$iAnneeNext."&iModeAff=".$this->iMode;
      
    for($k=1; $k<=$this->iNbDayVisible; $k++) {
      $strHtml .= "<td class='header'>".
        ( $k==1
          ? "<a onclick=\"AlkGoUrl('".$strUrlPrev."')\" title='Mois précédent'><<</a>&nbsp;&nbsp;"
          : "" ).
        $this->ConvertDateToFrench(0, $k).
        ( $k==$this->iNbDayVisible
          ? "&nbsp;&nbsp;<a onclick=\"AlkGoUrl('".$strUrlNext."')\" title='Mois suivant'>>></a>"
          : "" ).
        "</td>";
    }

    $cptLayer = 0;
    $iNbCellCalendar = 35;
    if( ($dayone==6 || $dayone==7) && $this->GetDaysInMonth($this->iCurrentMonth, $this->iCurrentYear)>=30 )
      $iNbCellCalendar = 42;
    if( $dayone==1 && $this->GetDaysInMonth($this->iCurrentMonth, $this->iCurrentYear)==28 )
      $iNbCellCalendar = 28;
    
    for($i=1; $i<=$iNbCellCalendar; $i++) {
      $iDelta = -$dayone+1;
      $iDayTest = date("w", strtotime(($iDelta<0 ? $iDelta : "+".$iDelta)." day", mktime(0, 0, 0, $this->iCurrentMonth, $i, $this->iCurrentYear)));
      if( $iDayTest==0 )
        $iDayTest = 7;
        
        
      if( $iDayTest <= $this->iNbDayVisible ) {
        if( $i % 7 == 1 ) {
          $strUrlWeek = $this->strUrlPagine."&j=".($i-$dayone+1)."&m=".$this->iCurrentMonth.
            "&a=".$this->iCurrentYear."&iModeAff=".ALK_CAL_MODE_WEEK;
          $strHtml .= "</tr><tr>".
            ( $this->bTimeVisible == true
              ? "<td class='header'>".
                "<a onclick=\"AlkGoUrl('".$strUrlWeek."')\" style='text-decoration: none'".
                " title='Afficher le calendrier de cette semaine'>".
                "S ".date("W", strtotime(($iDelta<0 ? $iDelta : "+".$iDelta)." day", mktime(0, 0, 0, $this->iCurrentMonth, $i, $this->iCurrentYear))).
                "</a></td>"
              : "" );
        }
        $iNumDay = $i-$dayone+1;
        $bJrFerie = $this->isClosedDay($iNumDay, $this->iCurrentMonth, $this->iCurrentYear);
        if( !$bJrFerie && $i < $this->GetDaysInMonth($this->iCurrentMonth, $this->iCurrentYear)+$dayone && $i >= $dayone ) {
          $strUrlDay = $this->strUrlPagine."&j=".$iNumDay."&m=".$this->iCurrentMonth.
            "&a=".$this->iCurrentYear."&iModeAff=".ALK_CAL_MODE_DAY;
          $dateDayCalStart = mktime(0, 0, 0, $this->iCurrentMonth, $iNumDay, $this->iCurrentYear);
          $dateDayCalEnd = mktime(23, 59, 0, $this->iCurrentMonth, $iNumDay, $this->iCurrentYear);
          
          $bOnClickCell = $this->bOnClickCell == true 
                      && ($this->bOnClickOnlyAfter == true 
                        ? $dateDayCalStart >= mktime(0, 0, 0, date("m", AlkFactory::getLocalDate()), date("d", AlkFactory::getLocalDate()), date("Y",AlkFactory::getLocalDate())) 
                        : true);
      
          $strHtml .= "<td ".
            " class=\"".( date("d/m/Y", $dateDayCalStart)===date("d/m/Y") ? "current" : "" )."\"".
            " valign='top' align='left' style='height:".$this->iCalCellHeight."px'".
            " onmouseover=\"onMouseOverCalCell(this, 'roll', ".($bOnClickCell ? "true" : "false").")\"".
            " onmouseout=\"onMouseOutCalCell(this)\"".
            ( $bOnClickCell
              ? " onclick=\"onClickCalCell".$this->name."(".$iNumDay.", ".$this->iCurrentMonth.", ".$this->iCurrentYear.", 12, 0)\""
              : "" ).
            ( $bOnClickCell
              ? " title=\"".$this->titleOnClickCell."\""
              : "" ).
            ">".
            "<div><a onclick=\"AlkGoUrl('".$strUrlDay."')\" style='text-decoration: none'".
            "  title='Afficher le calendrier de ce jour'>".
            ( $iNumDay < 10 ? "0" : "" ).$iNumDay."</a></div>";
          
          $strEvtDay="";
          foreach ($this->tabCalEvt as $evt) {
            $dateStartEvt = $this->getEvtTimestamp($evt, "deb");
            if( $evt["datefin"] != "" )
              $dateEndEvt = $this->getEvtTimestamp($evt, "fin");
            else
              $dateEndEvt = $dateStartEvt;

            if( ($dateStartEvt<=$dateDayCalStart and $dateDayCalEnd<=$dateEndEvt) ||
                ($dateDayCalStart<=$dateStartEvt && $dateStartEvt<=$dateDayCalEnd) ||
                ($dateDayCalStart<=$dateEndEvt && $dateEndEvt<=$dateDayCalEnd) ) {
              $strParamDiv = " class='".$this->cssCellCalendar."'".
                " style='background-color:".$evt["bgcolor"]."; width:".($iLargeurCellCal-$iBorderWidth)."; margin-bottom:1px;".$evt["csstxt"]."' ";
              $strEvtDay .= $this->getHtmlCalEvt($evt, $strParamDiv);
            }
          }
          
          $strHtml .=  $strEvtDay;
          
          if( $cptLayer == 0 ) {
            $strHtml .= $this->getHtmlLayerDesc();
            $cptLayer++;
          }
           
          $strHtml .= "</td>";
        } else {
          $strHtml .= "<td valign='top'".
            ( $bJrFerie
              ? " class='closed'><div align='left'>".( $iNumDay < 10 ? "0" : "" ).$iNumDay."</div><br/>Fermé"
              : " class='nullDate'>&nbsp;" ).
            "</td>";
        }
      }
    }
    
    $strHtml .= "</tr></tbody><tfoot><tr>".
      ( $this->bTimeVisible == true ? "<td class='none'></td>" : "" );
    
    for($k=1; $k<=$this->iNbDayVisible; $k++) {
      $strHtml .= "<td class='header'>".
        ( $k==1
          ? "<a onclick=\"AlkGoUrl('".$strUrlPrev."')\" title='Mois précédent'><< ".
            $this->ConvertDateToFrench(1, ($this->iCurrentMonth-1==0 ? 12 : $this->iCurrentMonth-1))."</a>"
          : "" ).
        ( $k==$this->iNbDayVisible
          ? "<a onclick=\"AlkGoUrl('".$strUrlNext."')\" title='Mois suivant'>".
            $this->ConvertDateToFrench(1, ($this->iCurrentMonth+1==13 ? 1 : $this->iCurrentMonth+1))." >></a>"
          : "" ).
        "</td>";
    }
    $strHtml .= "</tr></tfoot></table>";
    
    $this->AddScriptJs(ALK_ALKANET_ROOT_URL.ALK_ROOT_CLASSE."form/alkhtmlcalendar.js");

    return $strHtml;
  }
  
  /**
   *  Retourne le code html du calendrier annuel
   *
   * @return Retourne un string
   */
  protected function getHtmlYearlyCalendar()
  {
    $cptLayer = 0;

    $iBorderWidth = 9;

    $iCalWidth = ( $this->bTimeVisible==true 
                   ? $this->iCalWidth-22
                   : $this->iCalWidth );
    
    $iLargeurCellCal = floor(($iCalWidth-($this->iNbMonthVisible-1)*2) / $this->iNbMonthVisible);

    $nbDayPrev = date("t", strtotime(-$this->iNbMonthVisible." month", mktime(0, 0, 0, $this->iCurrentMonth, 1, $this->iCurrentYear)));
    $iDateCurPrev = mktime(0, 0, 0, $this->iCurrentMonth, min($this->iCurrentDay, $nbDayPrev), $this->iCurrentYear);
    $nbDayNext = date("t", strtotime($this->iNbMonthVisible." month", mktime(0, 0, 0, $this->iCurrentMonth, 1, $this->iCurrentYear)));
    $iDateCurNext = mktime(0, 0, 0, $this->iCurrentMonth, min($this->iCurrentDay, $nbDayNext), $this->iCurrentYear);
    
    $iDayPrev  = date("j", strtotime(-$this->iNbMonthVisible." month", $iDateCurPrev));
    $iMonthPrev  = date("n", strtotime(-$this->iNbMonthVisible." month", $iDateCurPrev));
    $iAnneePrev = date("Y", strtotime(-$this->iNbMonthVisible." month", $iDateCurPrev));
    
    $iDayNext  = date("j", strtotime("+".$this->iNbMonthVisible." month", $iDateCurNext));
    $iMonthNext  = date("n", strtotime("+".$this->iNbMonthVisible." month", $iDateCurNext));
    $iAnneeNext = date("Y", strtotime("+".$this->iNbMonthVisible." month", $iDateCurNext));
    
    $strHtml = "<table summary='' class='".$this->cssCalendar."' cellpadding='0' cellspacing='".$this->iCalCellSpacing."' border='0' align='".$this->strCalAlign."'>".
      "<thead>".
      "<tr>".
      ( $this->bTimeVisible == true ? "<td class='rowDimension' style='width:20px'></td>" : "" );

    for($k=0; $k<$this->iNbMonthVisible; $k++) {
      $strHtml .= "<td class='rowDimension' style='width:".$iLargeurCellCal."px'></td>";
    }

    $strHtml .= "</tr>".
      "<tr>".
      ( $this->bTimeVisible == true ? "<td class='none'></td>" : "" ).
      "<td colspan='".$this->iNbMonthVisible."'>".
      "<div>".
      ( $this->strCalTitle != ""
        ? $this->strCalTitle.( $this->bAffTitleAuto==true ? " - " : "" ) 
        : "" ).
      $this->iCurrentYear.
      ( $this->bAffMenuBar==true 
        ? "</div><div class='toolBar'>".$this->getHtmlCalMenu()
        : "" ).
      "<div>".
      "</td></tr>".
      "</thead>".
      "<tbody>".
      "<tr>".
      ( $this->bTimeVisible == true ? "<td class='none'></td>" : "" );
         
    $strUrlPrev = $this->strUrlPagine."&j=".$iDayPrev."&m=".$iMonthPrev."&a=".$iAnneePrev."&iModeAff=".$this->iMode."&nbmois=".$this->iNbMonthVisible;
    $strUrlNext = $this->strUrlPagine."&j=".$iDayNext."&m=".$iMonthNext."&a=".$iAnneeNext."&iModeAff=".$this->iMode."&nbmois=".$this->iNbMonthVisible;
    $strUrlDay = $this->strUrlPagine."&iModeAff=".ALK_CAL_MODE_MONTH."&j=1";
    
    for ($k=0; $k<$this->iNbMonthVisible; $k++) {
      $iM = date("n", strtotime("+".$k." month", mktime(0, 0, 0, $this->iCurrentMonth, 1, $this->iCurrentYear)));
      $iA = date("Y", strtotime("+".$k." month", mktime(0, 0, 0, $this->iCurrentMonth, 1, $this->iCurrentYear)));

      $strHtml .= "<td align='center' valign='top' class='header'>".
        ( $k==0 
          ? "<a onclick=\"AlkGoUrl('".$strUrlPrev."')\" title='".$this->iNbMonthVisible." mois précédents'><<</a>&nbsp;&nbsp;&nbsp; "
          : "").
        "<a style='text-decoration:none; border:none; padding:none' title='Afficher le calendrier de ce mois'".
        " onclick=\"AlkGoUrl('".$strUrlDay."&m=".$iM."&a=".$iA."')\">".
        $this->ConvertDateToFrench(1, $iM).
        ( $this->bAffMonthAnnee==true ? " ".$iA : "" ).
        "</a>".
        ( $k==$this->iNbMonthVisible-1
          ? "&nbsp;&nbsp;&nbsp;<a onclick=\"AlkGoUrl('".$strUrlNext."')\" title='".$this->iNbMonthVisible." mois suivants'>>></a>"
          : "" ).
        "</td>";
    }
        
    $strHtml .= "</tr>";

    // mémorise le nombre de jour par mois, le mois et l'année pour chaque colonne affichée
    $tabNbDays = array();
    for($k=0; $k<$this->iNbMonthVisible; $k++) {
      $iM = date("n", strtotime("+".$k." month", mktime(0, 0, 0, $this->iCurrentMonth, 1, $this->iCurrentYear)));
      $iA = date("Y", strtotime("+".$k." month", mktime(0, 0, 0, $this->iCurrentMonth, 1, $this->iCurrentYear)));
      $tabNbDays[$k]["nb"] = $this->GetDaysInMonth($iM, $iA);
      $tabNbDays[$k]["iM"] = $iM;
      $tabNbDays[$k]["iA"] = $iA;
    }    

    for($i=1; $i<=31; $i++) {
      $strHtml .= "<tr>".
        ( $this->bTimeVisible == true 
          ? "<td class='header' valign='top' align='left' style='height:".$this->iCalCellHeight."px'>".
            (strlen($i)<2 ? "0" : "" ).$i."</td>"
          : "" );
      
      for ($k=0; $k<$this->iNbMonthVisible; $k++) {
        $iM = $tabNbDays[$k]["iM"];
        $iA = $tabNbDays[$k]["iA"];
        $nbDaysInMonth = $tabNbDays[$k]["nb"];

        if( $i<=$nbDaysInMonth ) {
          $iJs = date("w", mktime(0, 0, 0, $iM, $i, $iA));
          $bJrFerie = $this->isClosedDay($i, $iM, $iA);
          
          if( !$bJrFerie ) {
            $strUrlDay = $this->strUrlPagine."&j=".$i."&m=".$iM."&a=".$iA."&iModeAff=".ALK_CAL_MODE_DAY;
            $dateDayCalStart = mktime(0, 0, 0, $iM, $i, $iA);
            $dateDayCalEnd = mktime(23, 59, 0, $iM, $i, $iA);
            
            $bOnClickCell = $this->bOnClickCell == true && ($this->bOnClickOnlyAfter == true ? $dateDayCalStart >= mktime(0, 0, 0, date("m",AlkFactory::getLocalDate()), date("d",AlkFactory::getLocalDate()), date("Y",AlkFactory::getLocalDate())) : true);

            $strHtml .= "<td".
              " class=\"".( date("d/m/Y", $dateDayCalStart)===date("d/m/Y") ? "current" : "" )."\"".
              " valign='top' align='left' style='height:".$this->iCalCellHeight."px'".
              " onmouseover=\"onMouseOverCalCell(this, 'roll', ".($bOnClickCell == true ? "true" : "false").")\"".
              " onmouseout=\"onMouseOutCalCell(this)\"".
              ( $bOnClickCell == true
                ? " onclick=\"onClickCalCell".$this->name."(".$i.", ".$iM.", ".$iA.", 12, 0)\""
                : "" ).
              ( $bOnClickCell == true
                ? " title=\"".$this->titleOnClickCell."\""
                : "" ).
              ">".
              "<div><a onclick=\"AlkGoUrl('".$strUrlDay."')\" style='text-decoration: none'".
              "  title='Afficher le calendrier de ce jour'>".
              $this->ConvertDateToFrench(2, $iJs, false, true).($this->bTimeVisible == true ? "" : " ".$i)."</a></div>";
          
            $strEvtDay="";
            foreach ($this->tabCalEvt as $evt) {
              $dateStartEvt = $this->getEvtTimestamp($evt, "deb");
              if( $evt["datefin"] != "" )
                $dateEndEvt = $this->getEvtTimestamp($evt, "fin");
              else
                $dateEndEvt = $dateStartEvt;

              if( ($dateStartEvt<=$dateDayCalStart and $dateDayCalEnd<=$dateEndEvt) ||
                  ($dateDayCalStart<=$dateStartEvt && $dateStartEvt<=$dateDayCalEnd) ||
                  ($dateDayCalStart<=$dateEndEvt && $dateEndEvt<=$dateDayCalEnd) ) {
                $strParamDiv = "  class='".$this->cssCellCalendar."'".
                  " style='background-color:".$evt["bgcolor"]."; width:".($iLargeurCellCal-$iBorderWidth)."; margin-bottom:1px;".$evt["csstxt"]."' ";
                $strEvtDay .= $this->getHtmlCalEvt($evt, $strParamDiv);
              }
            }
          
            $strHtml .=  $strEvtDay;
          
            if( $cptLayer == 0 ) {
              $strHtml .= $this->getHtmlLayerDesc();
              $cptLayer++;
            }
           
            $strHtml .= "</td>";
          } else {
            $strHtml .= "<td valign='top' class='closed'><div align='left'>".
              $this->ConvertDateToFrench(2, $iJs, false, true).($this->bTimeVisible == true ? "" : " ".$i)."</div><br/>Fermé</td>";
          }
        } else {
          $strHtml .= "<td valign='top' class='nullDate'>&nbsp;</td>";
        }
      }
      $strHtml .= "</tr>";
    }

    $strHtml .= "</tbody><tfoot><tr>".
      ( $this->bTimeVisible == true ? "<td class='none'></td>" : "" );

    for($k=0; $k<$this->iNbMonthVisible; $k++) {
      $iM = date("n", strtotime("+".$k." month", mktime(0, 0, 0, $this->iCurrentMonth, 1, $this->iCurrentYear)));
      $iA = date("Y", strtotime("+".$k." month", mktime(0, 0, 0, $this->iCurrentMonth, 1, $this->iCurrentYear)));

      $strHtml .= "<td>".
        ( $k==0
          ? "<a onclick=\"AlkGoUrl('".$strUrlPrev."')\" title='".$this->iNbMonthVisible." mois précédents'><<</a>&nbsp;&nbsp;&nbsp; "
          : "" ).
        "<a style='text-decoration:none; border:none; padding:none' title='Afficher le calendrier de ce mois'".
        " onclick=\"AlkGoUrl('".$strUrlDay."&m=".$iM."&a=".$iA."')\">".
        $this->ConvertDateToFrench(1, $iM).
        ( $this->bAffMonthAnnee==true ? " ".$iA : "" ).
        "</a>".
        ( $k==$this->iNbMonthVisible-1
          ? "&nbsp;&nbsp;&nbsp;<a onclick=\"AlkGoUrl('".$strUrlNext."')\" title='".$this->iNbMonthVisible." mois suivants'>>></a>"
          : "" ).
        "</td>";
    }
    $strHtml .= "</tr></tfoot></table>";

    $this->addScriptJs(ALK_ALKANET_ROOT_URL.ALK_ROOT_CLASSE."form/alkhtmlcalendar.js");
    
    return $strHtml;
  }
   
  /**
   *  Retourne le code html du formulaire complet
   *
   * @return Retourne un string
   */
  public function getHtml()
  {
    $this->addDefaultCss();
    $strHtml = "";
    switch( $this->iMode ) {
    case ALK_CAL_MODE_DAY:
      $strHtml .= $this->getHtmlDailyCalendar();
      break;
      
    case ALK_CAL_MODE_WEEK:
      $strHtml .= $this->getHtmlWeeklyCalendar();
      break;
      
    case ALK_CAL_MODE_MONTH:
      $strHtml .= $this->getHtmlMonthlyCalendar();
      break;
      
    case ALK_CAL_MODE_YEAR:
      $strHtml .= $this->getHtmlYearlyCalendar();
      break; 
    } 
    return $strHtml;      
  }

  /**
   *  Méthode abstraite non implémentée
   *        qui se charge de retourner le code html du ctrl de saisie seul
   *        Méthode appelée par GetHtml qui se chargera d'ajouter le code JS
   *        les textes before et after, les drapeaux, le bouton d'aide
   */
  protected function GetCtrlHtml($key=0, $tabLg=array()) { /** rien à faire */ }

}

?>