<?php
/*licence/ 

Module écrit, supporté par la société Alkante SAS <alkante@alkante.com>

Nom du module : Alkanet::Class::Form
Module fournissant les classes d'affichage Alkanet.
Ce module appartient au framework Alkanet.

Ce logiciel est régi par la licence CeCILL-C soumise au droit français et
respectant les principes de diffusion des logiciels libres. Vous pouvez
utiliser, modifier et/ou redistribuer ce programme sous les conditions
de la licence CeCILL-C telle que diffusée par le CEA, le CNRS et l'INRIA
sur le site http://www.cecill.info.

En contrepartie de l'accessibilité au code source et des droits de copie,
de modification et de redistribution accordés par cette licence, il n'est
offert aux utilisateurs qu'une garantie limitée. Pour les mêmes raisons,
seule une responsabilité restreinte pèse sur l'auteur du programme, le
titulaire des droits patrimoniaux et les concédants successifs.

A cet égard l'attention de l'utilisateur est attirée sur les risques
associés au chargement, à l'utilisation, à la modification et/ou au
développement et à la reproduction du logiciel par l'utilisateur étant
donné sa spécificité de logiciel libre, qui peut le rendre complexe à
manipuler et qui le réserve donc à des développeurs et des professionnels
avertis possédant des connaissances informatiques approfondies. Les
utilisateurs sont donc invités à charger et tester l'adéquation du
logiciel à leurs besoins dans des conditions permettant d'assurer la
sécurité de leurs systèmes et ou de leurs données et, plus généralement,
à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.

Le fait que vous puissiez accéder à cet en-tête signifie que vous avez
pris connaissance de la licence CeCILL-C, et que vous en avez accepté les
termes.

/licence*/

require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CLASSE."form/alkhtmlpanel.class.php");
require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CLASSE."form/alkhtmlblock.class.php");
require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CLASSE."form/alkformdata.class.php");

/**
 * @package Alkanet_Class_Form
 * @class AlkHtmlForm
 * 
 * Abstract -- classe d'affichage des blocs de controle d'un formulaire
 */
abstract class AlkHtmlForm extends AlkHtmlPanel
{  
  /** Application propriétaire du formulaire */
  protected $oAppli;
  
  /** Type d'onglet initial */
  protected $iTypeSheet;
  
  /** Indice d'onglet initial */
  protected $iSheet;
  
  /** Indice de sous-onglet initial */
  protected $iSSheet;
  
  
  /** mode du formulaire */
  protected $iMode;

  /** Ensemble des objets formdata du formulaire */
  protected $tabFormData; 

  /** Ensemble des controles du formulaire */
  protected $tabCtrls; 

  /** Block par défaut du formulaire */
  protected $oDefaultBlock; 

  /** Action du formulaire */
  protected $strAction;

  /** Méthode du formulaire */
  protected $iMethod; 

  /** vrai si upload nécessaire sur le formulaire */
  protected $bUpload;

  /** cible du formulaire */
  protected $target;

  /** vrai si vérification des droits sur le formulaire : si vrai et constante n'existe pas => pas de droit, =false par défaut */
  protected $bVerifFieldRight;

  /** classe css sur le formulaire */
  protected $cssForm;

  /** Identifie le bouton de type submit du formulaire. Un bouton par formulaire obligatoire */
  protected $tabSubmitButton = array();
  
  /** Identifie le bouton de type reset du formulaire. Un bouton par formulaire non obligatoire */
  protected $oResetButton = null;
  
  /**
   * false par défaut, true pour afficher le code js validator ou autre au côté du controle de saisie au lieu du header
   * utile pour les formulaires construits en dehors du contexte alkHtmlPage 
   */
  protected $bWriteJsInner;

  /**
   *  Constructeur par défaut
   *
   * @param oAppli    AlkAppli   Application propriétaire du formulaire
   * @param name      Nom du formulaire
   * @param label     Titre du formulaire (de la page)
   */
  public function __construct(AlkAppli &$oAppli, $name, $label="")
  {
    parent::__construct($name, $label);

    $this->oAppli       = $oAppli;
    $this->oAppli->getCurrentSheet($this->iTypeSheet, $this->iSheet, $this->iSSheet);
    
    $this->iMode     = ALK_FORM_MODE_READ;
    $this->strAction = "";
    $this->iMethod   = ALK_FORM_METHOD_GET; 
    $this->bUpload   = false;
    $this->target    = "";
    
    $this->bVerifFieldRight = false;
    $this->bWriteJsInner = false;
    
    $this->oDefaultBlock = $this->addBlock($this->name."_block0");

    $this->tabCtrls     = array();
    $this->tabFormData  = array();

    $this->cssForm = "formForm";
    
    $this->bWithComment = false;
    $this->tabCommentData = array();
    
    $this->setDefaultCss("alkhtmlform.css");
  }
  
  /**
   *  Etabli les propriétés du formulaire
   * 
   * @param iMode     Mode du formulaire : =1 ajout, =2 maj, =3 lecture
   * @param strAction Action du formulaire (url appelée sur validation)
   * @param iMethod   ALK_FORM_METHOD_GET (par défaut) ou ALK_FORM_METHOD_POST
   * @param bUpload   faux par défaut, vrai si upload nécessaire
   * 
   */
  protected function setProperties($iMode, $iMethod=ALK_FORM_METHOD_GET, $strAction="")
  {
    $this->iMode     = $iMode;
    $this->strAction = $strAction;
    $this->iMethod   = $iMethod; 
  }
  
  /**
   *  Abstract : Initialise les données (FormData) du formulaire selon un ou plusieurs types d'initialisation
   */
  abstract protected function setData($iTypeInit);
  
  /**
   *  Abstract : Appel des actions SQL sur les données du formulaire
   */
  abstract public function doSql();  
  
  /**
   *  Abstract : Définition du contenu affiché du formulaire
   */
  abstract protected function setContents();
  
  /**
   *  Initialise les données (FormData) du formulaire
   * 
   * @param iTypeInit Type(s) d'initialisation demandés
   * @param oDr       Datarow données de la base pour le formulaire
   * @param reqMethod Méthode de récupération de request
   */
  protected function initData($iTypeInit, $oDr, $reqMethod)
  {
    foreach ( $this->tabFormData as $oFormData ){
      $calcTypeInit = $iTypeInit;
      $iModeInit = parent::getTypeInit($calcTypeInit);
      while ( $iModeInit!=-1 ){
        switch ( $iModeInit ){
          case ALK_INIT_MODE_DATAROW :
            $oFormData->initData($iModeInit, $oDr, $reqMethod);
          break;
          
          case ALK_INIT_MODE_REQUEST :
            $oFormData->initData($iModeInit, $oDr, $reqMethod);
          break;
          
          case ALK_INIT_MODE_DEFAULT : 
          case ALK_INIT_MODE_DATASET :
          default : 
          break;
        }
        $iModeInit = parent::getTypeInit($calcTypeInit);
      }
    }
  }

  /**
   *  Evalue l'ajout d'un objet de la hiérarchie panel au tableau de panels courant
   * 
   * @param  oPanel   Objet de type panel que l'on souhaite ajouter au tableau des panels
   * @return boolean
   */
  protected function canAddPanel($oPanel)
  {
    return ( $oPanel->isTypeOf(self::ALK_CLASS_HTMLBLOCK) || get_class($oPanel)=="AlkHtmlPanel" );
  } 
  
  /**
   *  Ajoute et créé un onglet au panel
   * @param idSheet             Identifiant de l'onglet
   * @param sheetTitle          titre de l'onglet
   * @param oUrl                url sur l'onglet, chaine vide pour sélection par javascript
   *                            tableau associatif pour construire le token vers alkanet.php. le tableau doit comporter bPopup=true pour 
   *                            appeler alkanet_popup.php
   * @param bAddDefaultBlock    Indique si le block par défaut doit être affecté à cet onglet (false par défaut)
   * @return null ou AlkHtmlSheet
   */
  public function addSheet($idSheet, $sheetTitle, $oUrl="", $bAddDefaultBlock=false)
  { 
    $oSheet = parent::addSheet($idSheet, $sheetTitle, $oUrl);
    if ( $bAddDefaultBlock && !is_null($oSheet) && $this->equals($this->oDefaultBlock->getParent()) ){
      $oSheet->addPanel($this->oDefaultBlock);
    }
    return $oSheet;
  }
  
  /**
   *  ajoute un controle au formulaire (à la liste des controles)
   *
   * @param oCtrl   AlkHtmlCtrl   controle à ajouter
   */
  public function addCtrl(AlkHtmlCtrl $oCtrl)
  {
    $this->tabCtrls[$oCtrl->getGUID()] = $oCtrl;
  }
  
  /**
   *  ajoute une donnée de formulaire au formulaire
   *
   * @param oFormData   AlkFormData   donnée de formulaire à ajouter
   */
  public function addData(AlkFormData $oFormData)
  {
    $this->tabFormData[] = $oFormData;
  }
  
  /**
   *  Construit un bloc, l'ajoute au formulaire puis retourne une référence sur l'objet créé
   *
   * @param idBlock     identifiant du bloc
   * @param strTitle    titre du bloc (= chaine vide par défaut, dans ce cas, le cadre titre n'est pas affiché)
   * @param strDesc     description du bloc (= chaine vide par défaut)     
   * @param iWidthLabel largeur en px de la colonne label (=180 par défaut)
   * @param iWidthCtrl  largeur en px de la colonne ctrl  (=390 par défaut)
   * @param bMaintitle  vrai si titre principal, faux si titre de bloc classique (par défaut)
   * @return Retourne une référence sur le bloc créé
   */
  public function &addBlock($idBlock, $strTitle="", $strDesc="", $iWidthLabel="180", $iWidthCtrl="390", $bMaintitle=false)
  {
    $oBlock = $this->createBlock($idBlock, $strTitle, $strDesc, $iWidthLabel, $iWidthCtrl, $bMaintitle);
    if ( $this->addPanel($oBlock) )
      return $oBlock;
    return null;
  }
  
  /**
   *  Construit un bloc puis retourne une référence sur l'objet créé
   *
   * @param idBlock     identifiant du bloc
   * @param strTitle    titre du bloc (= chaine vide par défaut, dans ce cas, le cadre titre n'est pas affiché)
   * @param strDesc     description du bloc (= chaine vide par défaut)     
   * @param iWidthLabel largeur en px de la colonne label (=180 par défaut)
   * @param iWidthCtrl  largeur en px de la colonne ctrl  (=390 par défaut)
   * @param bMaintitle  vrai si titre principal, faux si titre de bloc classique (par défaut)
   * @return Retourne une référence sur le bloc créé
   */
  public function &createBlock($idBlock, $strTitle="", $strDesc="", $iWidthLabel="180", $iWidthCtrl="390", $bMaintitle=false)
  {
    $oBlock = new AlkHtmlBlock($this, $idBlock, $strTitle, $strDesc, $iWidthLabel, $iWidthCtrl, $bMaintitle);
    return $oBlock;
  }

  /**
   *  Retourne le code html l'entête du formulaire
   *
   * @return Retourne un string
   */
  protected function getHtmlHeader()
  {
    $this->addScriptJs(ALK_ALKANET_ROOT_URL.ALK_ROOT_CLASSE."form/alkhtmlform.js");
    
    $onSubmit = "";
    $glue = ( defined("ALK_B_CSSJS_MINIMIZE") && ALK_B_CSSJS_MINIMIZE==true ? " " : "\n" );
    foreach($this->tabSubmitButton as $submitName=>$submitAction) {
      $tabSubmitAction = explode(";", $submitAction);
      $tabSubmitAction = array_map("trim", $tabSubmitAction);
      $tabSubmitAction = array_diff($tabSubmitAction, array(""));
      $onSubmit .= 
        $glue.'if( this.alkActionType==\''.$submitName.'\' ) {' .
        $glue.'  return ('.str_replace("javascript:", "", implode(" && ", $tabSubmitAction)).');' .
        $glue.'}';
    }

    if ( $onSubmit!="" ) {
      $this->addJs(
        $glue.'function onSubmit_'.$this->getName().'() {'.
        $glue.'this.target = "'.$this->target.'";' .
        $glue.'this.action = "'.$this->strAction.'";' .
        $glue.$onSubmit.
        $glue.'return false;'.
        $glue.'};'
        );
      
      $this->addEvent("onsubmit", "return onSubmit_".$this->getName()."();");
    }

    $this->addDefaultCss();

    $strHtml = '<form name="'.$this->name.'" id="'.$this->name.'"'.
      ( $this->cssForm != "" ? ' class="'.$this->cssForm.'"' : "" ).
      ( $this->cssStyle != "" ? ' style="'.$this->cssStyle.'"' : "" ).
      ' action="'.$this->strAction.'"'.
      ' method="'.( $this->iMethod == ALK_FORM_METHOD_GET ? 'get' : 'post' ).'"'.
      ( $this->bUpload == true ? ' enctype="multipart/form-data"' : "" ).
      ( $this->target != "" ? ' target="'.$this->target.'"' : "" ).
      $this->GetHtmlEvent().
      '>';
    return $strHtml;
  }

  /**
   *  Retourne le code html de la fin du formulaire
   *
   *  Retourne un string
   */
  protected function getHtmlFooter()
  {
    return ( $this->bWriteJsInner ? $this->getHtmlJs().$this->getHtmlCss() : "" ).
      '</form>';
  }

  /**
   *  Retourne le code html du formulaire complet
   *
   * @return Retourne un string
   */
  public function getHtml()
  {
    $this->setContents();
    
    $this->checkCtrls();
    return parent::getHtml();
  }

  /**
   *  Construit un lien, l'ajoute au formulaire puis retourne une référence sur l'objet créé
   *
   * @param idLink        identifiant du lien
   * @param strButtonType type de bouton : button, buttonPage ou buttonForm  
   * @param strlink       Lien placé sur le texte ou l'image
   * @param strTxt        Texte associé au lien ou à l'image en info-bulle
   * @param strImg        Nom de l'image
   * @param strImgRol     Nom de l'image roll
   * @param strUrlBaseImg Url de base des images
   * @return Retourne une référence sur le lien créé
   */
  public function &addButton($idLink, $strLink, $strTxt, $strToolTip="", $bInline=false, 
                             $strButtonType="buttonPage", $strTarget="", $strPosition="after", $bAddOnRead=false)
  {
    if( $this->iMode==ALK_FORM_MODE_READ && !$bAddOnRead ) 
      return self::$oNull;
    return parent::addButton($idLink, $strLink, $strTxt, $strToolTip, $bInline, $strButtonType, $strTarget, $strPosition, $bAddOnRead);
  }
  
  /**
   *  Construit un lien Valider par défaut. 
   *        Le tableau des actions est une liste des fonctions javascript à appeler après la vérification
   *        des champs du formulaire. 
   *        Chaque fonction aura l'objet formulaire en paramètre et retournera un booléen
   * 
   * @param tabActionPostVerif  Liste des fonctions javascript à appeler après la vérification des champs du formulaire
   * @param strAction           Valeur de l'attribut action du formulaire (toute l'url)
   * @param strTxt              Texte du lien ("Valider" par défaut)
   * @param strToolTip          Info-bulle ("Valider la saisie" par défaut)
   * 
   * @return Retourne une référence sur le lien créé
   */
  public function &addValidButton($tabActionPostVerif=array(), $strAction="", $strTxt="Valider", $strToolTip="Valider la saisie")
  {
    $strTxt     = ( $strTxt == "Valider" ? _t("Valider") : $strTxt );
    $strToolTip = ( $strToolTip == "Valider la saisie" ? _t("Valider la saisie") : $strToolTip );
    
    $oButton =  $this->addButton("btValid", 
                                 "javascript:alkFormValidation('".$this->name."', '".$strAction."' ".
                                                               ( !empty($tabActionPostVerif) 
                                                                 ? ", '".implode("', '", $tabActionPostVerif)."'"
                                                                 : "").")", 
                                 $strTxt, 
                                 $strToolTip);
   $this->addSubmitButton($oButton);
   return $oButton;
  }
  
  /**
   *  Construit un lien Annuler par défaut. 
   * 
   * @param iTypeSheet        Type d'onglet
   * @param iSheet            Identifiant d'onglet
   * @param iSSheet           Identifiant de sous-onglet
   * @param strParamEncode    Paramètres supplémentaires encodés
   * @param strTxt            Texte du lien ("Annuler" par défaut)
   * @param strToolTip        Info-bulle ("Annuler les modifications" par défaut)
   * @param strParamDecode    Paramètres supplémentaires décodés
   * 
   * @return Retourne une référence sur le lien créé
   */
  public function &addCancelButton($iTypeSheet, $iSheet, $iSSheet, $strParam, 
                                   $strTxt="Annuler", $strToolTip="Annuler les modifications", $strParamDecode="")
  {
    $strTxt     = ( $strTxt == "Annuler" ? _t("Annuler") : $strTxt );
    $strToolTip = ( $strToolTip == "Annuler les modifications" ? _t("Annuler les modifications") : $strToolTip );
    $strUrl = ALK_ALKANET."?token=".$this->oAppli->getToken($iTypeSheet, $iSheet, $iSSheet, $strParam).$strParamDecode;
    return $this->addButton("btCancel", $strUrl, $strTxt, $strToolTip);
  }
  
  /**
   *  Verifie le droit de l'utilisateur pour l'ensemble des controles du formulaire
   *        Si pas de droit : change le mode du controle en -1 => ALK_FIELD_NOT_VIEW
   *
   * @return Retourne un bool
   */
  protected function verifRights()
  {
    foreach ( $this->tabCtrls as $oCtrl ){
      // regarde la valeur de la constante construite de la manière suivante :
      // majuscule([nom formulaire]_[nom champ])
      // valeur possible de la constante : 
      //   0 = aucun droit (le champ n'est pas visible)
      //   1 = lecture
      //   2 = modif (le champ est accessible en saisie pour la modification si ce bit est à 1)
      //   4 = ajout (le champ est accessible en saisie pour l'ajout si ce bit est à 1)
      $bAddCtrl = false;
      $strDroitChamp = "";
      $strDroitChamp = strtoupper($this->name."_"./*$this->iSheet."_".*/$oCtrl->getName());
      
  
      // Si la constante n'existe pas => pas de gestion de droit donc tous les droits
      $iDroitAgent = ( defined($strDroitChamp) ? constant($strDroitChamp) : ( $this->bVerifFieldRight ? 0 : 7) );
      switch( $this->iMode ) {
      case ALK_FORM_MODE_READ: // mode lecture
        if( $iDroitAgent & ALK_RIGHT_READ == ALK_RIGHT_READ ) {
          $oCtrl->setMode(1);
          $bAddCtrl = true;
        }
        break;
        
      case ALK_FORM_MODE_UPDATE: // mode modif
        if( ($iDroitAgent & ALK_RIGHT_UPDT) == ALK_RIGHT_UPDT ) {
          //$oCtrl->setMode(0);
          $bAddCtrl = true;
        } elseif( ($iDroitAgent & 1) == 1 ) {
          $oCtrl->setMode(1);
          $bAddCtrl = true;
        }
        break;
        
      case ALK_FORM_MODE_ADD: // mode ajout
        if( ($iDroitAgent & ALK_RIGHT_UPDT) == ALK_RIGHT_UPDT || ($iDroitAgent & ALK_RIGHT_ADD) == ALK_RIGHT_ADD ) {
          //$oCtrl->setMode(0);
          $bAddCtrl = true;
        } elseif( ($iDroitAgent & ALK_RIGHT_READ) == ALK_RIGHT_READ ) {
          $oCtrl->setMode(1);
          $bAddCtrl = true;
        }
        break;
      }
      if ( !$bAddCtrl ){
        if ( $this->iMode>ALK_FORM_MODE_READ )
          $oCtrl->setMode(-1); 
      }
    }
  }
  
  /**
   *  Vérifie que chaque controle du formulaire est dans un block du formulaire et 
   *        sinon, l'ajoute au block par défaut du formulaire
   */
  protected function checkCtrls()
  {
    $this->verifRights();
    foreach ($this->tabCtrls as $iCtrl=>$oCtrl) {
      $this->bUpload = $this->bUpload || $oCtrl->isUpload();
      if ( !$oCtrl->isInBlock() ) {
        $this->oDefaultBlock->addCtrl($oCtrl);
      }
    }
    $this->verifContents();
  }
  
  /**
   * @todo Méthode inutile. A supprimer.
   *  vérifie que le controle est non null et appelle la fonction addValidator sur le controle
   * 
   * @param oCtrl       Objet controle à vérifier
   * @param strFormName nom du formulaire
   * @param strType     type du controle de saisie : 'memo', 'text', 'int', 'date10', 'radio', 'check', 'select', 'mail', 'checkgroup',...
   * @param bRequire    =true si champ obligatoire, =false sinon
   * @param oMin        valeur minimale requise pour le controle (ou longueur min en fonction de strType)
   * @param oMax        valeur maximale requise pour le controle (ou longueur max en fonction de strType)
   * @param strExcept  valeurx qui font exceptions pour la vérification (séparées par "|" si plusieurs)
   * @param bStrict    true si la comparaison entre min et max est stricte, faux si éventuellement égale (=true par défaut)
   * @param iSheet     indice du layer htmlFormSheet (=-1 par défaut)
   */ 
  protected function setCtrlValidator($oCtrl, $iModeVerif, $oMin="", $oMax="", $strMsgErr="", $strExcept="", $bStrict=true, $iSheet="-1")
  {
    if ( is_null($oCtrl) ) return;
    $oCtrl->addValidator($iModeVerif, $oMin, $oMax, $strMsgErr, $strExcept, $bStrict, $iSheet);
  }
  
  /**
   *  vérifie que le controle est non null et appelle la fonction setIsRequire sur le controle
   * 
   * @param bRequire    =true si champ obligatoire, =false sinon
   */
  protected function setCtrlIsRequire($oCtrl, $bRequire)
  {
    if ( is_null($oCtrl) ) return;
    $oCtrl->setIsRequire($bRequire);
  }

  /**
   *  Modifie le mode du formulaire
   * @param iMode   nouveau mode du formulaire
   */
  public function setMode($iMode)
  {
    $this->iMode = $iMode;
  }
  
  /**
   *  initialise le panel à partir des requests
   */
  protected function initializeByRequest()
  {
    foreach ($this->tabPanels as $oPanel){
      $oPanel->initializeByRequest();
    }
    $this->GetTransformFromRequest();
  }
  
  /**
   *  initialise le panel à partir d'un datarow
   */
  protected function initializeByDatarow()
  {
    foreach ($this->tabPanels as $oPanel){
      $oPanel->initializeByDatarow();
    }
    $this->GetTransformFromDatarow();
  }
    
  /**
   *  Methode virtuelle chargée d'appliquer des formules aux champs 
   *        après une initialisation par request 
   */
  protected function GetTransformFromRequest() { }
  
  /**
   *  Methode virtuelle chargée d'appliquer des formules aux champs
   *        après une initialisation par datarow
   */
  protected function GetTransformFromDatarow() { }

  /**
   *  Retourne un tableau associatif conforme aux méthodes AlkQuery::_GetPartInsertSql et AlkQuery::_GetPartUpdateSql
   *        Le tableau retourné est de type array("pk" => $tabResPk, "field" => $tabResField);
   *        Les tableaux $tabResPk et $tabResField sont de type associatif fieldName => array(fieldType, fieldValue)
   *        fieldType  : = 0 string, 1 number, 2 date, 3 expression sql
   *        fieldValue : valeur brute php
   *
   *  tabFieldPk     Tableau contenant le nom des champs utilisés comme clé primaire
   *  tabFieldIgnore Tableau contenant le nom des champs à ignorer (= vide par défaut correspondant à aucun champ)
   *  tabFieldSelect Tableau contenant le nom des champs à prendre en compte (= vide par défaut correspondant à tous les champs)
   *  tabFieldAlias  Tableau contenant une liste d'association entre (champs-alias => champ-réel)
   * @return Retourne un array
   */
  function getTabQuery($tabFieldPk=array(), $tabFieldIgnore=array(), $tabFieldSelect=array(), $tabFieldAlias=array())
  {
    
    $tabResField = array();
    $tabResPk = array();
    foreach($this->tabFormData as $oFormData) {
      if ( !$oFormData->getIsVisible() ) continue;
      $strAttrib = $oFormData->name;
      $strField = strtoupper($strAttrib);
      if(   !in_array(strtoupper($strAttrib), $tabFieldIgnore) 
         && (empty($tabFieldSelect) || in_array(strtoupper($strAttrib), $tabFieldSelect)) 
      ){        
        
        if( array_key_exists($strField, $tabFieldAlias) ) {
          $strField = $tabFieldAlias[$strField];
        }
        if( in_array($strField, $tabFieldPk ) ) {
          $tabResPk[$strField] = array($oFormData->typeSql, $oFormData->getValueForQuery());
        } else {
          if( $oFormData->bMultiLanguage == true ) {
            foreach($this->tabLangue as $strKey => $tabLg) {
              $tabResField[$strField.$tabLg["bdd"]] = array($oFormData->typeSql, $oFormData->getValueForQuery($strKey));
            }
          } else {
            $tabResField[$strField] = array($oFormData->typeSql, $oFormData->getValueForQuery());
          }
        }
      }
    }
      
    return array("pk" => $tabResPk, "field" => $tabResField);
  }
  
  /**
   * @abstract 
   * Vérifie le contenu (appel de checkCtrlValue si souhaité)
   */
  protected function verifContents(){/**Si souhaité, appel de $this->checkCtrlValue();*/}
  
  /**
   * Si bVisible écrit le texte "Non renseigné" sur tous les controles en mode lecture ayant une valeur vide. 
   * Sinon setHidden(true) des controles. A appeler si souhaité
   * @param bVisible (false) Montre (avec Non renseigné) / Cache tous les champs vides en mode lecture
   */
  protected function checkCtrlValue($bVisible=false)
  {
    foreach ( $this->tabCtrls as $oCtrl ) {
      $value = $oCtrl->getProperty("value", "");
      if ( $oCtrl->getProperty("iMode", "0")==1 && empty($value) ){
        if ( $bVisible )
          $oCtrl->setProperty("defaultMsg", "<i>"._t("Non renseigné")."</i>");
        else
          $oCtrl->setHidden(true);
      }
    }
  }

  /**
   * Accesseur de l'attribut bWriteJsInner
   * @return boolean  
   */
  public function getWriteJsInner()
  {
    return $this->bWriteJsInner;
  }
  
  /**
   * Modificateur de l'attribut bWriteJsInner
   * @param bWriteJsInner   true par défaut pour écrire les validators et autre scripts js à l'interieur du formulaire,
   *                        false pour les écrire dans le header  
   */
  public function setWriteJsInner($bWriteJsInner=true)
  {
    $this->bWriteJsInner = $bWriteJsInner;
  }
 
  /**
   *  Genere puis retourne tous les evenements liés au controle html
   *
   * @return Retourne les événements concaténés dans une chaine pour être inclus dans
   *         dans un tag html de type controle de saisie
   */
  protected function getHtmlEvent()
  {
    return parent::getHtmlEvent();
  }
  
  /**
   * Détermine le bouton submit du formulaire. Ce bouton est obligatoire
   * @param AlkHtmlButton oSubmitButton
   */
  public function addSubmitButton($oSubmitButton, $buttonName="")
  {
    if ( is_null($oSubmitButton) || !$oSubmitButton->isTypeOf(self::ALK_CLASS_HTMLBUTTON)) return;
    
    $onSubmitAction = $oSubmitButton->getProperty("value", ""); // action js appelée sur le submit
    if ( $onSubmitAction!="" ) {
      $buttonName = ( $buttonName!="" 
                      ? $buttonName 
                      : mb_strtolower($oSubmitButton->getName()) );
      $this->tabSubmitButton[$buttonName] = $onSubmitAction;
      $oSubmitButton->addEvent("onclick", "setAlkFormActionType('".$buttonName."');");
    }

    $oSubmitButton->setButtonType(ALK_BUTTONTYPE_SUBMIT);
  }
  
  /**
   * Détermine le bouton reset du formulaire.
   * @param AlkHtmlButton oSubmitButton
   */
  public function setResetButton($oResetButton)
  {
    if ( is_null($oResetButton) || !$oResetButton->isTypeOf(self::ALK_CLASS_HTMLBUTTON)) return;
    $this->oResetButton = $oResetButton;
    $this->oResetButton->setButtonType(ALK_BUTTONTYPE_RESET);
  }
}

?>