<?php
/*licence/ 

Module écrit, supporté par la société Alkante SAS <alkante@alkante.com>

Nom du module : Alkanet::Class::Pattern
Module fournissant les classes de base Alkanet.
Ce module appartient au framework Alkanet.

Ce logiciel est régi par la licence CeCILL-C soumise au droit français et
respectant les principes de diffusion des logiciels libres. Vous pouvez
utiliser, modifier et/ou redistribuer ce programme sous les conditions
de la licence CeCILL-C telle que diffusée par le CEA, le CNRS et l'INRIA
sur le site http://www.cecill.info.

En contrepartie de l'accessibilité au code source et des droits de copie,
de modification et de redistribution accordés par cette licence, il n'est
offert aux utilisateurs qu'une garantie limitée. Pour les mêmes raisons,
seule une responsabilité restreinte pèse sur l'auteur du programme, le
titulaire des droits patrimoniaux et les concédants successifs.

A cet égard l'attention de l'utilisateur est attirée sur les risques
associés au chargement, à l'utilisation, à la modification et/ou au
développement et à la reproduction du logiciel par l'utilisateur étant
donné sa spécificité de logiciel libre, qui peut le rendre complexe à
manipuler et qui le réserve donc à des développeurs et des professionnels
avertis possédant des connaissances informatiques approfondies. Les
utilisateurs sont donc invités à charger et tester l'adéquation du
logiciel à leurs besoins dans des conditions permettant d'assurer la
sécurité de leurs systèmes et ou de leurs données et, plus généralement,
à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.

Le fait que vous puissiez accéder à cet en-tête signifie que vous avez
pris connaissance de la licence CeCILL-C, et que vous en avez accepté les
termes.

/licence*/

require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CLASSE."pattern/alkobject.class.php");
require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CLASSE."pattern/alkintsender.int.php");

/**
 * classe qui peut se connecter au service alkanet ou au service ovh
 */
class AlkSms extends AlkObject implements AlkIntSender
{
 
 protected $oClient = null;
 
 protected $smsSession;
 
 
 /** numero expéditeur du sms **/
 protected $numberFrom; 
 
 /** numero destinataire **/
 protected $tabTo;
 
 /** contenu du sms **/
 protected $strMessage;
 
 protected $bEnvoiSms ;
 
  
  /**
   * Constructeur par défaut
   */
  public function __construct($ts=ALK_SMS_SENDER_ALKANET)
  {    
    $options = array(
        'wsdl_cache' => 0,
        'trace' => 1
    );

    try {
       $service = ALK_ALKANET_SOAP."soapi/soapi-alk".strtolower($ts).".wsdl";
       $this->oClient = new SoapClient($service, $options);
    } catch (SoapFault $fault){
       $this->oClient = null;
    }
    
    if (!is_null($this->oClient)){
      $this->smsSession = $this->oClient->login(ALK_SMS_USER_ID, ALK_SMS_USER_PWD);
      $this->lastRequest =  $this->oClient->__getLastRequest();
    } else {
      $this->smsSession = null;
    }

    $this->numberFrom  = ALK_SMS_NUMBER;
    $this->tabTo       = array();
    $this->strMessage  = "";
    $this->bEnvoiSms   = ALK_B_SMS;
    
  }
  
  /**
   * vérifie si un soapClient a été bien défini
   * @return false si le client n'est pas défini
   */
  protected function isActivated()
  {
    return !is_null($this->oClient);
  }

  /**
   * établit la connexion
   * @param nic          the SMS account
   * @param password     the user login
   */
  public function login($nic, $password)
  {
    if( !$this->isActivated() ) {
       return false;
    }
    return  $this->oClient->login($nic, $password);
  }
  
  /**
   * se déconnecte
   */
  public function logout()
  {
    if( !$this->isActivated() ) {
       return false;
    }
    $this->oClient->logout($this->smsSession);
    $this->smsSession = "";
  }
   

  /**
   * ajoute un utilisateur sms
   * @param login : the user login
   * @param password : the user password
   * @param return false si échec
   */
  public function telephonySmsUserAdd($login, $password)
  {
    if( !$this->isActivated() ) {
       return false;
    }
    try {
      $this->oClient->telephonySmsUserAdd($this->smsSession, $login, $password);
    } catch(SoapFault $fault) {
      return false;
    }
  }
  
  /**
   * retourne la liste des utilisateurs
   * @param  smsAccount : the SMS account
   * @return string[]
   */
  public function telephonySmsUserList(){
    if( !$this->isActivated() ) {
       return false;
    }
    return $this->oClient->telephonySmsUserList($this->smsSession);
  }
  
  /**
   * met à jour un utilisateur (change le mot de passe)
   * @param login : the user login
   * @param password : the new user password
   */
  public function telephonySmsUserPassword($login, $password)
  {
    if( !$this->isActivated() ) {
       return false;
    }
    return $this->oClient->telephonySmsUserPassword($this->smsSession, $login, $password);
  }
  
  /** supprime un utilisateur sms
   * @param login : the user login
   */
  public function telephonySmsUserDel($login)
  {
    if( !$this->isActivated() ) {
       return false;
    }
    $this->oClient->telephonySmsUserDel($this->smsSession, $login);
  }
 
 /**
   * Add a sender to a SMS account
   * @param number : the sender number
   * @retour int
   */
  public function telephonySmsSenderAdd($number){
    if( !$this->isActivated() ) {
       return false;
    }
    return $this->oClient->telephonySmsSenderAdd($this->smsSession, $number);
  }
  
  /**
   * @param number : the sender number
   */
  public function telephonySmsSenderDelete($number){
    if( !$this->isActivated() ) {
       return false;
    }
    return $this->oClient->telephonySmsSenderDelete($this->smsSession, $number);
  }
  
 /**
   * Custom description for sender
   * @param number : the sender number
   * @param description : the new sender description
   */
  public function telephonySmsSenderDescription($number, $description )
  {
    if( !$this->isActivated() ) {
      return false;
    }
    return $this->oClient->telephonySmsSenderDescription($this->smsSession, $number, $description);
  }
  
 /**
   * List SMS account senders
   * @return telephonySmsSenderStruct[] 
   */
  public function telephonySmsSenderList(){
    if( !$this->isActivated() ) {
      return false;
    }
    return $this->oClient->telephonySmsSenderList($this->smsSession);
  }
  
  /**
   * Validate a SMS account sender (for mobile number)
   * @param number : the sender number
   * @param code : the validation code
   */
  public function telephonySmsSenderValidate($number, $code)
  {
    if( !$this->isActivated() ) {
      return false;
    }
    return $this->oClient->telephonySmsSenderValidate($this->smsSession, $number, $code);
  }
  
  /**
   * envoie un sms à un unique contact en utilisant un compte utilisateur
   * @param login : the user login
   * @param password : the user password
   * @param smsValidity : the maximum time -in minute(s)- before the message is dropped, defaut is 48 hours
   * @param smsClass : the sms class: flash(0),phone display(1),SIM(2),toolkit(3)
   * @param smsDeferred : the time -in minute(s)- to wait before sending the message, default is 0
   * @param smsPriority : the priority of the message (0 to 3), default is 3
   * @param smsCoding : the sms coding : 1 for 7 bit or 2 for unicode, default is 1
   * @param tag : an optional tag 
   * */
  public function telephonySmsUserSend($login, $password, $smsValidity=2880, $smsClass=2, $smsDeferred=0, $smsPriority=3, $smsCoding=2, $tag="")
  {
    if( ! ($this->isActivated() && $this->bEnvoiSms) ) {
      return false;
    }    
    if (count($this->tabTo)==1) {
      // envoi à un destinataire unique
      return $this->oClient->telephonySmsUserSend($login, $password, $this->numberFrom, $this->tabTo[0], 
                                                     $this->strMessage, $smsValidity, $smsClass, $smsDeferred, $smsPriority, $smsCoding, $tag);
    }
    
  }
  
  /**
   * envoie un sms à plusieurs contacts en utilisant un compte utilisateur
   * @param login : the user login
   * @param password : the user password
   * @param smsValidity : the maximum time -in minute(s)- before the message is dropped, defaut is 48 hours
   * @param smsClass : the sms class: flash(0),phone display(1),SIM(2),toolkit(3)
   * @param smsDeferred : the time -in minute(s)- to wait before sending the message, default is 0
   * @param smsPriority : the priority of the message (0 to 3), default is 3
   * @param smsCoding : the sms coding : 1 for 7 bit or 2 for unicode, default is 1
   * @param tag : an optional tag 
   * @return the SMS id 
   * */
  public function telephonySmsUserMultiSend($login, $password, $smsValidity=2880, $smsClass=2, $smsDeferred=0, $smsPriority=3, $smsCoding=2, $tag=""){
    if( ! ($this->isActivated() && $this->bEnvoiSms) ){
      return false;
    }
    
    if (count($this->tabTo)>1) {
      return $this->oClient->telephonySmsUserMultiSend($login, $password, $this->numberFrom, $this->tabTo, 
                                                     $this->strMessage, $smsValidity, $smsClass, $smsDeferred, $smsPriority, $smsCoding, $tag);
    }
  }
  
  /**
   * retourne la liste des consommations pour tous les utilisateurs
   * @param login the user login
   * @param password the user password
   * @param smsId the SMS id - optional
   * @param user the SMS user - optional
   * @param pagingStart paging: the first id, default is 0
   * @param pagingLimit paging: the limit, default is none
   * @param sortField sort: the field type (status|date|text|numberTo|numberFrom|user|tag)
   * @param sortOrder sort: the order (asc|desc) 
   * @param tag if specified show only sms history for given tag
   * @return telephonySmsHistoryStruct[] 
   */
  public function telephonySmsUserHistory($login, $password, $smsId="", $user="", $pagingStart=0, $pagingLimit="", $sortField="date", $sortOrder="desc", $tag="")
  {
    if( !$this->isActivated() ) {
      return false;
    }
    return $this->oClient->telephonySmsUserHistory($login, $password, $smsId, $user, $pagingStart, $pagingLimit, $sortField, $sortOrder, $tag);
  }
  
  /**
   * retourne la liste des consommations pour tous les utilisateurs
   * @param smsId the SMS id - optional
   * @param user the SMS user - optional
   * @param pagingStart paging: the first id, default is 0
   * @param pagingLimit paging: the limit, default is none
   * @param sortField sort: the field type (status|date|text|numberTo|numberFrom|user|tag)
   * @param sortOrder sort: the order (asc|desc) 
   * @param tag if specified show only sms history for given tag
   * @return telephonySmsHistoryStruct[] 
   */
  public function telephonySmsHistory($smsId="", $user="", $pagingStart=0, $pagingLimit="", $sortField="date", $sortOrder="desc", $tag="")
  {
    if( !$this->isActivated() ) {
      return false;
    }
    return $this->oClient->telephonySmsHistory($this->smsSession, $smsId, $user, $pagingStart, $pagingLimit, $sortField, $sortOrder, $tag);
  }
  
  /**
   * Get the available credit quota for an user
   * @param  login the user login
   * @return telephonySmsUserQuotaStruct
   */
  public function telephonySmsGetUserQuota($login){
    if( !$this->isActivated() ) {
      return false;
    }
    return $this->oClient->telephonySmsGetUserQuota($this->smsSession, $login);
  }
  
  /**
   * Set the available credit quota for an user
   * @param  login the user login
   * @param  status active or inactive quota (active|inactive) 
   * @param  quotaDiff the differential quota changed
   */
  public function telephonySmsSetUserQuota($login, $status="active", $quotaDiff="0"){
    if( !$this->isActivated() ) {
      return false;
    }
    $this->oClient->telephonySmsSetUserQuota($this->smsSession, $login, $status, $quotaDiff);
  }
  
    /**
   * Display quota notification info
   * @param login : the user login
   * @return telephonyNotificationSmsUserStruct
   */
  public function telephonySmsGetQuotaNotification($login)
  {
    if( !$this->isActivated() ) {
      return false;
    }
    return $this->oClient->telephonySmsGetQuotaNotification($this->smsSession, $login);
  }
  
 /**
   * Display quota notification info
   * @param login : the user login
   * @param password : the user password
   * @return telephonyNotificationSmsUserStruct
   */
  public function telephonySmsGetQuotaNotificationForUser($login, $password)
  {
    if( !$this->isActivated() ) {
      return false;
    }
    return $this->oClient->telephonySmsGetQuotaNotificationForUser($login, $password);
  }
  
  /**
   * Change quota notification info
   * @param login : the user login
   * @param support : the support where the alert is sent (mail|sms|both)
   * @param alertNumber : the number phone recipient in case of sms support
   * @param alertThreshold : the value of quota reached before the notification is sent (-1 to disable notification) 
   */
  public function telephonySmsSetQuotaNotification($login, $support="sms", $alertNumber="ALK_SMS_NUMBER", $alertThreshold="0")
  {
    if( !$this->isActivated() ) {
      return false;
    }
    $this->oClient->telephonySmsSetQuotaNotification($this->smsSession, $login, $support, $alertNumber, $alertThreshold);
    
  }
  
  /**
   * Get the SMS account credit left
   * @return string
   */
  public function telephonySmsCreditLeft(){
    if( !$this->isActivated() ) {
      return false;
    }
    return $this->oClient->telephonySmsCreditLeft($this->smsSession);
  }
  
  /**
   * Get the SMS account credit left with a user
   * @param login : the user login
   * @param password : the user password
   * @return string
   */
  public function telephonySmsUserCreditLeft($login, $password){
    if( !$this->isActivated() ) {
      return false;
    }
    return $this->oClient->telephonySmsUserCreditLeft($this->smsSession, $login, $password);
  }
  
  /**
   * retourne la liste des comptes
   * @return string[]
   */
  public function telephonySmsAccountList(){
    if( !$this->isActivated() ) {
      return false;
    }
    return $this->oClient->telephonySmsAccountList($this->smsSession);
  }
   
  
 /**
  * affecte le le contenu du sms
  * @param strMessage contenu du sms
  */
   
  public function setBody($strMessage)
  {
    $this->strMessage = $strMessage;
  }
  
  /**
   * affecte l'expéditeur du message
   * @param strName
   * @param numberFrom
   */
  public function setFrom($strName="", $numberFrom)
  {
    $this->numberFrom = $numberFrom;
  }
  
  /**
   * affecte le destinataire du message
   * @param string numberTo
   */
  public function addTo($strName, $numberTo, $tabAssoc=array())
  {
    $this->tabTo[] = $numberTo;
  }
  
}