<?php
/*licence/ 

Module écrit, supporté par la société Alkante SAS <alkante@alkante.com>

Nom du module : Alkanet::Module::Annu
Module annuaire Alkanet.
Ce module appartient au framework Alkanet.

Ce logiciel est régi par la licence CeCILL-C soumise au droit français et
respectant les principes de diffusion des logiciels libres. Vous pouvez
utiliser, modifier et/ou redistribuer ce programme sous les conditions
de la licence CeCILL-C telle que diffusée par le CEA, le CNRS et l'INRIA
sur le site http://www.cecill.info.

En contrepartie de l'accessibilité au code source et des droits de copie,
de modification et de redistribution accordés par cette licence, il n'est
offert aux utilisateurs qu'une garantie limitée. Pour les mêmes raisons,
seule une responsabilité restreinte pèse sur l'auteur du programme, le
titulaire des droits patrimoniaux et les concédants successifs.

A cet égard l'attention de l'utilisateur est attirée sur les risques
associés au chargement, à l'utilisation, à la modification et/ou au
développement et à la reproduction du logiciel par l'utilisateur étant
donné sa spécificité de logiciel libre, qui peut le rendre complexe à
manipuler et qui le réserve donc à des développeurs et des professionnels
avertis possédant des connaissances informatiques approfondies. Les
utilisateurs sont donc invités à charger et tester l'adéquation du
logiciel à leurs besoins dans des conditions permettant d'assurer la
sécurité de leurs systèmes et ou de leurs données et, plus généralement,
à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.

Le fait que vous puissiez accéder à cet en-tête signifie que vous avez
pris connaissance de la licence CeCILL-C, et que vous en avez accepté les
termes.

/licence*/


require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CLASSE."pattern/alkquery.class.php");

/**
 * @package Alkanet_Module_Annu
 *
 * @class AlkQueryAnnu 
 * Ensemble des requetes liées à l'application annuaire
 * Les méthodes intègrent également les actions ldap en selection et recherche
 * Ces méthodes ne vérifie pas ALK_LDAP. La vérif est à effectuer avant l'appel
 */
class AlkQueryAnnu extends AlkQuery
{
  /** référence sur la connexion ldap */
  protected $connLdap;
  
  /** tableau associatif annuaire ldap-alkanet */
  protected $tabLdapAssoc;
  
  /** 
   * Constructeur 
   * 
   */
  public function __construct(&$oDb)
  {
    parent::__construct($oDb);
    
    // connLdap is null if ALK_LDAP==false
    $this->connLdap = AlkFactory::getLdapConn();
    $this->tabLdapAssoc = ( ALK_LDAP ? unserialize(ALK_LDAP_ASSOC_ANNU) : array() );
  }

  /**
   *  Retourne le sigle associé au service_id
   *        Retourne la valeur de ALK_DEFAULTPWD_ANNU si pas trouvé
   * @param service_id  identifiant du service
   * @param bShort      true par défaut pour retourner le sigle, false pour l'intitule long
   * @return string 
   */
  public function getStr_ServiceSigle($service_id, $bShort=true)
  {
    $strSigle = ALK_ANNU_DEFAULTPWD;
    $strSql = "select SERVICE_INTITULE_COURT, SERVICE_INTITULE".
      " from SIT_SERVICE ".
      " where SERVICE_ID=".$service_id;
    $dsServ = $this->dbConn->InitDataSet($strSql);
    if( $drServ = $dsServ->getRowIter() ) {
      $strSigle = ( $bShort 
                    ? $drServ->getValueName("SERVICE_INTITULE_COURT")
                    : $drServ->getValueName("SERVICE_INTITULE") );
    }
    return $strSigle;
  }

  /**
   *  Retourne le type de service etat=1 ou non etat=0
   * @param service_id  identifiant du service
   * @return int
   */
  public function getStrServiceEtat($service_id)
  {
    $strRes = "0";
    $strSql = "select SERVICE_ETAT ".
      "from SIT_SERVICE ".
      "where SERVICE_ID=".$service_id;
    $dsServ = $this->dbConn->InitDataSet($strSql);
    if( $drServ = $dsServ->getRowIter() )
      $strRes = $drServ->getValueName("SERVICE_ETAT");

    return $strRes;
  }

  /**
   * Retourne l'identifiant du service de base (niveau 0)
   * retourne chaine vide si pas trouvé.
   * 07_service_list.php
   */
  function GetStr_idServiceBase()
  {
    $strRes = "";
    $strSql = "select SERVICE_ID from SIT_SERVICE where SERVICE_NIVEAU=0";
    $dsServ = $this->dbConn->InitDataSet($strSql);
    if( $drServ = $dsServ->getRowIter() )
      $strRes = $drServ->getValueName("SERVICE_ID");

    return $strRes;
  }

  /**
   * Retourne l'intitule du profil
   * @param profil_id  identifiant du profil
   * @return string
   */
  public function GetStr_profilIntitule($profil_id)
  {
    $strRes = "";
    $strSql = "select PROFIL_INTITULE from SIT_PROFIL where PROFIL_ID=".$profil_id;
    $dsProfil = $this->dbConn->InitDataSet($strSql);
    if( $drProfil = $dsProfil->getRowIter() )
      $strRes = $drProfil->getValueName("PROFIL_INTITULE");

    return $strRes;
  }

  /**
   * Retourne le nom de l'agent
   * 
   * @param agent_id  Identifiant de l'agent
   * @param bSens     =true pour NOM Prenom, = false pour Prenom NOM (par défaut)
   */
  function GetStr_agentNomPrenom($agent_id, $bSens=false)
  {
    $strRes = "";
    $strSql = "select AGENT_PRENOM, AGENT_NOM from SIT_AGENT where AGENT_ID=".$agent_id;
    $dsAgent = $this->dbConn->InitDataSet($strSql);
    if( $drAgent = $dsAgent->getRowIter() ) {
      $strRes = ( $bSens
                  ? trim($drAgent->getValueName("AGENT_NOM")." ".$drAgent->getValueName("AGENT_PRENOM"))
                  : trim($drAgent->getValueName("AGENT_PRENOM")." ".$drAgent->getValueName("AGENT_NOM")) );
    }
    return $strRes;
  }
  
  /**
   * récupère les données d'un agent via son mail
   * @param strMail  adresse mail
   * @return AlkDs
   */
  public function getDsAgentByMail($strMail)
  {
     $strSql = "select * from SIT_AGENT where AGENT_MAIL='".$strMail."'";
     return $this->dbConn->InitDataSet($strSql);
  }
  
  /**
   * Retourne la liste des services de niveau iNiv fils de idServPere pour un combobox
   * @param iNiv        Niveau de service souhaité
   * @param idServPere  identifiant du service parent ou liste d'identifiants séparés par une virgule
   * @param bLong       fixe une longueur limite
   * @param lgMax       longueur max, =55 par défaut
   * @param iEtat       filtre les services de type etat si !="", ="" par défaut
   * @param $dept_id    id de l'entité géographique liée, =-2 ou -1 si non utilisé, =-2 par défaut
   */
  public function getDsListeServiceForCombo($iNiv="1", $idServPere="-1", $bLong=true, $lgMax=55, $iEtat="", $dept_id="-2", $idServSelect="-1")
  {
    $strChamp = "SERVICE_INTITULE_COURT";
    if( $iNiv != "1" ) 
      $strChamp = $this->dbConn->GetSubstring("SERVICE_INTITULE", 0, $lgMax);

    if( $bLong == true ) {
	    $strChamp = "SERVICE_INTITULE";
	    if( $lgMax != 55 )
	      $this->dbConn->GetSubstring("SERVICE_INTITULE", 0, $lgMax);
	  }

    $strSql = "select ".$strChamp." as intitule, SERVICE_ID".
      " from SIT_SERVICE".
      " where SERVICE_NIVEAU=".$iNiv.
      " and SERVICE_VALIDE =1".
      ( $iNiv!="1" ? " and SERVICE_PERE in (".$idServPere.")" : "" ).
      ( $iEtat!="" ? " and SERVICE_ETAT=".$iEtat : "" ).
      ( $dept_id!="-2" && $dept_id!="-1" ? " and DEPT_ID=".$dept_id : "" ).
      ( $idServSelect!="-1" ? " and SERVICE_ID=".$idServSelect : "" ).
      " order by ".$strChamp;
    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   *  Retourne la liste de tous les services pour un combobox
   *        01_recherche_form.php, 02_agent_add_form.php
   *
   * @param 
   * @return Retourne un dataSet
   */
  function GetDs_listeAllServiceForCombo($lgMax=-1)
  {
    $strChamp = "SERVICE_INTITULE";
    if( $lgMax != -1 )
      $strChamp = $this->dbConn->GetSubstring("SERVICE_INTITULE", 0, $lgMax);

    $strSql = "select ".$strChamp." as intitule, SERVICE_ID, SERVICE_NIVEAU".
      " from SIT_SERVICE".
      " where SERVICE_NIVEAU>0".
      " order by ".$strChamp;

    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   *  Retourne la filiation complete d'un agent ordonnée par niveau
   *        Retourne null si l'agent connecté n'est pas administrateur de l'agent_id
   * @param agent_id   identifiant de l'agent à traiter
   * @param user_priv   privilège de l'agent connecté
   * @param userServ_id identifiant du service de niveau 1 de l'agent connecté
   * @return AlkDs
   */
  public function getDsListServiceByAgent($agent_id, $user_priv="", $userServ_id="")
  {
    $strWhere = ( ($user_priv=="" && $userServ_id=="") ||
                  (($user_priv && ALK_PRIV_ANNU_ALL)==ALK_PRIV_ANNU_ALL)
                  ? ""
                  : ( ($user_priv && ALK_PRIV_ANNU_SERV)==ALK_PRIV_ANNU_SERV
                      ? " and s.SERVICE_RACINE=".$userServ_id
                      : " and 1=0" ));

    $strSql = "select s.SERVICE_ID, s.SERVICE_NIVEAU, s.SERVICE_INTITULE, s.SERVICE_INTITULE_COURT, s.SERVICE_LOGO".
      " from SIT_SERVICE s".
      "   inner join SIT_AGENT_SERVICE ase on ase.SERVICE_ID=s.SERVICE_ID".
      " where ase.AGENT_ID=".$agent_id.
      $strWhere.
      " order by s.SERVICE_NIVEAU";
    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   *  Retourne la filiation complete d'un agent ordonnée par niveau
   *        Retourne null si l'agent connecté n'est pas administrateur de l'agent_id
   * @param agent_id   identifiant de l'agent à traiter
   * @param user_priv   privilège de l'agent connecté
   * @param userServ_id identifiant du service de niveau 1 de l'agent connecté
   * @return AlkDs
   */
  public function getServiceLdapUidByAgent($agent_id, $user_priv="", $userServ_id="")
  {
    $strWhere = ( ($user_priv=="" && $userServ_id=="") ||
                  (($user_priv && ALK_PRIV_ANNU_ALL)==ALK_PRIV_ANNU_ALL)
                  ? ""
                  : ( ($user_priv && ALK_PRIV_ANNU_SERV)==ALK_PRIV_ANNU_SERV
                      ? " and s.SERVICE_RACINE=".$userServ_id
                      : "-1" ));
    if( $strWhere == "-1" ) 
      return null;

    $strSql = "select s.SERVICE_ID, s.SERVICE_NIVEAU, s.SERVICE_INTITULE, s.SERVICE_INTITULE_COURT".
      " from SIT_SERVICE s".
      "   inner join SIT_AGENT_SERVICE ase on ase.SERVICE_ID=s.SERVICE_ID".
      " where ase.AGENT_ID=".$agent_id.
      $strWhere.
      " order by s.SERVICE_NIVEAU";
    $dsServ = $this->dbConn->InitDataSet($strSql);
    $nbElt = $dsServ->getCountDr();
    if( $nbElt > 0 ){
      $i = 1;
      while( $drService = $dsServ->getRowIter() ) {
        
        if($nbElt == $i){
          $service_id = $drService->getValueName("SERVICE_ID");
          $strSql="select s.SERVICE_LDAPUID".
            " from SIT_SERVICE s ".
            " where SERVICE_ID =".$service_id;
          $dsServiceIUD = $this->dbConn->InitDataSet($strSql);
          if( $drServiceIUD = $dsServiceIUD->getRowIter() ) {
            $serviceLdapIud = $drServiceIUD->getValueName("SERVICE_LDAPUID"); 
          }  
   
          return $serviceLdapIud; 
        }
        $i++;
      }
    }else{
      return null;
    }
  }


  
  /**
   * Retourne la fiche d'un service
   * 07_service_list.php, 07_service_form.php
   */
  public function getDs_ficheService($service_id)
  {
    $strSql = "select s.*, ".
      $this->dbConn->getConcat("r.AGENT_NOM", "' '", "r.AGENT_PRENOM")." as responsable, ".
      $this->dbConn->getDateFormat("DD/MM/YYYY", "s.SERVICE_DATE_CREATION", false). "as SERVICE_DATE_CREATION". 
      " from SIT_SERVICE s".
      "   left join SIT_AGENT r on s.AGENT_ID=r.AGENT_ID".
      " where SERVICE_ID=".$service_id;
    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   * Retourne la liste des départements visibles
   * @param bAffDept  true pour afficher le nom département, false pour un tiret
   * @return AlkDs
   */
  public function getDsListeDptForCombo($bAffDept=false, $bAffDept0=true)
  {
    $strRegion = ( $bAffDept ? "r&eacute;gion" : "R&eacute;gion" );
    $strSql = "select ".
      $this->dbConn->getConcat(
        $this->dbConn->compareSql("DEPT_PERE", "=", "0", 
                                  $this->dbConn->compareSql("DEPT_ID", "=", "0", "''", "'".$strRegion." '"), 
                                  "'&nbsp;&nbsp;".( $bAffDept ? "d&eacute;partement" : "-")." '"),
        "DEPT_INTITULE")." as DEPT_LABEL, DEPT_ID, DEPT_PERE, DEPT_INTITULE".
      " from SIT_DEPT".
      " where DEPT_VISIBLE=1".
      ( !$bAffDept0 ? " and DEPT_ID>0" : "" ).
      " order by DEPT_PERE, DEPT_LABEL";
    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   *  Retourne la liste des missions pour un combobox
   * @param iWidthMax  lg max de l'intitulé, =55 par défaut
   * @return AlkDs
   */
  public function GetDsListMissionForCombo($iWidthMax=55)
  {
    $strSql = "select ".$this->dbConn->GetSubstring("MISSION_INTITULE", 0, $iWidthMax).", MISSION_ID ".
      "from SIT_MISSION order by MISSION_INTITULE";

    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   * Retourne la liste des missions d'un agent pour un combobox
   * @param agent_id  identifiant de l'agent à traiter
   * @param bLong     true pour tronquer l'intitulé à 55 caractères, false intitulé entier (par défaut)
   * @param bExcept   false par défaut, true pour retourner la liste des missions non associées à l'agent
   * @return AlkDs
   */
  public function getDsListeMissionByAgentForCombo($agent_id, $bLong=false, $bExcept=false)
  {
    $strChamp = "m.MISSION_INTITULE";
    if( $bLong == false )
      $strChamp = $this->dbConn->GetSubstring("m.MISSION_INTITULE", 0, 55)." as MISSION_INTITULE";

    if( !$bExcept ) {
      $strSql = "select ".$strChamp.", m.MISSION_ID".
        " from SIT_AGENT_MISSION am".
        "   inner join SIT_MISSION m on am.MISSION_ID=m.MISSION_ID".
        " where am.AGENT_ID=".$agent_id. 
        " order by m.MISSION_INTITULE";
    } else {
      $strSql = "select ".$strChamp.", m.MISSION_ID".
        " from SIT_MISSION m".
        " where m.MISSION_ID not in (select MISSION_ID from SIT_AGENT_MISSION where AGENT_ID=".$agent_id.")".
        " order by m.MISSION_INTITULE";
    }
    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   * Retourne la fiche d'une mission
   * 06_mission_form.php
   */
  function GetDs_ficheMission($mission_id)
  {
    $strSql = "select * from SIT_MISSION where MISSION_ID=".$mission_id;
      
    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   *  Retourne la liste des profils pour un combobox
   * @param iWidth            longueur max de l'intitulé affiché, =55 par défaut
   * @param cont_id           identifiant de l'espace, =0 pour un profil principal, >0 pour un profil lié à un espace, <0 pour filtrer 0 et -cont_id
   * @param profil_id_exclude identifiant du profil à exclure, =-1 par défaut pour non prise en compte 
   * @return AlkDs
   */
  public function getDsListProfilForCombo($iWidth=55, $cont_id=0, $profil_id_exclude=-1)
  {
    $strIntitule = $this->dbConn->GetSubstring("PROFIL_INTITULE", 0, $iWidth);
    if( $cont_id < 0 ) {
       $cont_id = "0,".(-1*$cont_id);
       $strIntitule = $this->dbConn->getConcat($strIntitule, $this->dbConn->compareSql("CONT_ID", "=", "0", "'*'", "''"));
    }
    $strWhere = "";
    if( $profil_id_exclude != -1 ) {
      $strWhere = " and PROFIL_ID<>".$profil_id_exclude;
    }
    $strSql = "select ".$strIntitule." as PROFIL_INTITULE, PROFIL_ID ".
      " from SIT_PROFIL".
      " where CONT_ID in (".$cont_id.")".$strWhere.
      " order by PROFIL_INTITULE";
    return $this->dbConn->InitDataSet($strSql);      
  }
  
  /**
   *  Retourne la liste des profils d'un agent pour un combobox
   * @param agent_id        identifiant de l'utilisateur
   * @param bExceptPrimary  true = exclu le groupe primaire déjà sélectionné, false sinon (par défaut)
   * @parambExceptAll       true = exclu tous les groupes déjà associés, false sinon (par défaut)
   * @param iWidth          longueur max de l'intitulé affiché, =55 par défaut
   * @param cont_id         identifiant de l'espace, =0 pour un profil principal, >0 pour un profil lié à un espace 
   * @return AlkDs
   */
  public function getDsListProfilAgentForCombo($agent_id, $bExceptPrimary=false, $bExceptAll=false, $iWidth=55, $cont_id=0)
  {
    if( !$bExceptPrimary && !$bExceptAll ) {
      // tous les profils de agent_id
      $strSql = "select ".$this->dbConn->GetSubstring("p.PROFIL_INTITULE", 0, $iWidth)." as PROFIL_INTITULE, p.PROFIL_ID ".
        " from SIT_PROFIL p".
        "   inner join SIT_AGENT_PROFIL ap on ap.PROFIL_ID=p.PROFIL_ID".
        " where ap.AGENT_ID=".$agent_id." and p.CONT_ID=".$cont_id.
        " order by p.PROFIL_INTITULE";
    } elseif( $bExceptPrimary && !$bExceptAll ) {
      // tous les profils de agent_id sauf le principal => tous les profils secondaires de agent_id
      $strSql = "select ".$this->dbConn->GetSubstring("p.PROFIL_INTITULE", 0, $iWidth)." as PROFIL_INTITULE, p.PROFIL_ID ".
        " from SIT_PROFIL p".
        "   inner join SIT_AGENT_PROFIL ap on ap.PROFIL_ID=p.PROFIL_ID".
        "   inner join SIT_AGENT a on a.AGENT_ID=ap.AGENT_ID".
        " where ap.AGENT_ID=".$agent_id." and a.PROFIL_ID<>p.PROFIL_ID and p.CONT_ID=".$cont_id.
        " order by p.PROFIL_INTITULE";
      
    } else {
      // tous les profils non affectés à agent_id
      $strSql = "select ".$this->dbConn->GetSubstring("p.PROFIL_INTITULE", 0, $iWidth)." as PROFIL_INTITULE, p.PROFIL_ID ".
        " from SIT_PROFIL p".
        " where p.PROFIL_ID not in (select PROFIL_ID from SIT_AGENT_PROFIL where AGENT_ID=".$agent_id.")".
        "   and p.CONT_ID=".$cont_id.
        " order by p.PROFIL_INTITULE";
    }
    return $this->dbConn->InitDataSet($strSql);      
  }

  /**
   * Retourne la fiche d'un profil
   * 09_profil_form.php
   */
  function GetDs_ficheProfil($profil_id)
  {
    $strSql = "select * from SIT_PROFIL where PROFIL_ID=".$profil_id;
      
    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   * Retourne la liste des types de fiche annuaire pour un combobox
   * 01_recherche_form.php, 02_agent_add_form.php 02_agent_maj_form.php
   */
  function GetDs_listeTypeFicheAnnuaireForCombo()
  {
    $strSql = "select ".$this->dbConn->GetSubstring("AGTYPE_INTITULE", 0, 55).", AGTYPE_ID ".
      "from SIT_AGENT_TYPE order by AGTYPE_RANG";

    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   * Retourne la liste des coordonnees telecoms d'un agent
   * 02_agent_maj_form.php, lib_fiche_agent
   *
  function GetDs_listeCoordonneeTelecomByAgentForCombo($agent_id, $bValeurOnly=false)
  {
    $strChamp = "c.COOR_VALEUR";
    if( $bValeurOnly == false ) 
      $strChamp = $this->dbConn->GetSubstring($this->dbConn->GetConcat("t.TELECOM_INTITULE", "' : '", "c.COOR_VALEUR"), 0, 55);

    $strSql = "select ".$strChamp.", c.COOR_ID, t.TELECOM_ID,".
      " c.COOR_COMPLEMENT, c.COOR_PUBLICATION, t.TELECOM_INTITULE".
      " from SIT_COORDONNEE c".
      "   inner join SIT_TELECOM t on c.TELECOM_ID=t.TELECOM_ID".
      " where c.AGENT_ID=".$agent_id. 
      " order by t.TELECOM_RANG, c.COOR_ID";
    return $this->dbConn->InitDataSet($strSql);
  }*/

  /**
   * Retourne la fiche telecom
   * 02_telecom_form.php
   *
  function GetDs_ficheCoordonnee($coor_id)
  {
    $strSql = "select c.*, t.TELECOM_INTITULE".
      " from SIT_COORDONNEE c".
      "   inner join SIT_TELECOM t on c.TELECOM_ID=t.TELECOM_ID".
      " where c.COOR_ID=".$coor_id;
    return $this->dbConn->InitDataSet($strSql);
  }*/

  /**
   * Retourne la liste des types telecom pour un combobox
   * 02_telecom_form.php
   *
  function GetDs_listeTypeTelecomForCombo()
  {
    $strSql = "select TELECOM_INTITULE, TELECOM_ID from SIT_TELECOM order by TELECOM_RANG";
    return $this->dbConn->InitDataSet($strSql);      
  }*/

  /**
   *  Retourne la liste des admins de service liés à service_id
   * @param service_id   identifiant du service de niveau 1
   * @param bForCombo    false par défaut, true pour rendre un dataset pour un combo : text = civilite prenom nom, id=agent_id
   * @return AlkDs
   */
  public function getDs_listeAdminService($service_id, $bForCombo=false)
  {
    if( $bForCombo==false ) {
      $strSql = "select a.*".
        " from SIT_AGENT a".
        "   inner join SIT_AGENT_SERVICE ase on a.AGENT_ID=ase.AGENT_ID".
        "   inner join SIT_SERVICE s on ase.SERVICE_ID=s.SERVICE_ID".
        " where (a.AGENT_PRIV & ".ALK_PRIV_ANNU_SERV.")=".ALK_PRIV_ANNU_SERV." and a.AGENT_VALIDE=1".
        " and ase.SERVICE_ID=".$service_id." and s.SERVICE_NIVEAU=1".
        " order by a.AGENT_NOM, a.AGENT_PRENOM";
    } else {
      $strSql = "select ".$this->dbConn->getConcat("a.AGENT_CIVILITE", "' '", "a.AGENT_PRENOM", "' '", "a.AGENT_NOM")." as nom,".
        " a.AGENT_ID".
        " from SIT_AGENT a".
        "   inner join SIT_AGENT_SERVICE ase on a.AGENT_ID=ase.AGENT_ID".
        "   inner join SIT_SERVICE s on ase.SERVICE_ID=s.SERVICE_ID".
        " where (a.AGENT_PRIV & ".ALK_PRIV_ANNU_SERV.")=".ALK_PRIV_ANNU_SERV." and a.AGENT_VALIDE=1".
        "   and ase.SERVICE_ID=".$service_id." and s.SERVICE_NIVEAU=1".
        " order by nom";      
    }
    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   *  Retourne l'identifiant de l'agent à partir de son nom, prénom et l'id du service de niveau 1
   *
   * @param agent_nom    nom de l'agent
   * @param agent_prenom prénom de l'agent
   * @param service_id   identifiant du service de niv1
   * @return Retourne un entier, -1 si pas trouvé
   */
  function getStr_agentIdByNomPrenomServId($agent_nom, $agent_prenom, $service_id)
  {
    $agent_id = "-1";
    $strSql = "select a.AGENT_ID ".
      " from SIT_AGENT a".
      "   inner join  SIT_AGENT_SERVICE sas on a.AGENT_ID=sas.AGENT_ID".
      "   inner join SIT_SERVICE s on sas.SERVICE_ID=s.SERVICE_ID".
      " where ".$this->dbConn->GetUpperCase("a.AGENT_NOM")."=".
      $this->dbConn->GetUpperCase("'".$this->dbConn->analyseSql($agent_nom)."'").
      " and ".$this->dbConn->GetUpperCase("a.AGENT_PRENOM")."=".
      $this->dbConn->GetUpperCase("'".$this->dbConn->analyseSql($agent_prenom)."'").
      " and s.SERVICE_NIVEAU=1 and s.SERVICE_ID=".$service_id;
    $dsAgent = $this->dbConn->InitDataSet($strSql);
    if( $drAgent = $dsAgent->getRowIter() ) {
      $agent_id = $drAgent->getValueName("AGENT_ID");
    }
    return $agent_id;
  }

  /**
   *  Retourne la fiche agent avec la filiation de niveau 1
   * @param agent_id  identifiant (ou liste d'id) de l'agent
   * @param strCond   condition complémentaire, ="" par défaut
   */
  public function getDs_ficheAgent($agent_id, $strCond="")
  {
    $strSql = "select a.*, s.SERVICE_INTITULE, s.SERVICE_ID," .
      " ad.ADR_ADR1, ad.ADR_ADR2, ad.ADR_ADR3, ad.ADR_POST, ad.ADR_VILLE, ad.ADR_PAYS, ad.ADR_DESC, ".
      " p.PROFIL_INTITULE, ".
      $this->dbConn->getConcat("ac.AGENT_PRENOM", "' '", "ac.AGENT_NOM")." as agent_nomcrea, ".
      $this->dbConn->getConcat("am.AGENT_PRENOM", "' '", "am.AGENT_NOM")." as agent_nommaj, ".
      $this->dbConn->getConcat("a.AGENT_PRENOM", "' '", "a.AGENT_NOM")." as NOM, ".
      $this->dbConn->GetDateFormat("DD/MM/YYYY, HH:MI", "a.AGENT_DATE_CONN", false)." as AGENT_DATECONN, ".
      $this->dbConn->GetDateFormat("DD/MM/YYYY, HH:MI", "a.AGENT_DATE_MAJ", false)." as AGENT_DATEMAJ, ".
      $this->dbConn->GetDateFormat("DD/MM/YYYY, HH:MI", "a.AGENT_DATE_CREA", false)." as AGENT_DATECREA, ".
      $this->dbConn->GetDateFormat("DD/MM/YYYY, HH:MI", "a.AGENT_DATELIMIT", false)." as AGENT_DATE_LIMIT, ".
      $this->dbConn->GetDateFormat("DD/MM/YYYY, HH:MI", "a.AGENT_DATEVALID", false)." as AGENT_DATE_VALID ".
      " from SIT_AGENT a".
      "   inner join SIT_AGENT_SERVICE sas on a.AGENT_ID=sas.AGENT_ID".
      "   inner join SIT_SERVICE s on sas.SERVICE_ID=s.SERVICE_ID".
      "   inner join SIT_AGENT ac on a.AGENT_ID_CREA=ac.AGENT_ID".
      "   inner join SIT_AGENT am on a.AGENT_ID_MAJ=am.AGENT_ID".
      "   inner join SIT_ADRESSE ad on ad.AGENT_ID=a.AGENT_ID".
      "   inner join SIT_PROFIL p on a.PROFIL_ID=p.PROFIL_ID".
      " where a.AGENT_ID in (".$agent_id.") and s.SERVICE_NIVEAU=1 ".$strCond;
      //echo $strSql."<br/>";
    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   * Retourne la fiche adresse
   * 02_adresse_form.php
   */
  function GetDs_ficheAdresse($agent_id)
  {
    $strSql = "select * from SIT_ADRESSE where AGENT_ID=".$agent_id;

    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   *  Retourne la fiche agent avec la filiation et l'adresse de l'agent et du service
   * 
   * @param agent_id id de l'agent
   * @param strCond condition complémentaire
   * @return dataset 
   */
  function GetDs_ficheAgentPublipostage($agent_id, $strCond="", $bSql=false)
  {
    $strSql = "select a.*, s.SERVICE_ID, s.SERVICE_ADRESSE, " .
    	"  adr.ADR_ADR1 as ADR1, adr.ADR_ADR2 as ADR2, adr.ADR_ADR3 as ADR3, " .
      "  adr.ADR_POST as CODEPOSTAL, adr.ADR_VILLE	as VILLE, adr.ADR_PAYS as PAYS".
      " from " .
      "   SIT_SERVICE s " .
      "   left join SIT_AGENT_SERVICE sas on s.SERVICE_ID = sas.SERVICE_ID".
      "   left join SIT_AGENT a on a.AGENT_ID=sas.AGENT_ID" .
      "   left join SIT_ADRESSE adr on adr.AGENT_ID = a.AGENT_ID" .      
      " where a.AGENT_ID in (".$agent_id.")". 
      "   and a.AGENT_PUBLIPOSTAGE = 1".
			"   and s.SERVICE_NIVEAU=1 ".$strCond;
    if( $bSql )
    	return $strSql;
    
    return $this->dbConn->InitDataSet($strSql);
  }

   /**
   * Retourne la liste des départements non encore utilisé pour une connexion externe
   * 02_agent_login.php
   *
   * @todo Méthode à implémenter pour Mysql
   */
  function GetDs_listeLoginDptByAgentForCombo($agent_id)
  {
    if( ALK_BDD_TYPE != ALK_SGBD_MYSQL ) {
      $strSql = "select ".$this->dbConn->GetConcat("SITDEPT_ID", "'-'", "SITDEPT_INTITULE")." as DEPT_INTITULE,".
        " SITDEPT_ID DEPT_ID from SIT_DEPT".
        " minus (".
        "  select ".$this->dbConn->GetConcat("d.SITDEPT_ID", "'-'", "d.SITDEPT_INTITULE")." as DEPT_INTITULE,".
        "   d.SITDEPT_ID DEPT_ID".
        "  from SIT_AGENT_LOGINDPT ad, SIT_DEPT d".
        "  where ad.AGENT_ID=".$agent_id." and ad.SITDEPT_ID=d.SITDEPT_ID)".
        " order by DEPT_ID";
    } else {
      die("Méthode query.GetDs_listeLoginDptByAgentForCombo non implémenté pour Mysql");
    }
    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   * Retourne la liste des connexions externes pour un agent
   * 02_agent_login.php
   * lib_sqlmenu.php
   */
  function GetDs_listeLoginDptByAgent($agent_id)
  {
    $strSql = "select ald.*, d.SITDEPT_INTITULE, d.* ".
      " from SIT_AGENT_LOGINDPT ald".
      "  inner join SIT_DEPT d on d.SITDEPT_ID=ald.SITDEPT_ID".
      " where AGENT_ID=".$agent_id;

    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   * Retourne la liste des agents pour l'affectation de droits
   * pour l'une des applis SIT
   * bdalk/06_champ_droit.php
   */
  function GetDs_listeDroitAgentByFiltre($strNom, $service_id, $profil_id, $strSelectAppli)
  {
    // $strSelectAppli : requete sql possédant une interface AGENT_ID, DROIT_ID

    $strWhere = "";
    if( $service_id != -1 )
      $strWhere .= " and ase.SERVICE_ID=".$service_id;
    if( $profil_id != -1 )
      $strWhere .= " and ag.PROFIL_ID=".$profil_id;
    
    $strSql = "select ag.AGENT_ID as id, ".
      $this->dbConn->GetConcat("ag.AGENT_NOM", "' '", "ag.AGENT_PRENOM")." as nom, ".
      "p.PROFIL_ID, p.PROFIL_INTITULE as profil, ".
      "s.SERVICE_INTITULE_COURT as service, aa.DROIT_ID as droit ".
      "from SIT_AGENT ag".
      "  inner join SIT_PROFIL p on ag.PROFIL_ID=p.PROFIL_ID".
      "  inner join SIT_AGENT_SERVICE ase on ag.AGENT_ID=ase.AGENT_ID".
      "  inner join SIT_SERVICE s on ase.SERVICE_ID=s.SERVICE_ID".
      "  inner join (".$strSelectAppli.") aa on ag.AGENT_ID=aa.AGENT_ID".
      " where ag.AGENT_VALIDE=1".
      "  and s.SERVICE_NIVEAU=1 ".
      "  and ".$this->dbConn->GetLowerCase("ag.AGENT_NOM")." like ".
      $this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($strNom)."%'").
      $strWhere.
      " order by nom";
    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   *  Retourne le dataset de l'agent trouvé par son login et son mot de passe
   * @param strLogin            login saisi (testé en minuscule)
   * @param strPwd              mot de passe saisi (n'est pas testé en mode SSO)
   * @param strCondAgentValide  condition sur le champ AGENT_VALIDE, "=1" par défaut
   * @param iSSO                =0 par défaut, =1 si authentification SSO
   * @return AlkDs
   */
  public function getDs_ficheAgentByLoginPwd($strLogin, $strPwd, $strCondAgentValide="=1", $iSSO="0")
  {
    $strSql = "select ag.AGENT_ID, ag.AGENT_PRIV, ag.AGENT_MAIL, ag.AGENT_VALIDE,".
      " ag.CONT_ID_START, ag.AGENT_NOM, ag.AGENT_PRENOM, ag.AGENT_LOGIN, ag.AGENT_PWD,".
      " s.SERVICE_ID, s.SERVICE_INTITULE, s.SERVICE_INTITULE_COURT as SERVICE_SIGLE, s.DEPT_ID,".
      $this->dbConn->getConcat("ag.AGENT_PRENOM", "' '", "ag.AGENT_NOM")." as NOM, ag.PROFIL_ID," .
      $this->dbConn->GetDateFormat('DD/MM/YYYY', 'ag.AGENT_DATE_CONN', false)." as AGENT_DATE_CONN ,".
      " ag.AGENT_ADMINAUTH, ag.AGENT_LG".
      " from SIT_AGENT ag".
      "   inner join SIT_AGENT_SERVICE ags on ags.AGENT_ID=ag.AGENT_ID".
      "   inner join SIT_SERVICE s on ags.SERVICE_ID=s.SERVICE_ID".
      " where s.SERVICE_NIVEAU=1".
      "   and ag.AGENT_VALIDE".$strCondAgentValide.
      "   and ".$this->dbConn->GetLowerCase("ag.AGENT_LOGIN")."=".
      $this->dbConn->GetLowerCase("'".$this->dbConn->analyseSQL($strLogin)."'").
      ( $iSSO != "1" 
        ? " and ag.AGENT_PWD='".$this->dbConn->analyseSQL($strPwd)."'"
        : "" );
    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   * obtention du ds d'un utilisateur identifié par un marqueur d'identification défini dans les mails
   * @param strIdentification encodage du login et mot de passe par MD5('ALK_IDENTIFICATION=##'.$login.'##'.$motDePasse.'##') (à comparer sous meme formule à la base)
   * @param strCondAgentValide  condition sur le champ AGENT_VALIDE, "=1" par défaut
   * @param iSSO                =0 par défaut, =1 si authentification SSO
   * @return Retourne un dataset
   */
  function getDs_ficheAgentByMarkupIdentification($strIdentification, $strCondAgentValide="=1", $iSSO="0")
  {
    $strSql = "select ag.AGENT_ID, ag.AGENT_PRIV, ag.AGENT_MAIL, ag.AGENT_VALIDE,".
      " ag.CONT_ID_START, s.SERVICE_ID, ag.AGENT_NOM, ag.AGENT_PRENOM, ag.AGENT_LOGIN, ag.AGENT_PWD,".
      $this->dbConn->getConcat("ag.AGENT_PRENOM", "' '", "ag.AGENT_NOM")." as NOM, s.DEPT_ID, ag.PROFIL_ID," .
      " ag.AGENT_ADMINAUTH, ag.AGENT_LG, ".
      $this->dbConn->GetDateFormat('DD/MM/YYYY', 'ag.AGENT_DATE_CONN', false)." as AGENT_DATE_CONN".
      " from SIT_AGENT ag".
      "   inner join SIT_AGENT_SERVICE ags on ags.AGENT_ID=ag.AGENT_ID".
      "   inner join SIT_SERVICE s on ags.SERVICE_ID=s.SERVICE_ID".
      " where s.SERVICE_NIVEAU=1".
      "   and ag.AGENT_VALIDE".$strCondAgentValide.
      "   and MD5(".$this->dbConn->getConcat("'ALK_IDENTIFICATION=##'", "ag.AGENT_LOGIN", "'##'", "ag.AGENT_PWD", "'##'").")=" .
          "'".$this->dbConn->analyseSql($strIdentification)."'";      
       
    return $this->dbConn->initDataset($strSql);
  } 
  
  /**
   *  Retourne la fiche de l'agent trouvé par son login et son mot de passe
   *        ainsi que les paramètres pour se connecter à un site distant
   * sit_menu/login_verif.php
   *
   * @param strLogin Login saisi
   * @param strPwd   Mot de passe saisi
   * @return Retourne un dataSet
   */
  function GetDs_ficheAgentDptByLoginPwd($strLogin, $strPwd, $iDept)
  {
    $strSql = "select ag.*, s.SERVICE_ID, ald.LOGIN_DPT, ald.PWD_DPT, sd.* ".
      " from".
      "  (select AGENT_ID, PROFIL_ID, AGENT_ADMIN_ANNU ".
      "   from SIT_AGENT".
      "   where AGENT_VALIDE=1 and ".
      $this->dbConn->GetLowerCase("AGENT_LOGIN")."=".
      $this->dbConn->GetLowerCase("'".$this->dbConn->AnalyseSql($strLogin)."'").
      " and AGENT_PWD='".$this->dbConn->AnalyseSql($strPwd)."'".
      "  ) ag,".
      "  (select SERVICE_ID from SIT_SERVICE where SERVICE_NIVEAU=1) s,".
      "  SIT_AGENT_SERVICE ags,".
      "  SIT_DEPT sd,".
      "  (select * from SIT_AGENT_LOGINDPT where SITDEPT_ID='".$iDept."') ald ".
      "where".
      "  s.SERVICE_ID=ags.SERVICE_ID and ags.AGENT_ID=ag.AGENT_ID".
      "  and ag.AGENT_ID=ald.AGENT_ID and ald.SITDEPT_ID=sd.SITDEPT_ID";
    return $this->dbConn->InitDataSet($strSql);
  }
  
  
  /**
   *  fonction pour le recuperation de  l'intitule des champs et l'id leurs correspandants.
   *        11_import
   *
   * @param $strTable nom de la table.
   * @param $strChamp nom du champs.
   * @param $strIdChamp nom de l'id du champ.
   * @return un tableau 
   */
	function GetDs_champsImport($strTable, $strChamp, $strIdChamp)
	{
		$tabChamp =array();
	    $strSql = "select " .$strChamp." , ".$strIdChamp." from ".$strTable ;
	    $dsImport = $this->dbConn->InitDataSet($strSql);
		while( $drImport = $dsImport->getRowIter() ) {
			$strName = $drImport->getValueName($strChamp); 
			$idName = $drImport->getValueName($strIdChamp);
			$tabChamp[$idName] = $strName;
		}
		return $tabChamp;
	}
	
  /**
   *  Retourne l'identifiant du service recherché
   *        11_import
   *
   * @param service_intitule  nom du service recherché
   * @param service_pere      identifiant du service pere
   * @param service_niveau    niveau du service recherché
   * @return Retourne un entier
   */
  function getstr_serviceIdByNameByParent($service_intitule, $service_pere, $service_niveau)
  {
    $service_id = "-1";
    $strSql = "";
    if( $service_pere=="-1" && $service_niveau=="1" && $service_intitule!="" )
      $strSql = "select SERVICE_ID".
        " from SIT_SERVICE".
        " where SERVICE_INTITULE_COURT='".$this->dbConn->AnalyseSql($service_intitule).
        "' and SERVICE_NIVEAU=".$service_niveau;
    elseif( $service_pere!="-1" && $service_niveau!="1" && $service_intitule!="" ) {
      $strSql = "select SERVICE_ID".
        " from SIT_SERVICE".
        " where SERVICE_INTITULE like '".$this->dbConn->AnalyseSql($service_intitule).
        "' and SERVICE_PERE=".$service_pere." and SERVICE_NIVEAU=".$service_niveau;
    }
    
    if( $strSql != "" ) {
      $dsService = $this->dbConn->InitDataSet($strSql);
      if( $drService = $dsService->getRowIter() ) {
        $service_id = $drService->getValueName("SERVICE_ID"); 
      }
    }
    return $service_id;
  }
	
	/**
   *  fonction pour la recuperation de  l'intitule du service fils de $service et de niv int_niv.
   *        12_export_service
   *
   * @param int_niv le niveau.
   * @param service intitule du service.
   * @return intitule du service correspandant
   */
	function Get_intitule_Pere($int_Niv, $service)
	{
		$service_fils = "";
    //$strSelect = ( $int_Niv == "1" ? "SERVICE_INTITULE_COURT" : "SERVICE_INTITULE" );
    $strSelect = " SERVICE_INTITULE ";
    
		$strSql = "select ".$strSelect." as intitule from SIT_SERVICE".
      " WHERE SERVICE_ID in".
      " (select SERVICE_PERE from SIT_SERVICE where SERVICE_INTITULE ='".$this->dbConn->AnalyseSql($service)."')".
      " and SERVICE_NIVEAU=".$int_Niv;
		
		$dsService = $this->dbConn->InitDataSet($strSql);
		if( $drService = $dsService->getRowIter() ) {
			$service_fils = $drService->getValueName("INTITULE"); 
		}
		return $service_fils;
	}
	
  /**
   *  12_export_service
   * 
   */
	function GetDs_Service($service_id=-1)
	{
		$strSql = "select * from SIT_SERVICE where SERVICE_NIVEAU !=0 ".($service_id != -1 ? " AND SERVICE_ID=".$service_id." OR SERVICE_PERE=".$service_id." OR SERVICE_RACINE=".$service_id : "")." order by SERVICE_NIVEAU asc";
		return $this->dbConn->InitDataSet($strSql);
	}
	
  /**
   *  12_export_mission
   *
   */
	function GetDs_Mission()
	{
		$strSql = "select * from SIT_MISSION ";
		return $this->dbConn->InitDataSet($strSql);
	}
  
  /**
   *  12_export_profil
   *
   */
  function GetDs_Profil()
	{
		$strSql = "select * from SIT_PROFIL ";
		return $this->dbConn->InitDataSet($strSql);
	}

  /**
   *   Retourne la fiche d'une fonction avancée
   *
   * @param fonc_id  Identifiant de la fiche
   * @return Retourne un dataSet
   */
  function getDs_ficheFonctionAv($fonc_id)
  {
    $strSql = "select * from SIT_FONCTION where FONC_ID=".$fonc_id;
      
    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   *  Retourne la liste des fonctions associées à un agent
   *
   * @param agent_id  Identifiant de l'agent
   * @return Retourne un dataSet
   */
  function GetDs_listeFonctionAvByAgentForCombo($agent_id)
  {
    $strSql = "select f.FONC_INTITULE, f.FONC_ID".
      " from SIT_FONCTION f".
      "   inner join SIT_AGENT_FONCTION af on f.FONC_ID=af.FONC_ID".
      " where af.AGENT_ID=".$agent_id.
      " order by f.FONC_INTITULE";

    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   *  Retourne la liste des fonctions associées à un service
   *
   * @param agent_id  Identifiant de l'agent
   * @return Retourne un dataSet
   */
  function GetDs_listeFonctionAvByServiceForCombo($service_id)
  {
    $strSql = "select f.FONC_INTITULE, f.FONC_ID".
      " from SIT_FONCTION f".
      "   inner join SIT_SERVICE_FONCTION af on f.FONC_ID=af.FONC_ID".
      " where af.SERVICE_ID=".$service_id.
      " order by f.FONC_INTITULE";

    return $this->dbConn->InitDataSet($strSql);
  }
  
     /**
   *  si le service existe retourne l'id , -1 sinon, 11_import_serv.php
   *
   * @param strIntitule  = intitule du service à rechercher
   * @param $strNiveau  =  niveau du service à rechercher
   * @param $strIntitulePere  =  intitule du service père du service à rechercher
   */
  function get_IdServByIntituleNivPere($strIntitule, $strNiveau , $strIntitulePere)
  {
  	$service_id=-1;
  	
    $strSql="select s.SERVICE_ID".
      " from SIT_SERVICE ps, SIT_SERVICE s ".
      " where ".$this->dbConn->GetLowerCase("ps.SERVICE_INTITULE").
      "=".$this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($strIntitulePere)."'").
      " and   ".$this->dbConn->GetLowerCase("s.SERVICE_INTITULE").
      "=".$this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($strIntitule)."'").
      " and s.SERVICE_NIVEAU=".$strNiveau.
      " and s.SERVICE_PERE=ps.SERVICE_ID ";
    
    $dsService = $this->dbConn->InitDataSet($strSql);
  	if( $drService = $dsService->getRowIter() ) {
			$service_id = $drService->getValueName("SERVICE_ID"); 
		}  
   
    return $service_id;
  } 


   /**
   *  si le service existe retourne l'id , -1 sinon, 11_import_serv.php
   *
   * @param strLdapIud  = intitule du service à rechercher
   */
  function get_IdServByldapUid($strLdapUid)
  {
    $service_id=-1;
    
    $strSql="select s.SERVICE_ID ".
      " from SIT_SERVICE s ".
      " where ".$this->dbConn->GetLowerCase("s.SERVICE_LDAPUID").
      "=".$this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($strLdapUid)."'");
    $dsService = $this->dbConn->InitDataSet($strSql);
    if( $drService = $dsService->getRowIter() ) {
      $service_id = $drService->getValueName("SERVICE_ID"); 
    }  
   
    return $service_id;
  } 


   /**
   *  si le service existe retourne l'id , -1 sinon, 11_import_serv.php
   *
   * @param strLdapIud  = intitule du service à rechercher
   */
  function get_IntituleServByldapUid($strLdapUid)
  {
    $service_intitule = "";
    
    $strSql="select s.SERVICE_INTITULE ".
      " from SIT_SERVICE s ".
      " where ".$this->dbConn->GetLowerCase("s.SERVICE_LDAPUID").
      "=".$this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($strLdapUid)."'");
    $dsService = $this->dbConn->InitDataSet($strSql);
    if( $drService = $dsService->getRowIter() ) {
      $service_intitule = $drService->getValueName("SERVICE_INTITULE"); 
    }  
   
    return $service_intitule;
  } 


     /**
   *  retourne l'id du service de niveau 0, 11_import_serv.php
   *
   * @param 
   */
  function get_IdServ0()
  {
    $service_id=-1;
    
    $strSql="select SERVICE_ID from SIT_SERVICE where SERVICE_NIVEAU=0";
            
    $dsService = $this->dbConn->InitDataSet($strSql);
  	if( $drService = $dsService->getRowIter() ) {
			$service_id = $drService->getValueName("SERVICE_ID"); 
		}  
   
    return $service_id;
  } 
  
  /**
   *  retourne l'intitule du service de niveau 0, 11_import_serv.php
   *
   * @param 
   */
  function get_IntituleServ0()
  {
    $service_intitule="";
    
    $strSql="select SERVICE_INTITULE from SIT_SERVICE where SERVICE_NIVEAU=0";
            
    $dsService = $this->dbConn->InitDataSet($strSql);
  	if( $drService = $dsService->getRowIter() ) {
			$service_intitule = $drService->getValueName("SERVICE_INTITULE"); 
		}  
   
    return $service_intitule;
  } 
  
  /**
   * Retourne la fiche d'une liste de diffusion
   *
   * @param liste_id  Identifiant de la liste de diffusion
   * @return AlkDs
   */  
  public function getDsFicheListe($liste_id)
  {
    $strSql = "select * from SIT_LISTE where LISTE_ID=".$liste_id;
    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   * retourne la liste des listes de diffusion qu'un utilisateur peut utiliser sur l'espace en cours (hors liste d'abonnement)
   * listes publiques tout espace + listes privées tout espace + listes publiques de cont_id + listes privées de cont_id
   *
   * @param agent_id        Identifiant de l'utilisateur connecté
   * @param cont_id         Identifiant de l'espace en cours
   * @param bForCombo       vrai si liste pour combo, faux pour liste classique
   * @param iFirst          indice de début pour la pagination
   * @param iLast           indice de fin pour la pagination
   * @param bAdminAll       vrai si l'utilisateur connecté est administrateur/animateur de tous les espaces
   * @param bAnimSpace      vrai si l'utilisateur connecté est animateur de l'espace cont_id
   * @param bWithoutSubList faux par défaut, vrai si l'on retire de la liste, les listes qui dépendent d'autres listes...
   * @param liste_type      type de la liste : 
   *                          =0 liste diff (mes listes), 
   *                          =1 liste circulaire (module synd), 
   *                          =2 liste abonnement à une catégorie ou donnée (liste implicite)
   *                          =3 liste d'abonnement explicite (lettre d'information)
   *                          =4 liste des utilisateurs de l'espace
   * @return Retourne un dataSet
   */
  public function getDsListes($agent_id, $cont_id, $bForCombo=false, $iFirst=0, $iLast=-1,
                              $bAdminAll=false, $bAnimSpace=false, $bWithoutSubList=false, $liste_type="0, 3")
  {
    /**
     * Droits mis en place :
     * 
     *           | Privé espace | Privé tous esp | Publiq espace | Publiq tous esp
     * Admin Gen |      X       |       X        |       X       |       X
     * Anim Tous |      X       |       X        |       X       |       X
     * Anim Esp  | X(si espace) | X(si createur) | X(si espace)  |       X
     * Util      | X(crea & esp)| X(si createur) | X(si espace)  |       X
     */
    $user_id = AlkFactory::getSProperty("user_id", "-1");
    
    $strWhereAgent = ( $agent_id != "-1" ? " and l.AGENT_ID=".$agent_id : "" );
    $strWhereCont  = ( $cont_id!= "-1" ? " and ac.CONT_ID is not null and l.CONT_ID=".$cont_id : "" );
    
    $strPriveEsp = $strWhereCont.$strWhereAgent; // voit toutes les listes privées de cet espace dont il est créateur
    $strPriveAll = $strWhereAgent;               // voit toutes les listes privées à tous les espaces dont il est créateur
    $strPublicEsp = $strWhereCont;               // voit toutes les listes publiques à cet espace
    $strPublicAll = "";                          // voit toutes les listes publiques à tous les espaces
    
    if ( $bAnimSpace==true ){      
      $strPriveEsp = $strWhereCont; // voit toutes les listes privées de cet espace 
      $strPriveAll = $strWhereAgent;// voit toutes les listes privées à tous les espaces dont il est créateur
      $strPublicEsp = $strWhereCont;// voit toutes les listes publiques de cet espace
      $strPublicAll = "";           // voit toutes les listes publiques à tous les espaces
    }
    
    if ( $bAdminAll==true ){
      // voit toutes les listes
      $strPriveEsp = "";
      $strPriveAll = "";
      $strPublicEsp = "";
      $strPublicAll = "";
    }

    $strSql = "select ".($bForCombo == true ? "l.LISTE_INTITULE_FR, l.LISTE_ID" : "l.*").", ac.CONT_ID as ACONT_ID, c.CONT_INTITULE, n.nbAbonnes".
      " from SIT_LISTE l".
      "   left join (select LISTE_ID, count(ABONNE_ID) as nbAbonnes from SIT_ABONNE_LISTE group by LISTE_ID) n on l.LISTE_ID=n.LISTE_ID".
      "   left join SIT_AGENT_CONT ac on (l.CONT_ID=ac.CONT_ID and ac.AGENT_ID=".$user_id.")".
      "   left join SIT_CONTENEUR c on l.CONT_ID=c.CONT_ID".
      ( $bWithoutSubList
        ? " left join (select LISTE_ID, count(SLISTE_ID) as nbSListe from SIT_LISTE_SLISTE group by LISTE_ID) ll on l.LISTE_ID=ll.LISTE_ID"
        : "" ).
      " where l.LISTE_TYPE in (".$liste_type.") and l.LISTE_VALIDE > -1 " .
      ( $bWithoutSubList
        ? " and (ll.nbSListe is null or ll.nbSListe=0)"
        : "" ).
      " and (" .
      "      (l.LISTE_PRIVE=1 ".$strPriveEsp.")".                   // prive pour espace
      "   or (l.LISTE_PRIVE=3 and l.CONT_ID=0 ".$strPriveAll.")".   // prive tout espace
      "   or (l.LISTE_PRIVE=0 ".$strPublicEsp.")".                  // public pour espace
      "   or (l.LISTE_PRIVE=2 and l.CONT_ID=0 ".$strPublicAll.")".  // public tout espace
      "   )";

    $strSql .= " order by l.LISTE_INTITULE_FR, n.nbAbonnes";
    return $this->dbConn->InitDataSet($strSql, $iFirst, $iLast);
  }
  
  /**
   * retourne toutes les listes privées/publiques associées à l'espace
   */
  public function getDsListesForEspace($cont_id, $bForCombo=false, $iFirst=0, $iLast=-1, $liste_type="0, 3"){
      
      
      $strWhereCont  = ( $cont_id!= "-1" ? " and l.CONT_ID=".$cont_id : "" );
      $strSql = "select ".($bForCombo == true ? "l.LISTE_INTITULE_FR, l.LISTE_ID" : "l.*").", c.CONT_INTITULE, n.nbAbonnes".
      " from SIT_LISTE l".
      "   left join (select LISTE_ID, count(ABONNE_ID) as nbAbonnes from SIT_ABONNE_LISTE group by LISTE_ID) n on l.LISTE_ID=n.LISTE_ID".
      "   left join SIT_CONTENEUR c on l.CONT_ID=c.CONT_ID".
      " where l.LISTE_TYPE in (".$liste_type.") and l.LISTE_VALIDE > -1 " .
      $strWhereCont;
    $strSql .= " order by l.LISTE_INTITULE_FR, n.nbAbonnes";
    return $this->dbConn->InitDataSet($strSql, $iFirst, $iLast);
  }

  /**
   *  Retourne la liste des liste de diff gérés par agent_id
   *
   * @param agent_id   Identifiant de l'agent connecté
   * @param liste_type type de la liste : 
   *                     =0 liste diff (mes listes), 
   *                     =1 liste circulaire (module synd), 
   *                     =2 liste abonnement à une catégorie ou donnée (liste implicite)
   *                     =3 liste d'abonnement explicite (lettre d'information)
   * @param iFirst     indice de début pour la pagination
   * @param iLast      indice de fin pour la pagination
   * @return AlkDs
   */
  public function getDsListesGerees($agent_id, $liste_type, $iFirst=0, $iLast=-1)
  {
    $strSql = "select l.*, n.nbAbonnes, c.CONT_INTITULE".
      " from SIT_LISTE l".
      "   left join (select LISTE_ID, count(ABONNE_ID) as nbAbonnes from SIT_ABONNE_LISTE group by LISTE_ID) n".
      "     on l.LISTE_ID=n.LISTE_ID".
      "   left join SIT_CONTENEUR c on l.CONT_ID=c.CONT_ID".
      " where ".
      ( $agent_id != "-1" 
        ? "l.AGENT_ID=".$agent_id." and" 
        : "" ).
      "   l.LISTE_VALIDE > -1".
      "   and l.LISTE_TYPE in (".$liste_type.")".
      " order by l.LISTE_INTITULE_FR, n.nbAbonnes";

    return $this->dbConn->InitDataSet($strSql, $iFirst, $iLast);    
  }

  /**
   * Retourne la liste des abonnés d'une liste de diffusion
   * fonction liée à getStrDsListeAgentMailForLsDif() => une union est effectuée, les champs sélectionnés doivent être en correspondance
   * 
   * @param liste_id            Identifiant de la liste
   * @param bDataSet            true = retourne un dataSet, false par défaut = retourne la requête
   * @param iFirst              indice du premier élément pour la pagination
   * @param iLast               indice du dernier élément pour la pagination
   * @param bUnion              =false par défaut pour ordonner par nom et prénom, = vrai si la requête est utilisée dans une sous-requête de type union
   * @param bAgent              permet de sélectionner les abonnés ayant agent_id != ALK_USER_ID_INTERNET
   * @param abonne_id           identifiant de l'abonné (par défaut = -1)
   * @param tabAbonneIdInclude  tableau des identifiants des abonnés à conserver (exclus tous les autres), conserve tous si vide
   * @return AlkDs si bDataSet = true, string sinon
   */
  public function getDsAbonnesByListe($liste_id, $bDataSet=false, $iFirst=0, $iLast=-1, $bUnion=false, $bAgent=false, $abonne_id=-1, $tabAbonneIdInclude=array())
  {
    // attention, ne pas rajouter de champs dans le selection, requete utilisée dans des unions
    $strSql = "select ab.ABONNE_ID, ab.AGENT_ID,".
      " ab.ABONNE_PRENOM as prenom, ab.ABONNE_NOM as nom, ab.ABONNE_MAIL as mail".
      ( $bDataSet
        ? ", al.ABONNELISTE_DATE".
          ", ab.ABONNE_VALIDE, ab.ABONNE_DATE_DESINSCR"
        : "" ).
      " from SIT_ABONNE_LISTE al".
      "  inner join SIT_ABONNE ab on al.ABONNE_ID=ab.ABONNE_ID".
      " where al.LISTE_ID=".$liste_id.
      ( $abonne_id != -1 
        ? " and al.ABONNE_ID=".$abonne_id 
        : ( !empty($tabAbonneIdInclude) 
            ? ( $bAgent
                ? " and ab.ABONNE_ID in (".implode(",", $tabAbonneIdInclude).")"
                : " and ( ab.ABONNE_ID in (".implode(",", $tabAbonneIdInclude).") or ab.AGENT_ID=".ALK_USER_ID_INTERNET." )") 
            : "")).
      ( $bAgent 
        ? " and ab.AGENT_ID<>".ALK_USER_ID_INTERNET 
        : "").
      ( $bUnion == false 
        ? " order by nom, prenom" 
        : "" );

    if( $bDataSet == true )
      return $this->dbConn->InitDataSet($strSql, $iFirst, $iLast);

    return $strSql;  
  }

  /**
   *  Retourne l'id de l'auteur de la liste, -1 si non trouvé
   *
   * @param liste_id  Identifiant de la liste de diffusion
   * @return int
   */
  public function getStrIdAuteurListeDif($liste_id)
  {
    $agent_id = "-1";
    
    $strSql = "select AGENT_ID from SIT_LISTE where LISTE_ID=".$liste_id;
    $dsListe = $this->dbConn->InitDataSet($strSql);
    if( $drListe = $dsListe->getRowIter() ) {
      $agent_id = $drListe->getValueName("AGENT_ID");
    }
    return $agent_id;
  }

  /**
   * Retourne la liste des abonnés avec l'email
   * fonction liée à getDsAbonnesByListe() => une union est effectuée, les champs sélectionnés doivent être en correspondance
   *
   * @param strListAgentId liste d'id d'agents ou d'abonnés en fonction du param bAgentId
   * @param bDs            true retourne un dataset, false retourne la requete sql dans un string (par défaut)
   * @param bAgentId       true par défaut pour indiquer que la liste contient des agent_id, false indique que la liste contient des abonne_id
   * @return AlkDs si bDs=true, string si bDs=false
   */
  public function getStrDsListeAgentMailForLsDif($strListId, $bDs=false, $bAgentId=true)
  {
    $strSql = "select distinct ABONNE_ID, AGENT_ID, ABONNE_PRENOM as prenom, ABONNE_NOM as nom, ABONNE_MAIL as mail".
      " from SIT_ABONNE".
      " where ".( $bAgentId ? "AGENT_ID" : "ABONNE_ID" )." in (".$strListId.")";

    if( $bDs == true )
      return $this->dbConn->InitDataSet($strSql);
    return $strSql;
  }

  /**
   *  Retourne la requete sql pour la recherche d'agent
   *        01_recherche_list.php
   *
   * @param cont_id        Identifiant de l'espace courant
   * @param serv1_id       Identifiant du service de niveau 1 sélectionné ou -1
   * @param serv2_id       Identifiant du service de niveau 2 sélectionné ou -1
   * @param serv3_id       Identifiant du service de niveau 3 sélectionné ou -1
   * @param strAgentNom    Nom de l'agent recherché
   * @param mission_id     Identifiant de la mission sélectionnéé
   * @param profil_id      Identifiant du profil sélectionné
   * @param espaceCurOnly  si différent de vide, effectue le filtre sur l'espace courant
   * @param strAgentValide Identifiant de l'état de l'agent
   * @param liste_id       Identifiant de la liste sélectionnée ou -1
   * @param strSelect      Champ complémentaire dans le select, =vide par défaut
   */
  function getSqlListeAgentForSearch($cont_id, $dept_id, $serv1_id, $serv2_id, $serv3_id, 
                                     $strAgentNom, $mission_id, $profil_id, 
                                     $espaceCurOnly, $strAgentValide, $liste_id="-1", $strSelect = "", 
                                     $agent_adminauth="-1", $agent_priv="-1")
  {
  	
    $strSelect = " distinct a.AGENT_ID, a.AGENT_NOM, a.AGENT_PRENOM, a.AGENT_FONCTION, ".
      " a.PROFIL_ID, p.PROFIL_INTITULE,".
      " a.AGENT_MAIL as MAIL, a.AGENT_TEL as TEL, a.AGENT_TELPOSTE as TELPOSTE,".
      " ss.SERVICE_ID, ss.SERVICE_INTITULE_COURT, ss.SERVICE_VALIDE".
      ($strSelect != "" ? ", ".$strSelect : "");
    
    $strFrom = " from SIT_AGENT a".
      " inner join SIT_AGENT_SERVICE ass on a.AGENT_ID=ass.AGENT_ID".
      " inner join SIT_SERVICE ss on ass.SERVICE_ID=ss.SERVICE_ID".
      " inner join SIT_PROFIL p on a.PROFIL_ID=p.PROFIL_ID";
    
    $strOrder = " order by a.AGENT_NOM, a.AGENT_PRENOM";
    $strWhere = " where a.AGENT_ID>0 and ss.SERVICE_NIVEAU=1";
      
    // filliation
    $nivService = "-1";
    if( $serv1_id != "-1" ) {
      if( $serv2_id != "-1" ) {
        if( $serv3_id != "-1" ) {
          $nivService = "3";
          $idService = $serv3_id;
        } else {
          $nivService = "2";
          $idService = $serv2_id;
        }
      } else {
        $nivService = "1";
        $idService = $serv1_id;
      }
    }

    if( $nivService != "-1" ) {
      $strFrom .= " inner join SIT_AGENT_SERVICE agr on agr.AGENT_ID=a.AGENT_ID".
        " inner join SIT_SERVICE s on  agr.SERVICE_ID=s.SERVICE_ID";
      $strWhere .= " and s.SERVICE_ID=".$idService." and s.SERVICE_NIVEAU=".$nivService;
    }
      
    // nom
    if( $strAgentNom != "" ) {
      $strWhere .= " and (1=0";
      $tabAgentNom = explode("+", $strAgentNom);
      foreach($tabAgentNom as $key => $strAgNom) {
        $strWhere .= " or ".$this->dbConn->GetLowerCase("a.AGENT_NOM")." like ".
          $this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($strAgNom."%")."'");
      }
      $strWhere .= ")";
    }

    if( $dept_id!="-2" && $dept_id!="-1" ) {
      $strWhere .= " and ss.DEPT_ID=".$dept_id;
    }
    
    if( $mission_id != "-1" ) {
      $strFrom .= " inner join SIT_AGENT_MISSION am on am.AGENT_ID=a.AGENT_ID";
      $strWhere .= " and am.MISSION_ID=".$mission_id;
    }
    
    if( $liste_id != "-1" ) {
      $strFrom .= " inner join SIT_ABONNE ab on ab.AGENT_ID=a.AGENT_ID".
        " inner join SIT_ABONNE_LISTE al on al.ABONNE_ID=ab.ABONNE_ID";
      $strWhere .= " and al.LISTE_ID=".$liste_id;
    }  
      
    // restreint au conteneur courant
    if( $espaceCurOnly ) {
      $strFrom .= " inner join SIT_AGENT_CONT ac on ac.AGENT_ID=a.AGENT_ID";
      $strWhere .= " and ac.CONT_ID=".$cont_id;
    }
      
    // recherche sur le profil
    if( $profil_id != "-1" ) {
      $strFrom .= " inner join SIT_AGENT_PROFIL ap on ap.AGENT_ID=a.AGENT_ID";
      $strWhere .= " and ap.PROFIL_ID=".$profil_id;
    }
      
    // recherche sur l'état de la fiche agent
    if( $strAgentValide != "-1"  ) {
      $strWhere .= " and a.AGENT_VALIDE =".$strAgentValide;
    }
    
    // recherche sur le type de connexion
    if ($agent_adminauth != -1){
      $strWhere .= " and a.AGENT_ADMINAUTH =".$agent_adminauth;
    }
    
    // recherche sur le privilège
    if( $agent_priv != "-1") {
      $strWhere .= " and (a.AGENT_PRIV & ".$agent_priv.")=".$agent_priv;
    }
    
      
    $strSql = "select ".$strSelect." ".$strFrom." ".$strWhere." ".$strOrder;
    return $strSql;
  }

  /**
   *  Retourne la liste des profils pour l'administation
   * @param cont_id  =0 pour les profils généraux, >0 pour les profils liés aux espaces
   * @return Retourne un dataSet
   */
  public function getDs_listeProfil($cont_id=0)
  {
    $strWhere = "p.CONT_ID=".$cont_id;
    if( $cont_id < 0 ) {
      $strWhere = "p.CONT_ID=0 or p.CONT_ID=".(-1*$cont_id);
    }
    // on ne compte que les groupes principaux
    $strSql = "select p.*,".
      $this->dbConn->compareSql("n.nbr", "is", "null", "0", "n.nbr")." as NB_AG ".
      " from SIT_PROFIL p".
      "  left join (select PROFIL_ID, count(AGENT_ID) as nbr from SIT_AGENT_PROFIL group by PROFIL_ID) n".
      "  on p.PROFIL_ID=n.PROFIL_ID".
      " where ".$strWhere.
      " order by p.PROFIL_INTITULE";

    return $this->dbConn->initDataset( $strSql ) ;
  }

  /**
   *  Retourne la liste des missions pour l'administation
   *        06_mission_list.php
   *
   * @param strRecherche Filtre de recherche
   * @param idFirst      Indice début de pagination
   * @param idLast       Indice fin de pagination
   * @return Retourne un dataSet
   */
  function GetDs_listeMission($strRecherche, $idFirst, $idLast)
  {
    $strSql = "select m.MISSION_ID, m.MISSION_INTITULE, count(am.AGENT_ID) as NB_AG".
      " from SIT_MISSION m left join SIT_AGENT_MISSION am on m.MISSION_ID=am.MISSION_ID".
      ( $strRecherche != ""
        ? " where ".$this->dbConn->GetLowerCase("m.MISSION_INTITULE")." like ".
          $this->dbConn->GetLowerCase("'%".$this->dbConn->analyseSql($strRecherche)."%'")
        : "").
      " group by m.MISSION_ID, m.MISSION_INTITULE".
      " order by m.MISSION_INTITULE";
    
    return $this->dbConn->InitDataSet($strSql, $idFirst, $idLast);
  }

  /**
   *  Retourne la liste des services pour l'organigramme
   * @param user_id    Identifiant de l'utilisateur connecté
   * @param user_priv  Privilège de l'utilisateur connecté
   * @return AlkDs
   */
  public function getDs_arboServiceNoeud($user_id, $user_priv, $userServ_id)
  {
    $iDroit = 1;
    if( ($user_priv & ALK_PRIV_ANNU_ALL) == ALK_PRIV_ANNU_ALL ) {
      $iDroit = 2;
    } else {
      if( ($user_priv & ALK_PRIV_ANNU_SERV) == ALK_PRIV_ANNU_SERV ) {
        $iDroit = $this->dbConn->compareSql("(select count(s1.SERVICE_ID) ".
                                            " from SIT_SERVICE s1".
                                            "  inner join SIT_SERVICE s2 on s2.SERVICE_ID=s1.SERVICE_RACINE".
                                            "  inner join SIT_AGENT_SERVICE ase1 on ase1.SERVICE_ID=s2.SERVICE_ID".
                                            "  inner join SIT_AGENT a1 on ase1.AGENT_ID=a1.AGENT_ID".
                                            " where ase1.AGENT_ID=".$user_id." and (a1.AGENT_PRIV & ".ALK_PRIV_ANNU_SERV.")=".ALK_PRIV_ANNU_SERV.
                                            "   and s2.SERVICE_NIVEAU=1".
                                            "   and s0.SERVICE_ID=s1.SERVICE_ID)",
                                            "=", "0", "1", "2");
      }
    }

    $strSep = "''" ;
    
    $strSql = "select s.SERVICE_ID as ID, ".
      $this->dbConn->getConcat("s.SERVICE_INTITULE",
                               $this->dbConn->compareSql("count(ase.AGENT_ID)", "=", "0", 
                                                         "''",
                                                         $this->dbConn->getConcat("' ('", "count(ase.AGENT_ID)", "')'")))." as NOM, ".
      "  s.SERVICE_PERE as ID_PERE, s.SERVICE_NIVEAU as NIV, sd.DROIT,".
      $this->dbConn->getConcat("'javascript:OpenPopupFiche(".$strSep."service".$strSep.", '", "s.SERVICE_ID", "')'")." as URL ".
      " from (SIT_SERVICE s".
      "  left join SIT_AGENT_SERVICE ase on s.SERVICE_ID=ase.SERVICE_ID)".
      "  inner join (select s0.SERVICE_ID, ".$iDroit." as DROIT from SIT_SERVICE s0) sd on sd.SERVICE_ID=s.SERVICE_ID".
      " where s.SERVICE_VALIDE=1".
      ($userServ_id != -1 ? " AND (s.SERVICE_ID=1 OR s.SERVICE_ID=".$userServ_id." OR s.SERVICE_PERE=".$userServ_id." OR s.SERVICE_RACINE=".$userServ_id.")" : "").
      " group by s.SERVICE_ID, s.SERVICE_INTITULE, s.SERVICE_PERE, s.SERVICE_RANG, s.SERVICE_NIVEAU, sd.DROIT ".
      " order by NIV, NOM";
    
    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   * Retourne les services et les utilisateurs pour construire l'organigramme svg
   * @param service_id Identifiant du service racine
   * @return AlkDs
   */
  public function getDsOrgaServiceNoeud($service_id)
  {
    // liste des services avec chef de service dans le champ descr
    $strSql = "select s.SERVICE_ID as ID, s.SERVICE_PERE as ID_PERE, s.SERVICE_NIVEAU as NIVEAU,".
      "   s.SERVICE_INTITULE as INTITULE, ".
      $this->dbConn->GetConcat("a.AGENT_NOM", "' '", "a.AGENT_PRENOM")." as DESCR, '' as PJ, 'serv' as TYPE,".
      $this->dbConn->GetConcat("'serv-'", "s.SERVICE_ID")." as ID2,".
      $this->dbConn->GetConcat("'serv-'", "s.SERVICE_PERE")." as ID2PERE".
      " from SIT_SERVICE s".
      "   left join SIT_AGENT a on s.AGENT_ID=a.AGENT_ID".
      " where s.SERVICE_ARBRE like ".$this->dbConn->GetConcat("'%-'", "'".$service_id."'", "'-%'").
      
      // liste des utilisateurs avec fonction dans champ descr et photo dans pj
      " union".
      " (select a.AGENT_ID as ID, s.SERVICE_ID as ID_PERE, s.SERVICE_NIVEAU+1 as NIVEAU,".
      $this->dbConn->GetConcat("a.AGENT_NOM", "' '", "a.AGENT_PRENOM")." as INTITULE,".
      " a.AGENT_FONCTION as DESCR, a.AGENT_PHOTO as PJ, 'util' as TYPE,".
      $this->dbConn->GetConcat("'util-'", "a.AGENT_ID")." as ID2,".
      $this->dbConn->GetConcat("'serv-'", "s.SERVICE_ID")." as ID2PERE".
      " from SIT_SERVICE s".
      "   inner join SIT_AGENT_SERVICE ase on s.SERVICE_ID=ase.SERVICE_ID".
      "   inner join (select count(SERVICE_ID) as SERVICE_NIVEAU, AGENT_ID from SIT_AGENT_SERVICE group by AGENT_ID) n on ase.AGENT_ID=n.AGENT_ID".
      "   inner join SIT_AGENT a on n.AGENT_ID=a.AGENT_ID".
      " where s.SERVICE_ARBRE like ".$this->dbConn->GetConcat("'%-'", "'".$service_id."'", "'-%'").   
      "   and s.SERVICE_NIVEAU=n.SERVICE_NIVEAU".
      "   and a.AGENT_VALIDE=1)".
      // tri par : service d'abord, niveau, puis intitulé
      " order by TYPE, NIVEAU, INTITULE";
    
    return $this->dbConn->InitDataSet($strSql);  
  }
  
  /**
   * Retourne le CRC correspondant à la date de dernière modif de l'annuaire
   * @return int
   */
  public function getCRCLastUpdateAnnu()
  {
    $iRes = md5(time());
    $strSql = "select ".
      $this->dbConn->getDateFormat("YYYY-MM-DD-HH-MI-SS", "a.maxAgent", false)." as maxAgent,".
      $this->dbConn->getDateFormat("YYYY-MM-DD-HH-MI-SS", "s.maxServ", false)." as maxServ".
      " from".
      "  (select max(AGENT_DATE_MAJ) as maxAgent from SIT_AGENT) a," .
      "  (select max(SERVICE_DATE_MAJ) as maxServ from SIT_SERVICE) s";
      
    $dsAgent = $this->dbConn->InitDataSet($strSql);
    if( $drAgent = $dsAgent->getRowIter() ) {
      $iDate1 = str_replace("-", "", $drAgent->getValueName("AGENT_DATEMAJ"));
      $iDate2 = str_replace("-", "", $drAgent->getValueName("SERVICE_DATEMAJ"));
      $iRes = md5($iDate1+$iDate2); 
    } 
    return $iRes;
  }

  /**
   *  Retourne la liste des agents pour l'organigramme
   * @return AlkDs
   */
  public function GetDs_arboServiceFeuille()
  {
    $strSep = "''";

    $strSql = "select a.AGENT_ID as ID, ".
      $this->dbConn->getConcat("a.AGENT_NOM", "' '", "a.AGENT_PRENOM")." as NOM,".
      "  0 as DROIT, ase.SERVICE_ID as ID_NOEUD, ".
      $this->dbConn->getConcat("'javascript:OpenPopupFiche(".$strSep."agent".$strSep.", '", "a.AGENT_ID", "')'")." as URL, ".
      "  a.AGENT_PRIV as ICONE ".
      " from SIT_SERVICE s".
      "   inner join SIT_AGENT_SERVICE ase on s.SERVICE_ID=ase.SERVICE_ID".
      "   inner join SIT_AGENT a on ase.AGENT_ID=a.AGENT_ID".
      "   inner join ".
      "     (select ase0.AGENT_ID, MAX(s0.SERVICE_NIVEAU) as NIVEAU".
      "      from SIT_SERVICE s0 inner join SIT_AGENT_SERVICE ase0 on s0.SERVICE_ID=ase0.SERVICE_ID".
      "      group by ase0.AGENT_ID) n".
      "     on (s.SERVICE_NIVEAU=n.NIVEAU and a.AGENT_ID=n.AGENT_ID)".
        " where a.AGENT_VALIDE=1".
      " order by NOM";
    
    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   *  Retourne la liste des services enfant à idServPere
   *        07_service_list.php
   *
   * @return Retourne un dataSet
   */
  function GetDs_listeService($user_priv, $service_id, $servNivPere, $idServPere, $dept_id=-1)
  {
    $strSqlService = ""; 
    if( ($user_priv & ALK_PRIV_ANNU_SERV)==ALK_PRIV_ANNU_SERV && $servNivPere=="1" ) {
      $strSqlService = " and s.service_id = ".$service_id." ";
    }

    $strValAgInter = "0";
    $strJointureInter = "";
      
    $strSql = "select s.*,".
      $this->dbConn->compareSql("sp.NBR", "is", "null", "0", "sp.NBR")." as NB_FILS, ".
      $this->dbConn->compareSql("ag.NBR", "is", "null", "0", "ag.NBR")." as NB_AG, ".
      $strValAgInter." as NB_AGINTER".
      " from SIT_SERVICE s".
      "   left join (select SERVICE_PERE, count(SERVICE_PERE) as NBR from SIT_SERVICE group by SERVICE_PERE) sp".
      "              on s.SERVICE_ID=sp.SERVICE_PERE ".
      "   left join (select SERVICE_ID,count(SERVICE_ID) as NBR from SIT_AGENT_SERVICE group by SERVICE_ID) ag ".
      "              on s.SERVICE_ID=ag.SERVICE_ID".
      $strJointureInter.
      " where s.SERVICE_PERE=".$idServPere
      .$strSqlService.
      ( $dept_id != -1
        ? " and dept_id=".$dept_id
        : "" ).
      " order by s.SERVICE_INTITULE";

    return $this->dbConn->InitDataSet($strSql);
  }
  
  /**
   * Retourne l'identifiant du département associé à
   *  - service_id si <>-1 
   *  - l'utilisateur par l'intermédiaire de son service si service=-1
   * Retourne -1 si non trouvé.
   * @param user_id    identifiant de l'utilisateur
   * @param service_id identifiant du service sélectionné
   * @return int
   */
  public function getDefaultUserDeptId($user_id, $service_id=-1)
  {
    $dept_id = -1;
    $strSql = "select s.DEPT_ID".
      " from SIT_SERVICE s".
      ( $service_id == -1
        ? " inner join SIT_AGENT_SERVICE ase on ase.SERVICE_ID=s.SERVICE_ID"
        : "" ).
      " where ".
      ( $service_id == -1
        ? "ase.AGENT_ID=".$user_id." and s.SERVICE_NIVEAU=1"
        : "s.SERVICE_ID=".$service_id );
    $dsUser = $this->dbConn->InitDataSet($strSql);
    if( $drUser = $dsUser->getRowIter() ) {
      $dept_id = $drUser->getValueName("DEPT_ID");
    } 
    return $dept_id;
  }

  /**
   *  Retourne la fiche agent avec l'email
   *        agent_id peut être une suite d'id séparés par une virgule
   *        lsDif_01/04_user_list.php, lsDif_01/01_message_sql.php
   *
   * @param agent_id Identifiant de l'agent
   * @param bDs      =true pour retourner un dataSet, =false pour 
   * @param bUnion   =true pour fournir les colonnes pour une union, =false pour les colonnes normales
   * @return Retourne un dataSet ou un string selon la valeur de bDs
   */
  function GetStrDs_ficheAgentMail($agent_id, $bDs=true, $bUnion=true, $bkeyLdap = false)
  {
    if( $bUnion == true )
      $strSelect = "AGENT_PRENOM as PRENOM, AGENT_NOM as NOM, AGENT_ID";
    else
      $strSelect = "AGENT_PRENOM as PRENOM, AGENT_NOM as NOM, AGENT_ID, AGENT_LOGIN as LOGIN, AGENT_PWD as PWD";
    
    if($bkeyLdap){
      $strSql = "select ".$strSelect.", AGENT_MAIL , 7 as COOR_PUBLICATION".
                " from SIT_AGENT ".
                " where AGENT_ID in (".$agent_id.")";
    }else{
      $strSql = "select ".$strSelect.", AGENT_MAIL as MAIL, 7 as COOR_PUBLICATION".
                " from SIT_AGENT ".
                " where AGENT_ID in (".$agent_id.")";
    }
      
    if( $bDs == true )
      return $this->dbConn->InitDataSet($strSql);
    return $strSql;
  }

  /**
   *  Retourne un tableau de listes des agents appartenant au service de niveau 1 service_id
   *        Le découpage des listes en tableau pour éviter d'avoir des requêtes trop longues
   *
   * @param  service_id   Identifiant du service de niveau 1
   * @return Retourne un tableau contenant des listes d'agent_id séparés par une virgule
   */
  function GetStr_listeAgentIdByService($service_id)
  {
    $cpt = 0;
    $tabListAgentID[$cpt] = "-1";

    $strSql = "select distinct a.AGENT_ID".
      " from SIT_AGENT a".
      "   inner join SIT_AGENT_SERVICE ase on a.AGENT_ID=ase.AGENT_ID".
      " where a.AGENT_VALIDE>=0 and ase.SERVICE_ID=".$service_id;
    
    if( ALK_BDD_TYPE != ALK_SGBD_MYSQL ) {
      $tabListAgentID = $strSql;
    } else {
      $dsAgent = $this->dbConn->InitDataSet($strSql);
      while( $drAgent = $dsAgent->getRowIter() ) {
        $agent_id = $drAgent->getValueName("AGENT_ID");
        $tabListAgentID[$cpt] .= ",".$agent_id;
        if( strlen($tabListAgentID[$cpt]) > 500 ) {
          $cpt++;
          $tabListAgentID[$cpt] = "-1";
        }
      }
    }
    
    return $tabListAgentID;
  }

  /**
   *  Retourne la liste des numéros de téléphone répondant aux critères
   * 
   * @param strNom       Recherche sur le début du nom
   * @param strPrenom    Recherche sur le début du prénom
   * @param strTrigramme Recherche sur le trigramme
   * @param bEtatOnly    true par défaut pour filtrer les agents appartenants aux services de type etat, faux pour ne pas appliquer de filtre
   * @param $idFirst     indice début de pagination, =0 par  défaut
   * @param $idLast      indice fin de pagination, =-1 par  défaut pour ne pas appliquer de pagination
   * @return Retourne un dataSet
   */
  public function getDsListeAnnuTel($strNom, $strPrenom, $strTrigramme="", $bEtatOnly=true, $idFirst=0, $idLast=-1)
  {
    $strWhere = "";
    if( $strNom != "" ) {
      $strNom = str_replace("*", "%", $strNom);
      $strWhere .= " and ".$this->dbConn->GetLowerCase("a.AGENT_NOM")." like ".
        $this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($strNom)."%'");
    }
    if( $strPrenom != "" ) {
      $strPrenom = str_replace("*", "%", $strPrenom);
      $strWhere .= " and ".$this->dbConn->GetLowerCase("a.AGENT_PRENOM")." like ".
        $this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($strPrenom)."%'");
    }
    if( $strTrigramme != "" ) {
      $strTrigramme = str_replace("*", "%", $strTrigramme);
      $strWhere .= " and ".$this->dbConn->GetLowerCase("a.AGENT_TRIGRAMME")." like ".
        $this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($strTrigramme)."%'");      
    }   

    $strSql = "select distinct a.AGENT_ID, a.AGENT_NOM, a.AGENT_PRENOM, a.AGENT_TRIGRAMME, a.AGENT_CIVILITE,".
      " a.AGENT_TEL, a.AGENT_TELPOSTE, a.AGENT_MOBILE, a.AGENT_MAIL, a.AGENT_FAX".
      " from SIT_AGENT a".
      ( $bEtatOnly
        ? " inner join SIT_AGENT_SERVICE ase on a.AGENT_ID=ase.AGENT_ID".
          " inner join SIT_SERVICE s on ase.SERVICE_ID=s.SERVICE_ID"
        : "" ).
      " where a.AGENT_VALIDE>0 ".
      $strWhere.
      ( $bEtatOnly
        ? " and s.SERVICE_ETAT=1"
        : "" ).
      " order by a.AGENT_NOM, a.AGENT_PRENOM, a.AGENT_TRIGRAMME";
    return $this->dbConn->InitDataSet($strSql, $idFirst, $idLast);
  }

  /**
   *   Retourne la liste des fonctions avancées
   *
   * @param strRecherche  Critère de recherche
   * @param idFirst       Indice début de pagination
   * @param idLast        Indice fin de pagination
   * @return Retourne un dataSet
   */
  function getDs_listeFonctionAv($strRecherche, $idFirst=0, $idLast=-1)
  {
    $strWhere = "";
    if( $strRecherche != "" ) {
      $strWhere = " where ".$this->dbConn->GetLowerCase("FONC_INTITULE")." like ".
        $this->dbConn->GetLowerCase("'%".$this->dbConn->analyseSql($strRecherche)."%'");
    }

    $strSql = "select f.FONC_INTITULE, f.FONC_ID, fa.NB_AGENT, fs.NB_SERV".
      " from".
      "  SIT_FONCTION f".
      "  left join".
      "    (select count(AGENT_ID) as NB_AGENT, FONC_ID".
      "     from SIT_AGENT_FONCTION group by FONC_ID) fa on f.FONC_ID=fa.FONC_ID".
      "   left join".
      "    (select count(SERVICE_ID) as NB_SERV, FONC_ID".
      "     from SIT_SERVICE_FONCTION group by FONC_ID) fs on f.FONC_ID=fs.FONC_ID".
      ( $strWhere != "" ? $strWhere : "" ).
      " order by f.FONC_INTITULE";
      
    return $this->dbConn->InitDataSet($strSql, $idFirst, $idLast);
  }

  /**
   *  Retourne la liste des agents pour le trombinoscope répondant aux critères
   * 
   * @param strNom       Recherche sur le début du nom
   * @param strPrenom    Recherche sur le début du prénom
   * @param strTrigramme Recherche sur le trigramme
   * @return Retourne un dataSet
   */
  function getDsListeAnnuTrombi($strNom, $strPrenom, $idServ, $strTrigramme="", $bDirecteur=false, $bEtatOnly=true, $bRec=false)
  {
    $strWhere = "";
    if( $strNom != "" ) {
      $strNom = str_replace("*", "%", $strNom);
      $strWhere .= " and ".$this->dbConn->GetLowerCase("a.AGENT_NOM")." like ".
        $this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($strNom)."%'");
      }
    if( $strPrenom != "" ) {
      $strPrenom = str_replace("*", "%", $strPrenom);
      $strWhere .= " and ".$this->dbConn->GetLowerCase("a.AGENT_PRENOM")." like ".
        $this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($strPrenom)."%'");
    }
    if( $strTrigramme != "" ) {
      $strTrigramme = str_replace("*", "%", $strTrigramme);
      $strWhere .= " and ".$this->dbConn->GetLowerCase("a.AGENT_TRIGRAMME")." like ".
        $this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($strTrigramme)."%'");      
    }   
    if( $idServ != "-1" ) {
      $strWhere .= " and ase.SERVICE_ID=".$idServ;
    }   
    if( $bEtatOnly ) {
      $strWhere .= " and s.SERVICE_ETAT=1";
    }
      
    /*  $strSql = "create temporary table TMP_AGENT_".$agent_id.
        " select a0.AGENT_ID, a0.AGENT_NOM, a0.AGENT_PRENOM, a0.AGENT_PHOTO, a0.AGENT_TRIGRAMME, max(s0.SERVICE_NIVEAU) as NIV".
        " from SIT_AGENT a0, SIT_AGENT_SERVICE ase0, SIT_SERVICE s0".
        " where a0.AGENT_VALIDE>0 and a0.AGENT_ID=ase0.AGENT_ID and ase0.SERVICE_ID=s0.SERVICE_ID".
        " group by a0.AGENT_ID, a0.AGENT_NOM, a0.AGENT_PRENOM, a0.AGENT_PHOTO, a0.AGENT_TRIGRAMME";
      $this->dbConn->executeSql($strSql);
      
      $strSql = "select distinct a.AGENT_ID, a.AGENT_NOM, a.AGENT_PRENOM, a.AGENT_PHOTO, a.AGENT_TRIGRAMME,".
        "  a.NIV, 1 as ordre".
        " from ".
        "   TMP_AGENT_".$agent_id." a,".
        "   SIT_AGENT_SERVICE ase,".
        "   (select SERVICE_ID, SERVICE_INTITULE, SERVICE_NIVEAU from SIT_SERVICE) s".
        " where a.AGENT_ID=ase.AGENT_ID and ase.SERVICE_ID=s.SERVICE_ID and s.SERVICE_NIVEAU=a.NIV".
        $strWhere;
      
      if( $bDirecteur==true && $idServ!="-1" ) {
        $strSql .= " union select distinct a.AGENT_ID, a.AGENT_NOM, a.AGENT_PRENOM, a.AGENT_PHOTO, a.AGENT_TRIGRAMME, s.SERVICE_NIVEAU as NIV, 0 as ordre".
          " from ".
          "   (select SERVICE_ID, AGENT_ID, SERVICE_NIVEAU from SIT_SERVICE where SERVICE_ID=".$idServ.") s left join SIT_AGENT a on s.AGENT_ID=a.AGENT_ID".
          " where a.AGENT_VALIDE>0".
          " order by ordre, AGENT_NOM, AGENT_PRENOM, AGENT_TRIGRAMME";
      } else {
        $strSql .= " order by AGENT_NOM, AGENT_PRENOM, AGENT_TRIGRAMME";
      }
    */
    $strSql = "select distinct".
      " a.AGENT_ID, a.AGENT_CIVILITE, a.AGENT_NOM, a.AGENT_PRENOM, a.AGENT_PHOTO, a.AGENT_TRIGRAMME, a.AGENT_FONCTION,".
      " a.AGENT_MAIL, a.AGENT_TEL, a.AGENT_MOBILE, a.AGENT_TELPOSTE".
      " from SIT_SERVICE s".
      "  inner join SIT_AGENT_SERVICE ase on s.SERVICE_ID=ase.SERVICE_ID".
      ( !$bRec
        ? "  inner join (select count(SERVICE_ID) as SERVICE_NIVEAU, AGENT_ID from SIT_AGENT_SERVICE group by AGENT_ID) n on ase.AGENT_ID=n.AGENT_ID".
          "  inner join SIT_AGENT a on n.AGENT_ID=a.AGENT_ID"
        : "  inner join SIT_AGENT a on ase.AGENT_ID=a.AGENT_ID" ).
      " where a.AGENT_VALIDE>0".
      $strWhere.
      ( !$bRec
        ? " and s.SERVICE_NIVEAU=n.SERVICE_NIVEAU"
        : "" ).
      " order by a.AGENT_NOM, a.AGENT_PRENOM";
      
    /*$strSql = "select a.AGENT_ID, a.AGENT_CIVILITE, a.AGENT_NOM, a.AGENT_PRENOM, a.AGENT_PHOTO, a.AGENT_TRIGRAMME, a.AGENT_FONCTION".
      " from SIT_AGENT a".
        "   inner join SIT_AGENT_SERVICE ase on ase.AGENT_ID=a.AGENT_ID".
        "   inner join (select SERVICE_ID, SERVICE_NIVEAU
        " where a.AGENT_VALIDE>0 and a.AGENT_ID>9 ".
        $strWhere.
        " order by a.AGENT_NOM, a.AGENT_PRENOM";*/

    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   * Retourne la liste des services de niveau iNiv fils de idServPere pour un combobox
   * @param iNiv        Niveau de service souhaité
   * @param idServPere  identifiant ou liste d'identifiant séparé par une virgule des services parents
   * @param bLong       fixe une longueur limite
   * @param lgMax       longueur max, =55 par défaut
   * @param iEtat       filtre les services de type etat si !="", ="" par défaut
   * @param $dept_id    id de l'entité géographique liée, =-2 ou -1 si non utilisé, =-2 par défaut
   */
  public function getDsListeServiceForTrombi($iNiv="1", $idServPere="-1", $bLong=true, $lgMax=55, $iEtat="", $dept_id="-2", $idServSelect="-1")
  {
    $strChamp = "SERVICE_INTITULE_COURT";
    if( $iNiv != "1" ) 
      $strChamp = $this->dbConn->GetSubstring("SERVICE_INTITULE", 0, $lgMax);

    if( $bLong == true ) {
      $strChamp = "SERVICE_INTITULE";
      if( $lgMax != 55 )
        $this->dbConn->GetSubstring("SERVICE_INTITULE", 0, $lgMax);
    }
       
    $strSql = "select ".$strChamp." as intitule, SERVICE_ID".
      " from SIT_SERVICE".
      " where SERVICE_NIVEAU>0 and SERVICE_NIVEAU<=".$iNiv.
      ( $iNiv>1 ? " and SERVICE_PERE in (".$idServPere.")" : "" ).
      ( $iEtat!="" ? " and SERVICE_ETAT=".$iEtat : "" ).
      ( $dept_id!="-2" && $dept_id!="-1" ? " and DEPT_ID=".$dept_id : "" ).
      ( $idServSelect!="-1" ? " and SERVICE_ID=".$idServSelect : "" ).
      " order by SERVICE_NIVEAU, ".$strChamp;

    return $this->dbConn->InitDataSet($strSql);
  }
  
  /**
   *  retourne vrai si le login existe dans la base
   * @param strLogin login de l'agent
   */
  function getAgentIdByLogin($strLogin){
    $bExist = false;
    $strSql = "select AGENT_ID from SIT_AGENT where AGENT_LOGIN = '".$this->dbConn->analyseSql($strLogin)."'";
    return $this->dbConn->InitDataSet($strSql);
  }

	
  /**
   *  Retourne les infos d'un agent à partir du nom, prénom, mail
   *
   * @param agent_nom     nom de l'agent
   * @param agent_prenom  prenom de l'agent
   * @param agent_mail    mail de l'agent
   * @return Retourne un dataset
   */
  function GetDsAgentByNomPrenomMail($agent_nom, $agent_prenom, $agent_mail)
  {	
    $strSql = "select a.*, a.AGENT_MAIL as MAIL".
      " from SIT_AGENT a " .
      " where ".
      $this->dbConn->GetUpperCase("a.AGENT_NOM")." = ".$this->dbConn->GetUpperCase("'".$this->dbConn->AnalyseSql($agent_nom)."'").
      " and ".$this->dbConn->GetLowerCase("a.AGENT_PRENOM")." = ".$this->dbConn->GetLowerCase("'".$this->dbConn->AnalyseSql($agent_prenom)."'").
      " and ".$this->dbConn->GetLowerCase("a.AGENT_MAIL")." = ".$this->dbConn->GetLowerCase("'".$this->dbConn->AnalyseSql($agent_mail)."'");
    return $this->dbConn->InitDataSet($strSql);
  }

  /**
   *  calcul puis retourne le mot de passe par défaut à partir du sigle du service
   *
   * @param strServiceSigle Sigle du service
   * @return Retourne un string : le mot de passe par défaut
   */
  public function verifyPassword($strServiceSigle=ALK_DEFAULTPWD_ANNU)
  {
    // supprime tout caractere différent de 0..9A-Z
    $strServiceSigle = mb_strtoupper($strServiceSigle);
    $strPwd = mb_ereg_replace("([^A-Z0-9])", "", $strServiceSigle);
    $strPwd = mb_ereg_replace("\\\\", "", $strPwd);
    if( mb_strlen($strPwd) > ALK_ANNU_LG_MAX_PWD ) {
      $strPwd = mb_substr($strPwd, 0, ALK_ANNU_LG_MAX_PWD);
    }
    if( mb_strlen($strPwd) < ALK_ANNU_LG_MIN_PWD ) {
      $strPwd = ALK_ANNU_DEFAULTPWD;
    }
  
    // cas particulier du mot de passe des collectivités
    if( ALK_B_ANNU_COLLECT_PWD_SIT && $strPwd == ALK_ANNU_COLLECT_OLD_PWD ) 
      $strPwd = ALK_ANNU_COLLECT_NEW_PWD;
  
    return $strPwd;
  }
  
  /**
   * Retourne la liste des utilisateurs d'un service (service_id est le dernier niveau de agent_id si bRec=false)
   * Si bRec=true, la liste contient tous les utlisateurs présents sur les noeuds de la sous-arbo de service_id
   * @param service_id  identifiant du service
   * @param bRec        =true pour prendre en compte les sous-service, =false par défaut
   * @return AlkDs    
   */
  public function getDsListUserByServ($service_id, $bRec=false, $bValideOnly=false)
  {
    $strSql = "select s.SERVICE_ID, s.SERVICE_INTITULE, a.AGENT_ID, a.AGENT_NOM, a.AGENT_PRENOM, a.AGENT_MAIL, a.AGENT_LOGIN, a.PROFIL_ID".
      " from SIT_SERVICE s".
      "  inner join SIT_AGENT_SERVICE ase on s.SERVICE_ID=ase.SERVICE_ID".
      ( !$bRec
        ? "  inner join (select count(SERVICE_ID) as SERVICE_NIVEAU, AGENT_ID from SIT_AGENT_SERVICE group by AGENT_ID) n on ase.AGENT_ID=n.AGENT_ID".
          "  inner join SIT_AGENT a on n.AGENT_ID=a.AGENT_ID"
        : "  inner join SIT_AGENT a on ase.AGENT_ID=a.AGENT_ID" ).
      " where a.AGENT_VALIDE>".( $bValideOnly ? "0" : "-1" ).
      "   and ase.SERVICE_ID=".$service_id.
      ( !$bRec
        ? " and s.SERVICE_NIVEAU=n.SERVICE_NIVEAU"
        : "" ).
      " order by a.AGENT_NOM, a.AGENT_PRENOM";
        
    return $this->dbConn->InitDataSet($strSql);
  }
  
  /**
   * Retourne la liste des utilisateurs ayant l'adresse mail passée en paramètre
   * @param agent_mail     mail de l'agent recherché
   * @param service_niveau niveau du service en cours
   */
  public function getDsUserInfoByMail($agent_mail, $service_niveau=1)
  {
    $strSql = "select a.*, ase.SERVICE_ID, s.SERVICE_LDAPUID".
      " from SIT_AGENT a".
      "   inner join SIT_AGENT_SERVICE ase on a.AGENT_ID=ase.AGENT_ID".
      "   inner join SIT_SERVICE s on ase.SERVICE_ID=s.SERVICE_ID".
      " where ".$this->dbConn->getLowerCase("a.AGENT_MAIL")."=".
      $this->dbConn->getLowerCase("'".$this->dbConn->analyseSql($agent_mail)."'").
      " and s.SERVICE_NIVEAU=".$service_niveau;
    return $this->dbConn->InitDataSet($strSql);
  }


  /**
   * Retourne la liste des utilisateurs ayant l'adresse mail passée en paramètre
   * @param agent_mail     mail de l'agent recherché
   * @param service_niveau niveau du service en cours
   */
  public function getDsUserInfoByLdapKey($strLdapKey, $strKeyField, $service_niveau=1)
  {
    $strSql = "select a.*, ase.SERVICE_ID, s.SERVICE_LDAPUID".
      " from SIT_AGENT a".
      "   inner join SIT_AGENT_SERVICE ase on a.AGENT_ID=ase.AGENT_ID".
      "   inner join SIT_SERVICE s on ase.SERVICE_ID=s.SERVICE_ID".
      " where a.AGENT_VALIDE>=0 and ".$this->dbConn->getLowerCase("a.".$strKeyField)."=".
      $this->dbConn->getLowerCase("'".$this->dbConn->analyseSql($strLdapKey)."'").
      " and s.SERVICE_NIVEAU=".$service_niveau;
    return $this->dbConn->InitDataSet($strSql);
  }

 
  /**
   * Retourne dans un tableau les identifiants de service père successif jusqu'au service de niveau 1
   * @param service_id      identifiant d'un service de référence
   * @return array   
   */
  public function getTabServiceTree($service_id, $service_ldapuid, $service_ldapdn)
  {
    $tabRes = array();
    $strSql = "select SERVICE_ARBRE".
      " from SIT_SERVICE".
      " where 1=1".
      ( $service_id>-1       ? " and SERVICE_ID=".$service_id : "" ).
      ( $service_ldapuid!="" ? " and SERVICE_LDAPUID='".$this->dbConn->analyseSql($service_ldapuid)."'" : "").
      ( $service_ldapdn!=""  ? " and SERVICE_LDAPDN='".$this->dbConn->analyseSql($service_ldapdn)."'" : "" );
    $dsServ = $this->dbConn->InitDataSet($strSql);
    $nbServ = $dsServ->getCountTotDr(); 
    if( $nbServ==1 && $drServ = $dsServ->getRowIter() ) {
      $serv_arbre = $drServ->getValueName("SERVICE_ARBRE");
      $serv_arbre = mb_substr($serv_arbre, 1, -1); //retire les tirets en début et fin de chaine
      $tabRes = explode("-", $serv_arbre);
    }
    return $tabRes;
  }
    
  /* méthodes ldap */
  
  /**
   * Retourne la fiche ldap d'un utilisateur
   * @param strUserKey  identifiant de l'utilisateur
   * @return AlkDs
   */
  public function getDsLdapUserInfo($strUserKey, $strDn="")
  {
    if( is_null($this->connLdap) ) return new AlkDsLdap(null);
    $tabAttrib = array_keys($this->tabLdapAssoc);
    if( empty($tabAttrib) ) return;
    $strFieldKey = $tabAttrib[0]; 
    return $this->connLdap->initDataSet(ALK_LDAPQUERY_SEARCH, "(".$strFieldKey."=".$strUserKey.")", $strDn, $tabAttrib);
  }

  /**
   * Retourne la fiche ldap d'un utilisateur
   * @param strUserDN  identifiant de l'utilisateur
   * @return AlkDs
   */
  public function getAgentKeyFromLdapDn($strUserDN)
  {
    $strKey_value = "";
    if( is_null($this->connLdap) ) return new AlkDsLdap(null);
    $tabAttrib = array_keys($this->tabLdapAssoc);
    if( empty($tabAttrib) ) return;
    $dsAgentKey =  $this->connLdap->initDataSet(ALK_LDAPQUERY_SEARCH, "(objectclass=*)", $strUserDN, $tabAttrib);
    if( $drAgentKey = $dsAgentKey->getRowIter() ) {
          return $strKey_value = $drAgentKey->getValueName($this->tabLdapAssoc[$tabAttrib[0]]["field"]);
    }
    return $strKey_value;
  }

  /**
   * Retourne la liste des noeuds racines de chaque service de l'annuaire ldap
   * @return AlkDs
   */
  public function getDsLdapListServNode()
  {
    if( is_null($this->connLdap) ) return new AlkDsLdap(null);
    $tabAttrib = array("dn");
    return $this->connLdap->initDataSet(ALK_LDAPQUERY_LIST, "(objectclass=*)", ALK_LDAP_BASE_DN_NODE, $tabAttrib);
  }

  /**
   * Retourne la liste des service placé sous un noeud ldap
   * @param strNode  identifiant de l'utilisateur
   * @return AlkDs
   */
  public function getDsLdapListServ($strNode)
  {
    if( is_null($this->connLdap) ) return new AlkDsLdap(null);
    
    $tabAttribTmp = array_keys($this->tabLdapAssoc);
    if( empty($tabAttribTmp) ) return;
    $tabAttrib = array($tabAttribTmp[1]);

    $this->connLdap->setSizeLimit(10000);
    $this->connLdap->setTimeLimit(300);
    ini_set("memory_limit", "32M");
    ini_set("max_execution_time", "300");

    return $this->connLdap->initDataSet(ALK_LDAPQUERY_SEARCH, "(objectclass=*)", $strNode, $tabAttrib, $tabAttrib[0]);
  }
  
  /**
   * Retourne la liste des utilisateurs de l'annuaire ldap appartenant au service 
   * @param strLdapDn   dn racine où chercher les utilisateurs
   * @param strLdapUid  identifiant du service filtre
   * @return AlkDs 
   */
  public function getDsLdapListUserFromServ($strLdapDn, $strLdapUid, $strLdapKeyParam="AGENT_MAIL")
  {
    if( is_null($this->connLdap) ) return new AlkDsLdap(null);
    //cas d'un annuaire LDAP classique de type MI
    if (!ALK_B_LDAP_AD){
      $tabAttrib = array();
      $strServiceUid = "";
      foreach($this->tabLdapAssoc as $strLdapKey => $tabAnnu) {
        if( $tabAnnu["field"] == "AGENT_NOM" ||
            $tabAnnu["field"] == "AGENT_PRENOM" ||
            $tabAnnu["field"] == "AGENT_MAIL" ||
            $tabAnnu["field"] == $strLdapKeyParam ) {
          $tabAttrib[] = $strLdapKey;
        }
        if( $tabAnnu["field"] == "SERVICE_LDAPUID" ) {
          $strServiceUid = $strLdapKey;
        }
      }
      $this->connLdap->setSizeLimit(10000);
      $this->connLdap->setTimeLimit(300);
      ini_set("memory_limit", "32M");
      ini_set("max_execution_time", "300");
      return $this->connLdap->initDataSet(ALK_LDAPQUERY_SEARCH, "(objectClass=*)", 
                                  $strLdapUid.($strLdapUid != "" ? "," : "").$strLdapDn, $tabAttrib, $tabAttrib[0], 0, -1, false);
    } else {
      //cas d'active Directory (Les utilisateurs d'un service sont dans Users)
      //(&(objectclass=*)(memberOf=CN=Direction informatique,OU=Direction informatique,OU=Mairie,OU=Ermont,DC=alkante,DC=domdev1))
      $tabAttrib = array();
      //$strUID = spliti("OU=", $strLdapUid);
      //$strLdapDnSearch = "CN=".str_replace(",","",$strUID[1]).",".$strLdapUid.",".$strLdapDn;
      $this->connLdap->setSizeLimit(10000);
      $this->connLdap->setTimeLimit(300);
      ini_set("memory_limit", "32M");
      ini_set("max_execution_time", "300");
      
      $strServiceUid = "";
      foreach($this->tabLdapAssoc as $strLdapKey => $tabAnnu) {
        if( $tabAnnu["field"] == "AGENT_NOM" ||
            $tabAnnu["field"] == "AGENT_PRENOM" ||
            $tabAnnu["field"] == "AGENT_MAIL" ||
            $tabAnnu["field"] == $strLdapKey ) {
          $tabAttrib[] = $strLdapKey;
        }
        if( $tabAnnu["field"] == "SERVICE_LDAPUID" ) {
          $strServiceUid = $strLdapKey;
        }
      } 
      return $this->connLdap->initDataSet(ALK_LDAPQUERY_SEARCH, "(&(objectclass=*)(memberOf=".$strLdapUid."))", 
                                  ALK_LDAP_BASE_DN_NODE_AGENTS, $tabAttrib, $tabAttrib[0], 0, -1, false);
    }
  }
  
  /**
   * Retourne la liste des services de l'annuaire ldap distant
   * @param
   * @param
   * @return AlkDs 
   */
  public function getDsLdapListService()
  {
    if( is_null($this->connLdap) ) return new AlkDsLdap(null);
    
    $tabAttrib = array();
    $this->connLdap->setSizeLimit(10000);
    $this->connLdap->setTimeLimit(300);
    ini_set("memory_limit", "32M");
    ini_set("max_execution_time", "300");
      
    foreach($this->tabLdapAssoc as $strLdapKey => $tabAnnu) {
      if( $tabAnnu["field"] == "SERVICE_INTITULE" || $tabAnnu["field"] == "AGENT_ID" ) {
        $tabAttrib[] = $strLdapKey;
      }
    } 
          
    return $this->connLdap->initDataSet(ALK_LDAPQUERY_SEARCH, "(objectClass=organizationalUnit)", 
                                        ALK_LDAP_BASE_DN_NODE_SERV, $tabAttrib, $tabAttrib[0], 0, -1, false);
  }
 
  /**
   * Retourne la liste des utilisateurs de l'annuaire ldap appartenant au service 
   * @param strLdapDn   dn racine où chercher les utilisateurs
   * @param strLdapUid  identifiant du service filtre
   * @return AlkDs 
   */
  public function getDsLdapListUsers($strLdapUid)
  {
    if( is_null($this->connLdap) ) return new AlkDsLdap(null);
    //cas d'un annuaire LDAP classique de type MI
    if (defined(ALK_B_LDAP_AD) && ALK_B_LDAP_AD != true){
      //Ldap classique
    }else{
      $tabAttrib = array();
      $this->connLdap->setSizeLimit(10000);
      $this->connLdap->setTimeLimit(300);
      ini_set("memory_limit", "32M");
      ini_set("max_execution_time", "300");
      
      $strServiceUid = "";
      foreach($this->tabLdapAssoc as $strLdapKey => $tabAnnu) {
        if( $tabAnnu["field"] == "AGENT_NOM" ||
            $tabAnnu["field"] == "AGENT_PRENOM" ||
            $tabAnnu["field"] == "AGENT_MAIL" ) {
          $tabAttrib[] = $strLdapKey;
        }
        if( $tabAnnu["field"] == "SERVICE_LDAPUID" ) {
          $strServiceUid = $strLdapKey;
        }
      } 
      return $this->connLdap->initDataSet(ALK_LDAPQUERY_SEARCH, "(|(objectClass=User) (objectClass=contact))", 
                                  ALK_LDAP_BASE_DN_NODE_AGENTS, $tabAttrib, $tabAttrib[0], 0, -1, false);
    }
  }
  
  // ********** gestion des abonnés ********** //
  
  /**
   * Retourne l'identifiant de l'abonné d'un agent
   * @param agent_id  identifiant de l'utilisateur connecté
   * @return array
   */
  public function getAbonneIdByAgentId($agent_id)
  {
    $abonne_id = -1;
    $abonne_pseudo = "";
    $strSql = "select ABONNE_ID, ABONNE_PSEUDO from SIT_ABONNE where AGENT_ID=".$agent_id;
    $dsAb = $this->dbConn->InitDataSet($strSql);
    if( $drAb = $dsAb->getRowIter() ) {
      $abonne_id     = $drAb->getValueName("ABONNE_ID");
      $abonne_pseudo = $drAb->getValueName("ABONNE_PSEUDO"); 
    }
    return array($abonne_id, $abonne_pseudo);
  }

  /**
   * Retourne la fiche de l'abonné correspondant aux paramètres et à la méthode sélectionnée
   * @param abonne_id      identifiant 
   * @param abonne_nom     nom
   * @param abonne_prenom  prénom
   * @param abonne_mail    mail
   * @parap iMethode       =0 affectue un ou dans l'ordre suivant : abonne_id ou abonne_mail ou (nom et prénom)
   *                       =1 affectue un ou dans l'ordre suivant : abonne_id et (abonne_mail ou (nom et prénom))
   * @return AlkDs
   */
  public function getDsAbonneByIdOrNameOrMail($abonne_id, $abonne_nom, $abonne_prenom, $abonne_mail, $iMethode=0)
  {
    $strSql = "select ABONNE_ID" .
      " from SIT_ABONNE" .
      " where " .
      ( $iMethode == "0"
        ? "ABONNE_ID=".$abonne_id.
          " or ABONNE_MAIL like '".$this->dbConn->analyseSql($abonne_mail)."'".
          " or (".
          $this->dbConn->GetLowerCase("ABONNE_NOM")." like ".$this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($abonne_nom)."'").
          " and ".
          $this->dbConn->GetLowerCase("ABONNE_PRENOM")." like ".$this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($abonne_prenom)."'").
          ")"
        : ( $iMethode == "1"
            ? "ABONNE_ID=".$abonne_id.
              " and (ABONNE_MAIL like '".$this->dbConn->analyseSql($abonne_mail)."'".
              " or (".
              $this->dbConn->GetLowerCase("ABONNE_NOM")." like ".$this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($abonne_nom)."'").
              " and ".
              $this->dbConn->GetLowerCase("ABONNE_PRENOM")." like ".$this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($abonne_prenom)."'").
              "))"
            : "1=0" ));

    return $this->dbConn->InitDataSet($strSql);
  }
  
  /**
   * retourne les infos d'un abonné à partir de son mail
   * @param abonne_mail  adresse mail de l'abonné recherché
   * @return AlkDs
   */
  public function getDsAbonneByMail($abonne_mail)
  {
    $strSql = "select  a.*, ".
      $this->dbConn->GetDateFormat('DD/MM/YYYY', 'a.ABONNE_DATE_INSCR', false)." as ABONNE_DATE_INSCR ,".
      $this->dbConn->GetDateFormat('DD/MM/YYYY', 'a.ABONNE_DATE_DESINSCR', false)." as ABONNE_DATE_DESINSCR ".
      " from SIT_ABONNE a ".
    " where a.ABONNE_MAIL like '".$this->dbConn->analyseSql($abonne_mail)."'";
    return $this->dbConn->InitDataSet($strSql);
  }
  
  /**
   * Retourne les infos d'un abonné
   *
   * @param idAbonne Identifiant de l'abonné
   * @return AlkDs
   */
  public function getDsAbonne($idAbonne) 
  {
    $strSql = "select a.*, ".$this->dbConn->GetConcat("a.ABONNE_NOM", "' '", "a.ABONNE_PRENOM")." as AGNOM, ".
      $this->dbConn->GetDateFormat('DD/MM/YYYY', 'a.ABONNE_DATE_INSCR', false)." as ABONNE_DATE_INSCR ,".
      $this->dbConn->GetDateFormat('DD/MM/YYYY', 'a.ABONNE_DATE_DESINSCR', false)." as ABONNE_DATE_DESINSCR ".
      " from SIT_ABONNE a ".
      " where a.ABONNE_ID=".$idAbonne;
    return $this->dbConn->initDataset( $strSql );
  }
  
  /**
   * Retourne lesinfos d'un abonné en fonction de son adresse mail
   *
   * @param strMail email de l'abonné
   * @param strMdp  mot de passe de l'abonné (=vide par défaut)
   * @return AlkDs
   */
  public function getDsAbonneByMailMdp($strMail, $strMdp="") 
  {
    $strSql = "select a.*, " .
      $this->dbConn->GetDateFormat('DD/MM/YYYY', 'a.ABONNE_DATE_INSCR', false)." as ABONNE_DATE_INSCR ,".
      $this->dbConn->GetDateFormat('DD/MM/YYYY', 'a.ABONNE_DATE_DESINSCR', false)." as ABONNE_DATE_DESINSCR ".
      " from SIT_ABONNE a".
      " where a.ABONNE_MAIL like '".$this->dbConn->analyseSql($strMail)."'".
      ( $strMdp!="" 
        ? " and a.ABONNE_PWD='".$this->dbConn->analyseSql($strMdp)."'" 
        : "" ).
      " and a.ABONNE_VALIDE=1";
    return $this->dbConn->initDataset( $strSql );
  }
  
   /**
   *  Vérifie si le mail saisi par l'abonné est déjà utilisé par un autre
   * @param idAbonne  Identifiant de l'abonne
   * @param strMail Mail proposé 
   * @return boolean 
   */
  public function getBoolVerifyMail($idAbonne, $strMail)
  {
    
    $strSql = " select ABONNE_MAIL from SIT_ABONNE ".
              " where ".$this->dbConn->GetLowerCase("ABONNE_MAIL")."=".
               $this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($strMail)."'").
              " and ABONNE_ID<>".$idAbonne;

    $dsPseudo = $this->dbConn->InitDataSet($strSql);
    
    return ($dsPseudo->getCountTotDr() > 0);  
  }
  /**
   * Retourne un abonné en fonction de l'agent et d'une application
   *
   * @param idAgent Identifiant de l'agent
   * @return AlkDs
   */
  public function getDsAbonneByAgent($idAgent) 
  {
    $strSql = "select sa.*, ".
      $this->dbConn->getConcat("sa.ABONNE_PRENOM", "' '", "sa.ABONNE_NOM")." as AGNOM, ".
      $this->dbConn->GetDateFormat('DD/MM/YYYY', 'sa.ABONNE_DATE_INSCR', false)." as ABONNE_DATE_INSCR ,".
      $this->dbConn->GetDateFormat('DD/MM/YYYY', 'sa.ABONNE_DATE_DESINSCR', false)." as ABONNE_DATE_DESINSCR ".
      " from SIT_ABONNE sa ".
      " where sa.AGENT_ID=".$idAgent.
      " and sa.ABONNE_VALIDE=1";
    return $this->dbConn->initDataset( $strSql );
  }
  
  /**
   * Retourne la liste des abonnes en fonction d'une application
   *
   * @param appli_id       identifiant de l'application
   * @param bValide        Retourne les abonnés valide si =1, non valide si =0, tous si =-1
   * @param strTextFiltre  filtre optionnel sur nom, prénom, mail ou id
   * @param iFirst         indice de début pagination, =0 par défaut
   * @param iLast          indice de fin pagination, =-1 par défaut pour aucune pagination
   * @return AlkDs
   *
  public function getDsListeAbonnesByAppli($appli_id, $bValide="1", $strTextFiltre="", $iFirst=0, $iLast=-1)
  {
    if ( $strTextFiltre!="" ){
      $strTextFiltreLarge  = $this->dbConn->GetLowerCase("'%".$strTextFiltre."%'");
      $strTextFiltreStrict = $this->dbConn->GetLowerCase("'".$strTextFiltre."%'");
      $strSqlWhere = " and " .
          "(   ".$this->dbConn->GetLowerCase("a.ABONNE_NOM") ." like ".$strTextFiltreLarge."".
          " or ".$this->dbConn->GetLowerCase("a.ABONNE_PRENOM") ." like ".$strTextFiltreStrict."".
          " or ".$this->dbConn->GetLowerCase("a.ABONNE_MAIL")." like ".$strTextFiltreLarge."".
          " or ".$this->dbConn->GetLowerCase("a.ABONNE_ID")  ." like ".$strTextFiltreStrict.")";
    }
    else
      $strSqlWhere = "";
          
    $strSql = "select distinct a.ABONNE_ID,".
      " a.ABONNE_NOM,".
      " a.ABONNE_PRENOM,".
      " a.ABONNE_PSEUDO,".
      " a.ABONNE_MAIL,".
      " a.ABONNE_VALIDE,".
      " a.AGENT_ID, " .
      " a.ABONNE_DATE_INSCR as ABONNE_DATEINSCR,".
      $this->dbConn->GetDateFormat('DD/MM/YYYY', 'a.ABONNE_DATE_INSCR', false)." as ABONNE_DATE_INSCR ,".
      $this->dbConn->GetDateFormat('DD/MM/YYYY', 'a.ABONNE_DATE_DESINSCR', false)." as ABONNE_DATE_DESINSCR".
      " from SIT_ABONNE a".
      "   inner join SIT_ABONNE_LISTE abl on a.ABONNE_ID=abl.ABONNE_ID".
      "   inner join SIT_APPLI_LISTE al on abl.LISTE_ID=al.LISTE_ID".
      " where al.APPLI_ID=".$appli_id.
      ( $bValide != "-1"
        ? " and a.ABONNE_VALIDE=".($bValide=="1" ? "1" : "0")
        : "" ).
      $strSqlWhere.
      " order by ABONNE_DATEINSCR desc, a.ABONNE_NOM asc";
    
    return $this->dbConn->InitDataSet($strSql, $iFirst, $iLast);
  }*/
  
  /**
   * récupère la liste de tous les abonnés (tous les champs)
   * @return AlkDs
   *
  public function GetDs_ListeAllAbonneAll()
  {
    $strSql = "select a.*, s.*, al.LISTE_ID,".
      ",".$this->dbConn->GetDateFormat('DD/MM/YYYY', 'a.ABONNE_DATE_INSCR', false)." as ABONNE_DATE_INSCR".
      ",".$this->dbConn->GetDateFormat('DD/MM/YYYY', 'a.ABONNE_DATE_DESINSCR', false)." as ABONNE_DATE_DESINSCR".
      " from SIT_ABONNE a".
      "   inner join SIT_ABONNE_LISTE abl on a.ABONNE_ID=abl.ABONNE_ID".
      "   inner join SIT_APPLI_LISTE al on abl.LISTE_ID=al.LISTE_ID".
      "   inner join SIT_APPLI s on al.APPLI_ID = s.APPLI_ID".
      " order by ABONNE_MAIL ";
       
    return $this->dbConn->InitDataSet($strSql);
  } */
  
  /**
   * Retourne la requête sql
   */
  public function getSqlListeAbonneForSearch($cont_id, $rabonne_type, $rappli_id, $rliste_id, $rabonne_nom, $rabonne_valide)
  {
    $strSelect = " distinct ab.*,  ".
      $this->dbConn->GetDateFormat("DD/MM/YYYY", "ab.ABONNE_DATE_INSCR", false)." as ABONNE_DATEINSCR,".
      $this->dbConn->GetDateFormat("DD/MM/YYYY", "ABONNE_DATE_DESINSCR", false)." as ABONNE_DATEDESINSCR";
    
    $strFrom = " from SIT_ABONNE ab".
      ( $rliste_id != "-1"
        ? " inner join SIT_ABONNE_LISTE al on ab.ABONNE_ID=al.ABONNE_ID"
        : "" );
      
    $strWhere = " where ".
      ( $rabonne_valide == "-1" 
        ? " ab.ABONNE_VALIDE>=0"
        : ( $rabonne_valide == "1" 
            ? " ab.ABONNE_VALIDE=1"
            : ( $rabonne_valide == "0"
              ? " ab.ABONNE_VALIDE=0 and ab.ABONNE_DATE_DESINSCR is not null"
              : " ab.ABONNE_VALIDE=0 and ab.ABONNE_DATE_INSCR is not null and ab.ABONNE_DATE_DESINSCR is null" ))).
      ( $rabonne_type == "1"
        ? " and ab.AGENT_ID<>".ALK_USER_ID_INTERNET
        : ( $rabonne_type == "2"
            ? " and ab.AGENT_ID=".ALK_USER_ID_INTERNET
            : "" )).
      ( $rliste_id != "-1"
        ? " and al.LISTE_ID=".$rliste_id
        : "" );

    if( $rabonne_nom != "" ) {
      $strWhere .= " and (";
      $tabAgentNom = explode("+", $rabonne_nom);
      $strGlue = "";
      foreach($tabAgentNom as $strText) {
        $strText = trim($strText);
        $strWhere .= $strGlue.$this->dbConn->GetLowerCase("ab.ABONNE_NOM")." like ".
          $this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($strText."%")."'").
          " or ".$this->dbConn->GetLowerCase("ab.ABONNE_MAIL")." like ".
          $this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($strText."%")."'");
        $strGlue = " or ";
      }
      $strWhere .= ")";
    }

    $strOrder = " order by ABONNE_NOM, ABONNE_MAIL";

    $strSql = "select ".$strSelect." ".$strFrom." ".$strWhere." ".$strOrder;
    return $strSql;
    
  }  
  
  /**
   * retourne la date de l'abonnement d'un abonné à une liste
   * @param abonne_id   identifiant de l'abonné
   * @param liste_id    identifiant de la liste
   */
  public function getDateAbonnement($abonne_id, $liste_id)
  {
    $strDate = "";
    $strSql = "select ABONNELISTE_DATE" .
        " from SIT_ABONNE_LISTE" .
        " where ABONNE_ID=".$abonne_id." AND LISTE_ID=".$liste_id;
    $oDs = $this->dbConn->initDataset($strSql);
    if ( $oDr = $oDs->getRowIter() ) {
      $strDate = $oDr->getValueName("ABONNELISTE_DATE");
    }
    return $strDate;
  }
  
  /**
   * retourne le nombre d'agent du service
   * @param service_id l'id du service
   */
  public function getNombreAgent($service_id)
  {
     $strSql = "select COUNT(ase.AGENT_ID) as NOMBRE_AGENT".
      " from SIT_SERVICE s".
      "  inner join SIT_AGENT_SERVICE ase on s.SERVICE_ID=ase.SERVICE_ID".
      "  inner join SIT_AGENT a on a.AGENT_ID=ase.AGENT_ID".
      " where ase.SERVICE_ID=".$service_id.
      " AND a.AGENT_VALIDE=1";
     $oDs = $this->dbConn->initDataset($strSql);
     if ( $oDr = $oDs->getRowIter() ) {
      $iNombreAgent = $oDr->getValueName("NOMBRE_AGENT");
    }
    return $iNombreAgent;
  }
}
?>