<?php
/*licence/ 

Module écrit, supporté par la société Alkante SAS <alkante@alkante.com>

Nom du module : Alkanet::Module::GEdit
Module gestion éditoriale.
Ce module appartient au framework Alkanet.

Ce logiciel est régi par la licence CeCILL-C soumise au droit français et
respectant les principes de diffusion des logiciels libres. Vous pouvez
utiliser, modifier et/ou redistribuer ce programme sous les conditions
de la licence CeCILL-C telle que diffusée par le CEA, le CNRS et l'INRIA
sur le site http://www.cecill.info.

En contrepartie de l'accessibilité au code source et des droits de copie,
de modification et de redistribution accordés par cette licence, il n'est
offert aux utilisateurs qu'une garantie limitée. Pour les mêmes raisons,
seule une responsabilité restreinte pèse sur l'auteur du programme, le
titulaire des droits patrimoniaux et les concédants successifs.

A cet égard l'attention de l'utilisateur est attirée sur les risques
associés au chargement, à l'utilisation, à la modification et/ou au
développement et à la reproduction du logiciel par l'utilisateur étant
donné sa spécificité de logiciel libre, qui peut le rendre complexe à
manipuler et qui le réserve donc à des développeurs et des professionnels
avertis possédant des connaissances informatiques approfondies. Les
utilisateurs sont donc invités à charger et tester l'adéquation du
logiciel à leurs besoins dans des conditions permettant d'assurer la
sécurité de leurs systèmes et ou de leurs données et, plus généralement,
à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.

Le fait que vous puissiez accéder à cet en-tête signifie que vous avez
pris connaissance de la licence CeCILL-C, et que vous en avez accepté les
termes.

/licence*/

require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CLASSE."pattern/alkquery.class.php");
/**
 * @package Alkanet_Module_GEdit
 * @class Classe AlkQueryGEdit
 * Classe contenant la liste des requêtes selection de l'application gedit
 */
class AlkQueryGEdit extends AlkQuery 
{

  /**
   *  Constructeur par défaut
   * @param dbConn  référence sur l'objet de connexion de type AlkDb
   */
  public function __construct(&$dbConn)
  {
    parent::__construct($dbConn);   
  }
   
  /**
   * Destructeur par défaut
   */
  public function __destruct() { }

  /**
   * Retourne dans un tableau, la liste des applications qui possède une interface d'affichage pour GEDIT
   * @param cont_id identifiant de l'espace
   * @return array
   */
  public function getTabListGeditAppliType($cont_id="")
  {
    $tabRes = array();
    
    $tabFiles = array();
    
    $tabFilesTmp = getTabFilesByDir(ALK_ALKANET_ROOT_PATH.ALK_ROOT_TEMPLATE, array("tpl"), false, array("alkgedit_"), true);
    if( empty($tabFilesTmp) ) {
      $tabFilesTmp = getTabFilesByDir(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."gedit/templates/", array("tpl"), false, array("alkgedit_"), true);
    }
    foreach($tabFilesTmp as $strFileName => $strPathFileName) {
      $tabTmp = explode("_", $strFileName);
      if( count($tabTmp)>=3 ) {
        $strKeyFile = $tabTmp[0]."_".$tabTmp[1]."_".$tabTmp[2]."_".
          ( count($tabTmp)>=4 && $tabTmp[3]!='s' && is_numeric($tabTmp[3]) ? $tabTmp[3]."_" : "" );
        $tabFiles[$strKeyFile] = $strPathFileName;
      }  
    }
    if( !empty($tabFiles) ) {
      $_SESSION["alk_gedit_templates"] = $tabFiles;
    }
    
    
    $strSql = "select at.ATYPE_INTITULE, at.ATYPE_ID, gpt.PTYPE_ID, gpt.PTYPE_INTITULE".
      " from ".
      ( defined("ALK_B_ATYPE_IEDIT") && ALK_B_ATYPE_IEDIT==true
        ? "(select ATYPE_INTITULE, ATYPE_ID".
          " from SIT_APPLI_TYPE".
          " union".
          " select DATATYPE_INTITULE_FR as ATYPE_INTITULE, DATATYPE_ID as ATYPE_ID".
          " from IEDIT_DATATYPE".
          " where DATATYPE_ID<>".ALK_ATYPE_ID_RDV.") at"
        : "SIT_APPLI_TYPE at" ).
      "   inner join GEDIT_01_APPLI_TYPE gat on gat.ATYPE_ID=at.ATYPE_ID".
      "   inner join GEDIT_01_PAGE_TYPE gpt on gat.ATYPE_ID=gpt.ATYPE_ID".
      //" where gpt.PTYPE_SERVICENAME is not null and gpt.PTYPE_SERVICENAME<>''" .
      " group by at.ATYPE_INTITULE, at.ATYPE_ID, gpt.PTYPE_ID, gpt.PTYPE_INTITULE".
      " order by at.ATYPE_INTITULE, gpt.PTYPE_INTITULE";
    $dsType = $this->dbConn->initDataset($strSql);
    while( $drType = $dsType->getRowIter() ) {
      $atype_id       = $drType->getValueName("ATYPE_ID");
      $atype_intitule = $drType->getValueName("ATYPE_INTITULE");
      $ptype_id       = $drType->getValueName("PTYPE_ID");
      $ptype_intitule = $drType->getValueName("PTYPE_INTITULE");
      
      $strPrefix1 = "alkgedit_".$atype_id."_".$ptype_id."_";
      $strPrefix2 = ($cont_id != "" ? "alkgedit_".$cont_id."_".$atype_id."_".$ptype_id."_" : "");
      
      if( array_key_exists($strPrefix1, $tabFiles) ||
          array_key_exists($strPrefix2, $tabFiles) ) {
        $tabRes[$atype_intitule][$atype_id."_".$ptype_id] = $ptype_intitule;
      }  
    }
    return $tabRes;
  }

  /**
   * Retourne une chaine contenant le nom de l'application suivi du nom de la catégorie si <> -1
   * @param atype_id  type de l'application
   * @param appli_id  identifiant de l'application
   * @param cat_id    identifiant de la catégorie, =-1 par défaut
   * @return string
   */
  public function getAppliCatName($atype_id, $appli_id, $cat_id=-1)
  {
    $atype_id_appli = ( in_array($atype_id, array(ALK_ATYPE_ID_ACTU, ALK_ATYPE_ID_LIEN, ALK_ATYPE_ID_GLOS, ALK_ATYPE_ID_FAQS))
                        ? ALK_ATYPE_ID_IEDIT
                        : $atype_id);

    $oAppli = AlkFactory::getAppli($atype_id_appli, $appli_id);
    $strName = $oAppli->getAppliProperty("APPLI_INTITULE"); 
    if( $cat_id != -1 && method_exists($oAppli, "getGEditBlocTypeUpdate") ) {
      $tabBlocTypeUpdate = $oAppli->getGEditBlocTypeUpdate(-1, $atype_id);
      $strViewNameCat = ( $tabBlocTypeUpdate["iAssocCat"] == "2" && $tabBlocTypeUpdate["catTableName"]!="" 
                          ? $tabBlocTypeUpdate["catTableName"] 
                          : "" );
      if( $strViewNameCat != "" ) {
        $strSql = "select distinct CAT_INTITULE".ALK_LG_BDD." from ".$strViewNameCat." where CAT_ID=".$cat_id;
        $dsCat = $this->dbConn->initDataset($strSql);
        if( $drCat = $dsCat->getRowIter() ) {
          $strName .= " / ".$drCat->getValueName("CAT_INTITULE".ALK_LG_BDD);
        }  
      }
    }
    return $strName;
  }

  /**
   * Retourne l'identifiant de la première page brouillon. Si user_id est fourni, vérification des droits.
   * @param appli_id  identifiant de l'utilisateur connecté
   * @param user_id   identifiant de l'agent connecté
   * @param bDraft    =true pour filtrer les pages brouillons, = false pour filtrer les pages publiées (par défaut) 
   * @return AlkDs
   */
  public function getFirstPageIdByAppli($appli_id, $user_id="-1", $bDraft=false)
  {
    $page_id = "-1";
    
    // le order by page_brouillon permet de prendre systématiquement la page brouillon si elle existe, 
    // ou la page publiée si le brouillon n'existe pas encore

    $strSql = "select p.PAGE_ID".
      " from GEDIT_01_PAGE p".
      ( $user_id != "-1" ? " inner join V_GEDIT_USER_RIGHT_PAGE r on p.PAGE_ID=r.PAGE_ID" : "" ).
      " where p.PAGE_NIVEAU=1 and p.APPLI_ID=".$appli_id.
      ( $user_id != "-1" ? " and r.USER_ID=".$user_id." and r.RIGHT_ID>0" : "" ).
      " and p.PAGE_BROUILLON is ".( $bDraft ? "not" : "" )." null".
      " order by p.PAGE_RANG";
    $dsPage =  $this->dbConn->initDataset($strSql);
    if( $drPage = $dsPage->getRowIter() ) {
      $page_id = $drPage->getValueName("PAGE_ID");
    }
    return $page_id;
  }

  /**
   * Retourne la liste des rubriques accessibles.
   *        Uniquement les pages publiées si bPubOnly=true en vérifiant les dates de publication
   * @param user_id           identifiant de l'utilisateur connecté
   * @param appli_id          identifiant de l'application ou liste d'identifiants
   * @param bPubOnly          false par défaut pour filtrer les pages brouillons, true pour filtrer les pages publiées 
   * @param rootpage_id       identifiant de la page racine, =-1 pour aucun filtre
   * @param bAllowRedirected  true par défaut pour filtrer les pages de redirection, faux sinon pour aucun filtre
   * @param atype_id          filtre sur le type de l'appli associée à la page, =-1 par défaut pour aucun filtre
   * @param ptype_id          filtre sur le type de page, = -1 par défaut pour aucun filtre
   * @param iMenuVisible      =1 par défaut pour filtrer les pages visibles sur le menu, =0 ne filtrer les pages qui ne sont pas visible sur le menu, =-1 pour aucun filtre
   * @param bWithUrl          =false par défaut, =true pour ajouter les colonnes PAGE_URL_[LG] contenant l'url complète réécrite de chaque page
   * @return AlkDs
   */
  public function getArboAutorise($user_id, $appli_id, $bPubOnly=false, $rootpage_id="-1", $bAllowRedirected=true, $atype_id="-1", $ptype_id="-1", $iMenuVisible=1, $bWithUrl=false) 
  {
     $strIntituleChamp = "";
    foreach($this->tabLangue as $key => $tabLg) {
      $strIntituleChamp .= 
        ", p.PAGE_TITRE".$tabLg["bdd"]." as NOM".$tabLg["bdd"]. 
        ", p.PAGE_TITRE_COURT".$tabLg["bdd"]." as NOM_COURT".$tabLg["bdd"]. 
        ", p.PAGE_ALIAS".$tabLg["bdd"].
        ", p.PAGE_TITRENAV".$tabLg["bdd"].
        ", p.PAGE_DESC".$tabLg["bdd"].
        ", p. PAGE_MOTCLE".$tabLg["bdd"].
        ( $bWithUrl         
          ? (ALK_BDD_TYPE == ALK_SGBD_POSTGRES) 
          	?
          		", (select ".$this->dbConn->getConcat("'/'", $this->dbConn->getGroupConcat("p0.PAGE_ALIAS".$tabLg["bdd"], "'/'", "p0.PAGE_NIVEAU", false, "from GEDIT_01_PAGE p0".
            	"   where p0.PAGE_ID>0".
            	"    and p.PAGE_ARBRE like ".$this->dbConn->getConcat("'%-'", "p0.PAGE_ID", "'-%'")." order by p0.PAGE_NIVEAU").
          		") as PAGE_URL".$tabLg["bdd"])       
						: ", (select ".$this->dbConn->getConcat("'/'", $this->dbConn->getGroupConcat("p0.PAGE_ALIAS".$tabLg["bdd"], "'/'", "p0.PAGE_NIVEAU")).
            	"   from GEDIT_01_PAGE p0".
            	"   where p0.PAGE_ID>0".
            	"    and p.PAGE_ARBRE like ".$this->dbConn->getConcat("'%-'", "p0.PAGE_ID", "'-%'").") as PAGE_URL".$tabLg["bdd"]                        
          : "" );           
    }
    $strIntituleChamp.",p.ATYPE_ID";
    $strSql = "select p.PAGE_ID, p.APPLI_ID, sa.APPLI_INTITULE ".$strIntituleChamp.", p.PAGE_PARENT as ID_PERE, p.PAGE_ALIAS".ALK_LG_BDD." as PAGE_ALIAS, ".
      " p.PAGE_NIVEAU as NIV, p.PAGE_RANG as RANG, p.PAGE_ARBRE as ARBRE, p.PAGE_URL_REDIRECTION, p.PAGE_TARGET_REDIRECTION,".
      " p.PAGE_TEMPLATE, p.ATYPE_ID, p.PTYPE_ID, p.PAGE_APPLIID, p.PAGE_CATID, p.PAGE_VISIBLE_MENU, p.PAGE_DATEMAJ, r.RIGHT_ID, l.LISTE_ID, p.PAGE_NB_ELEMENTS ".
      ( $bPubOnly 
        ? "" 
        : ", (select ".$this->dbConn->compareSql("p0.PAGE_DATEMAJ", "=", "p.PAGE_DATEMAJ", "0", "1").
          "   from GEDIT_01_PAGE p0".
          "   where p.PAGE_BROUILLON=p0.PAGE_ID) as PAGE_MODIFIEE ").
      " , p.PAGE_VALIDEPUB as PAGE_PUBLIEE ".
      " from GEDIT_01_PAGE p".
      "   inner join V_GEDIT_USER_RIGHT_PAGE r on p.PAGE_ID=r.PAGE_ID".
      "   inner join SIT_APPLI sa on sa.APPLI_ID = p.APPLI_ID ".
      ( $bPubOnly 
        ? " left join GEDIT_01_PAGE_LISTE l on p.PAGE_ID=l.PAGE_ID" 
        : " left join GEDIT_01_PAGE_LISTE l on p.PAGE_BROUILLON=l.PAGE_ID" ).
      " where p.APPLI_ID in (".$appli_id.")".
      "   and r.USER_ID=".$user_id." and r.RIGHT_ID>0".
      ($bAllowRedirected ? "" : "   and p.PAGE_URL_REDIRECTION=''").
      ( $rootpage_id != "-1"
        ? " and p.PAGE_ARBRE like '%-".$rootpage_id."-%'"
        : "" ).
      ( $bPubOnly 
        ? " and p.PAGE_VALIDEPUB=1 and p.PAGE_BROUILLON is null".
          ( defined("ALK_B_GEDIT_VISIBLE_MENU") && ALK_B_GEDIT_VISIBLE_MENU==true && $iMenuVisible>-1 
            ? ( $iMenuVisible == 1
                ? " and PAGE_VISIBLE_MENU = 1"
                : " and PAGE_VISIBLE_MENU = 0" )
            : "" ).
          " and (PAGE_DATEPDEB is null or PAGE_DATEPDEB<=".$this->dbConn->getDateCur().")".
          " and (PAGE_DATEPFIN is null or PAGE_DATEPFIN>=".$this->dbConn->getDateCur().")"
        : " and p.PAGE_BROUILLON is not null" ).
      ( $atype_id != "-1"
        ? " and p.ATYPE_ID=".$atype_id
        : "").
      ( $ptype_id != "-1"
        ? " and PTYPE_ID=".$ptype_id
        : "").
      " order by NIV, ID_PERE, RANG";
        
   if( is_numeric($appli_id) && $bPubOnly ) {
     // mise en cache de la requête pendant 6h
     return $this->dbConn->initDataset($strSql, 0, -1, true, 6*3600, constant("ALK_ATYPE_ABREV_".ALK_ATYPE_ID_GEDIT)."_".$appli_id);
   }
     
   return $this->dbConn->initDataset($strSql);   
  }

  /**
   * Retourne la liste des profils et des droits associés à une page
   * @param page_id  identifiant de la page
   * @return AlkDs
   */
  public function getDsProfilListByPage($page_id)
  {
    $strSql = "select gp.DROIT_ID, p.*".
      " from SIT_PROFIL p ".
      "   left join GEDIT_01_PROFIL gp on p.PROFIL_ID=gp.PROFIL_ID".
      " where gp.PAGE_ID=".$page_id.
      " order by p.PROFIL_INTITULE";
    return $this->dbConn->InitDataSet($strSql);
  }
  
  /**
   * Retourne le titre de la page dans la langue sélectionnée
   * @param page_id   identifiant de la page
   * @param lg        suffixe de la langue sélectionnée, =ALK_LG_BDD par défaut
   * @param bLong     booleen pour signifier si l'intitulé long ou court
   * @return AlkDs
   */
  public function getPageTitle($page_id, $lg=ALK_LG_BDD, $bLong=true)
  {
    $strTitle = "";
    $strPageTitre = ($bLong ? "PAGE_TITRE" : "PAGE_TITRE_COURT").$lg;
    $strSql = "select ".$strPageTitre." from GEDIT_01_PAGE where PAGE_ID=".$page_id;
    $dsPage = $this->dbConn->initDataset( $strSql );
    if( $drPage = $dsPage->getRowIter() ) {
      $strTitle = $drPage->getValueName($strPageTitre);
    } 
    return $strTitle;
  }
  
  /**
   * Retourne les propriétés de la page avec le droit de l'utilisateur connecté
   * @param page_id   identifiant de la page ou tableau d'identifiants de page
   * @param user_id   identifiant de l'utilisateur connecté, =-1 pour ne pas vérifier les droits
   * @param bWithUrl  false par défaut, true pour ajouter les colonnes PAGE_URL_[LG] contenant l'url complète réécrite de chaque page
   * @return AlkDs
   */
  public function getPageById($page_id, $user_id, $bWithUrl=false)
  {
    if( is_array($page_id) ) {
      $page_id = implode(",", $page_id);
    }
    
    $strIntituleChamp = "";
    foreach($this->tabLangue as $key => $tabLg) {
      $strIntituleChamp .= 
        ", p.PAGE_TITRE".$tabLg["bdd"].
        ", p.PAGE_TITRE_COURT".$tabLg["bdd"].
        ", p.PAGE_ALIAS".$tabLg["bdd"].
        ", p.PAGE_TITRENAV".$tabLg["bdd"].
        ", p.PAGE_MOTCLE".$tabLg["bdd"].
        ", p.PAGE_DESC".$tabLg["bdd"].
      	( $bWithUrl
          ? (ALK_BDD_TYPE == ALK_SGBD_POSTGRES) 
          	?
          		", (select ".$this->dbConn->getConcat("'/'", $this->dbConn->getGroupConcat("p0.PAGE_ALIAS".$tabLg["bdd"], "'/'", "p0.PAGE_NIVEAU", false, "from GEDIT_01_PAGE p0".
            	"   where p0.PAGE_ID>0".
            	"    and p.PAGE_ARBRE like ".$this->dbConn->getConcat("'%-'", "p0.PAGE_ID", "'-%'")." order by p0.PAGE_NIVEAU").
          		") as PAGE_URL".$tabLg["bdd"])       
						: ", (select ".$this->dbConn->getConcat("'/'", $this->dbConn->getGroupConcat("p0.PAGE_ALIAS".$tabLg["bdd"], "'/'", "p0.PAGE_NIVEAU")).
            	"   from GEDIT_01_PAGE p0".
            	"   where p0.PAGE_ID>0".
            	"    and p.PAGE_ARBRE like ".$this->dbConn->getConcat("'%-'", "p0.PAGE_ID", "'-%'").") as PAGE_URL".$tabLg["bdd"]                        
          : "" );           
    }

    $strSql = "select p.PAGE_ID ".$strIntituleChamp.",".
      " p.PAGE_PARENT, p.PAGE_NIVEAU, p.PAGE_RACINE_DROIT, p.PAGE_RANG,".
      " p.PAGE_ARBRE, p.PAGE_VALIDEPUB, p.PAGE_TEMPLATE, ".
      $this->dbConn->getDateFormat("DD/MM/YYYY",  "PAGE_DATEPDEB", false)." as PAGE_DATEPDEB,".
      $this->dbConn->getDateFormat("DD/MM/YYYY",  "PAGE_DATEPFIN", false)." as PAGE_DATEPFIN,".
      $this->dbConn->getDateFormat("DD/MM/YYYY, HH:MI",  "PAGE_DATECREA", false)." as PAGE_DATE_CREA,".
      $this->dbConn->getDateFormat("DD/MM/YYYY, HH:MI",  "PAGE_DATEMAJ", false)." as PAGE_DATE_MAJ,".
      $this->dbConn->getConcat("ac.AGENT_PRENOM", "' '", "ac.AGENT_NOM")." as agent_nomcrea,".
      $this->dbConn->getConcat("am.AGENT_PRENOM", "' '", "am.AGENT_NOM")." as agent_nommaj,".
      " p.PAGE_REPERTOIRE, p.PAGE_VISIBLE_MENU, p.PAGE_URL_REDIRECTION, p.PAGE_TARGET_REDIRECTION,".
      " p.PAGE_BROUILLON, p.PAGE_TEMPLATE, p.PAGE_VALIDESYND, p.PAGE_APPLIID, p.PAGE_CATID, p.PAGE_PUBLIC,".
      $this->dbConn->getConcat("p.ATYPE_ID", "'_'", "p.PTYPE_ID")." as page_aptypeid,".
      " p.ATYPE_ID, p.PTYPE_ID, pt.PTYPE_SERVICENAME, p.PAGE_ORDRE, p.PAGE_TYPEASSOC, l.LISTE_ID, p.PAGE_NB_ELEMENTS" .
      ", ".$this->dbConn->getDateFormat("DD/MM/YYYY",  "PAGE_DATEDEB", false)." as PAGE_DATEDEB" .
      ", ".$this->dbConn->getDateFormat("DD/MM/YYYY",  "PAGE_DATEFIN", false)." as PAGE_DATEFIN".
      ( $user_id != "-1"
        ? ", r.RIGHT_ID"
        : ", 1 as RIGHT_ID" ).
      " from GEDIT_01_PAGE p".
      "   inner join SIT_AGENT ac on p.AGENT_ID_CREA=ac.AGENT_ID".
      "   inner join SIT_AGENT am on p.AGENT_ID_MAJ=am.AGENT_ID".
      ( $user_id != "-1"
        ? " inner join V_GEDIT_USER_RIGHT_PAGE r on p.PAGE_ID=r.PAGE_ID"
        : "" ).
      "   inner join GEDIT_01_PAGE_TYPE pt on (p.ATYPE_ID=pt.ATYPE_ID and p.PTYPE_ID=pt.PTYPE_ID)".
      "   left join GEDIT_01_PAGE_LISTE l on l.PAGE_ID=p.PAGE_ID".
      " where p.PAGE_ID in (".$page_id.")" .
      ( $user_id != "-1"
        ? " and r.USER_ID=".$user_id
        : "" ).
      " order by p.PAGE_NIVEAU";

    return $this->dbConn->initDataset( $strSql );
  }
 
  /**
   * Retourne la liste des mots clés
   * @param page_id   identifiant de l'appli
   * @param iFirst    Premier enreg à lire (pagination)
   * @param iLast     Dernier enreg à lire (pagination)
   * @return DataSet
   */
  public function getDsListMotsCles($appli_id, $iFirst=0,$iLast=-1,$stri="poids")
  {
  	if ($stri=="poids"){
  		$tri = "TAGCLOUD_POIDS DESC";
  	} else if ($stri=="mot"){
  		$tri = "TAGCLOUD_MOT ASC";
  	}
    $strSql = "select TAGCLOUD_ID, TAGCLOUD_MOT, TAGCLOUD_POIDS from GEDIT_01_TAGCLOUD where APPLI_ID = ".$appli_id." ORDER BY ".$tri;
    return $this->dbConn->initDataset( $strSql,$iFirst,$iLast );
  }
  
  /**
   * Retourne un enreg de mot clé
   * @param page_id   identifiant de l'appli
   * @param iFirst    Premier enreg à lire (pagination)
   * @param iLast     Dernier enreg à lire (pagination)
   * @return DataSet
   */
  public function getDsMotsCleByIDMotClé($appli_id, $tagCloud_id)
  {
    $strSql = "select TAGCLOUD_ID, TAGCLOUD_MOT, TAGCLOUD_POIDS" .
      " from GEDIT_01_TAGCLOUD" .
      " where APPLI_ID = ".$appli_id." AND TAGCLOUD_ID = ".$tagCloud_id;
    return $this->dbConn->initDataset( $strSql );
  }
  
  /**
   * @param appli_id
   * calcule la somme des mots clés
   */
  public function SommePoidsTagCloud($appli_id)
  {
    $strSql = "SELECT sum(TAGCLOUD_POIDS) as somme" .
      " FROM GEDIT_01_TAGCLOUD" .
      " where APPLI_ID = ".$appli_id;
    return $this->dbConn->initDataset( $strSql );
  }
  
  /**
   * retourne le tableau mots clés recherchés les plus demandés
   */
  public function getTableauMotsCleSearch($appli_id)
  {
    $tableau = array();
    $strSql = "select SSTAT_ID, SSTAT_MOT, SSTAT_NB" .
      " from GEDIT_01_SEARCH_STAT" .
      " where APPLI_ID = ".$appli_id.
      " order by SSTAT_NB desc";
    $dsListMotClesSearch = $this->dbConn->initDataset( $strSql );
    while( $drListMotClesSearch = $dsListMotClesSearch->getRowIter() ) {
       $mot = $drListMotClesSearch->getValueName("SSTAT_MOT")." (".$drListMotClesSearch->getValueName("SSTAT_NB").")";
       $tableau[$drListMotClesSearch->getValueName("SSTAT_ID")] = $mot;
     }
    return $tableau;
  }
  
  /**
   * 
   * @param $id_MotCleSearch
   * @return le mot clé et le nbre
   */
  public function GetDsMotCleSearchByIdMocleSearc($id_MotCleSearch)
  {
    $strSql = "select SSTAT_MOT, SSTAT_NB" .
      " from GEDIT_01_SEARCH_STAT" .
      " where SSTAT_ID = ".$id_MotCleSearch;
    return $this->dbConn->initDataset( $strSql );
  }
 
  /**
   * 
   * @return la liste des mots clés les plus utilisés (vient de la table bdict)
   */
  public function GetTableauMotCleMotsPlusUtilises()
  {
 	  $strSql = "SELECT word, sum(secno) as somme".
      " from bdict".
      " where cast( word AS SIGNED INTEGER )=0 ".
      "   and word NOT LIKE '%0%'".
      "   and word not like '#%'".
      "   and length(word)> 4 ".
      " group by word".
      " order by somme desc ";
 	  $tableau = array();
    $dsListMotUtil = $this->dbConn->initDataset( $strSql );
    while( $drListMotUtil = $dsListMotUtil->getRowIter() ) {
       $mot = $drListMotUtil->getValueName("word")." (".$drListMotUtil->getValueName("somme").")";
       $tableau[$drListMotUtil->getValueName("word")] = utf8_decode($mot);
     }
    return $tableau;
  }
  
  /**
   * 
   * @param MotCleUtil
   * @return le mot clé et le secno
   */
  public function GetDsMotCleUtilByIdMocleUtil($MotCleUtil)
  {
    $strSql = "SELECT *, sum(secno) as somme" .
      " from bdict" .
      " where word = "."'".$MotCleUtil."'"."group by word";
    return $this->dbConn->initDataset( $strSql );
  }
  
  /**
   * 
   * @param $MotCleSearch
   * @return la ligne motCleSearch
   */
  public function GetMotCleSearchByMocleSearch($MotCleSearch)
  {
    $strSql = "SELECT *" .
      " from GEDIT_01_SEARCH_STAT" .
      " where SSTAT_MOT = "."'".$this->dbConn->analyseSql(strtolower($MotCleSearch))."'";
    return $this->dbConn->initDataset( $strSql );
  }
  
  /**
   * Retourne la liste des blocs associés à la page
   *        avec possibilité de filtrer par type de bloc
   * @param page_id      identifiant de la page
   * @param idTypeBloc   identifiant du type de bloc, =0 par défaut pour tous les blocs, >0 pour filter
   * @param bRec         false par défaut, true pour également retourner les blocs de la sous-arbo de page_id
   * @param bWithPagePub true pour récupérer les blocs de la page publiée associée
   * @return AlkDs
   */
  public function getBlocByPageId($page_id, $idTypeBloc=0, $bRec=false, $bWithPagePub=false)
  {
    $strIntituleChamp = "";
    foreach($this->tabLangue as $key => $tabLg) {
      $strIntituleChamp .= ", b.BLOC_NOM".$tabLg["bdd"];
    }
    $page_id_pub = "";
    if( $bWithPagePub ) {
      $strSql = "select PAGE_BROUILLON from GEDIT_01_PAGE where PAGE_ID=".$page_id;
      $dsPagePub = $this->dbConn->initDataset($strSql);
      if( $drPagePub = $dsPagePub->getRowIter() ) {
        $page_id_pub = $drPagePub->getValueName("PAGE_BROUILLON");
      }
    } 
    
    $strSql = "select b.BLOC_ID, b.BLOC_NOM, b.BLOC_RANG, b.BLOC_COLONNE, b.BLOC_TYPEASSOC, b.BLOC_UNE, b.BLOC_ORDRE".
    $strIntituleChamp.",".
      $this->dbConn->getDateFormat("DD/MM/YYYY",  "b.BLOC_DATEDEB", false)." as BLOC_DATE_DEB,".
      $this->dbConn->getDateFormat("DD/MM/YYYY",  "b.BLOC_DATEFIN", false)." as BLOC_DATE_FIN,".
      " b.BLOC_DESC, b.BLOC_LIMIT,".
      " tb.TB_INTITULE, tb.TB_ID, tb.ATYPE_ID, p.PAGE_ID".
      " from GEDIT_01_PAGE p".
      "   inner join GEDIT_01_BLOC b on p.PAGE_ID=b.PAGE_ID".
      "   inner join GEDIT_01_TYPEBLOC tb on b.TB_ID = tb.TB_ID".
      " where ".
      ( $bRec
        ? " (p.PAGE_ARBRE like '%-".$page_id."-%'".($page_id_pub != "" ? " or p.PAGE_ARBRE like '%-".$page_id_pub."-%'" : "").")"
        : ( $page_id_pub != "" ? "p.PAGE_ID in (".$page_id.",".$page_id_pub.")" : "p.PAGE_ID=".$page_id )).
      ( $idTypeBloc>0
        ? " and b.TB_ID=".$idTypeBloc
        : "" ).
      " order by b.BLOC_RANG";
    
    if ($bWithPagePub){
      return $this->dbConn->initDataset($strSql, 0, -1, true, 600, "gedit");
    }else {
      return $this->dbConn->initDataset($strSql);
    }
  }

  /**
   * Retourne la fiche d'un bloc gedit
   * @param bloc_id       identifiant du bloc
   * @param iMaskFilter   masque appliqué à bloc_typeassoc pour obtenir l'entier correspondant au filtre de publication, -1 si non utilisé par défaut
   * @param iMaskCalendar masque appliqué à bloc_typeassoc pour obtenir l'entier correspondant au filtre calendaire, -1 si non utilisé par défaut
   * @return AlkDs
   */
  public function getDsBlocById($bloc_id, $iMaskFilter=-1, $iMaskCalendar=-1)
  {
    $strIntituleChamp = "";
    foreach($this->tabLangue as $key => $tabLg) {
      $strIntituleChamp .= ", b.BLOC_NOM".$tabLg["bdd"];
    }
    $strIntituleChamp.= ", BLOC_DESC, BLOC_LIMIT";
    $strSql = "select b.BLOC_ID, b.BLOC_COLONNE, b.BLOC_ORDRE, b.BLOC_TYPEASSOC,".
      $this->dbConn->getDateFormat("DD/MM/YYYY",  "b.BLOC_DATEDEB", false)." as BLOC_DATEDEB,".
      $this->dbConn->getDateFormat("DD/MM/YYYY",  "b.BLOC_DATEFIN", false)." as BLOC_DATEFIN".
      $strIntituleChamp.
      ( $iMaskFilter>=0 
        ? ", (b.BLOC_TYPEASSOC & ".$iMaskFilter.") as BLOC_FILTREPUB"
        : "" ).
      ( $iMaskCalendar>=0
        ? ", (b.BLOC_TYPEASSOC & ".$iMaskCalendar.") as BLOC_FILTRECAL"
        : "" ).
      " from GEDIT_01_BLOC b".
      " where b.BLOC_ID=".$bloc_id;
    return $this->dbConn->initDataset($strSql);
  }
  
  /**
   * Retourne la liste des blocs : id + typeAssoc (appli/cat/data) d'une page
   * @param tableCat  nom de la table association entre la catégorie et les blocs de la page
   * @param tableData nom de la table association entre la données et les blocs de la page
   * @param page_id   identifiant de la page
   * @return AlkDs
   */
  public function getDsBlocIdTypeAssocByPageId($tableCat, $tableData, $page_id)
  {
    $strSql = "select b.BLOC_ID, 1 as TYPEASSOC".
      " from GEDIT_01_APPLI_BLOC ab".
      "   inner join GEDIT_01_BLOC b on ab.BLOC_ID=b.BLOC_ID".
      " where b.PAGE_ID=".$page_id.
      ( $tableCat != ""
        ? " union select b.BLOC_ID, 2 as TYPEASSOC".
          " from ".$tableCat." c".
          "   inner join GEDIT_01_BLOC b on c.BLOC_ID=b.BLOC_ID".
          " where b.PAGE_ID=".$page_id
        : "" ).
      ( $tableData != "" 
        ? " union select b.BLOC_ID, 3 as TYPEASSOC".
          " from ".$tableData." d".
          "   inner join GEDIT_01_BLOC b on d.BLOC_ID=b.BLOC_ID".
          " where b.PAGE_ID=".$page_id
        : "" );
    
    return $this->dbConn->initDataset($strSql);
  }

  /**
   * Retourne la liste des blocs associés aux pages d'une application gedit
   * avec possibilité de filtrer par type de bloc
   * utiliser par oAppliGedit->delAppli(), pas de filtre sur page
   * 
   * @param appli_id    identifiant de l'appli
   * @param idTypeBloc  identifiant du type de bloc, =0 par défaut pour tous les blocs, >0 pour filter
   * @return AlkDs
   */
  public function getBlocByAppliId($appli_id, $idTypeBloc=0)
  {
    $strSql = "select b.BLOC_ID, b.BLOC_NOM as NOM, b.BLOC_RANG, b.BLOC_COLONNE, b.BLOC_UNE, b.BLOC_ORDRE,".
      $this->dbConn->getDateFormat("DD/MM/YYYY",  "b.BLOC_DATEDEB", false)." as BLOC_DATE_DEB,".
      $this->dbConn->getDateFormat("DD/MM/YYYY",  "b.BLOC_DATEFIN", false)." as BLOC_DATE_FIN,".
      " tb.TB_INTITULE, tb.TB_ID, tb.ATYPE_ID, b.PAGE_ID".
      " from GEDIT_01_PAGE p".
      "   inner join GEDIT_01_BLOC b on p.PAGE_ID=b.PAGE_ID".
      "   inner join GEDIT_01_TYPEBLOC tb on b.TB_ID=tb.TB_ID".
      " where p.APPLI_ID=".$appli_id.
      ( $idTypeBloc>0
        ? " and b.TB_ID=".$idTypeBloc
        : "" ).
      " order by b.BLOC_RANG";
    
    return $this->dbConn->initDataset($strSql);
  }

  /**
   * Retourne le cont_id, appli_id et cat_id correspondant aux 4 premiers paramètres 
   * @param bloc_typeassoc   champ de bits utilisé pour le filtre et l'ordonnencement des infos d'un bloc
   * @param strDataId        liste d'identifiants de l'info recherchée
   * @param dataTableName    nom de la table d'association bloc-data
   * @param catTableName     nom de la table d'association bloc-cat
   * @param cont_id          identifiant de l'espace trouvé, valeur non modifiée sinon
   * @param appli_id         identifiant de l'application trouvée, valeur non modifiée sinon
   * @param cat_id           identifiant de la catégorie trouvée, valeur non modifiée sinon
   */
  public function getBlocContAppliCat($bloc_typeassoc, $strDataId, $dataTableName, $catTableName, &$cont_id, &$appli_id, &$cat_id)
  {
    $strDataId = trim($strDataId);
    if( $strDataId=="" ) return;
    
    $bSelect = true;
    if( binmask_match($bloc_typeassoc, TASSOC_BYDATA) ) {
      $strTable = $dataTableName." t";
      $strChamp = "DATA_ID";
    } 
    elseif( binmask_match($bloc_typeassoc, TASSOC_BYCATEG) ) {
      $strTable = $catTableName." t".
        " inner join SIT_APPLI_CONT ac on ac.APPLI_ID=t.APPLI_ID";
      $strChamp = "CAT_ID";
    }
    else {
      // bloc associé à une appli
      $strTable = "(select ac.CONT_ID, ab.APPLI_ID, -1 as CAT_ID from GEDIT_01_APPLI_BLOC ab inner join SIT_APPLI_CONT ac on (ab.APPLI_ID=ac.APPLI_ID)) t";
      $strChamp = "APPLI_ID";
      $appli_id = $strDataId;
      //$bSelect = false;
    }

    if( $bSelect ) {
      $strSql = "select distinct CONT_ID, t.APPLI_ID, t.CAT_ID ".
         " from ".$strTable.
         " where t.".$strChamp." in (".$strDataId.")";
      $dsTmp = $this->dbConn->initDataSet($strSql);
      $bSelectCat = ( $dsTmp->getCountTotDr() == 1 ); 
      if($drTmp = $dsTmp->getRowIter()){
        $cont_id  = $drTmp->getValueName("CONT_ID");
        $appli_id = $drTmp->getValueName("APPLI_ID");
        if( $bSelectCat ) {
          $cat_id   = $drTmp->getValueName("CAT_ID");
        }
      }
    }
  }

  /**
   * Retourne la liste des catégories accessibles à l'utilisateur
   * @param appli_id    Identifiant de l'application
   * @param user_id     Idenfifiant de l'utilisateur connecté
   * @param strViewName Nom de la vue permettant de récupérer la liste des catégories
   */
  public function getDsListCat($appli_id, $user_id, $strViewName)
  {
    $strSql = "select distinct CAT_INTITULE".ALK_LG_BDD.", CAT_ID".
      " from ".$strViewName.
      " where USER_ID=".$user_id." and APPLI_ID=".$appli_id." and RIGHT_ID>0".
      " order by CAT_INTITULE".ALK_LG_BDD;
    return $this->dbConn->initDataset($strSql);
  }

  /**
   * Retourne la liste des catégories accessibles à l'utilisateur
   * @param appli_id    Identifiant de l'application
   * @param cat_id      Identifiant de la catégorie sélectionnée (-1 pour toutes les catégories de l'application)
   * @param user_id     Idenfifiant de l'utilisateur connecté
   * @param strViewName Nom de la vue permettant de récupérer la liste des catégories
   * @param bWithCatID Si faux, on ne récupère pas le CAT_ID dans le requete
   */
  public function getDsListData($appli_id, $cat_id, $user_id, $strViewName, $bWithCatID = true, $bOrderBy= false)
  {
    $strOrderBy = ( $strViewName == "V_GEDIT_FAQS_DATA" ? "DATA_DESC".ALK_LG_BDD : "DATA_TITRE".ALK_LG_BDD );
    
    $strSql = "select distinct d.DATA_ID, lower (d.DATA_TITRE".ALK_LG_BDD."), d.DATA_TITRE".ALK_LG_BDD.", d.DATA_DESC".ALK_LG_BDD.($bWithCatID == true ? ", d.CAT_ID" : "").
      ($strViewName!="V_GEDIT_FORM_DATA" ?  ", d.DATA_NEW " : "").
      " from ".$strViewName." d ".
      ( $bOrderBy && ALK_BDD_TYPE == ALK_SGBD_POSTGRES ? 
      " left join ".$strViewName." d2 ".
      " on  d.DATA_ID = d2.DATA_ID ".
      " and d.CAT_ID < d2.CAT_ID " : "").
      " where d.USER_ID=".$user_id." and d.APPLI_ID=".$appli_id." and d.RIGHT_ID>0 and d.DATA_VALIDEPUB=1".
      ( strtoupper($strViewName) == "V_GEDIT_IEDIT_DATA" ? " and d.DATA_BROUILLON is NULL" : "").
      ( $cat_id != "-1" ? " and d.CAT_ID=".$cat_id : "" ).
      ( $bOrderBy && ALK_BDD_TYPE == ALK_SGBD_POSTGRES ? " and d2.DATA_ID is null " : "" ).
      ( $bOrderBy && ALK_BDD_TYPE == ALK_SGBD_MYSQL ? " group by d.DATA_ID " : "").
      ( $bOrderBy ? " order by ".($strViewName!="V_GEDIT_FORM_DATA" ?  ", d.DATA_NEW desc, " : "")." lower (d.DATA_TITRE".ALK_LG_BDD.") " : "");
    return $this->dbConn->initDataset($strSql);
  }
  
  /**
   * Retourne les informations :
   *  - de la page page_id=object_id et typeassoc&3=3
   *  - des pages de appli_id=object_id et typeassoc&1=1
   *  - des pages de cont_id=object_id et typeassoc&3=0, -1 pour tous les espaces
   * filtre en fonction des bits allumés sur typeassoc
   * 
   * @param object_id     identifiant du bloc
   * @param typeAssoc     type d'association : 0=aucune, 1=appli, 3=data (non utilisé, toujours 3=data)
   *                       si bit 2 on : non utilisé
   *                       si bit 3 on : présente les pages publiés passées
   *                       si bit 4 on : présente les pages publiées présentes
   *                       si bit 5 on : présente les pages à valider
   *                       si bit 6 on : ordre les éléments d'abord par intitulé de catégorie (non utilisé)
   *                       si bit 7 on : présente les docs syndiqués 
   *                       si bit 8 on : non utilisé
   *                       si bit 9 on : présente les  infos en cours de validation 
   *                       si <=3, présente toutes les pages publiées : passés, présent, à venir
   * @param user_id        identifiant de l'utilisateur connecté, -1 pour non vérif de droit
   * @param strLg          langue utilisée, =_FR par défaut
   * @return AlkDs
   */
  public function getDsListPageById($object_id, $typeAssoc, $user_id, $strLg=ALK_LG_BDD)
  {
    $strWhere = "";
    $strSql = "";

    if( binmask_match($typeAssoc, TASSOC_BYDATA) ) {
      $strWhere = "p.PAGE_ID=".$object_id; 
    } elseif( binmask_match($typeAssoc, TASSOC_BYAPPLI) ) {
      $strWhere = "p.APPLI_ID=".$object_id; 
    } elseif( binmask_match($typeAssoc, TASSOC_BYDATA, 0) ) {
      $strWhere = ( $object_id != "-1" ? "r.CONT_ID=".$object_id : "r.CONT_ID>0" ); 
    } 

    // retourne toutes les données du bloc
    $strFilter = "";
    if( binmask_match($typeAssoc, TASSOC_ALAUNE) ) {
      // non utilisé
    }
    if( binmask_match($typeAssoc, TASSOC_ARCHIVE) ) {
      // filtre sur page passées
      if( !binmask_match($typeAssoc, TASSOC_PUBLIE) )
        $strFilter .= " and p.PAGE_DATEPFIN < ".$this->dbConn->getDateCur();
    }
    if( binmask_match($typeAssoc, TASSOC_PUBLIE) ) {
      // filtre sur pages présentes
      $strFilter .= " and (p.PAGE_DATEPDEB <= ".$this->dbConn->getDateCur()." or p.PAGE_DATEPDEB is null )".
        ( !binmask_match($typeAssoc, TASSOC_ARCHIVE)
          ? " and (p.PAGE_DATEPFIN >= ".$this->dbConn->getDateCur()." or p.PAGE_DATEPFIN is null )"
          : "" );
    }
    if( binmask_match($typeAssoc, TASSOC_AVALIDER) ) {
      $strFilter .= " and p.PAGE_VALIDEPUB=0 and p.PAGE_BROUILLON is not null";
    } else  if( binmask_match($typeAssoc, TASSOC_ENVALIDATION) ) {
      $strFilter .= " and p.PAGE_VALIDEPUB=2 and p.PAGE_BROUILLON is not null";
    } else {
      $strFilter .= " and p.PAGE_VALIDEPUB=1 and p.PAGE_BROUILLON is null";
    }
    if( binmask_match($typeAssoc, TASSOC_ESTSYNDIQUE) ) {
      $strFilter .= " and p.PAGE_VALIDESYND=1";
    } 
    
    $strSql = "select p.*, r.CONT_ID,".
      $this->dbConn->getDateFormat("DD/MM/YYYY",  "p.PAGE_DATEPDEB", false)." as PAGE_DATE_PDEB,".
      $this->dbConn->getDateFormat("DD/MM/YYYY",  "p.PAGE_DATEMAJ", false)." as PAGE_DATE_MAJ".
      " from GEDIT_01_PAGE p".
      ( $user_id != "-1" 
        ? " inner join V_GEDIT_USER_RIGHT_PAGE r on p.PAGE_ID=r.PAGE_ID" 
        : " inner join SIT_APPLI_CONT r on r.APPLI_ID=p.APPLI_ID" ).
      " where ".$strWhere.
      ( $user_id != "-1" 
        ? " and r.USER_ID=".$user_id." and r.RIGHT_ID>0"
        : "" ).
      $strFilter.
      " order by PAGE_TITRE".$strLg;
    
    return $this->dbConn->initDataset($strSql);
  }
  
  /**
   * Retourne la liste des documents d'un bloc gedit
   * En mode admin, retourne juste le nom de l'appli ou de la catégorie
   * @param bloc_id           identifiant du bloc
   * @param bloc_typeassoc    type d'association : 0=aucune, 1=appli, 2=cat, 3=data
   *                       si bit 2 on : présente les news uniquement
   *                       si bit 3 on : présente les docs publiés passés
   *                       si bit 4 on : présente les docs publiés présents
   *                       si <=3, présente tous les docs publiés : passés, présent, à venir
   * @param user_id           identifiant de l'utilisateur connecté
   * @param strAppliTypeAbrev abreviation de l'application contenant l'information
   * @param bAdmin            true si admin, false pour consultation
   * @return AlkDs
   */
  public function getDsListRssByBlocId($bloc_id, $bloc_typeassoc, $user_id, $strAppliTypeAbrev="", $bAdmin=false, $strLg=ALK_LG_BDD)
  {
    $strSql = "select b.PAGE_ID, r.RSS_ID as DATA_ID, '1' as DATA_RANG, r.RSS_TITRE".$strLg." as DATA_TITRE".$strLg.", ".
      " '' as DATA_DESC".$strLg.", r.RSS_URL as DATA_URL, r.RSS_NB_ELT_AFF as DATA_NB_ELT_AFF, ".
      " r.RSS_AFF_DETAIL as DATA_AFF_DETAIL, r.RSS_AFF_DATE as DATA_AFF_DATE, r.AGENT_ID ".
      " from GEDIT_01_BLOC b ".
      " inner join GEDIT_01_BLOC_RSS r on b.BLOC_ID=r.BLOC_ID ".
      " where b.BLOC_ID=".$bloc_id." ".
      " and (r.AGENT_ID is null or r.AGENT_ID=".$user_id.")".
      " order by DATA_RANG, DATA_TITRE".$strLg;
    return $this->dbConn->initDataset($strSql);
  }
  
  /**
   * Retourne le bloc RSS d'un bloc de contenu éditorial
   * @param bloc_id identifiant du bloc
   */
  public function getDsRssByBlocId($bloc_id)
  {
    $strSql = "select r.RSS_ID, b.BLOC_ID, r.RSS_TITRE".ALK_LG_BDD.", r.RSS_URL, ".
      " r.RSS_NB_ELT_AFF, r.RSS_AFF_DETAIL, r.RSS_AFF_DATE, b.BLOC_COLONNE ".
      " from GEDIT_01_BLOC b ".
      " inner join GEDIT_01_BLOC_RSS r on b.BLOC_ID=r.BLOC_ID ".
      " where b.BLOC_ID=".$bloc_id;
    return $this->dbConn->initDataset($strSql);
  }
  
  /**
   * Retourne un appli_id en fonction d'un espace et d'un type de donnée
   * Retourne -1 si non trouvé
   * @param user_id identifiant de l'utilisateur
   * @param cont_id identifiant de l'espace
   * @param datatype identifiant de type de donnée
   * @return int
   */
  public function getAppliIdBySpaceByDatatype($user_id, $cont_id, $datatype)
  {
    $bUnion = false;
    
    $strSql = "select distinct APPLI_ID from (";
    
    if (defined("ALK_B_ATYPE_ACTU") && ALK_B_ATYPE_ACTU==true) { 
        $strSql .= " select APPLI_ID ".
          " from V_GEDIT_ACTU_DATA where CONT_ID=".$cont_id." and DATA_TYPE=".$datatype." and USER_ID=".$user_id." and RIGHT_ID>0 ";
        $bUnion= true;
    }
        
    if (defined("ALK_B_ATYPE_FAQS") && ALK_B_ATYPE_FAQS==true) {
      $strSql .= ( $bUnion==true ? " union all" : "")." select APPLI_ID ".
        " from V_GEDIT_FAQS_DATA where CONT_ID=".$cont_id." and DATA_TYPE=".$datatype." and USER_ID=".$user_id." and RIGHT_ID>0 ";
      $bUnion= true;
    }
    
    if (defined("ALK_B_ATYPE_FDOC") && ALK_B_ATYPE_FDOC==true) { 
      $strSql .= ( $bUnion==true ? " union all" : "")." select APPLI_ID ".
        " from V_GEDIT_FDOC_DATA where CONT_ID=".$cont_id." and DATA_TYPE=".$datatype." and USER_ID=".$user_id." and RIGHT_ID>0 ";
      $bUnion= true;     
    }
    if (defined("ALK_B_ATYPE_FORM") && ALK_B_ATYPE_FORM==true) { 
      $strSql .= ( $bUnion==true ? " union all" : "")." select APPLI_ID ".
        " from V_GEDIT_FORM_DATA where CONT_ID=".$cont_id." and DATA_TYPE=".$datatype." and USER_ID=".$user_id." and RIGHT_ID>0 ";
      $bUnion= true;
    }
    if (defined("ALK_B_ATYPE_GLOS") && ALK_B_ATYPE_GLOS==true) { 
       $strSql .=  ( $bUnion==true ? " union all" : "")." select APPLI_ID ".
        " from V_GEDIT_GLOS_DATA where CONT_ID=".$cont_id." and DATA_TYPE=".$datatype." and USER_ID=".$user_id." and RIGHT_ID>0 ";
      $bUnion= true;
    }
    if (defined("ALK_B_ATYPE_LIEN") && ALK_B_ATYPE_LIEN==true) { 
      $strSql .= ( $bUnion==true ? " union all" : "")." select APPLI_ID ".
        " from V_GEDIT_LIEN_DATA where CONT_ID=".$cont_id." and DATA_TYPE=".$datatype." and USER_ID=".$user_id." and RIGHT_ID>0 ";
      $bUnion= true;
    }
    if (defined("ALK_B_ATYPE_LSDIF") && ALK_B_ATYPE_LSDIF==true) { 
      $strSql .= ( $bUnion==true ? " union all" : "")." select APPLI_ID ".
        " from V_GEDIT_LSDIF_DATA where CONT_ID=".$cont_id." and DATA_TYPE=".$datatype." and USER_ID=".$user_id." and RIGHT_ID>0 "; 
    }
    
    $strSql .= ") v_gedit";
    
    $ds = $this->dbConn->initDataset($strSql);
    if( $dr = $ds->GetRowIter() ) 
      return $dr->GetValueName("APPLI_ID");
    return -1;
  }
  
  /**
   * Retourne vrai si les champs page_alias_[fr|uk] sont uniques (pour les pages publiées)
   * @param page_id     identifiant de la page
   * @param page_alias  tableau des alias par langue de la page
   */
  public function isPageAliasUnique($page_id, $page_alias)
  {
    // pas de test si tous les page alias sont vide
    $strPageAliasAll = trim(implode("", $page_alias));
    if( $strPageAliasAll = "" ) return true; 
    
    // page_alias doit avoir des valeurs distinctes
    $bRes = true;
    for($i=0; $i<count($page_alias); $i++) {
      for($j=0; $j<$i; $j++) {
        if( $page_alias[$i]==$page_alias[$j] ) {
          $bRes = false;
          break(2);
        } 
      }
    }
    
    if( !$bRes ) return false;
    
    $strIntituleChamp = "";
    foreach($this->tabLangue as $key => $tabLg) {
      $strIntituleChamp .= ", p.PAGE_ALIAS".$tabLg["bdd"];
    }
    
    // Recherche les pages soeurs de page_id pour toutes les langues
    $strSql = "select 'brouillon' as type_page, p.PAGE_ID".$strIntituleChamp.
      " from GEDIT_01_PAGE pc".                                       // page courante brouillon
      "   inner join GEDIT_01_PAGE pp on pc.PAGE_PARENT=pp.PAGE_ID".  // page parente brouillon
      "   inner join GEDIT_01_PAGE p on p.PAGE_PARENT=pp.PAGE_ID".    // page soeur brouillon
      " where p.PAGE_ID<>pc.PAGE_ID".
      "   and p.PAGE_NIVEAU=pc.PAGE_NIVEAU".
      "   and pc.PAGE_ID=".$page_id.
      "   and p.PAGE_BROUILLON is not null".
      " union".
      " select 'publie' as type_page, p.PAGE_ID".$strIntituleChamp.
      " from GEDIT_01_PAGE pc".                                            // page courante brouillon
      "   inner join GEDIT_01_PAGE pcp on pc.PAGE_BROUILLON=pcp.PAGE_ID".  // page courante publiée
      "   inner join GEDIT_01_PAGE pp on pcp.PAGE_PARENT=pp.PAGE_ID".      // page parente publiée
      "   inner join GEDIT_01_PAGE p on p.PAGE_PARENT=pp.PAGE_ID".         // page soeur publiée
      " where p.PAGE_ID<>pcp.PAGE_ID".
      "   and p.PAGE_NIVEAU=pcp.PAGE_NIVEAU".
      "   and pc.PAGE_ID=".$page_id.
      "   and p.PAGE_BROUILLON is null";
    $dsPage = $this->dbConn->initDataSet($strSql);
    while( $drPage = $dsPage->getRowIter() ) {
      foreach($this->tabLangue as $key => $tabLg) {
        $strAliasBDD = $drPage->getValueName("PAGE_ALIAS".$tabLg["bdd"]);
        foreach($page_alias as $strPageAliasUpt) {
          if( $strPageAliasUpt != "" ) {
            if( $strAliasBDD == $strPageAliasUpt ) {
              $bRes = false;
              break(3);
            }
          }
        }
      }
    }
    
    return $bRes;
  }
  
  /**
   * Retourne les composantes primaires du token dans un tableau ayant pour clés : cont_id, appli_id, page_id, lg
   * page_id correspond forcément à une page publiée
   * 
   * @param strLastPage  intitule de la page consultée
   * @param cont_id      identifiant d'un espace spécifique, =-1 par défaut
   * @param bBrouillon   booléen indiquant si on souhaite acceder à la page brouillon =false par défaut
   * @param agent_id     =-1 par défaut, vérifie les droits si >0
   * @return Array
   */
  public function getPrimaryTokenData($strUrlPage, $cont_id=-1, $bBrouillon=false, $agent_id=-1, $bUseCache=false)
  { 
    $tabRes = array("cont_id" => -1, "appli_id" => -1, "page_id" => -1, "lg" => "fr", 
                    "page_atypeid"=>-1, "page_contid"=>-1, "page_appliid"=>-1);
    
    if (substr($strUrlPage, 0, 1)=="/") $strUrlPage = substr($strUrlPage, 1);
    if (substr($strUrlPage, -1)=="/") $strUrlPage = substr($strUrlPage, 0, -1);
    $tabPagesAlias = explode("/", $strUrlPage);
    $strArbre  = "-";
    $iCptAlias = 1;
    $strSql    = "";
    
    foreach ($tabPagesAlias as $pageAlias) {
      $strFields = "";
      $strWhere = "1=0";
      foreach($this->tabLangue as $key => $tabLg) {
        $strFields .= ", p".$iCptAlias.".PAGE_ALIAS".$tabLg["bdd"];
        $strWhere  .= " or p".$iCptAlias.".PAGE_ALIAS".$tabLg["bdd"]."='".$pageAlias."'";
      }
      $strSql = "select distinct p".$iCptAlias.".PAGE_ID".
        ($iCptAlias==count($tabPagesAlias) ? ", p".$iCptAlias.".APPLI_ID".$strFields.", ac".$iCptAlias.".CONT_ID, ac2".$iCptAlias.".CONT_ID as PAGE_CONTID, p".$iCptAlias.".PAGE_APPLIID, p".$iCptAlias.".ATYPE_ID as PAGE_ATYPEID" : "").
        " from GEDIT_01_PAGE p".$iCptAlias.
        " inner join SIT_APPLI_CONT ac".$iCptAlias." on p".$iCptAlias.".APPLI_ID=ac".$iCptAlias.".APPLI_ID".
        ($iCptAlias==count($tabPagesAlias) ? "   left join SIT_APPLI_CONT ac2".$iCptAlias." on p".$iCptAlias.".PAGE_APPLIID=ac2".$iCptAlias.".APPLI_ID" : "").
        ( $agent_id != -1
          ? " inner join V_GEDIT_USER_RIGHT_PAGE r".$iCptAlias." on (USER_ID=".$agent_id." and p".$iCptAlias.".PAGE_ID=r".$iCptAlias.".PAGE_ID)" 
          : "" ).
        " where (".$strWhere.")".
        ( $bBrouillon == true 
          ? " and p".$iCptAlias.".PAGE_BROUILLON is not null"
          : " and p".$iCptAlias.".PAGE_VALIDEPUB=1 and p".$iCptAlias.".PAGE_BROUILLON is null" ).
        ( $cont_id != -1
          ? " and ac".$iCptAlias.".CONT_ID=".$cont_id
          : "" ).
        ( $agent_id != -1
          ? " and r".$iCptAlias.".RIGHT_ID>0"
          : "" ).
        ($iCptAlias>1 ? " and p".$iCptAlias.".PAGE_PARENT=(".$strSql.")" : "");
      
      $iCptAlias += 1;
    }
    
    if( $bUseCache ) {
      $dsPage = $this->dbConn->initDataSet($strSql, 0, -1, true, 3600*4, constant("ALK_ATYPE_ABREV_".ALK_ATYPE_ID_GEDIT));
    } else  {
      $dsPage = $this->dbConn->initDataSet($strSql);
    }
    
    if( $drPage = $dsPage->getRowIter() ) {
      $tabRes["cont_id"]  = $drPage->getValueName("CONT_ID");
      $tabRes["appli_id"] = $drPage->getValueName("APPLI_ID");
      $tabRes["page_id"]  = $drPage->getValueName("PAGE_ID");
      $tabRes["page_atypeid"]  = $drPage->getValueName("PAGE_ATYPEID");
      $tabRes["page_contid"]   = $drPage->getValueName("PAGE_CONTID");
      $tabRes["page_appliid"]  = $drPage->getValueName("PAGE_APPLIID");
      
      $tabPagesAlias = explode("/", $strUrlPage);
      $strPageAlias     = $tabPagesAlias[count($tabPagesAlias)-1];
      foreach($this->tabLangue as $key => $tabLg) {
        $pageAlias = $drPage->getValueName("PAGE_ALIAS".$tabLg["bdd"]);
        if( $pageAlias == $strPageAlias ) {
          $tabRes["lg"] = $tabLg["rep"];
          break; 
        }
      }
    }
    return $tabRes;
    
  }
  
  
  /**
   * Retourne l'intitule url de la page d'accueil en mode publié
   *
   * @param cont_id identifiant de l'espace
   * @param appli_id identifiant de l'appli
   * @param lg langue de la page (fr par défaut)
   * @return string
   */
  public function getHomeUrl($cont_id, $appli_id=-1, $bWithRootUrl=true, $lg=ALK_LG_BDD, $bMobile=false) 
  {
    $strAlias = "error.php";
    $strSql = "select p.PAGE_ID, p.APPLI_ID, p.PAGE_ALIAS".$lg.
      " from GEDIT_01_PAGE p".
      "   inner join SIT_APPLI_CONT ac on p.APPLI_ID=ac.APPLI_ID".
      " where ac.CONT_ID=".$cont_id."".
      " and p.PAGE_VALIDEPUB=1 and p.PAGE_BROUILLON is null".
      " and p.PAGE_NIVEAU=1 ".
      ( $appli_id != -1
        ? " and p.APPLI_ID=".$appli_id 
        : "" ).
      ( $bMobile == true
        ? " and p.ATYPE_ID=23 and PTYPE_ID=7"
        : "" );
    
    $dsPage = $this->dbConn->initDataSet($strSql);
    if( $drPage = $dsPage->getRowIter() ) {
      $strAlias = $drPage->getValueName("PAGE_ALIAS".$lg);
    }
    return ($bWithRootUrl ? ALK_ALKANET_ROOT_URL : "").$strAlias;
  }
  
  /**
   * Retourne l'url correspondant à la page editoriale identifiée par page_id
   * si page_id n'est pas numérique, page_id correspond à page_url évitant ainsi un requête sgbd
   * Si page_id correspond à une page brouillon, retourne toute
   * L'url ne contient pas de / en début de chaine.
   * 
   * Ne tient pas compte du champ texte de l'URL de redirection (page_redirection).
   * Gère désormais les liens successifs entre plusieurs pages avec le module Link installé.
   * En cas de boucle, l'URL de la page retournée est celle rencontrée avant détection de la boucle.
   * 
   * @param page_id       identifiant de la page courante ou url de la page
   * @param strParam      liste de paramètres complémentaires à cont_id, appli_id, page_id, lg_id
   * @param bWithRootUrl  booléen à true pour ajouter le nom de domaine, false sinon
   * @param lg            langue utilisée, =fr par défaut
   * @param cat_id        identifiant de la catégorie ou dossier, =-1 par défaut
   * @param data_id       identifiant de la donnée, =-1 par défaut
   * @param iMode         mode d'accès à la donnée, =-1 par défaut
   * @param cat_name      intitulée la catégorie, ="" par défaut
   * @param data_name     intitulée la donnée, ="" par défaut
   * @param tabPageId     tableau contenant les identifiants des pages rencontrées lors des appels récurssifs
   * @return string
   */
  public function getPageUrl($page_id, $strParam, $bWithRootUrl=true, $lg=ALK_LG_BDD, $cat_id="-1", $data_id="-1", $iMode="-1", $cat_name="", $data_name="", $tabPageId=array())
  {
    $strUrl = "";
    
    // gestion des liens
    if ( is_int($page_id) || ctype_digit($page_id) ) {
      $oAppliLink = AlkFactory::getAppli(ALK_ATYPE_ID_LINK);
      if ( $oAppliLink ) {
        $oDsLinks = $oAppliLink->getDsLinks(ALK_ATYPE_ID_GEDIT, ALK_LTYPE_ID_PAGE, $page_id, -1, -1);
        if ( $oDrLinks = $oDsLinks->getRowIter() ) {
          if ( !in_array($page_id, $tabPageId) ) {
            array_push($tabPageId, $page_id);
            return $this->getPageUrl($oDrLinks->getValueName("DATA_TO_ID"), $strParam, $bWithRootUrl, $lg, $cat_id, $data_id, $iMode, $cat_name, $data_name, $tabPageId);
          } else {
            $page_id = array_pop($tabPageId);
          }
        }
      }
    }
    
    // pas nécessaire de filtrer sur page_visible et page_brouillon. le filtre est automatique en fonction de page_id
    if ( is_int($page_id) || ctype_digit($page_id) ) {
      $strSql = "select p.PAGE_ID, p.PAGE_ALIAS".$lg." as PAGE_ALIAS".
        " from GEDIT_01_PAGE p, GEDIT_01_PAGE pa".
        " where pa.PAGE_ID=".$page_id.
        "   and pa.PAGE_ARBRE like ".$this->dbConn->getConcat("'%-'", "p.PAGE_ID", "'-%'").
        "   and p.PAGE_ID>0".
        " order by p.PAGE_NIVEAU"; 
      $dsPage = $this->dbConn->initDataSet($strSql);
      while ( $drPage = $dsPage->getRowIter() ) {
        $strAlias = $drPage->getValueName("PAGE_ALIAS");
        if ( $strAlias != "" )
          $strUrl .= "/".$strAlias;
      }
    } else {
      $strUrl = $page_id;
      $page_id = "-1";
    }    
    if ($cat_id!="-1" || $data_id!="-1" || $iMode!="-1"){
      $strUrl .= "/".($cat_id!="-1" ? $cat_id : "")."_".
          ($data_id!="-1" ? $data_id : "").
          ($iMode!="-1" ? "_".$iMode : "").
          ($cat_name!="" ? "/".verifyPageName($cat_name) : "").
          ($data_name!="" ? "/".verifyPageName($data_name) : "");
    }
    
    if( $strUrl != "" ) {
      $strUrl = mb_substr($strUrl, 1).( $strParam != "" ? "?".$strParam : "" );
    }
    return ($bWithRootUrl 
             ? (defined("ALK_ALKANET_ROOT_URL_FRONTOFFICE")
                ? ALK_ALKANET_ROOT_URL_FRONTOFFICE
                : ALK_ALKANET_ROOT_URL
               )
             : "").$strUrl;
  }
  
  /**
   * retourne le dataset des contenus des pages (editoriaux, rss)
   * @param appli_id identifiant de l'appli gedit
   * @return AlkDs
   */
  public function getLinksPages()
  {
    $strIntituleChamp1 = "";
    $strIntituleChamp2 = "";
    foreach($this->tabLangue as $key => $tabLg) {
      $strIntituleChamp1 .= ", CONTENU".$tabLg["bdd"];
      $strIntituleChamp2 .= ", rss_url as CONTENU".$tabLg["bdd"];          
    }
    
    $strSql = " select * from " .
      " (select page.page_id, bloc.bloc_id, page.appli_id, ap.cont_id".$strIntituleChamp1.", tb_id".
      " from GEDIT_01_PAGE page ".
      " inner join GEDIT_01_BLOC bloc on (bloc.page_id = page.page_id)".
      " inner join SIT_APPLI_CONT ap on (ap.appli_id = page.appli_id)".
      " left join GEDIT_01_CONTENU contenu on (contenu.bloc_id = bloc.bloc_id)".
      " where page_brouillon is null".
      " union (".
      " select page.page_id, bloc.bloc_id, page.appli_id, ap.cont_id".$strIntituleChamp1.", tb_id".
      " from GEDIT_01_PAGE page ".
      " inner join GEDIT_01_BLOC bloc on (bloc.page_id = page.page_id)".
      " inner join SIT_APPLI_CONT ap on (ap.appli_id = page.appli_id)".
      " left join GEDIT_01_BLOC_RSS rss on (rss.bloc_id = bloc.bloc_id)".
      " where page_brouillon is null)) link".
      " order by link.bloc_id";
              
    return $this->dbConn->initDataSet($strSql);
  }
  
  /**
   * Retourne l'identifiant de l'espace et de l'appli de la page_id
   * si la page n'est pas trouvée, cont_id et appli_id ne sont pas modifiée
   * cont_id et appli_id sont passés en référence.
   * @param page_id   identifiant de la page éditoriale
   * @param cont_id   identifiant de l'espace
   * @param appli_id  identifiant de l'application gedit
   */
  public function getIdContAppliByPage($page_id, &$cont_id, &$appli_id)
  {
    $strSql = "select ac.CONT_ID, ac.APPLI_ID".
      " from GEDIT_01_PAGE p".
      "   inner join SIT_APPLI_CONT ac on ac.APPLI_ID=p.APPLI_ID".
      " where p.PAGE_ID=".$page_id; 
    $dsRes = $this->dbConn->initDataSet($strSql);
    if( $drRes = $dsRes->getRowIter() ) {
      $cont_id  = $drRes->getValueName("CONT_ID");
      $appli_id = $drRes->getValueName("APPLI_ID"); 
    } 
  }
  
  /**
   * retourne la requête de recherche des abonnements pour un agent
   * @param user_id
   * @return string
   */
  public function getListAbonnements($user_id){    
    $strSql = "select p.PAGE_ID as ID, p.PAGE_TITRE".ALK_LG_BDD." as NOM, p.PAGE_PARENT as ID_PERE,sc.CONT_ID, p.APPLI_ID, ".
      " p.PAGE_NIVEAU as NIV, p.PAGE_RANG as RANG, p.PAGE_ARBRE as ARBRE, l.ABONNEMENT, l.LISTE_ID".
      " from GEDIT_01_PAGE p".
      "   inner join V_GEDIT_USER_RIGHT_PAGE r on p.PAGE_ID=r.PAGE_ID".
      "   inner join GEDIT_01_PAGE_LISTE l on (p.PAGE_ID=l.PAGE_ID )" .
      "   inner join SIT_APPLI_CONT sc on (sc.APPLI_ID = p.APPLI_ID)".
      " where r.USER_ID=".$user_id." and r.RIGHT_ID>=1".
      " and l.ABONNEMENT=1 or ".
                                " p.PAGE_ID in (select PAGE_PARENT from GEDIT_01_PAGE p2 ".
                                                " inner join GEDIT_01_PAGE_LISTE l2 on (p2.PAGE_ID=l2.PAGE_ID and l2.ABONNEMENT=1)".
                                                " inner join V_GEDIT_USER_RIGHT_PAGE r2 on (p2.PAGE_ID=r2.PAGE_ID and r.RIGHT_ID>=1))".
      " and p.PAGE_VALIDEPUB=1 and p.PAGE_BROUILLON is null".
      " order by NIV, ID_PERE, RANG";
      
    return $strSql;
                   
  }
  
  /**
   * Retourne l'url de la page du glossaire en fonction du cont_id courant
  */
  public function getDsUrlPageGlossaire(){
     $strSql = "select page_id ".
    " from GEDIT_01_PAGE WHERE atype_id=24 and page_brouillon IS NULL";
    return $this->dbConn->initDataset($strSql);
  }
  
  /**
   * Retourne la ou les pages correspondant à l'appli_id, et cat_id == -1 si elle existe en fonction du cont_id courant
   * @param int $cont_id        : Identifiant de l'espace ou l'on doit chercher l'application de l'arborescence
   * @param int $appli_id_iedit : Identifiant de l'appli Article que l'on souhaite lier au lien
   * @param int $appli_id_gedit : Identifiant de l'appli d'Arborescence ou l'on veut récupérer la page
   * @return AlkDs contenant toutes les pages correspondant à la recherche
   */
  public function getDsPageByAppliID($cont_id, $appli_id_iedit, $appli_id_gedit="-1"){
     $strSql = " select p.PAGE_ID , p.PAGE_CATID ".
               " from GEDIT_01_PAGE p ".
               " inner join SIT_APPLI a on ( a.APPLI_ID = p.APPLI_ID ) ".
               " inner join SIT_APPLI_CONT c on ( c.APPLI_ID = a.APPLI_ID )".
               " where c.CONT_ID = ".$cont_id." ".
               " and p.PAGE_APPLIID = ".$appli_id_iedit." ".
               ( $appli_id_gedit != "-1" ? " and a.APPLI_ID = ".$appli_id_gedit." " : "").
               " and p.PAGE_BROUILLON is null ";
    return $this->dbConn->initDataset($strSql);
  }
  /**
   * retourne le rang max+1 pour initialiser le formulaire de création d'une page fille à partir de l'id pere
   * @param id_pere
   * @return AlkDs
   */  
  public function getNextRang($id_pere)
  {
    
    $strSql = "select  p.PAGE_RANG ".
      " from GEDIT_01_PAGE p WHERE p.page_parent  =".$id_pere;
    
    $oDsDataPage = $this->dbConn->initDataSet($strSql);   
    $rangMax = 0;
    while($oDrDataPage = $oDsDataPage->getRowIter()){ 

          $rangPageCourante = $oDrDataPage->getValueName("page_rang");
          if($rangPageCourante > $rangMax){
            $rangMax = $rangPageCourante;
          }
    }
    return $rangMax+1;
  }
  
  /**
   * retourne le rang de la page dont on a passer l'id
   * @param id_page
   * @return le rang
   */  
  public function getRangById($id_page)
  {
    
    $strSql = "select  p.PAGE_RANG ".
      " from GEDIT_01_PAGE p WHERE p.page_id  =".$id_page;
    return $this->dbConn->getScalarSql($strSql,1);
  }

  /**
   * Permet de trouver le nombre de page fille d'une page
   * @param int $page_id Id de la page pour laquelle on recherche le nombre de pages filles de premier niveau
   * @return int Retourne le nombre de pages filles trouvées
   */
  public function getNbPageFilleByPageId($page_id)
  {
    $strSql = "SELECT DISTINCT COUNT(PAGE_ID) FROM GEDIT_01_PAGE WHERE PAGE_PARENT= ".$page_id;
    return $this->dbConn->getScalarSql($strSql,0);
  }
  
}  
?>